/*
 * inotify.c - frontend for really simple inotify example
 *
 * Robert Love	<rml@novell.com>
 */

#include <config.h>

#include <glib.h>

#include "inotify-lib.h"

/*
 * my_cb - our callback.  Events are relative to the watch, which is given by
 * "wd".  This simple example ignores it, for the most part, but realworld uses
 * probably want to map the watch descriptor back to the object they are
 * watching.  "event" describes the event (see inotify.h) and "cookie" is used
 * to synchronize two separate IN_MOVED_{TO,FROM} events (not done here).
 */
static gboolean
my_cb (const char *name, int wd, unsigned int event, unsigned int cookie)
{
	switch (event) {
	case IN_ACCESS:
		g_print ("Event on wd=%d: %s was read\n", wd, name);
		break;
	case IN_MODIFY:
		g_print ("Event on wd=%d: %s was written\n", wd, name);
		break;
	case IN_ATTRIB:
		g_print ("Event on wd=%d: %s's inode was modified\n",
			 wd, name);
		break;
	case IN_CLOSE_WRITE:
		g_print ("Event on wd=%d: %s was closed (was writable)\n",
			 wd, name);
		break;
	case IN_CLOSE_NOWRITE:
		g_print ("Event on wd=%d: %s was closed (was not writable)\n",
			 wd, name);
		break;
	case IN_OPEN:
		g_print ("Event on wd=%d: %s was opened\n", wd, name);
		break;
	case IN_MOVED_FROM:
		g_print ("Event on wd=%d: %s was moved away\n", wd, name);
		break;
	case IN_MOVED_TO:
		g_print ("Event on wd=%d: %s was moved here\n", wd, name);
		break;
	case IN_DELETE_SUBDIR:
		g_print ("Event on wd=%d: %s, a directory, was deleted\n",
			 wd, name);
		break;
	case IN_DELETE_FILE:
		g_print ("Event on wd=%d: %s, a file, was deleted\n",
			 wd, name);
		break;
	case IN_CREATE_SUBDIR:
		g_print ("Event on wd=%d: %s, a directory, was created\n",
			 wd, name);
		break;
	case IN_CREATE_FILE:
		g_print ("Event on wd=%d: %s, a file, was created\n",
			 wd, name);
		break;
	case IN_DELETE_SELF:
		g_print ("Event on wd=%d: The watch was deleted!", wd);
		break;
	case IN_UNMOUNT:
		g_print ("Event on wd=%d: %s was unmounted\n", wd, name);
		break;
	case IN_Q_OVERFLOW:
		g_print ("The queue overflowed!");
		break;
	case IN_IGNORED:
		g_print ("Event on wd=%d: %s is no longer watched\n", wd, name);
		break;
	}

	return TRUE;
}

int
main (int argc, char *argv[])
{
	GMainLoop *mainloop;
	GIOChannel *gio;
	int i;

	if (argc < 2) {
		g_warning ("Usage: %s [directories or files ...]\n", argv[0]);
		return -1;
	}

	gio = inotify_open ();
	if (!gio)
		return -1;

	for (i = 1; i < argc; i++) {
		int wd;

		g_print ("Adding watch on %s\n", argv[i]);
		wd = inotify_add_watch (gio, argv[i]);
	}

	inotify_callback (gio, my_cb);

	mainloop = g_main_new (FALSE);
	g_main_run (mainloop);

	inotify_close (gio);

	return 0;
}
