/****************************
  Reception for ICQ v7-8 protocol (Oscar)
  Olivier Crete (c) 2001-2002
  GnomeICU
*****************************/

#include "common.h"
#include "gnomecfg.h"
#include "gnomeicu.h"
#include "gnomeicu-spinner.h"
#include "groups.h"
#include "gtkfunc.h"
#include "personal_info.h"
#include "changeinfo.h"
#include "response.h"
#include "search.h"
#include "sendcontact.h"
#include "showlist.h"
#include "util.h"
#include "v7recv.h"
#include "v7send.h"
#include "v7snac13.h"
#include "events.h"

#include <libgnomeui/libgnomeui.h>
#include <string.h>
#include <stdlib.h>

void     v7_connected (GTcpSocket *socket, GInetAddr *ia,
                   GTcpSocketConnectAsyncStatus status, V7Connection *conn);
gboolean v7_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn);
gboolean v7_error_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn);

/*  Handlers for the different type of packets */
void   v7_snac_channel_handler      (V7Packet *packet, V7Connection *conn);
void   v7_newconn_channel_handler   (V7Packet *packet, V7Connection *conn);

void   v7_make_requests             (V7Connection *conn);
void   v7_ack_ack_rate              (V7Connection *conn);
void   v7_receive_msg               (Snac *snac, V7Connection *conn);
void   v7_text_message              (UIN_T uin, gchar *here);
void   v7_special_message           (UIN_T uin, gchar *here);
void   v7_advanced_message          (UIN_T uin, V7Connection *conn,
				     gchar *here, int len, gchar *begofmes);
void   v7_advanced_message_ack      (UIN_T uin, V7Connection *conn, 
				     guchar *data, gchar *begofmes);
void   v7_incoming_user             (Snac *snac);
void   v7_outgoing_user             (Snac *snac);
void   v7_acked_adv_msg             (Snac *snac, V7Connection *conn);

void   v7_rec_contactlist           (UIN_T uin, gchar *msg, int msglen,
                                     time_t msg_time);

void   v7_server_status             (Snac *snac, V7Connection *conn);
void   v7_rate_changed              (Snac *snac, V7Connection *conn);
void   v7_saved_data                (V7Connection *conn, Snac *snac);
void   v7_offline_message           (DWORD reqid, gchar *data);
void   v7_saved_data_extension      (DWORD reqid, gchar *data, gchar *end);
gchar *v7_search_reply              (DWORD reqid, BYTE result, gchar *here);
void   v7_end_of_search_reply       (DWORD reqid, BYTE result, gchar *here,
                                     gchar *buffer_end);

void   v7_parse_main_home_info      (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_more_info           (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_about_info          (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_work_info           (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_emails_info         (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_homepage_category_info(Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void   v7_parse_interests_info      (Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);
void v7_parse_past_and_affiliation_info(Request *requestdat, USER_INFO_PTR info, 
                                     BYTE result, gchar *here);


V7Connection *v7_connect(gchar *address, guint port, gchar *cookie, guint cookielen)
{
  V7Connection *conn;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_connect\n" );
#endif

  conn = g_new0(V7Connection, 1);

  conn->cookie = g_memdup(cookie, cookielen);
  conn->cookielen = cookielen;

  conn->gsocketasyncid = gnet_tcp_socket_connect_async( address, port,
                             (GTcpSocketConnectAsyncFunc) v7_connected, conn);
  conn->last_reqid = 1;
  conn->status = NOT_CONNECTED;

  return conn;
}



void v7_connected (GTcpSocket *socket, GInetAddr *ia,
                   GTcpSocketConnectAsyncStatus status, V7Connection *conn)

{

#ifdef TRACE_FUNCTION
  g_print( "v7_connected\n" );
#endif
  
  if (status != GTCP_SOCKET_CONNECT_ASYNC_STATUS_OK) {

    g_free(conn);
    Current_Status = STATUS_OFFLINE;
    ready_set(FALSE);
    enable_online_events = FALSE;
    gnomeicu_spinner_stop();

    return;
  }
  
  
  conn->gsocket = socket;
  
  conn->status = CONNECTING;
  
  conn->iochannel = gnet_tcp_socket_get_iochannel (socket);
  
  conn->in_watch = g_io_add_watch ( conn->iochannel, G_IO_IN|G_IO_PRI,
                                    (GIOFunc) v7_handler, conn);
  conn->err_watch = g_io_add_watch ( conn->iochannel,
                                     G_IO_ERR|G_IO_HUP|G_IO_NVAL,
                                     (GIOFunc) v7_error_handler, conn);
  
}

gboolean v7_error_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn)
{
#ifdef TRACE_FUNCTION
  g_print( "v7_error_handler\n" );
#endif

  icq_set_status_offline( NULL, NULL);
  return FALSE; /* not sure what to return -- menesis */
}

gboolean v7_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn)
{
  V7Packet *packet;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_handler\n" );
#endif
  
  g_assert(conn != NULL);
  
  /* If we dont have the whole packet stop here */
  if ((packet = read_flap(conn)) == NULL)
    return TRUE;
  
  switch (packet->channel) {
    
  case CHANNEL_NEWCONN:
    v7_newconn_channel_handler (packet, conn) ;
    break;
  case CHANNEL_SNAC:
    v7_snac_channel_handler( packet, conn);
    break;
  case CHANNEL_ERROR:
    
    break;
  case CHANNEL_CLOSE:
    gnome_error_dialog(_("Server forced disconnect."));
/*    v7_quit();*/
      icq_set_status_offline (NULL, NULL);
    break;
  case CHANNEL_KEEPALIVE:
    
    break;
    
  default:
    g_warning(_("New unknown channel: %d"), packet->channel);
    break;
  }

  g_free(packet->data);
  g_free(packet);
  
  return TRUE;
}



void v7_snac_channel_handler(V7Packet *packet, V7Connection *conn) 
{
  Snac *snac;

#ifdef TRACE_FUNCTION
  g_print( "v7_snac_channel_handler\n" );
#endif

  snac = read_snac(packet);

  switch (snac->family) {
  case FAMILY_01_GENERIC:
    switch (snac->type) {
    case F01_SERVER_READY:
      v7_iam_icq(conn);
      break;
    case F01_SERVER_ACK_RATE:
      v7_ack_ack_rate(conn);
      v7_make_requests(conn);
      break;
    case F01_SERVER_SENDING_TOO_FAST:
      v7_rate_changed(snac, conn);
      break;
    case F01_SERVER_PERSONAL_INFO:
      v7_server_status(snac, conn);
      break;
    case F01_SERVER_MOTD:
      v7_request_rate (conn);
      break;
    case F01_SERVER_ACK_I_AM_ICQ:
      /* We dont understand the data, ignore it */
      break;
    default:
      break;
    }
    break;
  case FAMILY_02_LOCATION:
    switch (snac->type) {
    case F02_SERVER_RIGHTS_LOCATION: 
      /* We dont understand the data, ignore it */
      break;
    default:
      break;
    }
    break;
  case FAMILY_03_CONTACTLIST:
    switch (snac->type) {
    case F03_SERVER_CL_INFO:
      /* We dont understand the data, ignore it */
      break;
    case F03_SERVER_INCOMING_USER:
      v7_incoming_user(snac);
      break;
    case F03_SERVER_OUTGOING_USER:
      v7_outgoing_user(snac);
      break;
    default:
      break;
    }
    break;
  case FAMILY_04_MESSAGING:
    switch (snac->type) {
    case F04_SERVER_RIGHTS_MSG:
      /* We dont understand the data, ignore it */
      break;
    case F04_SERVER_RECV_MESSAGE:
      v7_receive_msg(snac, conn);
      break;
    case F04_BOTH_ACK_ADV_MSG:
      v7_acked_adv_msg(snac, conn);
      break;
    default:
      break;
    }
    break;
  case FAMILY_09_LISTS:
    switch (snac->type) {
    case F09_SERVER_LIST_RIGHTS:
      /* We dont understand the data, ignore it */
      break;
    default:
      break;
    }
    break;
  case FAMILY_0B_UNKNOWN_1:
    switch (snac->type) {
    default:
      break;
    }
    break;
  case FAMILY_13_SAVED_LIST:
    switch (snac->type) {
    case F13_SERVER_REQ_RIGHTS_ANS: /* 0x03 */
      /* reply to F13_SERVER_UNKNOWN_1 */
      break;
    case F13_SERVER_SAVED_LIST: /* 0x06 */
      v7_read_contacts_list (snac, conn);
      break;
    case F13_SERVER_ACK: /* 0x0E */
      v7_snac13_check_srv_ack (conn, snac);
      break;
    case F13_SERVER_LIST_NO_CHANGE: /* 0x0F */
      /* we get this because the value sent didn't change */
      /* send 13,07 */
      snac_send (conn, NULL, 0, FAMILY_13_SAVED_LIST,
		 F13_CLIENT_RDY_TO_USE_LIST, NULL, ANY_ID);
      v7_set_user_info (conn);
      v7_add_icbm (conn);
      v7_setstatus (conn, preset_status);
      v7_client_ready (conn);
      break;
    case F13_SERVER_AUTH_REQUEST: /* 0x19 */
      v7_recv_auth_request (snac);
      break;
    case F13_SERVER_USER_GRANT_AUTH: /* 0x1B */
      v7_user_grant_auth (snac);
      break;
    case F13_SERVER_USER_ADDED_YOU: /* 0x1C */
      v7_recv_added_you (snac);
      break;
    default:
      break;
    }
    break;
  case FAMILY_15_SAVED_DATA:
    switch (snac->type) {
    case F15_SERVER_SAVED_DATA:
      v7_saved_data(conn, snac);
    default:
      break;
    }
    break;
  case FAMILY_17_NEW_USER:
    switch (snac->type) {
    default:
      break;
   }
    break;
  default:
    break;
    
  }

  g_free (snac->data);
  g_free(snac);
  
}



void v7_newconn_channel_handler (V7Packet *packet, V7Connection *conn) 
{
  TLVstack *cookietlv;

#ifdef TRACE_FUNCTION
  g_print( "v7_newconn_channel_handler\n" );
#endif

  if ( packet->len == 4  &&
       packet->data[0] == packet->data[1] == packet->data[2] == 0 &&
       packet->data[3] == 1) {
    
    
    cookietlv  = new_tlvstack ("\0\0\0\x01", 4);
    
    add_tlv(cookietlv, 6, conn->cookie, conn->cookielen);
      
    if (! flap_send(conn, CHANNEL_NEWCONN, cookietlv->beg, cookietlv->len))
      conn->status = BROKEN;

    g_free(conn->cookie);
    conn->cookie = NULL;
    conn->cookielen = 0;

    free_tlvstack(cookietlv);

    
  } else
    g_warning (_("Unknown packet on login on main server\n"));
}



void v7_ack_ack_rate    (V7Connection *conn)
{
  BYTE v7_ack_ack_rate_array[10] = "\x00\x01\x00\x02\x00\x03\x00\x04\x00\x05";

#ifdef TRACE_FUNCTION
  g_print( "v7_ack_ack_rate\n" );
#endif

  snac_send(conn, v7_ack_ack_rate_array, 10, FAMILY_01_GENERIC,
            F01_CLIENT_ACK_ACK_RATE, NULL, ANY_ID);
}

void v7_receive_msg (Snac *snac, V7Connection *conn)
{
  gchar *here = snac->data + 8;
  WORD msgformat;
  BYTE uinlen;
  UIN_T uin;
  int len = snac->datalen;
  TLV *blah;
  int tlvcount,i;

#ifdef TRACE_FUNCTION
  g_print( "v7_receive_msg\n" );
#endif

  msgformat = CharsBE_2_Word(here);

  here += 2;
  len -= 2;
  
  uinlen = here[0];
  here += 1;
  len -=1;

  uin = g_malloc0(uinlen+1);
  g_memmove(uin, here, uinlen);

  here += uinlen;
  len -= uinlen;

  here += 2;
  len -= 2;
  
  tlvcount = CharsBE_2_Word(here);
  here += 2;
  len -=2;

  for(i=0; i<tlvcount; i++) {
    blah = new_tlv(here); 
    here += blah->len + 4; 
    len -=  (blah->len + 4);
    delete_tlv(blah); 
  }

  switch(msgformat) {
  case 0x01:
    v7_text_message(uin, here);
    break;
  case 0x02:
    v7_advanced_message(uin, conn, here, len, snac->data);
    break;
  case 0x04:
    v7_special_message(uin, here);
    break;
  default:
    g_warning("unknown message format: %x\n", msgformat);
    break;
  }
  
  g_free(uin);
}

void v7_text_message(UIN_T uin, gchar *here)
{
  
  WORD msglen;
  gchar *msg;
  TLV *msgtlv;
  time_t msg_time;

#ifdef TRACE_FUNCTION
  g_print( "v7_text_message\n" );
#endif
  
  msgtlv = new_tlv(here); 

  if (msgtlv->type == 4 && msgtlv->len == 0) {
    TLV *temptlv;
    temptlv = new_tlv(here + 4 + msgtlv->len);
    delete_tlv(msgtlv);
    msgtlv = temptlv;
  }
  
  if (msgtlv->type != 2) {
    g_warning(_("TLV is %d, expecting 2\n"), msgtlv->type);
    return;
  }
    

  /* I dont know why 3, but it does work */
  msglen = CharsBE_2_Word(msgtlv->value+ 6 + msgtlv->value[3]); 
  msglen -= 4;                /* to skip the 4 zeros */
  
  if (msglen + 13 > msgtlv->len) { 
    g_warning(_("MSGLEN TOO LONG: %d > %d\n"), msglen+12+msgtlv->value[3],
              msgtlv->len);
    msglen = msgtlv->len -12 -msgtlv->value[3];
  }
  
  msg = g_malloc0(msglen +1);
  
  g_memmove(msg, msgtlv->value + 12 + msgtlv->value[3], msglen);
  
  delete_tlv(msgtlv); 
  
  time(&msg_time);
  
  msg_received(uin, msg, msg_time);
  
  g_free(msg);

}

void v7_special_message(UIN_T uin, gchar *here)
{

  WORD msglen;
  gchar *msg, *url, *nick, *first, *last, *email, *tmp=NULL, *conv_msg;
  TLV *msgtlv;
  BYTE msgtype, msgflags;
  time_t msg_time;

#ifdef TRACE_FUNCTION
  g_print( "v7_special_message\n" );
#endif
  
  msgtlv = new_tlv(here); 
  
  msgtype = msgtlv->value[4];
  msgflags = msgtlv->value[5];

  msglen = Chars_2_Word(msgtlv->value+6);

  msg = g_malloc0(msglen +1);

  g_memmove(msg, msgtlv->value+8, msglen);

  delete_tlv(msgtlv); 

  time(&msg_time);
  
  switch(msgtype) {
  case 0x01:
    msg_received(uin, msg, msg_time);
    break;
  case 0x04:
    url = strchr( msg, '\xFE' ) + 1;
    *(url-1) = '\0';
    conv_msg = convert_to_utf8(msg);
    url_received (uin, url, conv_msg, msg_time);
    g_free(conv_msg);
    break;
  case 0x13:
    v7_rec_contactlist(uin, msg, msglen, msg_time);
    break;
  case 0x0C:
    user_added_you(uin, msg_time);
    break;
  case 0x06:
    nick = msg;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    url = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp ++;

    authorization_request(uin, nick, first, last, email, url, msg_time);
    break;
  default:
    g_warning(_("Special message type not handled, type: %x, text: %s\n"), msgtype, msg);
    break;
  }

  g_free(msg);
}

void v7_advanced_message(UIN_T uin, V7Connection *conn, gchar *data, int len, 
			 gchar *begofmes)
{

  TLV *maintlv, *insidetlv, *tlv;
  WORD type1, msglen, fileport;
  BYTE msgtype;
  gchar *msg=NULL, *here, *url, *nick, *first, *last, *email, *tmp=NULL, *file,
    *msgid, *conv_msg;
  DWORD fileip=0, filesize;
  time_t msg_time;
  gboolean isfileokack=FALSE;
  gboolean isfile = FALSE;

#ifdef TRACE_FUNCTION
  g_print( "v7_advanced_message\n" );
#endif


  maintlv = new_tlv(data);

  if (maintlv->type != 5) {
    g_warning("Advanced msg has a tlv of a bizzare type: %d\n", maintlv->type);
    print_data(maintlv->value, maintlv->len);
    delete_tlv(maintlv);
    return;
  } 

  type1 = CharsBE_2_Word (maintlv->value);

  if (type1 == 1) {
    g_warning("We have an abort request, we dont handle that right now!\n");
    delete_tlv(maintlv);
    return;
  } 
  
  here = maintlv->value+26;
  
  insidetlv = new_tlv(here);

  while (insidetlv->type != 0x2711) {
    switch (insidetlv->type) {
    case 0x000A:  /* file stuff */
      if (CharsBE_2_Word(insidetlv->value) == 2) {
	isfile = TRUE;
	isfileokack = TRUE;
      }
      else
	isfileokack = FALSE;
      break;
    case 0x0005: /* port */
      isfile = TRUE;
      fileport = Chars_2_Word(insidetlv->value);
      break;
    case 0x0003: /* ip */
      isfile = TRUE;
      fileip = IP_2_DW(insidetlv->value);
      break;
    case 0x000F: /* always empty ? */
      break;
    default:
      g_warning("We have a tlv that we dont know of type: %x\n",
                insidetlv->type);
      delete_tlv(insidetlv);
      delete_tlv(maintlv);
      print_data(data, len);
      return;
    }
    here += insidetlv->len+4;
    delete_tlv(insidetlv);
    insidetlv = new_tlv(here);
  }
  
  
  msgtype = insidetlv->value[45];
  
  if (msgtype != 0x1A) {
    msglen = Chars_2_Word(insidetlv->value+51);
    msg = g_malloc0(msglen +1);
    g_memmove(msg, insidetlv->value + 53, msglen);
    time(&msg_time);
  }

  switch(msgtype) {
  case 0x01: /* text message */
    msg_received(uin, msg, msg_time);
    break;
  case 0x02: /* Chat */
  case 0x03: /* File .. are you sure? */ 
    break;
  case 0x1A: /* File */
  case 0x13: /* Contacts */
    
    if (isfile) {
      here = insidetlv->value+54;
      msglen =  Chars_2_Word(here);
      here += 2 + msglen + 4;
      msglen = Chars_2_DW(here);
      here += 4;
      msg = g_malloc0(msglen+1);
      g_memmove(msg, here, msglen);
      here += msglen + 4; 
      msglen = Chars_2_Word(here);
      here += 2;
      file = g_malloc0(msglen);
      g_memmove(file, here, msglen);
      here += msglen;
      filesize = Chars_2_DW(here);
      here += 4;
      fileport = Chars_2_Word(here);
      here += 4;

      tlv = new_tlv(here);
      if (tlv->type == 4 && tlv->len == 4)
	fileip = IP_2_DW(tlv->value);
      delete_tlv(tlv);

      msgid = g_memdup(begofmes, 8);

      /*    g_print("msg: %s\n file:%s\n size:%d ip:%lX port:%d \n", msg,
	    file, filesize, fileip, fileport); */

      if (! isfileokack)
	Do_File( msg, uin, msgid, file, filesize, conn);
      else 
	start_transfer(uin, msgid, fileip, fileport);
      
      g_free(msgid);

    } else {

      g_free(msg);

      msglen = Chars_2_Word(insidetlv->value+105);
    
      msg = g_malloc0(msglen +1);
    
      g_memmove(msg, insidetlv->value + 109, msglen);
    
      v7_rec_contactlist(uin, msg, msglen, msg_time);
    }
    break;
  case 0x04: /* URL */
    url = strchr( msg, '\xFE' ) + 1;
    *(url-1) = '\0';
    conv_msg = convert_to_utf8(msg);
    url_received (uin, url, conv_msg, msg_time);
    g_free(conv_msg);
    break;
  case 0x0C: /* User Added you */
    user_added_you(uin, msg_time);
    break;
  case 0x06:  /* Auth request */
    nick = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;

    url = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp ++;

    authorization_request(uin, nick, first, last, email, url, msg_time);
    break;
  case 0xE8:  /* Request for away message */
  case 0xE9:
  case 0xEA:
  case 0xEB:
  case 0xEC:
    break;
  default:
    g_warning("Advanced message type not handled, type: %x, text: %s\n",
              msgtype, msg);
    delete_tlv(insidetlv);
    delete_tlv(maintlv);
    g_free(msg);
    print_data(data, len);
    return;
  }

  g_free(msg);

  v7_advanced_message_ack(uin, conn, insidetlv->value, begofmes); 
  
  delete_tlv(insidetlv);
  delete_tlv(maintlv);
  
}

void v7_advanced_message_ack (UIN_T uin, V7Connection *conn, guchar *data, 
			      gchar *begofmes) 
{

  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print( "v7_advanced_message_ack\n" );
#endif

  tlvs = new_tlvstack(begofmes,11+begofmes[10]);

  add_nontlv_w_be(tlvs, 0x0003);

  /* I dont know what all of that means, but it works */
  
  add_nontlv(tlvs, data, 26);
  add_nontlv(tlvs, "\0", 1);
  add_nontlv(tlvs, data+27, 20);
  add_nontlv_dw_be(tlvs,0); /* Always send 0 as accept status */

  /* Add away message if away */
  if ( Current_Status == STATUS_AWAY ||
       Current_Status == STATUS_NA ||
       Current_Status == STATUS_OCCUPIED ||
       Current_Status == STATUS_DND) {
    add_nontlv_w_le(tlvs, strlen(Away_Message)+1);
    add_nontlv(tlvs, Away_Message, strlen(Away_Message)+1);
  }
  else
    add_nontlv_w_le(tlvs, 0);  

  if (data[45] < 0xE0) {  /* add this if not autoawaymessage reply */
    add_nontlv_dw_be(tlvs, 0);
    add_nontlv_dw_be(tlvs, 0xFFFFFF00);
  }
  
  snac_send(conn, tlvs->beg, tlvs->len, FAMILY_04_MESSAGING,
            F04_BOTH_ACK_ADV_MSG, NULL, ANY_ID);

  free_tlvstack(tlvs);
}

void v7_incoming_user (Snac *snac)
{

  UIN_T uin = NULL; 
  GSList *contact;
  gchar *here;
  int tlvcount, i, j;
  TLV *tlv;
  gboolean status_has_changed = FALSE;
  WORD tempstatus;

#ifdef TRACE_FUNCTION
  g_print( "v7_incoming_user (%s)\n", snac->data+1 );
#endif
  
  uin = snac->data+1;

  contact = Find_User( uin );
  if( contact == NULL )
    return;
  
  here = snac->data + snac->data[0] + 3;

  tlvcount = CharsBE_2_Word(here);

  here += 2;
  
  for(i=0; i<tlvcount; i++) {
    tlv = new_tlv(here); 
    here += tlv->len + 4; 

    switch (tlv->type) {
    case 0x0C:
      kontakt->has_direct_connect = Chars_2_DW(tlv->value) ? TRUE : FALSE;
      kontakt->port = CharsBE_2_DW(tlv->value+4);
      kontakt->version = CharsBE_2_Word(tlv->value+9);
      kontakt->direct_cookie = CharsBE_2_DW(tlv->value+11);
      if (kontakt->version > TCP_VERSION)
        kontakt->used_version = TCP_VERSION;
      else
        kontakt->used_version  = kontakt->version;
      break;
    case 0x0A:
      kontakt->current_ip = IP_2_DW( tlv->value );
      break;
    case 0x06:
      tempstatus = CharsBE_2_Word(tlv->value+2);
      if (kontakt->status != tempstatus)
      	status_has_changed = TRUE;
      /* getting rid of non-standard statuses here */
      switch (tempstatus) {
      case STATUS_NA_99A: 
	tempstatus = STATUS_NA;
	break;
      case STATUS_OCCUPIED_MAC:
	tempstatus = STATUS_OCCUPIED;
	break;
      case STATUS_DND_MAC:
	tempstatus = STATUS_DND;
	break;
      }
      set_contact_status (contact, tempstatus);
      break;
    case 0x01:
	if ((CharsBE_2_Word(tlv->value+1) & 0xBFFF) == STATUS_AIM_ONLINE)
		set_contact_status (contact, STATUS_ONLINE);
	else if ((CharsBE_2_Word(tlv->value+1) & 0xBFFF) == STATUS_AIM_AWAY)
		set_contact_status (contact, STATUS_AWAY);
	break;
    case 0x0D: /* Capabilites */
      for (j=0 ; j<=tlv->len ; j+=16)
	if (!strncmp(tlv->value+j, "\x97\xB1\x27\x51" "\x24\x3C\x43\x34" "\xAD\x22\xD6\xAB" "\xF7\x3F\x14\x92", 16))
	  kontakt->has_rtf = TRUE;
    }

    delete_tlv(tlv); 
  }

  if (kontakt->status == STATUS_OFFLINE) {
    status_has_changed = TRUE;
    set_contact_status (contact, STATUS_ONLINE);
  }

  kontakt->last_time = time( NULL );

}

void v7_outgoing_user (Snac *snac)
{
  
  UIN_T uin = NULL;
  GSList *contact;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_outgoing_user\n" );
#endif
  
  uin = snac->data+1;
  
  contact = Find_User( uin );
  if( contact == NULL )
    return;
  
  kontakt->has_rtf = FALSE;

  User_Offline(contact);
}


void v7_acked_adv_msg (Snac *snac, V7Connection *conn)
{

  BYTE type;
  gchar *awaymessage;
  UIN_T sender_uin;
  WORD sender_status;

#ifdef TRACE_FUNCTION
  g_print( "v7_acked_adv_msg\n" );
#endif
  
  type = snac->data[11+snac->data[10]+2+26+1+18];

  switch (type) {
  case 0xE8:
    sender_status = STATUS_AWAY;
    break;
  case 0xE9:
    sender_status = STATUS_OCCUPIED;
    break;
  case 0xEA:
    sender_status = STATUS_NA;
    break;
  case 0xEB:
    sender_status = STATUS_DND;
    break;
  case 0xEC:
    sender_status = STATUS_FREE_CHAT;
    break;
  default:
    return;
  }

  sender_uin = snac->data+11;

  awaymessage = g_strndup(snac->data+11+snac->data[10]+2+26+1+20+1+3+2,
                          Chars_2_Word(snac->data+11+snac->data[10]+2+26+1+20+1+3));
  
  recv_awaymsg( sender_uin, sender_status, awaymessage);
}

void v7_saved_data(V7Connection *conn, Snac *snac)
{
  TLV *tlv;
  WORD type, reqid;

#ifdef TRACE_FUNCTION
  g_print( "v7_saved_data\n" );
#endif

  tlv = new_tlv(snac->data);
  
  type = CharsBE_2_Word(tlv->value+6);
  
  reqid = Chars_2_Word(tlv->value+8);

  switch(type) {
  case 0x4100:
    v7_offline_message(snac->reqid, tlv->value+10);
    break;
  case 0x4200: 
    v7_send_often(conn, snac->reqid);
    break; 
  case 0xDA07:
    v7_saved_data_extension(snac->reqid, tlv->value+10, tlv->value+tlv->len);
  }
  
  delete_tlv(tlv);

}

void v7_saved_data_extension (DWORD reqid, gchar *data, gchar *end)
{
  WORD subtype;
  BYTE result; 
  GSList *requestdat=NULL, *contact=Contacts;
  gchar *nick = NULL;
  USER_INFO_PTR info = NULL;

#ifdef TRACE_FUNCTION
  g_print( "v7_saved_data_extension\n" );
#endif
  
  subtype = CharsBE_2_Word(data);

  result = data[2];
  
  data += 3;


  /* We do these only if we have a full info request,
     see it as a type of "if" with 8 ORed conditions */
  switch (subtype) {
  case 0xC800:
	nick = data + 2;
  case 0xDC00:
  case 0xEB00:
  case 0x0E01:
  case 0xD200:
  case 0xE600:
  case 0xF000:
  case 0xFA00:
    
    requestdat = v7_get_request_by_reqid(reqid);

    if (requestdat == NULL)
      return;

    if (((Request*)requestdat->data)->type == OUR_INFO) {
      info = our_info;
      break;
    }

    contact = Find_User(((Request*)(requestdat->data))->uin);
    
    if (contact == NULL )
      return;

    /* Anti spam action here */
    if (((Request*)requestdat->data)->type == NEW_USER_VERIFICATION) {
      if (result != 0x0A) {
        remove_contact (contact, TRUE);
        g_free(((Request*)requestdat->data)->uin);
        g_free(requestdat->data);
        requests = g_slist_remove(requests, requestdat->data);
        return;
      }
      else {
        if (kontakt->confirmed == FALSE) {
          kontakt->confirmed = TRUE;
          kontakt->uid = contact_gen_uid();
          kontakt->gid = 0;
          gnomeicu_event (EV_MSGRECV, contact);
          if (nick != NULL) {
            g_free(kontakt->nick);
            kontakt->nick = g_strdup(nick);
	  }
          gnomeicu_tree_user_add (kontakt);
        }
      }
    }
    else if (((Request*)requestdat->data)->type != FULL_INFO)
      return;
        
    if (result != 0x0A)
      return;
    
    if( kontakt->info == NULL ) {
      if (!strcmp(kontakt->uin, our_info->uin))
	kontakt->info = our_info;
      else
	kontakt->info = g_malloc0( sizeof(USER_INFO_STRUCT));
    }

    g_free (kontakt->info->uin);
    kontakt->info->uin = g_strdup (kontakt->uin);
    info = kontakt->info;

    break;
  default:
    break;
  }

  switch (subtype) {
  case 0x9001:
  case 0xA401:
    v7_search_reply(reqid, result, data);
    break;
  case 0x9A01:
  case 0xAE01:
    v7_end_of_search_reply(reqid, result, data, end);
    break;
  case 0xC800:
    v7_parse_main_home_info(requestdat->data, info, result, data);
    break;
  case 0xDC00:
    v7_parse_more_info(requestdat->data, info, result, data);
    break;
  case 0xD200:
    v7_parse_work_info(requestdat->data, info, result, data);
    break;
  case 0xE600:
    v7_parse_about_info(requestdat->data, info, result, data);
    break;
  case 0xEB00:
    v7_parse_emails_info(requestdat->data, info, result, data);
	 break;
  case 0x0E01:
    v7_parse_homepage_category_info(requestdat->data, info, result, data);
	 break;
  case 0xF000:
    v7_parse_interests_info(requestdat->data, info, result, data);
	 break;
  case 0xFA00:
    v7_parse_past_and_affiliation_info(requestdat->data, info, result, data);

    /* This one seems to come last.. so we delete the requestdat here */
    if (requestdat) {      
      g_free(((Request*)requestdat->data)->uin);
      g_free(requestdat->data);
      requests = g_slist_remove(requests, requestdat->data);
    }
    break;
  }

}

void v7_offline_message(DWORD reqid, gchar *data)
{

  time_t msg_time;
  struct tm build_time;
  UIN_T uin;
  BYTE msgtype, msg_flags;
  WORD msglen;
  gchar *msg = data+14;
  gchar *url, *nick, *first, *last, *email, *tmp=NULL, *conv_msg;

#ifdef TRACE_FUNCTION
  g_print( "v7_offline_message\n" );
#endif
  
  uin = g_strdup_printf ("%d", Chars_2_DW(data));

  build_time.tm_year = Chars_2_Word(data+4) - 1900;
  build_time.tm_mon = data[6]-1;  /* jan = 1 */
  build_time.tm_mday = data[7];
  build_time.tm_hour = data[8];
  build_time.tm_min = data[9];
  build_time.tm_sec = 0;
  msg_time = mktime(&build_time);

  msgtype  = data[10];
  msg_flags = data[11];
  msglen = Chars_2_Word(data+12);

  switch(msgtype) {
  case 0x01:
    msg_received(uin, msg, msg_time);
    break;
  case 0x04:
    url = strchr( msg, '\xFE' ) + 1;
    *(url-1) = '\0';
    conv_msg = convert_to_utf8(msg);
    url_received (uin, url, conv_msg, msg_time);
    g_free(conv_msg);
    break;
  case 0x0C:
    user_added_you(uin, msg_time);
    break;
  case 0x06:
    nick = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    url = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp ++;

    authorization_request(uin, nick, first, last, email, url, msg_time);
    break;
  case 0x13:
    v7_rec_contactlist(uin, msg, msglen, msg_time);
    break;
  default:
    g_warning("Offline message type not handled, type: %x, text: %s\n", msgtype, msg);
    break;
  }

  g_free(uin);
}

/* 
   Returns the end of what has been read (a pointer to what's next)
   for the last packet in row type
*/


gchar *v7_search_reply(DWORD reqid, BYTE result, gchar *here)
{

  WORD len, templen;
  gchar *hereorig = here;
  UIN_T uin = NULL;
  gchar *nick=NULL, *first=NULL, *last=NULL, *email=NULL;
  gchar auth=0, status=0, sex=0, age=0;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_search_reply\n" );
#endif 

  len = Chars_2_Word (here);
  here += 2;

  uin = g_strdup_printf ("%d", Chars_2_DW(here));
  here += 4;

  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  nick = here;
  here += strlen(nick)+1;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  first = here;
  here += strlen(first)+1;;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  last = here;
  here += strlen(last)+1;;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  email = here;
  here += strlen(email)+1;
  
  if (hereorig+len >= here+5) {
    auth   = here[0] ? 0 : 1;
    status = here[1];
    sex    = here[3];
    age    = here[4];
  }
  else {
    auth = sex = age = 0;
    status = 2;
  }

 full:
  
  Display_Search_Reply(reqid, uin, nick, first, last, email, auth, status,
                       sex, age);

  g_free (uin);
  
  return hereorig+len;
}

void v7_end_of_search_reply(DWORD reqid, BYTE result, gchar *here,
                            gchar *buffer_end)
{

#ifdef TRACE_FUNCTION
  g_print( "v7_end_of_search_reply\n" );
#endif 
  if (result != 0x32) {
    here = v7_search_reply(reqid, result, here)+1;

    if (here < buffer_end && CharsBE_2_DW(here))
      search_finished (reqid, TOO_MANY_RESULTS);
    else
      search_finished (reqid, COMPLETE);
  }
  else
    search_finished (reqid, NO_RESULTS);

}


/* Retrieves:
 * - Nickname
 * - First/Last Name
 * - Email
 * - City, State
 * - Phone, Fax
 * - Street Address
 * - Cellular phone
 * - Zip code, Country
 * - Time zone
 */
void v7_parse_main_home_info (Request *requestdat, USER_INFO_PTR info, 
                              BYTE result, gchar *here)
{
  int len_str;
     
#ifdef TRACE_FUNCTION
  g_print( "v7_parse_main_home_info\n" );
#endif 

  g_free(info->nick);
  len_str = Chars_2_Word( here );
  here += 2;
  info->nick = g_strdup( here );
  here += len_str;
  if (requestdat->type == NEW_USER_VERIFICATION) {
    GSList *contact = Contacts;
    
    contact = Find_User( info->uin );
    
    if (contact != NULL ) {
		g_free(kontakt->nick);
		kontakt->nick = g_strdup(info->nick);
	  }
  }

  g_free(info->first);
  len_str = Chars_2_Word( here );
  here += 2;
  info->first = g_strdup( here );
  here += len_str;

  g_free(info->last);
  len_str = Chars_2_Word( here );
  here += 2;
  info->last = g_strdup( here );
  here += len_str;

  g_free(info->email);
  len_str = Chars_2_Word( here );
  here += 2;
  info->email = g_strdup( here );
  here += len_str;

  g_free(info->city);
  len_str = Chars_2_Word( here );
  here += 2;
  info->city = g_strdup( here );
  here += len_str;

  g_free(info->state);
  len_str = Chars_2_Word( here );
  here += 2;
  info->state = g_strdup( here );
  here += len_str;

  g_free(info->phone);
  len_str = Chars_2_Word( here );
  here += 2;
  info->phone = g_strdup( here );
  here += len_str;

  g_free(info->fax);
  len_str = Chars_2_Word( here );
  here += 2;
  info->fax = g_strdup( here );
  here += len_str;

  g_free(info->street);
  len_str = Chars_2_Word( here );
  here += 2;
  info->street = g_strdup( here );
  here += len_str;

  g_free(info->cellular);
  len_str = Chars_2_Word( here );
  here += 2;
  info->cellular = g_strdup( here );
  here += len_str;

  g_free(info->zip);
  len_str = Chars_2_Word( here );
  here += 2;
  info->zip = g_strdup( here );
  here += len_str;

  info->country = Chars_2_Word( here );
  here += 2;
  
  info->timezone = *here;
  here++;

  info->auth = *here ? 0 : 1;

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

/* Retrieves:
 * - Age
 * - Sex
 * - Homepage
 * - Birth year, month, day
 * - Languages (3)
 */
void v7_parse_more_info (Request *requestdat, USER_INFO_PTR info, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_more_info\n" );
#endif 
  
  if( Chars_2_Word( here ) != (WORD) -1 )
    info->age = Chars_2_Word( here );
  else
    info->age = 0;
  here += 2;

  /* 0x01: FEMALE, 0x02: MALE */
  info->sex = *here;
  if ( *here > 2 || *here == 0)
    info->sex = NOT_SPECIFIED;
  here++;

  g_free(info->homepage);
  len_str = Chars_2_Word( here );
  here += 2;
  info->homepage = g_strdup( here );
  here += len_str;

  info->birth_year = Chars_2_Word(here);
  info->birth_month = *(here+2);
  info->birth_day = *(here+3);
  here += 4;
  info->language1 = here[0];
  info->language2 = here[1];
  info->language3 = here[2];
  
  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

/* Retrieves:
 * - About
 */
void v7_parse_about_info (Request *requestdat, USER_INFO_PTR info, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_about_info\n" );
#endif 
  
  g_free(info->about);
  len_str = Chars_2_Word( here );
  here += 2;  info->about = convert_to_utf8( here );


  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

/* Retrieves:
 * - Work city, state
 * - Work phone, fax
 * - Work address, zip code
 * - Company name
 * - Department
 * - Job position
 * - Work homepage
 */
void v7_parse_work_info (Request *requestdat, USER_INFO_PTR info, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_work_info\n" );
#endif 
  
  g_free(info->work_city);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_city = convert_to_utf8( here );
  here += len_str;

  g_free(info->work_state);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_state = convert_to_utf8( here );
  here += len_str;

  g_free(info->work_phone);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_phone = g_strdup( here );
  here += len_str;

  g_free(info->work_fax);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_fax = g_strdup( here );
  here += len_str;

  g_free(info->work_address);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_address = convert_to_utf8( here );
  here += len_str;
  
  g_free(info->work_zip);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_zip = convert_to_utf8( here );
  here += len_str;

  info->work_country = Chars_2_Word( here );
  here += 2; /* Skip unknown data */

  g_free(info->company_name);
  len_str = Chars_2_Word( here );
  here += 2;
  info->company_name = convert_to_utf8( here );
  here += len_str;

  g_free(info->department);
  len_str = Chars_2_Word( here );
  here += 2;
  info->department = convert_to_utf8( here );
  here += len_str;

  g_free(info->job_pos);
  len_str = Chars_2_Word( here );
  here += 2;
  info->job_pos = convert_to_utf8( here );
  here += len_str;

  info->occupation = Chars_2_Word( here );
  here += 2;

  g_free(info->work_homepage);
  len_str = Chars_2_Word( here );
  here += 2;
  info->work_homepage = convert_to_utf8( here );
  here += len_str;

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

    
void v7_parse_emails_info(Request *requestdat, USER_INFO_PTR info, 
                          BYTE result, gchar *here)
{
  int len_str, count, i;
  gchar *email;
  GList *element;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_emails_info\n" );
#endif 
  
  count = here[0];
  here ++;

  /* we free all the data in info->emails */
  element = info->emails;
  while (element) {
	  g_free(element->data);
	  element = element->next;
  }
  g_list_free(info->emails);
  info->emails = NULL;

  for (i=0; i<count; i++) {
    here++; /* Skipping hide, if it was hidden, we wouldnt see it */
    len_str = Chars_2_Word( here );
    here += 2;
    email = g_malloc0(len_str+1);
    email[0] = here[-3]; /* save the hide value for use if it's our info */
    strncpy( email+1, here, len_str );
    here += len_str;
    
    info->emails = g_list_append(info->emails,
                                              email);
  }

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

void v7_parse_homepage_category_info(Request *requestdat, USER_INFO_PTR info, 
                          BYTE result, gchar *here)
{
  int len_str, count, i;
  gchar *category;
  GList *element;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_homepage_category_info\n" );
#endif 

  count = here[0];
  here ++;

  /* we free all the data in info->homepage_cat */
  element = info->homepage_cat;
  while (element) {
	  g_free(element->data);
	  element = element->next;
  }
  g_list_free(info->homepage_cat);
  info->homepage_cat = NULL;

  for (i=0; i<count; i++) {
    here += 2; /* skipping category for the moment */
    len_str = Chars_2_Word( here );
    here += 2;
    category = g_malloc0(len_str+2);
    *(WORD *) category = Chars_2_Word(here-4);
    strncpy( category+2, here, len_str );
    here += len_str;
    
    info->homepage_cat = g_list_append(info->homepage_cat, category);
  }

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

void v7_parse_interests_info(Request *requestdat, USER_INFO_PTR info, 
                          BYTE result, gchar *here)
{
  int len_str, count, i;
  gchar *interest;
  GList *element;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_interests_info\n" );
#endif 

  count = here[0];
  here ++;

  /* we free all the data in info->interests */
  element = info->interests;
  while (element) {
	  g_free(element->data);
	  element = element->next;
  }
  g_list_free(info->interests);
  info->interests = NULL;

  for (i=0; i<count; i++) {
    here += 2; /* skipping category for the moment */
    len_str = Chars_2_Word( here );
    here += 2;
    interest = g_malloc0(len_str+2);
    *(WORD *) interest = Chars_2_Word(here-4);
    strncpy( interest+2, here, len_str );
    here += len_str;
    
    info->interests = g_list_append(info->interests, interest);
  }

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

void v7_parse_past_and_affiliation_info(Request *requestdat, USER_INFO_PTR info, 
                          BYTE result, gchar *here)
{
  int len_str, count, i;
  gchar *buffer;
  GList *element;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_past_and_affiliation_info\n" );
#endif 

  count = here[0];
  here ++;

  /* we free all the data in info->past_background */
  element = info->past_background;
  while (element) {
	  g_free(element->data);
	  element = element->next;
  }
  g_list_free(info->past_background);
  info->past_background = NULL;

  for (i=0; i<count; i++) {
    here += 2; /* skipping category for the moment */
    len_str = Chars_2_Word( here );
    here += 2;
    buffer = g_malloc0(len_str+2);
    *(WORD *) buffer = Chars_2_Word(here-4);
    strncpy( buffer+2, here, len_str );
    here += len_str;
    
    info->past_background = g_list_append(info->past_background, buffer);
  }

  count = here[0];
  here ++;

  /* we free all the data in info->affiliations */
  element = info->affiliations;
  while (element) {
	  g_free(element->data);
	  element = element->next;
  }
  g_list_free(info->affiliations);
  info->affiliations = NULL;

  for (i=0; i<count; i++) {
    here += 2; /* skipping category for the moment */
    len_str = Chars_2_Word( here );
    here += 2;
    buffer = g_malloc0(len_str+2);
    *(WORD *) buffer = Chars_2_Word(here-4);
    strncpy( buffer+2, here, len_str );
    here += len_str;
    
    info->affiliations = g_list_append(info->affiliations, buffer);
  }

  if (requestdat->type == OUR_INFO) {
    update_our_info();
  } else
    update_personal_info(info->uin);
}

void v7_rec_contactlist(UIN_T uin, gchar *msg, int msglen, time_t msg_time)
{
  gchar *countstr, *tmp;
  GSList *contacts=NULL;
  ContactPair *cpair;
  int i, count;

#ifdef TRACE_FUNCTION
  g_print( "v7_rec_contactlist\n" );
#endif 

    countstr = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    count = atoi(countstr);

    if (count == 0)
      return;

    for (i=0; i < count; i++) {
      cpair = g_new0(ContactPair, 1);
      
      cpair->textuin = tmp;
      tmp = strchr( tmp, '\xFE' );
      if ( tmp == NULL )
        return; /* Bad Packet */
      *tmp = '\0';
      cpair->textuin = g_strdup(cpair->textuin);
      tmp++;
      
      cpair->nick = tmp;
      tmp = strchr( tmp, '\xFE' );
      if ( tmp == NULL && i != count-1) 
        return; /* Bad Packet */
      if (tmp)
        *tmp = '\0';
      cpair->nick = g_strdup(cpair->nick);
      tmp++;

      contacts = g_slist_append(contacts, cpair);
    }
    
    contact_list_received (uin, contacts, msg_time);
}

void  v7_rate_changed (Snac *snac, V7Connection *conn)
{

  WORD code, class;
  DWORD window_size, clear, alert, limit, disconnect, currentavg, maxavg;
  gchar *here;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_rate_changed\n" );
#endif 

  here = snac->data +8;  /* Dont ask why there is a +8 */

  code = CharsBE_2_Word(here);
  here += 2;
  
  /*
    1 = Changed
    2 = Warning
    3 = Limit
    4 = Limit cleared
  */

  class = CharsBE_2_Word(here);
  here += 2;
  
  window_size = CharsBE_2_DW(here);
  here += 4;
  
  clear = CharsBE_2_DW(here);
  here += 4;
  
  alert = CharsBE_2_DW(here);
  here += 4;
  
  limit = CharsBE_2_DW(here);
  here += 4;
  
  disconnect = CharsBE_2_DW(here);
  here += 4;
  
  currentavg = CharsBE_2_DW(here);
  here += 4;
  
  maxavg = CharsBE_2_DW(here);
  here += 4;

  g_print("Code: %d Class: %d\n", code, class);
  g_print("WinSize: %d Clear: %d Alert: %d Limit: %d Discon: %d Avg: %d MaxAvg:%d\n", window_size, clear, alert, limit, disconnect, currentavg, maxavg);

  if (code == 2)
    gnome_warning_dialog(_("You are sending too fast, please slow down your sending speed."));
  
  if (code == 3)
    gnome_warning_dialog(_("You are sending messages too fast, your last message has been dropped, please wait 10 seconds and resend. If you continue sending messages at that speed, you will be disconnect by the server. Please give it a break."));
  
}

void v7_server_status(Snac *snac, V7Connection *conn)
{

  BYTE *here;
  int tlvcount, i;
  TLV *tlv;

#ifdef TRACE_FUNCTION
  g_print( "v7_server_status\n" );
#endif 

  here = snac->data;

  here += here[0] + 1;
  
  here += 2; /* Warning level ??? */

  if (here+2 > snac->data+snac->datalen)
    return;

  tlvcount = CharsBE_2_Word(here);

  here += 2;

  for(i=0; i<tlvcount; i++) {
    
    if (here+4 > snac->data+snac->datalen)
      return;

    tlv = new_tlv(here); 
    here += tlv->len + 4; 

    switch (tlv->type) {
    case 0x0A:
      conn->ourip = IP_2_DW(tlv->value);
    }

    delete_tlv(tlv); 
  }
}
