/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "netdialog.h"
#include "netdialog.moc"

#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qwidget.h>
#include <qlayout.h>
#include <qfile.h>
#include <qregexp.h>

#include <klocale.h>
#include <kcombobox.h>
#include <kdebug.h>
#include <klineedit.h>
#include <kapplication.h>
#include <kurlrequester.h>

#ifdef __FreeBSD__
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>
#endif

#include <string.h>

NetDialog::NetDialog(QWidget *parent, const char *name)
   : QTabDialog(parent, name, true)
{
  m_clicked = false;
  setCaption(kapp->makeStdCaption(i18n("Network Interface")));

  m_generalTab = new QWidget(this);
  m_generalLayout = new QGridLayout(m_generalTab);
  m_generalLayout->setSpacing(6);
  m_generalLayout->setMargin(11);

  m_deviceLabel = new QLabel(m_generalTab);
  m_deviceLabel->setText(i18n("Interface:"));
  m_generalLayout->addMultiCellWidget(m_deviceLabel, 0, 0, 0, 0);

  m_deviceCombo = new KComboBox(true, m_generalTab);
  m_deviceCombo->setSizePolicy(QSizePolicy(QSizePolicy::Preferred,
     QSizePolicy::Fixed));
  m_deviceCombo->setFocus();
  m_deviceCombo->setDuplicatesEnabled(false);
  m_generalLayout->addMultiCellWidget(m_deviceCombo, 0, 0, 1, 1);

  QStringList output(createList());
  if (output.isEmpty()) {
    m_deviceCombo->insertItem("ppp0");
    m_deviceCombo->insertItem("eth0");
  }
  else
    m_deviceCombo->insertStringList(output);

  QSpacerItem *deviceSpacer = new QSpacerItem(20, 20,
     QSizePolicy::Expanding, QSizePolicy::Fixed);
  m_generalLayout->addMultiCell(deviceSpacer, 0, 0, 2, 2);

  m_timerBox = new QGroupBox(m_generalTab);
  m_timerBox->setTitle(i18n("Timer"));
  m_timerBox->setColumnLayout(0, Qt::Vertical);
  m_timerBox->layout()->setSpacing(0);
  m_timerBox->layout()->setMargin(0);
  m_timerBoxLayout = new QVBoxLayout(m_timerBox->layout());
  m_timerBoxLayout->setAlignment(Qt::AlignTop);
  m_timerBoxLayout->setSpacing(6);
  m_timerBoxLayout->setMargin(11);

  m_timerEdit = new KLineEdit(m_timerBox);
  m_timerEdit->setText("%h:%m:%s");
  m_timerEdit->setEnabled(false);

  m_showTimer = new QCheckBox(m_timerBox);
  m_showTimer->setText(i18n("Show timer"));
  connect(m_showTimer, SIGNAL(toggled(bool)),
      m_timerEdit, SLOT(setEnabled(bool)));
  m_timerBoxLayout->addWidget(m_showTimer);
  m_timerBoxLayout->addWidget(m_timerEdit);

  m_hFormat = new QLabel(m_timerBox);
  m_hFormat->setText(i18n("%h - Total hours online"));
  m_timerBoxLayout->addWidget(m_hFormat);

  m_mFormat = new QLabel(m_timerBox);
  m_mFormat->setText(i18n("%m - Total minutes online"));
  m_timerBoxLayout->addWidget(m_mFormat);

  m_sFormat = new QLabel(m_timerBox);
  m_sFormat->setText(i18n("%s - Total seconds online"));
  m_timerBoxLayout->addWidget(m_sFormat);
  m_generalLayout->addMultiCellWidget(m_timerBox, 1, 1, 0, 2);
  
  QSpacerItem *spacer = new QSpacerItem(20, 20,
      QSizePolicy::Minimum, QSizePolicy::Expanding);
  m_generalLayout->addMultiCell(spacer, 2, 2, 0, 0);
  addTab(m_generalTab, i18n("General"));

  m_commandTab = new QWidget(this);
  m_commandLayout = new QVBoxLayout(m_commandTab);
  m_commandLayout->setSpacing(6);
  m_commandLayout->setMargin(11);

  m_commandsBox = new QGroupBox(m_commandTab);
  m_commandsBox->setTitle(i18n("Commands"));
  m_commandsBox->setColumnLayout(0, Qt::Vertical);
  m_commandsBox->layout()->setSpacing(0);
  m_commandsBox->layout()->setMargin(0);
  m_commandsBoxLayout = new QGridLayout(m_commandsBox->layout());
  m_commandsBoxLayout->setAlignment(Qt::AlignTop);
  m_commandsBoxLayout->setSpacing(6);
  m_commandsBoxLayout->setMargin(11);

  m_enableCommands = new QCheckBox(m_commandsBox);
  m_enableCommands->setText(i18n("Enable connect/disconnect"));
  m_commandsBoxLayout->addMultiCellWidget(m_enableCommands, 0, 0, 0, 2);

  m_cCommand = new QLabel(m_commandsBox);
  m_cCommand->setText(i18n("Connect command:"));
  m_commandsBoxLayout->addMultiCellWidget(m_cCommand, 1, 1, 0, 0);

  m_connectRequester = new KURLRequester(m_commandsBox);
  m_connectRequester->setMinimumSize(145, 0);
  m_connectRequester->setEnabled(false);
  connect(m_enableCommands, SIGNAL(toggled(bool)),
      m_connectRequester, SLOT(setEnabled(bool)));
  m_commandsBoxLayout->addMultiCellWidget(m_connectRequester, 1, 1, 1, 2);

  m_dCommand = new QLabel(m_commandsBox);
  m_dCommand->setText(i18n("Disconnect command:"));
  m_commandsBoxLayout->addMultiCellWidget(m_dCommand, 2, 2, 0, 0);

  m_disconnectRequester = new KURLRequester(m_commandsBox);
  m_disconnectRequester->setMinimumSize(145, 0);
  m_disconnectRequester->setEnabled(false);
  connect(m_enableCommands, SIGNAL(toggled(bool)),
      m_disconnectRequester, SLOT(setEnabled(bool)));
  m_commandsBoxLayout->addMultiCellWidget(m_disconnectRequester, 2, 2, 1, 2);
  m_commandLayout->addWidget(m_commandsBox);
  
  QSpacerItem *commandSpacer = new QSpacerItem(20, 20,
      QSizePolicy::Minimum, QSizePolicy::Expanding);
  m_commandLayout->addItem(commandSpacer);
  addTab(m_commandTab, i18n("Commands"));

  m_graphTab = new QWidget(this);
  m_graphLayout = new QVBoxLayout(m_graphTab);
  m_graphLayout->setSpacing(6);
  m_graphLayout->setMargin(11);

  m_showGraph = new QCheckBox(m_graphTab);
  m_showGraph->setText(i18n("Show graph"));
  m_showGraph->setChecked(true);
  m_graphLayout->addWidget(m_showGraph);

  m_loadLabel = new QCheckBox(m_graphTab);
  m_loadLabel->setText(i18n("Include current load label"));
  m_loadLabel->setChecked(true);
  m_graphLayout->addWidget(m_loadLabel);

  QSpacerItem *spacer_2 = new QSpacerItem(20, 20,
      QSizePolicy::Minimum, QSizePolicy::Expanding);
  m_graphLayout->addItem(spacer_2);
  addTab(m_graphTab, i18n("Graph"));

  setOkButton(i18n("&OK"));
  setCancelButton(i18n("&Cancel"));
  connect(this, SIGNAL(applyButtonPressed()), SLOT(sendClicked()));
}

NetDialog::~NetDialog()
{
}

const QString NetDialog::deviceName() const
{
  return m_deviceCombo->currentText();
}

bool NetDialog::timer()
{
  return m_showTimer->isChecked();
}

const QString NetDialog::format() const
{
  return m_timerEdit->text();
}

bool NetDialog::commands()
{
  return m_enableCommands->isChecked();
}

const QString NetDialog::cCommand() const
{
  return m_connectRequester->url();
}

const QString NetDialog::dCommand() const
{
  return m_disconnectRequester->url();
}

bool NetDialog::graph()
{
  return m_showGraph->isChecked();
}

bool NetDialog::loadLabel()
{
  return m_loadLabel->isChecked();
}

void NetDialog::setDeviceName(const QString &text)
{
  m_deviceCombo->setCurrentItem(text, true);
}

void NetDialog::setShowTimer(bool value)
{
  m_showTimer->setChecked(value);
}

void NetDialog::setFormat(const QString &format)
{
  m_timerEdit->setText(format);
}

void NetDialog::setShowCommands(bool value)
{
  m_enableCommands->setChecked(value);
}

void NetDialog::setCCommand(const QString &text)
{
  m_connectRequester->setURL(text);
}

void NetDialog::setDCommand(const QString &text)
{
  m_disconnectRequester->setURL(text);
}

void NetDialog::setShowGraph(bool value)
{
  m_showGraph->setChecked(value);
}

void NetDialog::setShowLoadLabel(bool value)
{
  m_loadLabel->setChecked(value);
}

void NetDialog::sendClicked()
{
  m_clicked = true;
  kdDebug() << "ok was clicked" << endl;
}

QStringList NetDialog::createList() const
{
#ifdef __linux__
  QFile file("/proc/net/dev");
  if (!file.open(IO_ReadOnly))
    return QStringList();

  QStringList output;
  QTextStream textStream(&file);
  while (!textStream.atEnd())
    output.append(textStream.readLine());

  if (output.isEmpty())
    return QStringList();

  output.pop_front();
  output.pop_front();

  QStringList::Iterator it;
  QStringList list;
  for (it = output.begin(); it != output.end(); ++it) {
    list = QStringList::split(' ', (*it));
    (*it) = list[0].stripWhiteSpace();
    (*it).truncate((*it).find(':'));
  }

  return output;
#endif

#ifdef __FreeBSD__
  QStringList output;
  int mib[] = { CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0 };
  char *buf = 0;
  int alloc = 0;
  struct if_msghdr *ifm, *nextifm;
  struct sockaddr_dl *sdl;
  char *lim, *next;
  size_t needed;
  char s[32];

  if (sysctl(mib, 6, NULL, &needed, NULL, 0) < 0)
    return QStringList();

  if (alloc < needed) {
    buf = new char[needed];

    if (buf == NULL)
      return QStringList();

    alloc = needed;
  }

  if (sysctl(mib, 6, buf, &needed, NULL, 0) < 0)
    return QStringList();

  lim = buf + needed;

  next = buf;
  while (next < lim) {
    ifm = (struct if_msghdr *)next;
    if (ifm->ifm_type != RTM_IFINFO)
      return QStringList();

    next += ifm->ifm_msglen;

    // get an interface with a network address
    while (next < lim) {
      nextifm = (struct if_msghdr *)next;
      if (nextifm->ifm_type != RTM_NEWADDR)
        break;

      next += nextifm->ifm_msglen;
    }

    // if the interface is up
    if (ifm->ifm_flags & IFF_UP) {
      sdl = (struct sockaddr_dl *)(ifm + 1);
      if (sdl->sdl_family != AF_LINK)
        continue;

      strncpy(s, sdl->sdl_data, sdl->sdl_nlen);
      s[sdl->sdl_nlen] = '\0';

      output.append(s);
    }
  }

  if (buf)
    delete[] buf;

  return output;
#endif
}
