/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __ARTICLE_H__
#define __ARTICLE_H__

#include <time.h>

#include <glib.h>

#include <pan/base/group.h>
#include <pan/base/pan-callback.h>

#define ARTICLE(a) ((Article *)a)

enum {
	STATE_NONE		= 0,
	STATE_MULTIPART_ALL	= 1<<0,
	STATE_MULTIPART_SOME	= 1<<1,
	STATE_DOWNLOAD_FLAGGED	= 1<<2,
	STATE_SAVE_QUEUED	= 1<<3,
	STATE_DECODED		= 1<<4,
	STATE_ERROR		= 1<<5,
	STATE_CACHED		= 1<<7,
	STATE_DECODE_FAILED	= 1<<8
};

typedef struct _Article
{
        guint is_new;                 /* true if article is 'new' */
	guint self_passes_filter : 1; /* articlelist */
	guint tree_passes_filter : 1; /* articlelist */
	gint8 crosspost_qty;	      /* # of groups posted to */
	gint16 part;		      /* part #; 0 if no attachments */
	gint16 parts;		      /* # of parts in multipart message */
	gint16 score;		      /* scoring */
	guint16 linecount;	      /* # of lines in the body */
	guint16 state;                /* see the state enum in this file*/
	guint16 unread_children;      /* articlelist: unread kids in thread */
	guint16 new_children;         /* articlelist: new kids in thread */
	time_t date;                  /* date posted, as parsed into time_t */
	GSList * threads;             /* articlelist: list of child threads */
	struct _Article * parent;     /* articlelist: parent in threading */
	const gchar * xref;           /* article's subject */
	const gchar * subject;        /* article's subject */
	const gchar * author_addr;    /* article's author -- address*/
	const gchar * message_id;     /* article's message-id */
	const gchar * author_real;    /* (maybe NULL) article's author -- real name */
	const gchar * references;     /* (maybe NULL) references string for this article */

	/* At some point this may be an array for 'article-centric' */
	gulong number;               /* (PROTECTED) number for article */
	Group * group;               /* (PROTECTED) group this article is in */

	/* holds infrequently-existing headers.
	   It should be used sparingly to avoid the overhead of hashtable.
	   But, since it lets us remove char*'s from Article for lesser-used
	   headers, it actually decreases sizeof(Article) and reduces memory
	   use most of the time.  See the header #defines below...
	*/
	GHashTable* headers;
}
Article;

/* Headers commonly stored in article->headers. */
#define HEADER_FOLLOWUP_TO        "Followup-To"
#define HEADER_NEWSGROUPS         "Newsgroups"
#define HEADER_ORGANIZATION       "Organization"
#define HEADER_REFERENCES         "References"
#define HEADER_REPLY_TO           "Reply-To"
#define HEADER_SUPERSEDES         "Supersedes"
#define HEADER_XREF               "Xref"
/* Headers stored in their own fields but here for completeness */
#define HEADER_SUBJECT            "Subject"
#define HEADER_FROM               "From"
#define HEADER_MESSAGE_ID         "Message-Id"
#define HEADER_DATE               "Date"
/* Headers not stored (must be retrieved from the body). */
#define HEADER_USER_AGENT         "User-Agent"
#define HEADER_X_NEWSREADER       "X-Newsreader"
#define HEADER_X_MAILER           "X-Mailer"
/* Headers beginning with Pan- are for internal use. */
#define PAN_ATTRIBUTION           "Pan-Attribution"
#define PAN_ATTACH_FILE           "Pan-Attach-File"
#define PAN_HEADER_FILENAME       "Pan-Attached-Filename"
#define PAN_BODY                  "Pan-Body"
#define PAN_MAIL_TO               "Pan-Mail-To"
#define PAN_SERVER                "Pan-Server"
#define PAN_REVERSE_PATH          "Pan-Reverse-Path"
#define PAN_LINES_PER_PART        "Pan-Lines-Per-Part"
#define PAN_REPLY_PORTION         "Pan-Reply-Portion"

/**
***  PUBLIC LIFE CYCLE
**/

Article*     article_new                      (Group          * group);

Article*     article_dup                      (Group          * group,
                                               const Article  * article);

void         article_destructor               (Article        * article);

/**
***  PUBLIC MUTATORS
**/

/**
 * Populates the article object's author_addr and author_real fields
 * from a typical From: header; ie, "Charles Kerr <charles@rebelbase.com>".
 * Fires an ARTICLE_CHANGED_DIRTY event.
 */
void         article_set_author_from_header   (Article        * article,
                                               const gchar    * header_from);
/**
 * Identical to article_set_author_from_header() except that the
 * ARTICLE_CHANGED_DIRTY event is not fired.  This function should
 * be when initializing, rather than modifying, an Article object.
 */
void         article_init_author_from_header  (Article        * article,
                                               const gchar    * header_from);

/**
 * Sets the specified flag to TRUE on the specified articles, and fires
 * a single ARTICLE_CHANGED_DIRTY event for the entire batch of articles.
 */
void         articles_add_flag                (Article       ** articles,
                                               gint             qty,
                                               gushort          flag_to_set);
/**
 * Sets the specified flag to FALSE on the specified articles, and fires
 * a single ARTICLE_CHANGED_DIRTY event for the entire batch of articles.
 */
void         articles_remove_flag             (Article       ** articles,
                                               gint             qty,
                                               gushort          flag_to_remove);

/**
***  Headers
**/

/**
 * If DO_CHUNK is set, then the header field will be stored in the Article's
 * Group's StringChunk to simplify memory management.  Use this for
 * Message-IDs, Xrefs, and other strings that should be unique.
 *
 * If DO_CHUNK_SHARE is set, the header will be stored in the Article's Group's
 * StringChunk such that it's shared across articles.  Use this for subjects,
 * author names, References: strings, and anything else that can be shared.
 *
 * In general you always want to chunk the strings unless.
 */
typedef enum { DO_CHUNK=1, DO_CHUNK_SHARE=2 } HeaderAction;

/**
 * Sets the specified header name/value pair in the specified article
 * and fires an ARTICLE_CHANGED_DIRTY event to notify listeners.
 *
 * Headers that are always present in an Article have their own fields
 * in the article struct -- subject, message_id, and so on.  Less
 * frequently-used headers are stored in an internal table so that
 * we don't have to bloat the size of every  Article struct for specal
 * cases.
 *
 * The HeaderAction should almost always be DO_CHUNK.  See the HeaderAction
 * typedef for more information.
 */
void         article_set_header               (Article        * article,
                                               const gchar    * header_name,
                                               const gchar    * header_value,
                                               HeaderAction     action);
/**
 * Identical to article_set_header() except that the
 * ARTICLE_CHANGED_DIRTY event is not fired.  This function should
 * be when initializing, rather than modifying, an Article object.
 */
void         article_init_header              (Article        * article,
                                               const gchar    * header_name,
                                               const gchar    * header_value,
                                               HeaderAction     action);

/**
 * Sets the specified header name/value pair in the specified article
 * and fires an ARTICLE_CHANGED_DIRTY event to notify listeners.
 */
void         article_remove_header            (Article        * article,
                                               const gchar    * header_name);

/**
 * Returns the Article's value for the specified header, or NULL if no value.
 */
const gchar* article_get_extra_header         (const Article  * article,
                                               const gchar    * header_name);

/**
 * Returns a GPtrArray* of strings.  The array has 2N elements, where
 * N is the number of headers present in the Article -- index i has
 * the header name, and i+1 contains the value.
 *
 * When done, the array should be g_ptr_array_free (a, TRUE) but
 * do not g_free the strings themselves - these are managed by the Article.
 */
GPtrArray*   article_get_all_headers          (const Article  * article);

/**
 * Returns TRUE if the header is one only meant for Pan's internal bookkeeping
 * and should never be shown to the user.
 */
gboolean     article_header_is_internal       (const gchar    * key);

/**
 * This is a wart specific to message-window which should eventually be
 * moved out of article.h.  It returns true if the header isn't
 * internal, nor is followup_to, newsgroups, organization, or reply_to.
 */
gboolean     article_header_is_extra          (const gchar    * key);


/**
 * This is the iterator function for article_xref_foreach.
 */
typedef void (ArticleXRefFunc)(Group*,gulong,gpointer);

/**
 * For each cross-reference specified in Article a's Xref: header,
 * the specified ArticleXRefFunc is invoked.
 *
 * If skip_group_a is TRUE, then the group "a->group" is not included
 * in this foreach, even if it's listed in the Xref header.
 */
void         article_xref_foreach             (const Article   * a,
                                               ArticleXRefFunc   func,
                                               gpointer          user_data,
                                               ServerGroupsType  set,
                                               gboolean          skip_group_a);

/**
***  PUBLIC ACCESSORS
**/

/**
 * Sanity-checks an article.  Convenient for sanity checking at the 
 * beginning of functions that take Article pointers.
 */
gboolean     article_is_valid                 (const Article  * article);

gboolean     articles_are_valid               (const Article ** articles,
                                               gint             qty);

gboolean     article_has_attachment           (const Article  * article);

gboolean     article_has_body                 (const Article  * article);

gchar*       article_get_body                 (const Article  * article);

const gchar* article_get_message_id           (const Article  * article);

const gchar* article_get_subject              (const Article  * article);

gchar*       article_get_headers              (const Article  * article);

gchar*       article_get_message              (const Article  * article);

gchar*       article_get_author_str           (const Article  * article,
                                               gchar          * buf,
                                               gint             len);

void         article_get_short_author_str     (const Article  * article,
                                               gchar          * buf,
                                               gint             len);

void         article_set_from_raw_message     (Article        * article,
                                               const gchar    * text);

gulong       article_get_combined_linecount   (const Article  * article);

/**
***  READ
**/

/**
 * Returns true if this article is marked as read.
 */
gboolean     article_is_read                  (const Article  * article);

/**
 * Sets the specified articles as read/unread,
 * And fires a single ARTICLE_CHANGED_READ event for the batch of articles.
 */
void         articles_set_read                (Article      ** articles,
                                               int              article_qty,
                                               gboolean         read);

/**
 * Returns true if this article is new, where new is defined as having
 * been retrieved from the news server the last time the user tried to
 * fetch new headers.
 */
gboolean     article_is_new                   (const Article  * article);

void         articles_set_new                 (Article       ** articles,
                                               int              article_qty,
                                               gboolean         is_new);


#define article_flag_on(a, flag) (a && (a->state&flag))


/**
 * @param article the article whose thread is to be returned
 * @param GPtrArray a container to hold all the articles found in
 *                  this article's thread.
 */
void           article_get_entire_thread     (Article         * article,
                                              GPtrArray       * setme);

void           article_get_subthread         (Article         * article,
                                              GPtrArray       * setme);

void           article_get_references        (Article         * article,
                                              GPtrArray       * setme);

typedef enum
{
	GET_WHOLE_THREAD,
	GET_SUBTHREAD
}
ThreadGet;
 
GPtrArray*     article_get_unique_threads    (const GPtrArray    * articles,
                                              ThreadGet            thread_get);
 

gchar*         article_get_thread_message_id (const Article*);

extern const gchar * default_incoming_name_real;

extern const gchar * default_incoming_name_addr;

gboolean       article_get_header_run_hash_from_text  (const gchar    * article_text,
                                                       GHashTable    ** makeme_hash,
                                                       gpointer       * allocme_buf);

/***
****  Events
***/
 
typedef enum
{
	ARTICLE_CHANGED_READ,
	ARTICLE_CHANGED_NEW,
	ARTICLE_CHANGED_DIRTY
}
ArticleChangeType;
 
typedef struct
{
	Group * group;
	Article ** articles;
	gint article_qty;
	ArticleChangeType type;
}
ArticleChangeEvent;
 
/**
 * @call_obj: ArticleChangeEvent*
 * @call_arg: NULL
 **/
PanCallback *     article_get_articles_changed_callback      (void);                                                            



#endif /* __ARTICLE_H__ */
