/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif	/* !HAVE_CONFIG_H */

#include <gtk/gtk.h>
#include <libxfce4util/i18n.h>
#include <libxfcegui4/dialogs.h>

#include <xfprinterlist.h>

#include "propertiesdlg.h"

GtkWidget *mainwin;
GtkWidget *treeview;
GtkWidget *propdlg;

GtkWidget *menuitem_delprinter;
GtkWidget *menuitem_properties;
GtkWidget *menuitem_setdefault;

enum
{
	ICON_COLUMN,
	ALIAS_COLUMN,
	NAME_COLUMN,
	N_COLUMNS
};

static void tree_selection_changed_cb(GtkTreeSelection *, gpointer);
static void set_default_cb(GtkWidget *, gpointer);
static void add_printer_cb(GtkWidget *, gpointer);
static void del_printer_cb(GtkWidget *, gpointer);
static void properties_cb(GtkWidget *, gpointer);

extern XfPrinterList *printers;

void
mainwin_setup(void)
{
	GdkPixbuf *icon;
	GtkAccelGroup *accel_group;
	GtkListStore *store;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkWidget *menubar;
	GtkWidget *menuitem_printer;
	GtkWidget *menu_printer;
	GtkWidget *menuitem_addprinter;
	GtkWidget *menuitem_quit;
	GtkWidget *scrollwin;
	GtkWidget *separator;
	GtkWidget *vbox;

	XfPrinter *pr;

	accel_group = gtk_accel_group_new();

	mainwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_add_accel_group(GTK_WINDOW(mainwin), accel_group);
	gtk_window_set_default_size(GTK_WINDOW(mainwin), 450, 250);
	gtk_window_set_title(GTK_WINDOW(mainwin), _("Printer manager"));

	propdlg = properties_dlg_new(GTK_WINDOW(mainwin));

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(mainwin), vbox);

	menubar = gtk_menu_bar_new();
	gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);

	menuitem_printer = gtk_menu_item_new_with_mnemonic(_("_Printer"));
	gtk_container_add(GTK_CONTAINER(menubar), menuitem_printer);

	menu_printer = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_printer),menu_printer);

	menuitem_addprinter = gtk_image_menu_item_new_with_mnemonic(
		_("_Add printer"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem_addprinter),
		gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_MENU));
	g_signal_connect(G_OBJECT(menuitem_addprinter), "activate",
		G_CALLBACK(add_printer_cb), NULL);
	gtk_container_add(GTK_CONTAINER(menu_printer), menuitem_addprinter);

	menuitem_delprinter = gtk_image_menu_item_new_with_mnemonic(
		_("_Remove printer"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem_delprinter),
		gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
	g_signal_connect(G_OBJECT(menuitem_delprinter), "activate",
		G_CALLBACK(del_printer_cb), NULL);
	gtk_container_add(GTK_CONTAINER(menu_printer), menuitem_delprinter);

	separator = gtk_menu_item_new();
	gtk_widget_set_sensitive(separator, FALSE);
	gtk_container_add(GTK_CONTAINER(menu_printer), separator);

	menuitem_properties = gtk_image_menu_item_new_with_mnemonic(
		_("_Properties"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem_properties),
		gtk_image_new_from_stock(GTK_STOCK_PROPERTIES,
			GTK_ICON_SIZE_MENU));
	g_signal_connect(G_OBJECT(menuitem_properties), "activate",
		G_CALLBACK(properties_cb), NULL);
	gtk_container_add(GTK_CONTAINER(menu_printer), menuitem_properties);

	menuitem_setdefault = gtk_image_menu_item_new_with_mnemonic(
		_("Set _default"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem_setdefault),
		gtk_image_new_from_stock(GTK_STOCK_PRINT, GTK_ICON_SIZE_MENU));
	g_signal_connect(G_OBJECT(menuitem_setdefault), "activate",
		G_CALLBACK(set_default_cb), NULL);
	gtk_container_add(GTK_CONTAINER(menu_printer), menuitem_setdefault);

	separator = gtk_menu_item_new();
	gtk_widget_set_sensitive(separator, FALSE);
	gtk_container_add(GTK_CONTAINER(menu_printer), separator);

	menuitem_quit = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT,
		accel_group);
	g_signal_connect(G_OBJECT(menuitem_quit), "activate",
		G_CALLBACK(gtk_main_quit), NULL);
	gtk_container_add(GTK_CONTAINER(menu_printer), menuitem_quit);

	/* the printer list store */
	store = gtk_list_store_new(N_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING,
		G_TYPE_STRING, G_TYPE_STRING);

	pr = xfprinterlist_first(printers);
	while (pr != NULL) {
		icon = xfprinter_icon(pr, FALSE);

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
			ICON_COLUMN, icon,
			ALIAS_COLUMN, pr->alias,
			NAME_COLUMN, pr->name,
			-1);

		g_object_unref(G_OBJECT(icon));

		pr = xfprinterlist_next(printers);
	}

	/* The printer list view */
	scrollwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);

	treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_container_add(GTK_CONTAINER(scrollwin), treeview);

	/* configure the tree selection for the printer list */
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
	g_signal_connect(G_OBJECT(selection), "changed",
		G_CALLBACK(tree_selection_changed_cb), NULL);

	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview),
		ICON_COLUMN, _("Type"), gtk_cell_renderer_pixbuf_new(),
		"pixbuf", ICON_COLUMN, NULL);

	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview),
		ALIAS_COLUMN, _("Alias name"), gtk_cell_renderer_text_new(),
		"text", ALIAS_COLUMN, NULL);

	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview),
		NAME_COLUMN, _("System name"), gtk_cell_renderer_text_new(),
		"text", NAME_COLUMN, NULL);

	/* the view now holds a reference of the store */
	g_object_unref(G_OBJECT(store));

	/* connect signals for the main window */
	g_signal_connect_swapped(G_OBJECT(mainwin), "delete-event",
		G_CALLBACK(gtk_menu_item_activate), menuitem_quit);

	gtk_widget_show_all(mainwin);
}

static void
tree_selection_changed_cb(GtkTreeSelection *select, gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *alias;

	if (gtk_tree_selection_get_selected(select, &model, &iter)) {
		gtk_tree_model_get(model, &iter, ALIAS_COLUMN, &alias, -1);

		gtk_widget_set_sensitive(menuitem_delprinter, TRUE);
		gtk_widget_set_sensitive(menuitem_properties, TRUE);
		gtk_widget_set_sensitive(menuitem_setdefault,
			!xfprinterlist_is_default_by_alias(printers, alias));

		g_free(alias);
	}
	else {
		gtk_widget_set_sensitive(menuitem_delprinter, FALSE);
		gtk_widget_set_sensitive(menuitem_properties, FALSE);
		gtk_widget_set_sensitive(menuitem_setdefault, FALSE);
	}
}

static void
set_default_cb(GtkWidget *menuitem, gpointer user_data)
{
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;
	XfPrinter *pr;
	gchar *alias;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));

	if (gtk_tree_selection_get_selected(select, &model, &iter)) {
		gtk_tree_model_get(model, &iter, ALIAS_COLUMN, &alias, -1);

		if ((pr = xfprinterlist_lookup(printers, alias)) != NULL) {
			xfprinterlist_set_default(printers, pr);
			xfprinterlist_save(printers);
		
			/* disable "Set default", as the actual printer is now
		 	 * the default printer
			 */
			gtk_widget_set_sensitive(menuitem, FALSE);
		}

		g_free(alias);
	}
}

static void
add_printer_cb(GtkWidget *menuitem, gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GdkPixbuf *icon;
	XfPrinter *pr;
	const XfPrinter *newpr;

	properties_dlg_popup(PROPERTIES_DLG(propdlg), _("Add a new printer"),
		NULL);

	for (;;) {
		newpr = properties_dlg_run(PROPERTIES_DLG(propdlg));

		if (newpr == NULL)
			goto hide;

		if (xfprinterlist_lookup(printers, newpr->alias)) {
			xfce_err(_("A printer of that name is already "
				   "recorded. Please choose another "
				   "alias name for the new printer."));
			continue;
		}

		break;
	}

	if ((pr = xfprinter_dup(newpr)) != NULL) {
		/* insert printer into printer list and save the
		 * modified list
		 */
		xfprinterlist_insert(printers, pr);
		xfprinterlist_save(printers);

		model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));

		icon = xfprinter_icon(pr, FALSE);

		/* XXX - should be insert sorted */
		gtk_list_store_append(GTK_LIST_STORE(model), &iter);
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			ICON_COLUMN, icon,
			ALIAS_COLUMN, pr->alias,
			NAME_COLUMN, pr->name,
			-1);

		g_object_unref(G_OBJECT(icon));
	}

hide:
	properties_dlg_hide(PROPERTIES_DLG(propdlg));
}

static void
del_printer_cb(GtkWidget *menuitem, gpointer user_data)
{
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;
	XfPrinter *pr;
	gchar *alias;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));

	if (gtk_tree_selection_get_selected(select, &model, &iter)) {
		gtk_tree_model_get(model, &iter, ALIAS_COLUMN, &alias, -1);

		/* remove the printer from the printer list */
		if ((pr = xfprinterlist_lookup(printers, alias)) != NULL) {
			xfprinterlist_remove(printers, pr);
			xfprinterlist_save(printers);
		}

		g_free(alias);

		gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
	}
}

static void
properties_cb(GtkWidget *menuitem, gpointer user_data)
{
	GtkTreeSelection *select;
	GtkTreeModel *model;
	GtkTreeIter iter;
	XfPrinter *pr;
	GdkPixbuf *icon;
	const XfPrinter *modpr;
	gchar *alias;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));

	if (!gtk_tree_selection_get_selected(select, &model, &iter))
		return;

	gtk_tree_model_get(model, &iter, ALIAS_COLUMN, &alias, -1);

	if ((pr = xfprinterlist_lookup(printers, alias)) == NULL) {
		g_warning(_("Printer \"%s\" does not exists"), alias);
		goto finished;
	}

	properties_dlg_popup(PROPERTIES_DLG(propdlg),
		_("Printer properties"), pr);

	if ((modpr = properties_dlg_run(PROPERTIES_DLG(propdlg))) == NULL)
		goto hide;

	xfprinter_copy(pr, modpr);
	xfprinterlist_save(printers);

	icon = xfprinter_icon(pr, xfprinterlist_is_default(printers, pr));

	gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		ICON_COLUMN, icon,
		NAME_COLUMN, pr->name,
		-1);

	g_object_unref(G_OBJECT(icon));

hide:
	properties_dlg_hide(PROPERTIES_DLG(propdlg));

finished:
	g_free(alias);
}

