/*
 *  Copyright (C) 2002 Jorn Baayen <jorn@nl.linux.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libxml/tree.h>
#include <libxml/xmlwriter.h>
#include <string.h>
#include <sys/stat.h>

#include "galeon-embed-persist.h"
#include "galeon-config.h"
#include "pixbuf-cache.h"
#include "gul-general.h"

#include "galeon-favicon-cache.h"

static void galeon_favicon_cache_class_init (GaleonFaviconCacheClass *klass);
static void galeon_favicon_cache_init (GaleonFaviconCache *ma);
static void galeon_favicon_cache_finalize (GObject *object);
static void galeon_favicon_cache_sync_to_disc (GaleonFaviconCache *cache);
static void galeon_favicon_cache_sync_from_disc (GaleonFaviconCache *cache);
static gboolean galeon_favicon_cache_timeout_cb (GaleonFaviconCache *cache);
static void galeon_favicon_cache_insert (GaleonFaviconCache *cache,
			                 const char *url,
			                 const char *pixbuf_location);
static char *galeon_favicon_cache_dest (GaleonFaviconCache *cache,
					const char *url);
static void favicon_download_completed_cb (GaleonEmbedPersist *persist,
			                   GaleonFaviconCache *cache);

struct GaleonFaviconCachePrivate
{
	GHashTable *url_to_pixbuf;
	GHashTable *favicon_to_pixbuf;

	guint timeout;
	gboolean dirty;

	char *filename;
	char *directory;
};

enum
{
	CHANGED,
	LAST_SIGNAL
};

static guint galeon_favicon_cache_signals[LAST_SIGNAL] = { 0 };

static GObjectClass *parent_class = NULL;

GType
galeon_favicon_cache_get_type (void)
{
	static GType galeon_favicon_cache_type = 0;

	if (galeon_favicon_cache_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (GaleonFaviconCacheClass),
			NULL,
			NULL,
			(GClassInitFunc) galeon_favicon_cache_class_init,
			NULL,
			NULL,
			sizeof (GaleonFaviconCache),
			0,
			(GInstanceInitFunc) galeon_favicon_cache_init
		};

		galeon_favicon_cache_type = g_type_register_static (G_TYPE_OBJECT,
								    "GaleonFaviconCache",
								     &our_info, 0);
	}

	return galeon_favicon_cache_type;
}

static void
galeon_favicon_cache_class_init (GaleonFaviconCacheClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = galeon_favicon_cache_finalize;

	galeon_favicon_cache_signals[CHANGED] =
		g_signal_new ("changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GaleonFaviconCacheClass, changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_STRING);
}

static void
galeon_favicon_cache_init (GaleonFaviconCache *cache)
{
	cache->priv = g_new0 (GaleonFaviconCachePrivate, 1);

	cache->priv->url_to_pixbuf     = g_hash_table_new_full (g_str_hash,
							        g_str_equal,
						                (GDestroyNotify) g_free,
						                (GDestroyNotify) g_object_unref);
	cache->priv->favicon_to_pixbuf = g_hash_table_new_full (g_str_hash,
								g_str_equal,
								(GDestroyNotify) g_free,
								(GDestroyNotify) g_object_unref);

	cache->priv->filename  = g_build_filename (g_get_home_dir (),
				                   GALEON_DIR,
				                   "favicon_cache.xml",
				                   NULL);
	cache->priv->directory = g_build_filename (g_get_home_dir (),
						   GALEON_DIR,
						   "favicon_cache/",
						   NULL);

	cache->priv->dirty = FALSE;
	
	if (g_file_test (cache->priv->directory, G_FILE_TEST_IS_DIR) == FALSE)
	{
		if (g_file_test (cache->priv->directory, G_FILE_TEST_EXISTS))
		{
			g_error ("Please remove %s to continue.", cache->priv->directory);
		}

		if (mkdir (cache->priv->directory, 488) != 0)
		{
			g_error ("Couldn't mkdir %s.", cache->priv->directory);
		}
	}

	galeon_favicon_cache_sync_from_disc (cache);

	/* sync to disc every 10 minutes */
	cache->priv->timeout = g_timeout_add (10 * 60 * 1000,
					      (GSourceFunc) galeon_favicon_cache_timeout_cb,
					      cache);
}

static void
galeon_favicon_cache_finalize (GObject *object)
{
	GaleonFaviconCache *cache;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GALEON_IS_FAVICON_CACHE (object));

	cache = GALEON_FAVICON_CACHE (object);

	g_return_if_fail (cache->priv != NULL);

	g_source_remove (cache->priv->timeout);

	galeon_favicon_cache_sync_to_disc (cache);

	g_hash_table_destroy (cache->priv->url_to_pixbuf);
	g_hash_table_destroy (cache->priv->favicon_to_pixbuf);

	g_free (cache->priv->filename);
	g_free (cache->priv->directory);

	g_free (cache->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GaleonFaviconCache *
galeon_favicon_cache_new (void)
{
	GaleonFaviconCache *cache;

	cache = GALEON_FAVICON_CACHE (g_object_new (GALEON_TYPE_FAVICON_CACHE, NULL));

	g_return_val_if_fail (cache->priv != NULL, NULL);

	return cache;
}

GdkPixbuf *
galeon_favicon_cache_lookup (GaleonFaviconCache *cache,
			     const char *url)
{
	char *cache_url;
	GdkPixbuf *ret;

	g_return_val_if_fail (GALEON_IS_FAVICON_CACHE (cache), NULL);

	if (url == NULL)
	{
		return NULL;
	}

	cache_url = galeon_favicon_cache_url (url);
	ret = galeon_favicon_cache_lookup_direct (cache, cache_url);
	g_free (cache_url);

	return ret;
}

GdkPixbuf *
galeon_favicon_cache_lookup_direct (GaleonFaviconCache *cache,
				    const char *cache_url)
{
	GdkPixbuf *ret;

	if (cache_url == NULL)
	{
		return NULL;
	}
	
	ret = g_hash_table_lookup (cache->priv->url_to_pixbuf,
				   cache_url);
	return ret;
}

static void
galeon_favicon_cache_insert (GaleonFaviconCache *cache,
			     const char *url,
			     const char *pixbuf_location)
{
	GdkPixbuf *pixbuf = NULL;
	char *cache_url;

	cache_url = galeon_favicon_cache_url (url);

	if (cache_url == NULL) return;
	
	pixbuf = g_hash_table_lookup (cache->priv->favicon_to_pixbuf, 
				      pixbuf_location);
	if (pixbuf == NULL)
	{
		pixbuf = gdk_pixbuf_new_from_file (pixbuf_location, NULL);
		if (pixbuf == NULL) {
			g_free (cache_url);
			return;
		}

		if (gdk_pixbuf_get_width (pixbuf) > 16 ||
		    gdk_pixbuf_get_height (pixbuf) > 16)
		{
			GdkPixbuf *scaled = gdk_pixbuf_scale_simple (pixbuf, 16, 16,
						  		     GDK_INTERP_NEAREST);
			g_object_unref (G_OBJECT (pixbuf));
			pixbuf = scaled;
		}
		
		g_hash_table_insert (cache->priv->favicon_to_pixbuf,
				     g_strdup (pixbuf_location),
				     pixbuf);
	}

	if (pixbuf == NULL) {
		g_free (cache_url);
		return;
	}

	g_object_ref (G_OBJECT (pixbuf));

	g_object_set_data_full (G_OBJECT (pixbuf),
				"location",
				g_strdup (pixbuf_location),
				g_free); 
	
	g_hash_table_replace (cache->priv->url_to_pixbuf,
			      cache_url,
			      pixbuf);

	cache->priv->dirty = TRUE;
	g_signal_emit (G_OBJECT (cache), galeon_favicon_cache_signals[CHANGED], 0, cache_url);
}

static void
save_item (const char *key, GObject *value, xmlTextWriterPtr writer)
{
	xmlTextWriterStartElement (writer, "entry");

	xmlTextWriterWriteAttribute (writer, "url", key);
	xmlTextWriterWriteAttribute (writer, "favicon", 
				     g_object_get_data (value, "location"));

	xmlTextWriterEndElement (writer);
}

static int
galeon_favicon_cache_save_to_file (GaleonFaviconCache *cache,
				   xmlTextWriterPtr writer)
{
	int ret;

	ret = xmlTextWriterStartDocument (writer, "1.0", NULL, NULL);
	if (ret < 0) return ret;
	ret = xmlTextWriterStartElement (writer, "GaleonFaviconCache");
	if (ret < 0) return ret;
	
	g_hash_table_foreach (cache->priv->url_to_pixbuf, (GHFunc) save_item, writer);

	ret = xmlTextWriterEndElement (writer); /* root */
	if (ret < 0) return ret;

	ret = xmlTextWriterEndDocument (writer);
	if (ret < 0) return ret;
	
	return 0;
}

static void
galeon_favicon_cache_sync_to_disc (GaleonFaviconCache *cache)
{
	xmlTextWriterPtr writer;
	gchar *tmpfile;
	int ret;

	if (!cache->priv->dirty)
		return;

	tmpfile = g_strconcat (cache->priv->filename, ".tmp", NULL);

	writer = xmlNewTextWriterFilename (tmpfile, 0);
	xmlTextWriterSetIndent (writer, 1);
	xmlTextWriterSetIndentString (writer, "  ");

	ret = galeon_favicon_cache_save_to_file (cache, writer);
	
	if (ret < 0)
	{
		g_warning ("Failed writing to %s", cache->priv->filename);
	}
	else
	{
		gul_general_switch_temp_file (cache->priv->filename, tmpfile);
	}

	g_free (tmpfile);
	xmlFreeTextWriter (writer);

	cache->priv->dirty = FALSE;
}

static void
galeon_favicon_cache_sync_from_disc (GaleonFaviconCache *cache)
{
	xmlDocPtr doc;
	xmlNodePtr child;
	
	if (g_file_test (cache->priv->filename, G_FILE_TEST_EXISTS) == FALSE)
		return;

	doc = xmlParseFile (cache->priv->filename);

	if (doc == NULL)
		return;

	for (child = doc->children->children; child != NULL; child = child->next)
	{
		char *url = xmlGetProp (child, "url");
		char *favicon = xmlGetProp (child, "favicon");

		galeon_favicon_cache_insert (cache,
					     url,
					     favicon);

		g_free (url);
		g_free (favicon);
	}

	xmlFreeDoc (doc);
}

static gboolean
galeon_favicon_cache_timeout_cb (GaleonFaviconCache *cache)
{
	galeon_favicon_cache_sync_to_disc (cache);
	
	return TRUE;
}

char *
galeon_favicon_cache_url (const char *url)
{
	GnomeVFSURI *uri;
	char *result, *clean_url;
	int clean_url_len;

	if (url == NULL)
	{
		return NULL;
	}

	uri = gnome_vfs_uri_new (url);
	if (uri == NULL)
	{
		return NULL;
	}

	clean_url = gnome_vfs_uri_to_string (uri,
					     GNOME_VFS_URI_HIDE_USER_NAME |
					     GNOME_VFS_URI_HIDE_PASSWORD |
					     GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER);

	clean_url_len = strlen (clean_url);
	if (clean_url[clean_url_len - 1] == '/')
	{
		result = g_strndup (clean_url, clean_url_len - 1);
		g_free (clean_url);
	}
	else
	{
		result = clean_url;
	}

	gnome_vfs_uri_unref (uri);

	return result;
}

gboolean
galeon_favicon_cache_url_equal (const char *a,
				const char *b)
{
	char *aa = galeon_favicon_cache_url (a);
	char *ab = galeon_favicon_cache_url (b);
	gboolean ret;

	ret = ((aa != NULL && ab != NULL) && (strcmp (aa, ab) == 0));

	g_free (aa);
	g_free (ab);

	return ret;
}

static char *
galeon_favicon_cache_dest (GaleonFaviconCache *cache, const char *url)
{
	char *clean_url = galeon_favicon_cache_url (url);
	char *slashpos, *dest;

	if (clean_url == NULL)
	{
		return NULL;
	}

	while ((slashpos = strstr (clean_url, "/")) != NULL)
	{
		*slashpos = '_';
	}
	
	dest = g_build_filename (cache->priv->directory, clean_url, NULL);
	
	g_free (clean_url);

	return dest;
}

void
galeon_favicon_cache_insert_from_url (GaleonFaviconCache *cache,
				      const char *url,
				      const char *favicon_url)
{
	GaleonEmbedPersist *persist;
	char *dest;

	g_return_if_fail (GALEON_IS_FAVICON_CACHE (cache));
	g_return_if_fail (url != NULL);
	g_return_if_fail (favicon_url != NULL);

	dest = galeon_favicon_cache_dest (cache, favicon_url);
	if (dest == NULL)
		return;
	
	persist = galeon_embed_persist_new (NULL);

	galeon_embed_persist_set_max_size (persist, 100);
	galeon_embed_persist_set_flags    (persist, EMBED_PERSIST_BYPASSCACHE);
	galeon_embed_persist_set_source   (persist, favicon_url);
	galeon_embed_persist_set_dest     (persist, dest);

	g_object_set_data_full (G_OBJECT (persist), "url", g_strdup (url), g_free);
	g_object_set_data_full (G_OBJECT (persist), "favicon", dest, g_free);

	g_signal_connect (G_OBJECT (persist),
			  "completed",
			  G_CALLBACK (favicon_download_completed_cb),
			  cache);

	galeon_embed_persist_save (persist);

	g_object_unref (persist);
}

static void
favicon_download_completed_cb (GaleonEmbedPersist *persist,
			       GaleonFaviconCache *cache)
{
	galeon_favicon_cache_insert (cache,
				     g_object_get_data (G_OBJECT (persist), "url"),
				     g_object_get_data (G_OBJECT (persist), "favicon"));
}
