/* atmtcp.c - control ATM on TCP emulation */

/* Written 1995-1998 by Werner Almesberger, EPFL-LRC/ICA */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <atm.h>
#include <atmd.h>
#include <linux/atm_tcp.h>


#define PORT 8402	/* old ATMTCP used 8401, but now the header is in
			   network byte order, so we're incompatible in most
			   cases */

#define MAX_PACKET ATM_MAX_AAL5_PDU+sizeof(struct atmtcp_hdr)


static int s_tcp,s_krn;
static int debug = 0;


static void net_write(void *data,int size)
{
    int wrote;

    wrote = write(s_tcp,data,size);
    if (wrote < 0) {
	perror("write to TCP");
	exit(1);
    }
    if (!wrote) exit(0); /* EOF */
    if (wrote != size) {
	fprintf(stderr,"bad write (%d != %d)\n",wrote,size);
	exit(1);
    }
}


/*
 * This is a pretty poor implementation because we need to read twice for each
 * PDU. But then, atmtcp isn't exactly designed for throughput ...
 */


static void from_tcp(void)
{
    static char buf[MAX_PACKET];
    static struct atmtcp_hdr *hdr = (struct atmtcp_hdr *) buf;
    static int bytes = 0;
    int ret,size;

    size = sizeof(struct atmtcp_hdr)-bytes;
    if (size <= 0) size += ntohl(hdr->length);
    ret = read(s_tcp,buf+bytes,size);
    if (ret < 0) {
	perror("read from TCP");
	exit(1);
    }
    if (!ret) exit(0); /* EOF */
    bytes += ret;
    if (bytes < sizeof(struct atmtcp_hdr)) return;
    if (ntohl(((struct atmtcp_hdr *) buf)->length) > ATM_MAX_AAL5_PDU) {
	fprintf(stderr,"giant PDU (length = %d) received\n",
	  (unsigned int) ntohl(hdr->length));
	exit(1);
    }
    if (bytes < sizeof(struct atmtcp_hdr)+ntohl(hdr->length)) return;
    if (debug)
	fprintf(stderr,"TCP %d.%d, %d bytes\n",ntohs(hdr->vpi),ntohs(hdr->vci),
	  (unsigned int) ntohl(hdr->length));
    size = write(s_krn,buf,bytes);
    if (size < 0) perror("write to kernel");
    else if (size != bytes) {
	    fprintf(stderr,"bad write (%d != %d)\n",size,bytes);
	    exit(1);
	}
    bytes = 0;
}


static void to_tcp(void)
{
    char buf[MAX_PACKET];
    struct atmtcp_hdr *hdr = (struct atmtcp_hdr *) buf;
    int ret;

    ret = read(s_krn,buf,MAX_PACKET);
    if (ret < 0) {
	perror("read from kernel");
	exit(1);
    }
    if (!ret) exit(0); /* we don't use that yet */
    if (ret < sizeof(struct atmtcp_hdr)) {
	fprintf(stderr,"kernel message too small (%d)\n",ret);
	exit(1);
    }
    if (debug)
	fprintf(stderr,"KRN %d.%d, %d bytes\n",ntohs(hdr->vpi),ntohs(hdr->vci),
	  (unsigned int) ntohl(hdr->length));
    if (ret != sizeof(struct atmtcp_hdr)+ntohl(hdr->length)) {
	fprintf(stderr,"invalid kernel message\n");
	exit(1);
    }
    net_write(buf,ret);
}


static void usage(const char *name)
{
    fprintf(stderr,"%s [ -b ] [ -d ] [ -i itf ] -l [ <lclport> ]\n",name);
    fprintf(stderr,"%s [ -b ] [ -d ] [ -i itf ] -c <host> [ <rmtport> ]\n",
      name);
    fprintf(stderr,"%s [ -b ] [ -d ] [ -i itf ] -s <host> <line> "
      "[ <rmtport> ]\n",name);
    fprintf(stderr,"%s -p [ [ -i ] itf ]\n",name);
    fprintf(stderr,"%s -r [ [ -i ] itf ]\n",name);
    fprintf(stderr,"\n  -b  background\n  -d  debug\n  -i  interface\n");
    fprintf(stderr,"  -l  listen\n  -c  connect\n  -s  connect to switch\n");
    fprintf(stderr,"  -p  create persistent\n  -r  remove persistent\n");
    exit(1);
}


int main(int argc,char **argv)
{
    enum { m_undefined,m_listen,m_connect,m_switch,m_create,m_remove } mode;
    struct sockaddr_in addr;
    const char *name;
    char *end;
    char *host,*line;
    int background,port,itf;
    int s2;
    int c;

    name = argv[0];
    mode = m_undefined;
    background = 0;
    host = line = NULL; /* for gcc */
    itf = -1;
    port = 0; /* for gcc */
    while ((c = getopt(argc,argv,"bdi:lc:prs:")) != EOF)
	switch (c) {
	    case 'b':
		background = 1;
		break;
	    case 'd':
		debug = 1;
		break;
	    case 'i':
		if (itf != -1) usage(name);
		itf = strtoul(optarg,&end,10);
		if (*end) usage(name);
		break;
	    case 'l':
		if (mode != m_undefined) usage(name);
		mode = m_listen;
		break;
	    case 'c':
		if (mode != m_undefined) usage(name);
		mode = m_connect;
		host = optarg;
		break;
	    case 's':
		if (mode != m_undefined) usage(name);
		mode = m_switch;
		host = optarg;
		break;
	    case 'p':
		if (mode != m_undefined) usage(name);
		mode = m_create;
		break;
	    case 'r':
		if (mode != m_undefined) usage(name);
		mode = m_remove;
		break;
	    default:
		usage(name);
        }
    if (mode == m_undefined) usage(name);
    switch (mode) {
	case m_switch:
	    if (argc == optind) usage(name);
	    line = argv[optind++];
	    /* fall through */
	case m_connect:
	    /* fall through */
	case m_listen:
	    if (argc > optind+1) usage(name);
	    port = argc == optind+1 ? atoi(argv[optind]) : -1;
	    break;
	case m_remove:
	    /* fall through */
	case m_create:
	    if (background || debug) usage(name);
	    if (argc != optind && itf != -1) usage(name);
	    if (argc != optind+1) usage(name);
	    itf = strtoul(argv[optind],&end,10);
	    if (*end) usage(name);
	    break;
	default:
	    abort();
    }
    if ((s_krn = socket(PF_ATMSVC,SOCK_DGRAM,0)) < 0) {
	perror("socket");
	return 1;
    }
    if (mode == m_create || mode == m_remove) {
	if (mode == m_remove && itf == -1) itf = 0;
	if (ioctl(s_krn,mode == m_create ? ATMTCP_CREATE : ATMTCP_REMOVE,itf)
	  >= 0) return 0;
	perror(mode == m_create ?
	  "ioctl ATMTCP_CREATE" : "ioctl ATMTCP_REMOVE");
	return 1;
    }
    if ((s_tcp = socket(PF_INET,SOCK_STREAM,0)) < 0) {
	perror("socket");
	return 1;
    }
    if (!port) usage(name);
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port == -1 ?  PORT :  port);
    /*
     * Create atmtcp interface before setting up the TCP connection in order
     * to make the assignment of interface numbers a bit more predictable.
     */
    if ((itf = ioctl(s_krn,SIOCSIFATMTCP,itf)) < 0) {
	perror("ioctl SIOCSIFATMTCP");
	return 1;
    }
    if (mode == m_listen) {
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	if (bind(s_tcp,(struct sockaddr *) &addr,sizeof(addr)) < 0) {
	    perror("bind");
	    return 1;
	}
	printf("Listening on port %d ...\n",ntohs(addr.sin_port));
	if (listen(s_tcp,5) < 0) {
	    perror("listen");
	    return 1;
	}
	if ((s2 = accept(s_tcp,NULL,NULL)) < 0) {
	    perror("accept");
	    return 1;
	}
	(void) close(s_tcp);
	s_tcp = s2;
    }
    else {
	addr.sin_addr.s_addr = text2ip(host,NULL,T2I_NAME | T2I_ERROR);
	if (addr.sin_addr.s_addr == INADDR_NONE) return 1;
	if (connect(s_tcp,(struct sockaddr *) &addr,sizeof(addr)) < 0) {
	    perror("connect");
	    return 1;
	}
	if (mode == m_switch) net_write(line,strlen(line)+1);
    }
    printf("Connected, interface %d\n",itf);
    if (background) {
	pid_t pid;

	pid = fork();
	if (pid < 0) {
	    perror("fork");
	    return 1;
	}
	if (pid) return 0;
    }
    while (1) {
	fd_set set;
	int ret;

	FD_ZERO(&set);
	FD_SET(s_tcp,&set);
	FD_SET(s_krn,&set);
	ret = select(s_tcp > s_krn ? s_tcp+1 : s_krn+1,&set,NULL,NULL,0);
	if (ret < 0) {
	    if (errno != EINTR) perror("select");
	}
	else {
	    if (FD_ISSET(s_tcp,&set)) from_tcp();
	    if (FD_ISSET(s_krn,&set)) to_tcp();
	}
    }
    return 0;
}
