/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   lec.h
*   
* Overview:
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   13-Dec-94  TLR   Created.
*    3-Jan-95  TLR   Updated based on changes to Connection Manager interface.
*   10-Jan-95  TLR   Updated based on team review.
*   10-May-95  TLR   Added Support for Event Logging.
*   27-Jun-95  TLR   Added LEC Management Features.
*
* Dependencies:
*   types.h    - Generally used type definitions.
*   codes.h    - Completion codes.
*   lane_atm.h - ATM specific definitions and types.
*
*/

/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* LEC_EVENT
 *
 *   This enumerated type describes the possible events that the LEC reports
 *   to the registered LEC user.
 *
 *   LEC_NETWORK_AVAILABLE......The LEC has just successfully completed the
 *                              join process and is ready to provide
 *                              communication services with the newly
 *                              connected Emulated LAN.
 *
 *   LEC_NETWORK_UNAVAILABLE....The LEC has just detected a network error
 *                              that is severe enough to cause membership in
 *                              the ELAN to be dropped.  This is either due to
 *                              loss of a critical control connection or loss
 *                              of the ATM link to the adjacent switch.  In
 *                              either case, the LEC has begun to attempt to
 *                              rejoin the ELAN with the parameters supplied by
 *                              the registered user during registration.
 */
#ifndef LEC_H
#define LEC_H
#include "lane_atm.h"

typedef enum
   {
   LEC_NETWORK_AVAILABLE,
   LEC_NETWORK_UNAVAILABLE
   } LEC_EVENT;

/* LAN_TYPE
 *
 *   This type enumerates the supported Emulated LAN types.  With the
 *   exception of LAN_FDDI, these are derived directly from the ATM Forum
 *   specification.
 *
 *   LAN_802_3......ATM Forum standard 802.3/ethernet LAN type.
 *
 *   LAN_802_5......ATM Forum standard 802.5/token ring LAN type.
 *
 *   LAN_FDDI.......Non-ATM Forum standard FDDI LAN type.  This is a possible
 *                  Digital value added frame type
 */
typedef enum
   {
   LAN_802_3,
   LAN_802_5,
   LAN_FDDI
   } LAN_TYPE;

/* LAN_MTU
 *
 *   This enumerated type is used to request a maximum frame size when joining
 *   an emulated LAN.  These values are derived from the ATM Forum
 *   specification.
 *
 *   MTU_UNSPEC.....Unspecified maximum.  Will be assigned by the LES.
 *
 *   MTU_1516.......MTU for ethernet frame sizes.
 *
 *   MTU_4544.......MTU for FDDI frame sizes.
 *
 *   MTU_9234.......MTU for RFC1577 (Classical-IP) frame sizes.
 *
 *   MTU_18190......MTU for Token Ring frame sizes.
 */
typedef enum
   {
   MTU_UNSPEC,
   MTU_1516,
   MTU_4544,
   MTU_9234,
   MTU_18190
   } LAN_MTU;

/* INIT_METHOD
 *
 *   This type is used to indicate to an instance of the LEC how it is to
 *   join an ELAN.
 *
 *   INIT_NORMAL...........Normal mode of initialization.  In this case, the
 *                         LEC first attempts to determine the LECS ATM address
 *                         via ILMI.  If it is unable to do so, it will try the
 *                         well-known LECS ATM address, and then the LECS PVC.
 *
 *   INIT_MANUAL_LECS......Use a manually entered LECS address.  If
 *                         configuration fails using this address, the LEC will
 *                         next attempt to use the well-known LECS address.
 *
 *   INIT_WELL_KNOWN_LECS..Use the well-known LECS address.  If configuration
 *                         fails using this address, try the LECS PVC.
 *
 *   INIT_LECS_PVC.........Use the LECS PVC for configuration.
 *
 *   INIT_MANUAL_LES.......Skip the configuration phase altogether and directly
 *                         contact the LES using the manually entered ATM
 *                         address.
 */
typedef enum
   {
   INIT_NORMAL,
   INIT_MANUAL_LECS,
   INIT_WELL_KNOWN_LECS,
   INIT_LECS_PVC,
   INIT_MANUAL_LES
   } INIT_METHOD;

/*++
* ========================
* = (LEC_EVENT_CALLBACK) =
* ========================
*
* Overview:
*   Notifies the registered user of an important event that has taken place
*   within the LEC.
*
* Arguments:
*   context     - (IN) Context value provided at registration time.
*   event       - (IN) Indication of the event that took place.
*   mtu         - (IN) LAN MTU negotiated during the join.
*   p_elan_name - (IN) Name of ELAN that was joined (if event is
*                      "NETWORK_AVAILABLE").
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/

typedef void (*LEC_EVENT_CALLBACK) (HANDLE      context,
                                    LEC_EVENT   event,
                                    LAN_MTU     mtu,
                                    char       *p_elan_name);


/*++
* ======================
* = (LEC_RCV_CALLBACK) =
* ======================
*
* Overview:
*   Callback that is called when a packet is received on the emulated LAN.
*   A pointer to this routine is supplied at registration time by the user.
*   Note that this callback will not be invoked for packets that were received
*   but filtered by one of the receive packet filters.
*
* Arguments:
*   context     - (IN) Context value supplied at registration time.
*   length      - (IN) Length of the packet in bytes.
*   pp_pkt      - (IO) Pointer to a pointer to the received packet structure.
*                      The returned value of this pointer refers to a buffer
*                      that may be freed by the lower driver.  If the received
*                      packet was copied, the pointer may be returned
*                      unchanged.  If a fresh buffer is to replace the
*                      received one, a pointer to the fresh buffer must be
*                      returned.  If there are no free buffers, a NULL pointer
*                      must be returned.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
*
--*/
typedef void (*LEC_RCV_CALLBACK) (HANDLE  context,
                                  UINT32  length,
                                  void  **pp_pkt);


/*++
* ===========================
* = (LEC_XMT_DONE_CALLBACK) =
* ===========================
*
* Overview:
*   Callback for completed transmissions.  This call returns the context of
*   the registered user and the actual transmitted buffer that may now be
*   freed because the contents of the buffer were copied (via DMA or some
*   such mechanism).
*
* Arguments:
*   context     - (IN) Context value supplied at registration time.
*   p_pkt       - (IN) Pointer to the transmitted packet.
*              
* Returns:
*   None
*
* Preconditions:
*   The packet stored in the buffer is a packet that was previously
*   transmitted by this registered client.
*
* Postconditions:
*   None
*
--*/
typedef void (*LEC_XMT_DONE_CALLBACK) (HANDLE  context,
                                       void   *p_packet);


#endif /* LEC_H */
