/* 
 * Copyright (C) 2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: script_engine.c,v 1.19 2003/03/14 20:47:03 guenter Exp $
 *
 * gate to the spider monkey javascript interpreter
 *
 * provides functions to generate interpreter instances
 * customized for controlling the xine engine and the gxine
 * frontend. this is done by defining a set of global objects
 * and functions so any xine and gxine function can be invoked from
 * (java)scripts.
 *
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>
#include <pthread.h>

#define XP_UNIX
#include "spidermonkey/jsapi.h"

#include "globals.h"
#include "script_engine.h"
#include "utils.h"
#include "mediamarks.h"
#include "playlist.h"
#include "settings.h"
#include "preferences.h"
#include "key_events.h"
#include "log_window.h"
#include "stream_info.h"
#include "open_mrl.h"
#include "gtkxine.h"
#include "server.h"
#include "wizards.h"
#include "snapshot.h"

static JSRuntime *rt=NULL; /* global */

struct se_s {

  JSContext     *cx;
  JSObject      *global;
  
  jsval          rval;
  JSString      *str;

  se_print_cb_t  print_cb;
  void          *print_cb_data;
  se_view_cb_t   view_cb;
  void          *view_cb_data;
};

int se_eval (se_t *se, const gchar *script, 
	     se_print_cb_t print_cb, void *print_cb_data) {

  JSBool     ok;

  se->print_cb      = print_cb;
  se->print_cb_data = print_cb_data;

  ok = JS_EvaluateScript (se->cx, se->global, script, strlen(script),
			 "gxine", 0, &se->rval);
  return ok;
}

int se_eval_view (se_t *se, const gchar *script, 
		  se_view_cb_t view_cb, void *view_cb_data) {

  JSBool     ok;

  se->view_cb      = view_cb;
  se->view_cb_data = view_cb_data;
  se->print_cb     = NULL;

  ok = JS_EvaluateScript (se->cx, se->global, script, strlen(script),
			 "gxine", 0, &se->rval);
  return ok;
}

gchar *se_result_str (se_t *se) {

  if (!JSVAL_IS_STRING (se->rval))
    return NULL;

  se->str = JS_ValueToString (se->cx, se->rval);
  if (verbosity)
    printf("script_engine: script result: %s\n", JS_GetStringBytes (se->str));
  return JS_GetStringBytes (se->str);
}

int se_result_num (se_t *se, int *num) {

  if (!JSVAL_IS_INT (se->rval))
    return 0;

  return JS_ValueToInt32 (se->cx, se->rval, num);
}

/**************************************************************
 *
 * player.controls class & object
 *
 * adapted from http://users.skynet.be/saw/SpiderMonkey.htm
 *
 **************************************************************/

/*
 * properties
 */

#define CONTROLS_PROP_AUDIO_LANGUAGE_COUNT          0
#define CONTROLS_PROP_CURRENT_AUDIO_LANGUAGE        1
#define CONTROLS_PROP_CURRENT_AUDIO_LANGUAGE_INDEX  2
#define CONTROLS_PROP_CURRENT_ITEM                  3
#define CONTROLS_PROP_CURRENT_MARKER                4
#define CONTROLS_PROP_CURRENT_POSITION              5
#define CONTROLS_PROP_CURRENT_POSITION_STRING       6
#define CONTROLS_PROP_CURRENT_POSITION_TIMECODE     7
#define CONTROLS_PROP_IS_AVAILABLE                  8

JSBool controls_JSGetProperty (JSContext *cx, JSObject *obj, jsval id, jsval *vp) {

#ifdef LOG
  printf ("script_engine: get property called\n");
#endif

  if (JSVAL_IS_STRING (id)) {
    JSString *str;
    char     *prop;

#ifdef LOG
    printf ("script_engine: id is a string \n");
#endif

    str = JS_ValueToString (cx, id);
    prop = JS_GetStringBytes (str);

#ifdef LOG
    printf ("script_engine: id = '%s'\n", JS_GetStringBytes (str));
#endif

    if (!strcmp (prop, "audioLanguageCount")) {
      *vp = INT_TO_JSVAL (1);
    } else if (!strcmp (prop, "currentPosition")) {
      *vp = INT_TO_JSVAL (0);
    }
  }
#if 0
  if (JSVAL_IS_INT(id)) 
    {
      Customer *priv = (Customer *) JS_GetPrivate(cx, obj);
      switch(JSVAL_TO_INT(id))
	{
	case name_prop:
	  
	  break;
	case age_prop:
	  *vp = INT_TO_JSVAL(priv->getCustomer()->GetAge());
	  break;
	}
    }
#endif
  return JS_TRUE;
}
  

JSBool controls_JSSetProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp) {


#ifdef LOG
  printf ("script_engine: set property called\n");
#endif

#if 0
  if (JSVAL_IS_INT(id)) {
    Customer *priv = (Customer *) JS_GetPrivate(cx, obj);
    switch(JSVAL_TO_INT(id)) {
    case name_prop:
      break;
    case age_prop:
      priv->getCustomer()->SetAge(JSVAL_TO_INT(*vp));
      break;
    }
  }
#endif
  return JS_TRUE;
}

/* 
 * methods
 */

static JSBool controls_exit (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  gchar *fname;

#ifdef LOG
  printf ("script_engine: controls.exit called\n");
#endif

#if 0
  stop_server();

  printf ("script_engine: server stopped\n");
#endif

  mm_save (); 
  playlist_save (NULL);
  save_key_bindings ();

#ifdef LOG
  printf ("script_engine: saving config...\n");
#endif

  fname = g_strconcat(g_get_home_dir(), "/.gxine/config", NULL);

  gtk_xine_config_save (GTK_XINE (gtx), fname);
  
  g_free (fname);

  gtk_main_quit();

  exit (0);

  return JS_TRUE;
}

static JSBool controls_play (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  char *mrl = NULL;
  int  pos, pos_time;

#ifdef LOG
  printf ("script_engine: controls.play called\n");
#endif

  pos = -1; pos_time = -1;

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, "sorry, xine engine is busy");
    return JS_TRUE;
  }


  switch (argc) {
  case 0:
    break;

  case 1:
    {
      JSString *str;

      if (!JSVAL_IS_STRING (argv[0])) {
	se->print_cb (se->print_cb_data, "error, string value expected");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }

      str = JS_ValueToString (cx, argv[0]);

      mrl = JS_GetStringBytes (str);
#ifdef LOG
      printf ("script_engine: playing '%s' from start\n", mrl);
#endif
    }
    break;

  case 2: 
    {
      if (!JSVAL_IS_INT (argv[0])) {
	se->print_cb (se->print_cb_data, "error, integer value expected (arg 0)");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }
      if (!JSVAL_IS_INT (argv[1])) {
	se->print_cb (se->print_cb_data, "error, integer value expected (arg 1)");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }


      JS_ValueToInt32 (cx, argv[0], &pos);
      JS_ValueToInt32 (cx, argv[1], &pos_time);
      mrl = NULL;
#ifdef LOG
      printf ("script_engine: playing from %d, %d\n", pos, pos_time); 
#endif
    }
    break;

  case 3:
    {
      JSString *str;
      if (!JSVAL_IS_STRING (argv[0])) {
	se->print_cb (se->print_cb_data, "error, string value expected (arg 0)");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }
      if (!JSVAL_IS_INT (argv[1])) {
	se->print_cb (se->print_cb_data, "error, integer value expected (arg 1)");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }
      if (!JSVAL_IS_INT (argv[2])) {
	se->print_cb (se->print_cb_data, "error, integer value expected (arg 2)");
	pthread_mutex_unlock (&engine_lock);
	return JS_TRUE;
      }

      str = JS_ValueToString (cx, argv[0]);
      JS_ValueToInt32 (cx, argv[1], &pos);
      JS_ValueToInt32 (cx, argv[2], &pos_time);
      mrl = JS_GetStringBytes (str);
#ifdef LOG
      printf ("script_engine: playing '%s' from %d, %d\n", mrl, pos, pos_time); 
#endif
    }
    break;

  default:
    se->print_cb (se->print_cb_data, 
		  "error: play() called with too many parameters.\n");
    pthread_mutex_unlock (&engine_lock);
    return JS_TRUE;
  }

  pos *=655 ;

  if (mrl) {

    int item;

    item = playlist_add_mrl (mrl);

    if (pos<0)
      pos = 0;
    if (pos_time<0)
      pos_time = 0;

    playlist_play_from (item, pos, pos_time);

  } else {

    play_item_t *item;

    item = playlist_get_item (playlist_get_list_pos());

    if (item) {
      if (gtk_xine_get_status (GTK_XINE(gtx)) == XINE_STATUS_STOP) {

	if (pos<0)
	  pos = 0;
	if (pos_time<0)
	  pos_time = item->start_time;
	
	playlist_play_from (playlist_get_list_pos(), pos, pos_time);
      } else {
	
	if ( (pos>=0) || (pos_time>=0) ) {
	  
	  if (pos<0)
	    pos = 0;
	  if (pos_time<0)
	    pos_time = 0;
	  gtk_xine_play (GTK_XINE (gtx), pos, pos_time);
	}
      }
    } else
      printf ("script_engine: error, no valid play item available\n");
  } 

  if (gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_SPEED) == XINE_SPEED_PAUSE) {
    gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
  }

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool show_about (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval) {

  display_info ("About gxine",
		"About gxine %s\n\n"
		"A gtk media player frontend for xine\n"
		"(C) 2002-2003 by Guenter Bartsch "
		"and the xine project team\n", VERSION);

  return JS_TRUE;
}

static JSBool get_time (JSContext *cx, JSObject *obj, uintN argc, 
		     jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   pos, pos_time, len;

#ifdef LOG
  printf ("script_engine: get_time () called\n");
#endif

  gtk_xine_get_pos_length (GTK_XINE (gtx), 
			   &pos, &pos_time, &len);

  *rval = INT_TO_JSVAL (pos_time);

  return JS_TRUE;
}

static JSBool js_get_speed (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   speed;

#ifdef LOG
  printf ("script_engine: get_speed () called\n");
#endif

  speed = gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_SPEED);

  *rval = INT_TO_JSVAL (speed);

  return JS_TRUE;
}

static JSBool js_playlist_get_item (JSContext *cx, JSObject *obj, uintN argc, 
				    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   speed;

#ifdef LOG
  printf ("script_engine: playlist_get_item () called\n");
#endif

  speed = gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_SPEED);

  *rval = INT_TO_JSVAL (playlist_get_list_pos());

  return JS_TRUE;
}

static JSBool js_get_volume (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   volume;

#ifdef LOG
  printf ("script_engine: get_volume () called\n");
#endif

  volume = gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_AUDIO_VOLUME);

  *rval = INT_TO_JSVAL (volume);

  return JS_TRUE;
}

static JSBool js_get_zoom (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   zoom;

#ifdef LOG
  printf ("script_engine: get_zoom () called\n");
#endif

  zoom = gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_VO_ZOOM_X);

  *rval = INT_TO_JSVAL (zoom);

  return JS_TRUE;
}

static JSBool controls_pause (JSContext *cx, JSObject *obj, uintN argc, 
			      jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   pause_state=-1;

#ifdef LOG
  printf ("script_engine: controls_pause called\n");
#endif

  if (argc==1) {
    if (!JSVAL_IS_INT (argv[0])) {
      se->print_cb (se->print_cb_data, 
		    "error: play() called with non-int parameter.\n");
      return JS_TRUE; 
    }

    JS_ValueToInt32 (cx, argv[0], &pause_state);
  }

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, 
		  "sorry - engine busy.");
    return JS_TRUE; 
  }

  switch (pause_state) {
  case 0: /* pause off */
    gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
    break;
  case 1: /* pause on */
    gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    break;
  default: /* toggle */
    if (gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_SPEED) != XINE_SPEED_PAUSE)
      gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    else
      gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
  }

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool controls_stop (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

#ifdef LOG
  printf ("script_engine: controls_stop called\n");
#endif

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, 
		  "sorry - engine busy.");
    return JS_TRUE; 
  }

  gtk_xine_stop (GTK_XINE(gtx));

  playlist_logo ();

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_playlist_clear (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: playlist_clear called\n");
#endif

  playlist_clear ();

  return JS_TRUE;
}

static JSBool js_playlist_add (JSContext *cx, JSObject *obj, uintN argc, 
			       jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *mrl;
  int       item;

#ifdef LOG
  printf ("script_engine: playlist_add called\n");
#endif

  if (argc!=1) {
    se->print_cb (se->print_cb_data, 
		  "playlist_add: missing mrl argument");
    return JS_TRUE; 
  }

  if (!JSVAL_IS_STRING (argv[0])) {
    se->print_cb (se->print_cb_data, "playlist_add: string value expected (arg 0)");
    return JS_TRUE;
  }

  str = JS_ValueToString (cx, argv[0]);
  mrl = JS_GetStringBytes (str);

#ifdef LOG
  printf ("playlist_add: mrl=%s\n", mrl);
#endif
  
  item = playlist_add_mrl (mrl);

  *rval = INT_TO_JSVAL (item);

  return JS_TRUE;
}

static JSBool js_playlist_play (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   item;

#ifdef LOG
  printf ("script_engine: paylist_play called\n");
#endif

  if (argc!=1) {
    se->print_cb (se->print_cb_data, 
		  "playlist_play: missing position argument");
    return JS_TRUE; 
  }

  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "paylist_play called with non-int parameter.\n");
    return JS_TRUE; 
  }
  
  JS_ValueToInt32 (cx, argv[0], &item);

  if (item<0)
    item = 0;

  playlist_play (item);
  
  return JS_TRUE;
}

static JSBool js_playlist_show (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: paylist_show called\n");
#endif

  playlist_show ();
  
  return JS_TRUE;
}

static JSBool js_settings_show (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: settings_show called\n");
#endif

  settings_show ();
  
  return JS_TRUE;
}

static JSBool js_preferences_show (JSContext *cx, JSObject *obj, uintN argc, 
				   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: preferences_show called\n");
#endif

  preferences_show ();
  
  return JS_TRUE;
}

static JSBool js_keybindings_show (JSContext *cx, JSObject *obj, uintN argc, 
				   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: keybindings_show called\n");
#endif

  kb_edit_show ();
  
  return JS_TRUE;
}

static JSBool js_mm_add_show (JSContext *cx, JSObject *obj, uintN argc, 
			      jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: mm_add_show called\n");
#endif

  mm_add_show ();
  
  return JS_TRUE;
}

static JSBool js_mm_manage_show (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: mm_manage_show called\n");
#endif

  mm_manage_show ();
  
  return JS_TRUE;
}

static JSBool js_log_show (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: log_show called\n");
#endif

  log_window_show ();
  
  return JS_TRUE;
}

static JSBool js_open_show (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: open_show called\n");
#endif

  file_dialog_show ();
  
  return JS_TRUE;
}

static JSBool js_open_mrl_show (JSContext *cx, JSObject *obj, uintN argc, 
				jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: open_mrl_show called\n");
#endif

  open_mrl_show ();
  
  return JS_TRUE;
}

static JSBool js_stream_info_show (JSContext *cx, JSObject *obj, uintN argc, 
				   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: stream_info_show called\n");
#endif

  stream_info_show ();
  
  return JS_TRUE;
}

static JSBool js_run_wizards (JSContext *cx, JSObject *obj, uintN argc, 
			      jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: run_wizards called\n");
#endif

  run_wizards (TRUE);
  
  return JS_TRUE;
}

static JSBool js_import_mediamarks (JSContext *cx, JSObject *obj, uintN argc, 
				    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: import_mediamarks called\n");
#endif

  mm_import ();
  
  return JS_TRUE;
}

static JSBool js_set_speed (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   speed;

#ifdef LOG
  printf ("script_engine: set_speed called\n");
#endif

  if (argc!=1) {

    se->print_cb (se->print_cb_data, 
		  "error: set_speed() needs one numeric parameter\n");
    return JS_TRUE; 
  }
  
  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "error: set_speed() called with non-int parameter.\n");
    return JS_TRUE; 
  }

  JS_ValueToInt32 (cx, argv[0], &speed);

  if (speed<0)
    speed = 0;

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, 
		  "sorry - engine busy.");
    return JS_TRUE; 
  }

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPEED, speed);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_volume (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   volume;

#ifdef LOG
  printf ("script_engine: set_volume called\n");
#endif

  if (argc!=1) {

    se->print_cb (se->print_cb_data, 
		  "error: set_volume() needs one numeric parameter\n");
    return JS_TRUE; 
  }
  
  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "error: set_volume() called with non-int parameter.\n");
    return JS_TRUE; 
  }

  JS_ValueToInt32 (cx, argv[0], &volume);

  if (volume<0)
    volume = 0;

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, 
		  "sorry - engine busy.");
    return JS_TRUE; 
  }

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_AUDIO_VOLUME, volume);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_zoom (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   zoom;

#ifdef LOG
  printf ("script_engine: set_zoom called\n");
#endif

  if (argc!=1) {

    se->print_cb (se->print_cb_data, 
		  "error: set_zoom() needs one numeric parameter\n");
    return JS_TRUE; 
  }
  
  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "error: set_zoom() called with non-int parameter.\n");
    return JS_TRUE; 
  }

  JS_ValueToInt32 (cx, argv[0], &zoom);

  if (zoom<XINE_VO_ZOOM_MIN)
    zoom = XINE_VO_ZOOM_MIN;
  if (zoom>XINE_VO_ZOOM_MAX)
    zoom = XINE_VO_ZOOM_MAX;

#ifdef LOG
  printf ("script_engine: zoom is %d\n", zoom);
#endif

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, 
		  "sorry - engine busy.");
    return JS_TRUE; 
  }

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_VO_ZOOM_X, zoom);
  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_VO_ZOOM_Y, zoom);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_video_size (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   zoom;

#ifdef LOG
  printf ("script_engine: set_video_size called\n");
#endif

  if (argc!=1) {

    se->print_cb (se->print_cb_data, 
		  "error: set_video_size() needs one numeric parameter\n");
    return JS_TRUE; 
  }
  
  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "error: set_video_size() called with non-int parameter.\n");
    return JS_TRUE; 
  }

  JS_ValueToInt32 (cx, argv[0], &zoom);

  gtk_xine_set_resize_factor (GTK_XINE(gtx), (double) zoom / 100.0);

  return JS_TRUE;
}

static JSBool js_set_sub (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   channel;

#ifdef LOG
  printf ("script_engine: set_sub called\n");
#endif

  if (argc!=1) {

    se->print_cb (se->print_cb_data, 
		  "error: set_sub() needs one numeric parameter\n");
    return JS_TRUE; 
  }
  
  if (!JSVAL_IS_INT (argv[0])) {
    se->print_cb (se->print_cb_data, 
		  "error: set_sub() called with non-int parameter.\n");
    return JS_TRUE; 
  }

  JS_ValueToInt32 (cx, argv[0], &channel);

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_SPU_CHANNEL, channel);

  return JS_TRUE;
}

static JSBool js_set_fullscreen (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   fs;

#ifdef LOG
  printf ("script_engine: set_fullscreen called\n");
#endif

  if (argc==1) {

    if (!JSVAL_IS_INT (argv[0])) {
      se->print_cb (se->print_cb_data, 
		    "error: set_fullscreen() called with non-int parameter.\n");
      return JS_TRUE; 
    }

    JS_ValueToInt32 (cx, argv[0], &fs);

  } else {
    fs = !gtk_xine_is_fullscreen (GTK_XINE(gtx));
  }

  gtk_xine_set_fullscreen (GTK_XINE(gtx), fs);

  return JS_TRUE;
}

static JSBool js_set_deinterlace (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   di;

#ifdef LOG
  printf ("script_engine: set_deinterlace called\n");
#endif

  if (argc==1) {

    if (!JSVAL_IS_INT (argv[0])) {
      se->print_cb (se->print_cb_data, 
		    "error: set_deinterlace() called with non-int parameter.\n");
      return JS_TRUE; 
    }

    JS_ValueToInt32 (cx, argv[0], &di);

  } else {
    di = !gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_VO_DEINTERLACE);
  }

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_VO_DEINTERLACE, di);

  return JS_TRUE;
}

static JSBool js_set_aspect (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   aspect;

#ifdef LOG
  printf ("script_engine: set_aspect called\n");
#endif

  if (argc==1) {

    if (!JSVAL_IS_INT (argv[0])) {
      se->print_cb (se->print_cb_data, 
		    "error: set_aspect() called with non-int parameter.\n");
      return JS_TRUE; 
    }

    JS_ValueToInt32 (cx, argv[0], &aspect);

  } else {
    aspect = (gtk_xine_get_param (GTK_XINE(gtx), XINE_PARAM_VO_ASPECT_RATIO) + 1)%4;
  }

  gtk_xine_set_param (GTK_XINE(gtx), XINE_PARAM_VO_ASPECT_RATIO, aspect);

  return JS_TRUE;
}

static JSBool js_set_vis (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *id = NULL;
  
#ifdef LOG
  printf ("script_engine: js_set_vis called\n");
#endif

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, "sorry, xine engine is busy");
    return JS_TRUE;
  }

  if (argc!=1) {
    se->print_cb (se->print_cb_data, 
		  "error: set_vis() string argument expected.\n");
    pthread_mutex_unlock (&engine_lock);
    return JS_TRUE;
  }

  if (!JSVAL_IS_STRING (argv[0])) {
    se->print_cb (se->print_cb_data, "error, set_vis() string argument expected");
    pthread_mutex_unlock (&engine_lock);
    return JS_TRUE;
  }

  str = JS_ValueToString (cx, argv[0]);

  id = JS_GetStringBytes (str);

  if (!strcmp (id, "none"))
    gtk_xine_set_vis (GTK_XINE (gtx), NULL);
  else
    gtk_xine_set_vis (GTK_XINE (gtx), id);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_snapshot (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *fname = NULL;
  
#ifdef LOG
  printf ("script_engine: js_snapshot called\n");
#endif

  if (pthread_mutex_trylock (&engine_lock)) {
    se->print_cb (se->print_cb_data, "sorry, xine engine is busy");
    return JS_TRUE;
  }

  if (argc==1) {

    if (!JSVAL_IS_STRING (argv[0])) {
      se->print_cb (se->print_cb_data, "error, snapshot() string argument expected");
      pthread_mutex_unlock (&engine_lock);
      return JS_TRUE;
    }

    str = JS_ValueToString (cx, argv[0]);

    fname = JS_GetStringBytes (str);

  } 

  make_snapshot (fname);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_input_up (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_up called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_UP;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_down (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_down called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_DOWN;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_left (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_left called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_LEFT;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_right (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_right called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_RIGHT;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_select (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_select called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_SELECT;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_menu1 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_menu1 called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU1;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_menu2 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_menu2 called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU2;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_menu3 (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_menu3 called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_MENU3;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_previous (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_previous called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_PREVIOUS;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

static JSBool js_input_next (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  xine_event_t event;

#ifdef LOG
  printf ("script_engine: input_next called\n");
#endif

  event.type = 0;
  event.data = NULL;
  event.data_length = 0;

  event.type = XINE_EVENT_INPUT_NEXT;

  gtk_xine_event_send (GTK_XINE(gtx), &event);

  return JS_TRUE;
}

#if 0
static JSBool dummy (JSContext *cx, JSObject *obj, uintN argc, 
		     jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

#ifdef LOG
  printf ("script_engine: dummy called\n");
#endif

  /*
    argc

    JSString *str;
    str = JS_ValueToString (cx, argv[0]);
    mrl = JS_GetStringBytes (str);

    JS_ValueToInt32 (cx, argv[0], &pos);

    se->print_cb (se->print_cb_data, 
		  "error: \n");

    JSCustomer *p = JS_GetPrivate(cx, obj);
    *rval = INT_TO_JSVAL(5);
  */

  return JS_TRUE;
}
#endif

static JSBool show_help (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  se->print_cb (se->print_cb_data,
		"\navailable commands (gxine javascript interface):\n"
		"\nengine control:\n"
		"play ([mrl] [, pos, time]); /* time in msec */\n"
		"open (mrl); \n"
		"pause ([bool]); \n"
		"stop (); \n"
		"exit (); \n"
		"\n"
		"dialogs:\n"
		"about_show ();\n"
		"playlist_show ();\n"
		"settings_show ();\n"
		"preferences_show ();\n"
		"keybindings_show ();\n"
		"run_wizards ();\n"
		"import_mediamarks ();\n"
		"\n"
		"playlist cotrol:\n"
		"playlist_clear();\n"
		"playlist_add (mrl);\n"
		"playlist_play (int);\n"
		"playlist_get_item ();\n"
		"\n"
		"status:\n"
		"get_time(); \n"
		"\n"
		"properties:\n"
		"set_fullscreen([bool]);\n"
		"set_aspect([int]);\n"
		"set_speed(int);\n"
		"get_speed();\n"
		"set_volume(int);\n"
		"get_volume();\n"
		"set_zoom(int);\n"
		"get_zoom();\n"
		"set_video_size(int);\n"
		"set_sub(int);\n"
		"set_deinterlace([bool]);\n"
		"\ngenerate input events (dvd menus)\n"
		"input_up ();\n"
		"input_down ();\n"
		"input_left ();\n"
		"input_right ();\n"
		"input_select ();\n"
		"input_menu1 ();\n"
		"input_menu2 ();\n"
		"input_menu3 ();\n"
		"input_previous ();\n"
		"input_next ();\n");
  /*
   * set_video_size (int);
   * set_aspect_ratio (int);
   * set_deinterlace (int);
   * set_volume (int);
   * set_audio_channel (int);
   * set_spu_chanel (int);
   *
   * int get_pos ();
   * int get_time ();
   * int get_length ();
   *
   * show_playlist (0/1);
   * show_settings (0/1);
   * run_wizards (bool requested);
   * show_stream_info (0/1);
   *
   */
  return JS_TRUE;
}

static void controls_JSDestructor (JSContext *cx, JSObject *obj) {
#ifdef LOG
  printf ("script_engine: controls destructor called\n");
#endif
}

  
static JSClass controls_JSClass = 
  {
    "controls", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub,
    controls_JSGetProperty, controls_JSSetProperty,
    JS_EnumerateStub, JS_ResolveStub, 
    JS_ConvertStub, controls_JSDestructor
  };

static JSObject *create_controls_object (se_t *se) {

  JSObject *controls_obj;

  controls_obj = JS_DefineObject (se->cx, se->global, "controls",
				  &controls_JSClass, NULL, 0);

  JS_DefineFunction (se->cx, controls_obj, "play", controls_play, 0, 0);  

  return controls_obj;
}

/*
 * view object
 */


JSBool view_JSGetProperty (JSContext *cx, JSObject *obj, jsval id, jsval *vp) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

#ifdef LOG
  printf ("script_engine: view get property called\n");
#endif

  if (JSVAL_IS_STRING (id)) {
    JSString *str;
    char     *prop;

    str = JS_ValueToString (cx, id);
    prop = JS_GetStringBytes (str);

#ifdef LOG
    printf ("script_engine: looking for view property '%s'\n", prop);
#endif

    if (se->view_cb) 
      *vp = INT_TO_JSVAL (se->view_cb (se->view_cb_data, prop));
  }
  return JS_TRUE;
}
  

JSBool view_JSSetProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp) {


#ifdef LOG
  printf ("script_engine: set property called\n");
#endif

#if 0
  if (JSVAL_IS_INT(id)) {
    Customer *priv = (Customer *) JS_GetPrivate(cx, obj);
    switch(JSVAL_TO_INT(id)) {
    case name_prop:
      break;
    case age_prop:
      priv->getCustomer()->SetAge(JSVAL_TO_INT(*vp));
      break;
    }
  }
#endif
  return JS_TRUE;
}

static void view_JSDestructor (JSContext *cx, JSObject *obj) {
#ifdef LOG
  printf ("script_engine: view destructor called\n");
#endif
}

static JSClass view_JSClass = 
  {
    "views", JSCLASS_HAS_PRIVATE,
    JS_PropertyStub, JS_PropertyStub,
    view_JSGetProperty, view_JSSetProperty,
    JS_EnumerateStub, JS_ResolveStub, 
    JS_ConvertStub, view_JSDestructor
  };

static JSObject *create_view_object (se_t *se) {

  JSObject *view_obj;

  view_obj = JS_DefineObject (se->cx, se->global, "view",
				  &view_JSClass, NULL, 0);

  return view_obj;
}

/*
 * skins object hierarchy
 *
 *
 * theme
 *   view
 *     player
 *       controls
 *       settings
 *     video
 *     subview
 *       buttongroup
 *         buttonelement
 *       button
 *       subview
 *     
 */

static void error_reporter (JSContext *cx, const char *message,
			    JSErrorReport *report) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

#ifdef LOG
  printf ("scripte_engine: JSError '%s'\n", message);
#endif

  if (se->print_cb) 
    se->print_cb (se->print_cb_data, (char *) message);

}

se_t *se_new (void) {

  se_t    *se;
  static JSClass global_class = {
    "global",0,
    JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,JS_PropertyStub,
    JS_EnumerateStub,JS_ResolveStub,JS_ConvertStub,JS_FinalizeStub
  };

  /*
   * create a js engine instance 
   */

  if (!rt) 
    rt = JS_NewRuntime(0x100000);

  se = malloc (sizeof (se_t));

  se->cx       = JS_NewContext (rt, 0x1000);
  se->global   = JS_NewObject (se->cx, &global_class, NULL, NULL);
  JS_InitStandardClasses (se->cx, se->global);
  JS_SetErrorReporter (se->cx, error_reporter);
  JS_SetContextPrivate (se->cx, se);

  /*
   * create object oriented environment
   */
  
  create_controls_object (se) ;
  create_view_object (se) ;

  /*
   * hook up global xine functions for convenience
   */
  
  JS_DefineFunction (se->cx, se->global, "play", controls_play, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "exit", controls_exit, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "help", show_help, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "get_time", get_time, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "pause", controls_pause, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "stop", controls_stop, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "snapshot", js_snapshot, 0, 0);  

  JS_DefineFunction (se->cx, se->global, "about_show", show_about, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "playlist_show", js_playlist_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "settings_show", js_settings_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "open_show", js_open_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "open_mrl_show", js_open_mrl_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "preferences_show", js_preferences_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "keybindings_show", js_keybindings_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "mm_add_show", js_mm_add_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "mm_manage_show", js_mm_manage_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "log_show", js_log_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "stream_info_show", js_stream_info_show, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "run_wizards", js_run_wizards, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "import_mediamarks", js_import_mediamarks, 0, 0);  

  JS_DefineFunction (se->cx, se->global, "playlist_clear", js_playlist_clear, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "playlist_add", js_playlist_add, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "playlist_play", js_playlist_play, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "playlist_get_item", js_playlist_get_item, 0, 0);  

  JS_DefineFunction (se->cx, se->global, "set_speed", js_set_speed, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "get_speed", js_get_speed, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_volume", js_set_volume, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "get_volume", js_get_volume, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_zoom", js_set_zoom, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "get_zoom", js_get_zoom, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_video_size", js_set_video_size, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_fullscreen", js_set_fullscreen, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_aspect", js_set_aspect, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_vis", js_set_vis, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_sub", js_set_sub, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "set_deinterlace", js_set_deinterlace, 0, 0);  

  JS_DefineFunction (se->cx, se->global, "input_up", js_input_up, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_down", js_input_down, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_left", js_input_left, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_right", js_input_right, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_select", js_input_select, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_menu1", js_input_menu1, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_menu2", js_input_menu2, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_menu3", js_input_menu3, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_previous", js_input_previous, 0, 0);  
  JS_DefineFunction (se->cx, se->global, "input_next", js_input_next, 0, 0);  

  return se;
}
