/*

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/
module derelict.util.exception;

private
{
    import derelict.util.compat;
}

/**
* Base class for all exceptions thrown by Derelict packages.
*/
class DerelictException : Exception
{
public:
    this(string msg)
    {
        super(msg);
    }
}

/**
* Helper struct to facilitate throwing a single SharedLibException after failing
* to load a library using multiple names.
*/
struct FailedSharedLib
{
    string name;
    string reason;
}

/**
* This exception is thrown when a shared library cannot be loaded
* because it is either missing or not on the system path.
*/
class SharedLibLoadException : DerelictException
{
public:
    static void throwNew(in char[][] libNames, in char[][] reasons)
    {
        string msg = "Failed to load one or more shared libraries:";
        foreach(i, n; libNames)
        {
            msg ~= "\n\t" ~ n ~ " - ";
            if(i < reasons.length)
                msg ~= reasons[i];
            else
                msg ~= "Unknown";
        }
        throw new SharedLibLoadException(msg);
    }

    this(string msg)
    {
        super(msg);
        _sharedLibName = "";
    }

    this(string msg, string sharedLibName)
    {
        super(msg);
        _sharedLibName = sharedLibName;
    }

    string sharedLibName()
    {
        return _sharedLibName;
    }

private:
    string _sharedLibName;
}

/**
* This exception is thrown when a symbol cannot be loaded from a shared library,
* either because it does not exist in the library or because the library is corrupt.
*/
class SymbolLoadException : DerelictException
{
public:
    this(string msg)
    {
        super(msg);
    }

    this(string sharedLibName, string symbolName)
    {
        super("Failed to load symbol " ~ symbolName ~ " from shared library " ~ sharedLibName);
        _symbolName = symbolName;
    }

    string symbolName()
    {
        return _symbolName;
    }

private:
    string _symbolName;
}

alias SymbolLoadException SharedLibProcLoadException;

//******************************************************************************

/**
* The MissingSymbolCallback allows the app to handle the case of missing symbols. By default,
* a SharedLibSymbolLoadException is thrown. However, if a the app determines that
* particular symbol is not needed, the callback can return true. This will cause
* the shared library to continue loading. Returning false will cause the exception
* to be thrown.
*/
alias bool function(string libName, string symbolName) MissingSymbolCallback;
alias MissingSymbolCallback MissingProcCallback;

private MissingSymbolCallback missingSymbolCallback;

public void Derelict_HandleMissingSymbol(string libName, string symbolName)
{
    bool result = false;
    if(missingSymbolCallback !is null)
        result = missingSymbolCallback(libName, symbolName);
    if(!result)
        throw new SymbolLoadException(libName, symbolName);
}
alias Derelict_HandleMissingSymbol Derelict_HandleMissingProc;

void Derelict_SetMissingSymbolCallback(MissingSymbolCallback callback)
{
    missingSymbolCallback = callback;
}
alias Derelict_SetMissingSymbolCallback Derelict_SetMissingProcCallback;