/*
** Copyright 2001 Double Precision, Inc.
** See COPYING for distribution information.
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include "superfloppy.h"
#include "sysconfdir.h"
#include "bindir.h"

#include	<sys/types.h>
#if HAVE_SYS_WAIT_H
#include	<sys/wait.h>
#endif
#ifndef WEXITSTATUS
#define	WEXITSTATUS(stat_val)	((unsigned)(stat_val) >> 8)
#endif
#ifndef	WIFEXITED
#define	WIFEXITED(stat_val)	(((stat_val) & 255) == 0)
#endif

#define FLOPPY BINDIR "/floppy"

static const char rcsid[]="$Id: floppygtk.c,v 1.3 2001/08/05 15:44:35 mrsam Exp $";

GtkWidget *window;	/* Main window */
GtkWidget *window_vbox;	/* Vbox for menu bar, and dialog */
GtkWidget *main_menu;
GtkWidget *table;	/* The grid for our widgets are */

GtkWidget *floppypromptLabel;	/* Floppy: prompt */
GtkWidget *capacitypromptLabel; /* Format size: prompt */
GtkWidget *formattypepromptLabel; /* Format type: prompt */

GtkWidget *verifyoption;	/* Verify format toggle button */

GtkWidget *gobutton;
GtkWidget *formatprogress;	/* Format progress indicator */
GtkObject *formatprogressAdjustment;
GtkWidget *formatprogressLabel;
static char *drivename;

/*
** Calculate the approximate size of a text string rendered in a widget.
*/
static void getsize(GtkWidget *w, const char *str, int *ww, int *h)
{
	guint width;
	guint lbearing;
	guint rbearing;
	guint ascent;
	guint descent;

	if (!w || !w->style || !w->style->font)
		return;

	gdk_string_extents (w->style->font, str,
			    &lbearing, &rbearing, &width,
			    &ascent, &descent);

	width += 10;

	ascent += descent;
	ascent += 10;

	*ww=width;
	*h=ascent;
}

/*
** Initialize a combo widget's size based on the size of the dropdown
** list items.
*/

static void set_dyn_combo(GtkWidget *fld, GList *strlist, int *wptr)
{
	int w, h;

	if (strlist)
		gtk_combo_set_popdown_strings(GTK_COMBO(fld), strlist);
	gtk_widget_realize(fld);

	w=h=0;

	while (strlist)
	{
		int ww, hh;

		getsize(fld, (const char *)strlist->data, &ww, &hh);

		if (ww > w) w=ww;
		if (hh > h) h=hh;
		strlist=strlist->next;
	}

	gtk_widget_show(fld);

	w += (int)GTK_COMBO(fld)->button->allocation.width
		+ 10;

	gtk_widget_set_usize (fld, w, h);
	if (wptr)
		*wptr=w;

}

/*
** An error popup dialog.
*/

void popup(const char *title, const char *message)
{
	GtkWidget *dialog, *label, *dismiss;
   
	dialog = gtk_dialog_new();

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 10);
	label = gtk_label_new (message);
	dismiss = gtk_button_new_with_label(_("Dismiss"));
   
	gtk_signal_connect_object (GTK_OBJECT (dismiss), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   (gpointer)dialog);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
			   dismiss);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
			   label);

	gtk_window_set_title(GTK_WINDOW(dialog), title);
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_transient_for(GTK_WINDOW(dialog),
				     GTK_WINDOW(window));
	gtk_widget_show_all (dialog);
}

void errmsg(const char *message)
{
	popup(_("ERROR"), message);
}

static void (*warning_callback_func)(int);

static void warning_yes(GtkWidget *w)
{
	gtk_widget_destroy(w);
	(*warning_callback_func)(TRUE);
}

static void warning_no(GtkWidget *w)
{
	gtk_widget_destroy(w);
	(*warning_callback_func)(FALSE);
}

gint delete_dialog(GtkWidget *w)
{
	(*warning_callback_func)(FALSE);
	return (FALSE);
}

void warning(const char *message, const char *yesmsg, const char *nomsg,
	     void (*callback_func)(int))
{
	GtkWidget *dialog, *label, *box, *yes, *no;
   
	warning_callback_func=callback_func;

	dialog = gtk_dialog_new();

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 10);
	label = gtk_label_new (message);

	box=gtk_hbox_new(TRUE, 50);
	yes = gtk_button_new_with_label(yesmsg);
	no = gtk_button_new_with_label(nomsg);

	gtk_box_pack_start(GTK_BOX(box), yes, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), no, FALSE, FALSE, 0);

	gtk_signal_connect_object (GTK_OBJECT (yes), "clicked",
				   GTK_SIGNAL_FUNC (warning_yes),
				   (gpointer)dialog);

	gtk_signal_connect_object (GTK_OBJECT (no), "clicked",
				   GTK_SIGNAL_FUNC (warning_no),
				   (gpointer)dialog);
	gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
			   GTK_SIGNAL_FUNC(delete_dialog), NULL);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
			   box);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
			   label);

	gtk_window_set_title(GTK_WINDOW(dialog), _("WARNING"));
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_transient_for(GTK_WINDOW(dialog),
				     GTK_WINDOW(window));
	gtk_widget_show_all (dialog);
}


/****************************************************************************
 **
 ** Functions for running the 'floppy' binary in the background
 **
 ****************************************************************************/

static char floppy_stderr_buf[1024];	/* stderr from floppy */
static int floppy_stderr_bufptr;
static int floppy_stderr_fd;	/* fd from floppy stderr */
static gint floppy_stderr_tag;

static char floppy_stdout_linebuf[1024];	/* stdout from floppy */
static int floppy_stdout_bufptr;
static int floppy_stdout_fd;	/* fd from floppy stdout */
static void (*floppy_stdout_func)(const char *);
			/* Callback for floppy stdout */
static pid_t floppy_pid= -1;	/* Floppy pid */
static int floppy_stdout_tag;

static void (*floppy_completed_func)(int);

static void stderr_callback(gpointer, gint, GdkInputCondition);
static void stdout_callback(gpointer, gint, GdkInputCondition);

static int do_floppy_start(char **argv,
			   void (*stdout_func)(const char *),
			   void (*completed_func)(int));

int floppy_start(char **argv,
		 void (*stdout_func)(const char *),
		 void (*completed_func)(int))
{
	if (do_floppy_start(argv, stdout_func, completed_func) == 0)
		return (0);
	floppy_stderr_buf[0]=0;
	strncat(floppy_stderr_buf,
		errno == EAGAIN ?
		_("floppygtk is busy right now, try again later."):
		g_strerror(errno),
		sizeof(floppy_stderr_buf)-1);
	errmsg(floppy_stderr_buf);
	return (-1);
}

static int do_floppy_start(char **argv,
		 void (*stdout_func)(const char *),
		 void (*completed_func)(int))
{
	int pipe1[2];
	int pipe2[2];

	if (floppy_pid >= 0)	/* Something else is cooking */
	{
		errno=EAGAIN;
		return (-1);
	}

	signal(SIGCHLD, SIG_DFL);

	if (pipe(pipe1) < 0)
		return (-1);

	if (pipe(pipe2) < 0)
	{
		close(pipe1[0]);
		close(pipe1[1]);
		return (-1);
	}

	floppy_pid=fork();

	if (floppy_pid < 0)
	{
		close(pipe2[0]);
		close(pipe2[1]);
		close(pipe1[0]);
		close(pipe1[1]);
		return (-1);
	}

	if (floppy_pid == 0)
	{
		close(1);
		if (dup(pipe1[1]) != 1)
			exit(1);
		close(2);
		if (dup(pipe2[1]) != 2)
			exit(1);
		close(pipe1[0]);
		close(pipe1[1]);
		close(pipe2[0]);
		close(pipe2[1]);
		execv(FLOPPY, argv);
		perror(FLOPPY);
		exit(1);
	}

	close(pipe1[1]);
	close(pipe2[1]);
	floppy_stdout_fd=pipe1[0];
	floppy_stderr_fd=pipe2[0];
	floppy_stdout_func=stdout_func;
	floppy_completed_func=completed_func;
	floppy_stderr_bufptr=0;
	floppy_stdout_bufptr=0;

	floppy_stderr_tag=gdk_input_add(floppy_stderr_fd, GDK_INPUT_READ,
					&stderr_callback, NULL);
	floppy_stdout_tag=gdk_input_add(floppy_stdout_fd, GDK_INPUT_READ,
					&stdout_callback, NULL);
	return (0);
}

static void chk_floppy_completed();

/*
** Callback - read from floppy child proc stderr.  We collect up to the first
** 1024 bytes.
*/

static void stderr_callback(gpointer dummy1, gint dummy2,
			    GdkInputCondition dummy3)
{
	char buffer[BUFSIZ];

	int n=read(floppy_stderr_fd, buffer, sizeof(buffer));

	if (n > 0)
	{
		if (n > sizeof(floppy_stderr_buf)-1-floppy_stderr_bufptr)
			n=sizeof(floppy_stderr_buf)-1-floppy_stderr_bufptr;
		if (n > 0)
			memcpy(floppy_stderr_buf + floppy_stderr_bufptr,
			       buffer, n);
		floppy_stderr_bufptr += n;
		return;
	}
	floppy_stderr_buf[floppy_stderr_bufptr]=0;
	gdk_input_remove(floppy_stderr_tag);
	close(floppy_stderr_fd);
	floppy_stderr_fd = -1;
	chk_floppy_completed();
}

/*
** Callback - read from floppy child stdout.  We read lines of input and
** call a callback function to process each line of input.
*/

static void stdout_callback(gpointer dummy1, gint dummy2,
			    GdkInputCondition dummy3)
{
	char buffer[BUFSIZ];

	int n=read(floppy_stdout_fd, buffer, sizeof(buffer));

	if (n > 0)
	{
		const char *p=buffer;

		while (n > 0)
		{
			if (*p == '\n')
			{
				floppy_stdout_linebuf[floppy_stdout_bufptr]=0;
				(*floppy_stdout_func)(floppy_stdout_linebuf);
				floppy_stdout_bufptr=0;
			}
			else
			{
				if (floppy_stdout_bufptr <
				    sizeof(floppy_stdout_linebuf)-1)
				{
					floppy_stdout_linebuf[floppy_stdout_bufptr]
						=*p;
					++floppy_stdout_bufptr;
				}
			}
			++p;
			--n;
		}
		return;
	}
	floppy_stdout_linebuf[floppy_stdout_bufptr]=0;
	if (floppy_stdout_bufptr)
		(*floppy_stdout_func)(floppy_stdout_linebuf);

	gdk_input_remove(floppy_stdout_tag);
	close(floppy_stdout_fd);
	floppy_stdout_fd = -1;
	chk_floppy_completed();
}

/*
** Both stdout and stderr are now closed.  Check the child process exit code.
** Display an error for a non-zero exit status.
*/

static void chk_floppy_completed()
{
	int waitstat;
	pid_t pid2;

	if (floppy_stderr_fd >= 0 || floppy_stdout_fd >= 0)
		return;	/* Something is still open */

	/* Check exit status of child process */

	while ((pid2=wait(&waitstat)) != floppy_pid)
	{
		if (pid2 < 0 && errno == ECHILD)
			break;
	}

	if (pid2 == floppy_pid && WIFEXITED(waitstat)
	    && WEXITSTATUS(waitstat) == 0)
	{
		floppy_pid= -1;
		(*floppy_completed_func)(0);
	}
	else
	{
		errmsg(floppy_stderr_buf);
		floppy_pid= -1;
		(*floppy_completed_func)(-1);
	}
}

/*
** Drop down list of available floppy devices.
**
** Run "floppy --showrc --noprompt", save the results in a combo box.
*/

GtkWidget *floppylistComboHbox; /*
				** Hbox used to prevent combo box from being
				** stretch by GtkTable
				*/
GtkWidget *floppylistComboHboxFiller;	/* The filler label widget */

GtkWidget *floppylistCombo;	/* Drop-down list of available floppy drives */
GList *floppylistGlist=NULL;	/* Current glist */
guint floppylistComboSig;

GList *nextglist=NULL;	/* Next glist being created */

static void readfloppylist(const char *);
static void completedfloppylist(int);

void createfloppylist()
{
	char *argv[4];

	argv[0]=FLOPPY;
	argv[1]="--showrc";
	argv[2]="--noprompt";
	argv[3]=0;


	if (floppy_start(argv, readfloppylist, completedfloppylist) == 0)
	{
		while (nextglist)
		{
			gpointer *p=nextglist->data;

			nextglist=g_list_remove(nextglist, p);
			g_free(p);
		}
	}
}

static void readfloppylist(const char *s)
{
	char *p=g_strdup(s);

	if (p)
		nextglist=g_list_append(nextglist, p);
}

static void createcapacitylist();
static void drivechangeoff();
static void drivechangeon();

static void completedfloppylist(int rc)
{
	GList *listptr;
	char *first_drive="";
	char *p;

	/* If error, dump partial results */

	if (rc)
	{
		while (nextglist)
		{
			gpointer *p=nextglist->data;
			nextglist=g_list_remove(nextglist, p);
		}
	}

	for (listptr=g_list_first(nextglist);
	     listptr; listptr=g_list_next(listptr))
	{
		if (!*first_drive)
			first_drive=listptr->data;

		if ( drivename &&
		     g_strcasecmp( (const char *)listptr->data,
				   drivename) == 0)
			break;
	}

	drivechangeoff(); /* Disable createcapacitylist() */

	set_dyn_combo(floppylistCombo, nextglist, NULL);
	while (floppylistGlist)
	{
		gpointer *p=floppylistGlist->data;
		floppylistGlist=g_list_remove(floppylistGlist, p);
		g_free(p);
	}

	floppylistGlist=nextglist;
	nextglist=NULL;


	p=listptr ? listptr->data:first_drive;

	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(floppylistCombo)->entry), p);

	p=gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(floppylistCombo)
					      ->entry), 0, -1);

	if (p)
		g_free(p);

	drivechangeon();
	createcapacitylist();
}

/*
** Drop down list of available formattable capacities.
**
** Run "floppy --capacity --noprompt floppy:", save the results in a combo box.
*/

/* Do this when the drive name changed ... */

static void drivechanged(GtkWidget *w,
			 gpointer *data)
{
	createcapacitylist();
}

static void drivechangeon()
{
	floppylistComboSig=
		gtk_signal_connect(GTK_OBJECT(GTK_COMBO(floppylistCombo)
					      ->entry),
				   "changed",
				   GTK_SIGNAL_FUNC(drivechanged),
				   NULL);
}

void drivechangeoff()
{
	gtk_signal_disconnect(GTK_OBJECT(GTK_COMBO(floppylistCombo)->entry),
			      floppylistComboSig);
}

/* ... or the popdown list is closed */

gint drivepopdownclosed(GtkWidget *w,
			GdkEvent  *e,
			gpointer   d)
{
	createcapacitylist();
	return (FALSE);
}

struct fmtsizelist {
	struct fmtsizelist *next;

	char *floppysize;	/* From floppy */
	char *shownsize;	/* What's shown in the popdown list */
	int flag;
} ;

GtkWidget *capacitylistComboHbox; /*
				  ** Hbox used to prevent combo box from being
				  ** stretch by GtkTable
				  */
GtkWidget *capacitylistComboHboxFiller;	/* The filler label widget */

GtkWidget *capacitylistCombo;

struct fmtsizelist *capacitylistCurrent=NULL;	/* Current list being shown */
GList *capacitylistGlist=NULL;
struct fmtsizelist *capacitylistNext=NULL;	/* Next list being built */

static void readcapacitylist(const char *);
static void completedcapacitylist(int);

static void createcapacitylist()
{
	char *argv[5];
	char *new_drivename;

	if (floppy_pid >= 0)
		return;	/* Something else is cooking */

	new_drivename=
		gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(floppylistCombo)
						    ->entry), 0, -1);

	if (!new_drivename)
		return;

	if (drivename)
		g_free(drivename);
	drivename=new_drivename;

	argv[0]=FLOPPY;
	argv[1]="--capacity";
	argv[2]="--noprompt";
	argv[3]=drivename;
	argv[4]=0;

	if (!drivename || !*drivename ||
	    floppy_start(argv, readcapacitylist, completedcapacitylist) == 0)
	{
		struct fmtsizelist *p;

		while ((p=capacitylistNext) != 0)
		{
			capacitylistNext=p->next;
			if (p->floppysize)
				g_free(p->floppysize);

			if (p->shownsize)
				g_free(p->shownsize);
			g_free(p);
		}

		if (!drivename || !*drivename)
			completedcapacitylist(0);
	}
}

static void readcapacitylist(const char *c)
{
	struct fmtsizelist *p=g_malloc(sizeof(struct fmtsizelist));
	struct fmtsizelist **pp;

	memset(p, 0, sizeof(*p));
	p->floppysize=g_strdup(c);
	for (pp= &capacitylistNext; *pp; pp= &(*pp)->next)
		;
	*pp=p;
}

static void fixcapacity(struct fmtsizelist *);

static int parse_size(const char *sizestr,
		      unsigned *t, unsigned *s, unsigned *ss)
{
	*t=*s=*ss=0;

	if (sscanf(sizestr, "%ux%ux%u", t, s, ss) == 3)
		return (0);
	return (-1);
}

static void completedcapacitylist(int rc)
{
	struct fmtsizelist *p, *q;
	GList *newcapacitylist;

	static int first_popup=0;

	if (rc)
		return;

	/*
	** Display the formattable capacities in a nice friendly fashion.
	*/

	for (p=capacitylistNext; p; p=p->next)
	{
		unsigned t=0, s=0, ss=0;

		if (parse_size(p->floppysize, &t, &s, &ss) == 0)
			p->shownsize=g_strdup(fmtsize( (unsigned long)t * s,
						       ss));
		else
			p->shownsize=g_strdup(p->floppysize);
	}

	/*
	** If we somehow ended up with multiple formats of the same capacity,
	** make things more verbose.
	*/

	for (p=capacitylistNext; p; p=p->next)
	{
		if (p->flag)
			continue;	/* Kilroy was here */

		for (q=p->next; q; q=q->next)
			if (strcmp(p->shownsize, q->shownsize) == 0)
				break;

		if (q == NULL)
			continue;

		for (q=p->next; q; q=q->next)
			if (strcmp(p->shownsize, q->shownsize) == 0)
				fixcapacity(q);
		fixcapacity(p);
	}

	newcapacitylist=NULL;

	for (p=capacitylistNext; p; p=p->next)
	{
		newcapacitylist=g_list_append(newcapacitylist, p->shownsize);
	}

	set_dyn_combo(capacitylistCombo, newcapacitylist, NULL);
	gtk_widget_show(capacitylistCombo);
	gtk_widget_show(window);
	set_dyn_combo(capacitylistCombo, newcapacitylist, NULL);

	if (!first_popup)	/* First time, redraw */
	{
		char *save_selection=
			gtk_editable_get_chars(GTK_EDITABLE
					       (GTK_COMBO(floppylistCombo)
						->entry), 0, -1);

		drivechangeoff();
		first_popup=1;
		set_dyn_combo(floppylistCombo, floppylistGlist, NULL);
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(floppylistCombo)
					     ->entry), save_selection);
		drivechangeon();
	}

	while (capacitylistGlist)
	{
		gpointer *p=capacitylistGlist->data;

		capacitylistGlist=g_list_remove(capacitylistGlist, p);
	}

	capacitylistGlist=newcapacitylist;

	while (capacitylistCurrent)
	{
		struct fmtsizelist *p=capacitylistCurrent->next;

		g_free(capacitylistCurrent->shownsize);
		g_free(capacitylistCurrent->floppysize);
		g_free(capacitylistCurrent);
		capacitylistCurrent=p;
	}
	capacitylistCurrent=capacitylistNext;
	capacitylistNext=NULL;
}

static void fixcapacity(struct fmtsizelist *p)
{
	const char *fmt=_("%s (%s)");

	char *s=g_malloc(strlen(p->shownsize)+strlen(p->floppysize)
			 + strlen(fmt));

	sprintf(s, fmt, p->shownsize, p->floppysize);

	g_free(p->shownsize);
	p->shownsize=s;
	p->flag=1;
}

/*
** Format type.
*/
GtkWidget *formattypeComboHbox; /*
				** Hbox used to prevent combo box from being
				** stretch by GtkTable
				*/
GtkWidget *formattypeComboHboxFiller;	/* The filler label widget */

GtkWidget *formattypeCombo;

const char *type_DOS;
const char *type_LINUX;
const char *type_RAW;

GList *formattypelist;

gint delete_event (GtkWidget *w,
		   GdkEvent *e,
		   gpointer data)
{
	if (floppy_pid >= 0)
		return TRUE;	/* Ignore - something's happening */
	return FALSE;
}

void destroy(GtkWidget *w,
	     gpointer *data)
{
	gtk_main_quit();
}

/*
** Here we go!
*/

static void beginformat();

const char *get_format_size()
{
	const char *p=NULL;
	struct fmtsizelist *s;

	char *size=
		gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(capacitylistCombo)
						    ->entry), 0, -1);

	for (s=capacitylistCurrent; s; s=s->next)
		if (s->shownsize && s->floppysize && size &&
		    strcmp(s->shownsize, size) == 0)
		{
			p=s->floppysize;
		}

	if (size)
		free(size);
	return (p);
}

static void reallydoit(int flag)
{
	if (flag)
		beginformat();
}

void doit(GtkWidget *w, gpointer *data)
{
	const char *size_str=get_format_size();
	unsigned t, s, ss;

	if (size_str && parse_size(size_str, &t, &s, &ss) == 0 &&
	    kilobytes( (unsigned long)t * s, ss) >= 4096)
	{
		warning(_("You are trying to format a high-density disk.\n"
			  "Most high-density disks are not formattable, only\n"
			  "standard 3.5\" floppies can be formatted.  Trying\n"
			  "to format a high-density disk may permanently\n"
			  "damage it.  Please check the manual for your\n"
			  "floppy drive before proceeding any further."),
			_("Format"),_("Cancel"), reallydoit);
		return;
	}
	beginformat();
}

void enabledisable(int flag)
{
	gtk_widget_set_sensitive(floppylistCombo, flag);
	gtk_widget_set_sensitive(capacitylistCombo, flag);
	gtk_widget_set_sensitive(floppylistCombo, flag);
	gtk_widget_set_sensitive(formattypeCombo, flag);
	gtk_widget_set_sensitive(verifyoption, flag);
	gtk_widget_set_sensitive(gobutton, flag);
}

static void show_format_progress(const char *);
static void format_completed(int);

static void beginformat()
{
	char *drive;
	char *type;

	const char *floppysize=get_format_size();

	if (floppy_pid >= 0)
		return;	/* Something else is going on */

	drive=gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(floppylistCombo)
						  ->entry), 0, -1);
	type=gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(formattypeCombo)
						 ->entry), 0, -1);

	if (drive && floppysize && type && *drive && *type && *floppysize)
	{
		char *argv[15];
		char *sizebuf;

		if ((sizebuf=g_malloc(strlen(floppysize)+15)) != 0)
		{
			int n;

			argv[0]=FLOPPY;
			argv[1]="--format";

			strcat(strcpy(sizebuf, "--size="), floppysize);

			argv[2]=sizebuf;
			n=3;

			if (GTK_TOGGLE_BUTTON (verifyoption)->active)
				argv[n++]="-V";

			if (strcmp(type, type_DOS) == 0)
				argv[n++]="--fat";
			else if (strcmp(type, type_LINUX) == 0)
				argv[n++]="--ext2";

			argv[n++]="--noprompt";
			argv[n++]="--eject";
			argv[n++]=drive;
			argv[n]=0;

			show_format_progress("0");

			if (floppy_start(argv, show_format_progress,
					 format_completed) == 0)
			{
				gtk_label_set_text(GTK_LABEL(formatprogressLabel),
						   _("Formatting..."));
				enabledisable(FALSE);
			}
			free(sizebuf);
		}
	}

	if (drive)
		g_free(drive);
	if (type)
		g_free(type);
}

static void show_format_progress(const char *p)
{
	int i=atoi(p);

	if (i >= 0 && i <= 100)
		gtk_progress_set_value(GTK_PROGRESS(formatprogress), i);
}

static void format_completed(int status)
{
	enabledisable(TRUE);
	gtk_progress_set_value(GTK_PROGRESS(formatprogress), 100);
	gtk_label_set_text(GTK_LABEL(formatprogressLabel),
			   status ? _("Format failed") :
			   _("Format complete"));
}

FILE *rescan_fp;

#define TMPFILE SYSCONFDIR "/floppy.new"

static void write_rescan(const char *);
static void completed_rescan(int);

static void rescan( GtkWidget *dummy1,
		    gpointer dummy2 )
{
	char *argv[4];

	argv[0]=FLOPPY;
	argv[1]="--createrc";
	argv[2]="--noprompt";
	argv[3]=0;

	if (floppy_start(argv, write_rescan, completed_rescan) == 0)
	{
		rescan_fp=fopen(TMPFILE, "w");
	}
}

static void write_rescan(const char *l)
{
	if (rescan_fp)
		fprintf(rescan_fp, "%s\n", l);
}

static void completed_rescan(int bad)
{
	if (!rescan_fp)
	{
		errmsg(strerror(errno));
		return;
	}
	else
	{
		fclose(rescan_fp);
		if (!bad && rename(TMPFILE, SYSCONFDIR "/floppy") == 0)
		{
			createfloppylist();
		}
	}
}

char *floppy_info_buf;

static void save_about(const char *);
static void doabout(int);

static void about( GtkWidget *dummy1,
		   gpointer dummy2 )
{
	char *argv[4];

	if (floppy_info_buf)
	{
		g_free(floppy_info_buf);
		floppy_info_buf=0;
	}

	floppy_info_buf=g_strdup(PACKAGE " " VERSION
				 " Copyright 2001, Double Precision, Inc.\n\n"
				 "http://floppyutil.sourceforge.net\n\n");
	argv[0]=FLOPPY;
	argv[1]="--probe";
	argv[2]="--noprompt";
	argv[3]=0;

	floppy_start(argv, save_about, doabout);
}

static void save_about(const char *s)
{
	char *p;

	if (floppy_info_buf &&
	    (p=g_realloc(floppy_info_buf,
			 strlen(floppy_info_buf)+strlen(s)+2)) != 0)
	{
		floppy_info_buf=p;
		strcat(p, s);
		strcat(p, "\n");
	}
}

static void doabout(int dummy)
{
	if (floppy_info_buf)
		popup(_("About"), floppy_info_buf);
}

static GtkItemFactory *main_menu_item_factory;
static GtkItemFactoryEntry menu_items[3];


static int disable_rescan()
{
	GtkWidget *w;

	if (getuid() > 0)
	{
		/* Disable rescan option for non-root users */

		w=gtk_item_factory_get_widget(main_menu_item_factory,
					      _("/Floppy/Rescan"));
		gtk_widget_set_sensitive(w, FALSE);
	}
	return (FALSE);
}

static GtkWidget *get_main_menu( GtkWidget  *window)
{
	GtkAccelGroup *accel_group;
	gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

	memset(menu_items, 0, sizeof(menu_items));

	menu_items[0].path=_("/_Floppy");
	menu_items[0].item_type="<Branch>";

	menu_items[1].path=_("/Floppy/_Rescan");
	menu_items[1].accelerator=_("<control>R");
	menu_items[1].callback= &rescan;

	menu_items[2].path=_("/Floppy/_About");
	menu_items[2].accelerator=_("<control>A");
	menu_items[2].callback= &about;

	accel_group = gtk_accel_group_new ();

	main_menu_item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
						       "<main>", 
						       accel_group);


	gtk_item_factory_create_items (main_menu_item_factory, nmenu_items,
				       menu_items,
				       NULL);

	gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);

	gtk_signal_connect
		(GTK_OBJECT
		 (gtk_item_factory_get_widget(main_menu_item_factory,
					      _("/Floppy"))),
		 "realize",
		 GTK_SIGNAL_FUNC(disable_rescan), NULL);

	return (gtk_item_factory_get_widget(main_menu_item_factory, "<main>"));
}

int main(int argc, char **argv)
{
	int argn=1;

        setlocale(LC_ALL, "");
        /*      bindtextdomain(PACKAGE, LOCALEDIR); */
        textdomain(PACKAGE);
	gtk_init (&argc, &argv);

	if (argn < argc)
	{
		drivename=strdup(argv[argn++]);
		if (!drivename)
		{
			perror("malloc");
			exit(1);
		}
	}

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), _("Format floppy disk"));
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(window), "destroy",
			   GTK_SIGNAL_FUNC(destroy), NULL);
	gtk_container_set_border_width(GTK_CONTAINER(window), 0);

	/* Create the menu */

	window_vbox=gtk_vbox_new (FALSE, 1);
	main_menu=get_main_menu(window);
	gtk_box_pack_start (GTK_BOX (window_vbox), main_menu, FALSE, TRUE, 0);
	gtk_widget_show(main_menu);
	gtk_container_set_border_width(GTK_CONTAINER(window_vbox), 0);

	/* Create the dialog */

	table = gtk_table_new(8, 3, 0);

	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 10);

	/* Floppy drive prompt */

	floppypromptLabel=gtk_label_new(_("Floppy: "));
	floppylistCombo=gtk_combo_new();
	gtk_combo_set_value_in_list(GTK_COMBO(floppylistCombo), TRUE, FALSE);
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(floppylistCombo)->entry), FALSE);
	drivechangeon();
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(floppylistCombo)->popwin),
			   "unmap_event",
			   GTK_SIGNAL_FUNC(drivepopdownclosed),
			   NULL);
	gtk_table_attach_defaults(GTK_TABLE(table), floppypromptLabel,
				  0, 1, 0, 1);

	floppylistComboHbox=gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(floppylistComboHbox),
			   floppylistCombo,
			   FALSE,
			   FALSE,
			   0);

	floppylistComboHboxFiller=gtk_label_new("");

	gtk_widget_show(floppylistComboHboxFiller);

	gtk_box_pack_start(GTK_BOX(floppylistComboHbox),
			   floppylistComboHboxFiller,
			   TRUE,
			   TRUE,
			   0);

	gtk_table_attach(GTK_TABLE(table), floppylistComboHbox, 1, 2, 0, 1,
			 GTK_FILL, 0, 0, 0);

	/* Floppy capacity prompt */

	capacitypromptLabel=gtk_label_new(_("Format size: "));
	capacitylistCombo=gtk_combo_new();
	capacitylistComboHbox=gtk_hbox_new(FALSE, 0);

	gtk_box_pack_start(GTK_BOX(capacitylistComboHbox),
			   capacitylistCombo,
			   FALSE,
			   FALSE,
			   0);

	capacitylistComboHboxFiller=gtk_label_new("");

	gtk_widget_show(capacitylistComboHboxFiller);

	gtk_box_pack_start(GTK_BOX(capacitylistComboHbox),
			   capacitylistComboHboxFiller,
			   TRUE,
			   TRUE,
			   0);

	gtk_combo_set_value_in_list(GTK_COMBO(floppylistCombo), TRUE, FALSE);
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(floppylistCombo)->entry),
			       FALSE);
	gtk_combo_set_value_in_list(GTK_COMBO(capacitylistCombo), TRUE, FALSE);
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(capacitylistCombo)->entry),
			       FALSE);
	gtk_table_attach_defaults(GTK_TABLE(table), capacitypromptLabel,
				  0, 1, 1, 2);
	gtk_table_attach(GTK_TABLE(table), capacitylistComboHbox, 1, 2, 1, 2,
			 GTK_FILL, 0, 0, 0);

	gtk_widget_set_usize(capacitylistCombo, 10, 0);

	/* Format type: prompt */

	formattypepromptLabel=gtk_label_new(_("Format type:"));

	type_DOS=_("Dos (FAT)");
	type_LINUX=_("Linux (ext2)");
	type_RAW=_("Raw (none)");

	formattypelist=g_list_append(NULL, (char *)type_DOS);
	formattypelist=g_list_append(formattypelist, (char *)type_LINUX);
	formattypelist=g_list_append(formattypelist, (char *)type_RAW);

	formattypeCombo=gtk_combo_new();
	formattypeComboHbox=gtk_hbox_new(FALSE, 0);

	gtk_box_pack_start(GTK_BOX(formattypeComboHbox),
			   formattypeCombo,
			   FALSE,
			   FALSE,
			   0);
	gtk_combo_set_value_in_list(GTK_COMBO(formattypeCombo), TRUE, FALSE);
	gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(formattypeCombo)->entry),
			       FALSE);

	formattypeComboHboxFiller=gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(formattypeComboHbox),
			   formattypeComboHboxFiller,
			   TRUE,
			   TRUE,
			   0);
	gtk_widget_show(formattypeComboHboxFiller);
	gtk_table_attach_defaults(GTK_TABLE(table), formattypepromptLabel,
				  0, 1, 2, 3);
	gtk_table_attach(GTK_TABLE(table), formattypeComboHbox, 1, 2, 2, 3,
			 GTK_FILL, 0, 0, 0);

	/* Verify checkbox */

	verifyoption=
		gtk_check_button_new_with_label(_("Verify after format"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(verifyoption), TRUE);
	gtk_table_attach(GTK_TABLE(table), verifyoption, 0, 2, 3, 4,
			 0, 0, 0, 0);

	/* Go button */
	gobutton = gtk_button_new_with_label (_("Begin format"));
	gtk_signal_connect(GTK_OBJECT(gobutton), "clicked",
			   GTK_SIGNAL_FUNC(doit), NULL);

	gtk_table_attach(GTK_TABLE(table), gobutton, 0, 2, 4, 5,
			 0, 0, 0, 0);

	/* Format progress bar */

	formatprogressAdjustment=
		gtk_adjustment_new(0, 0, 100, 1, 1, 1);

	formatprogress=
		gtk_progress_bar_new_with_adjustment(GTK_ADJUSTMENT(formatprogressAdjustment));
	gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(formatprogress),
				       GTK_PROGRESS_CONTINUOUS);
	gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(formatprogress),
					 GTK_PROGRESS_LEFT_TO_RIGHT);
	gtk_table_attach(GTK_TABLE(table), formatprogress, 0, 2, 5, 6,
			 GTK_EXPAND | GTK_FILL, 0, 0, 0);
	formatprogressLabel=gtk_label_new("");
	gtk_table_attach(GTK_TABLE(table),formatprogressLabel, 0, 2, 6, 7,
			 GTK_EXPAND | GTK_FILL, 0, 0, 0);

	gtk_misc_set_alignment(GTK_MISC(floppypromptLabel), (gfloat)0,
			       (gfloat)0.5);
	gtk_misc_set_alignment(GTK_MISC(capacitypromptLabel), (gfloat)0,
			       (gfloat)0.5);
	gtk_misc_set_alignment(GTK_MISC(formattypepromptLabel), (gfloat)0,
			       (gfloat)0.5);

	gtk_container_set_border_width(GTK_CONTAINER(table), 10);
	gtk_box_pack_start(GTK_BOX(window_vbox), table, FALSE, TRUE, 10);
	gtk_container_add(GTK_CONTAINER(window), window_vbox);
	gtk_widget_show(window_vbox);

	gtk_widget_show(floppypromptLabel);
	gtk_widget_show(floppylistComboHbox);
	gtk_widget_show(floppylistCombo);
	gtk_widget_show(capacitypromptLabel);
	gtk_widget_show(capacitylistComboHbox);
	gtk_widget_show(formattypepromptLabel);
	gtk_widget_show(formattypeComboHbox);
	set_dyn_combo(formattypeCombo, formattypelist, NULL);
	gtk_widget_show(formattypeCombo);

	gtk_widget_show(gobutton);
	gtk_widget_show(table);
	gtk_widget_show(verifyoption);
	gtk_widget_show(formatprogress);
	gtk_widget_show(formatprogressLabel);

	createfloppylist();
	gtk_main ();
	return(0);
}
