/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include "_scm.h"

#ifdef __EMX__
# include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
#   include <unistd.h>
#endif

#include <sys/stat.h>
extern char *getcwd ();

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif






PROC (s_read_line, "read-line", 0, 1, 0, scm_read_line);
#ifdef __STDC__
SCM 
scm_read_line (SCM port)
#else
SCM 
scm_read_line (port)
     SCM port;
#endif
{
  register int c;
  register int j = 0;
  sizet len = 30;
  SCM tok_buf = scm_makstr ((long) len, 0);
  register char *p = CHARS (tok_buf);
  if (UNBNDP (port))
    port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_read_line);
  if (EOF == (c = scm_gen_getc (port)))
    return EOF_VAL;
  while (1)
    {
      switch (c)
	{
	case LINE_INCREMENTORS:
	case EOF:
	  if (len == j)
	    return tok_buf;
	  return scm_resizuve (tok_buf, (SCM) MAKINUM (j));
	default:
	  if (j >= len)
	    {
	      p = scm_grow_tok_buf (tok_buf);
	      len = LENGTH (tok_buf);
	    }
	  p[j++] = c;
	  c = scm_gen_getc (port);
	}
    }
}



PROC (s_read_line_x, "read-line!", 1, 1, 0, scm_read_line_x);
#ifdef __STDC__
SCM 
scm_read_line_x (SCM str, SCM port)
#else
SCM 
scm_read_line_x (str, port)
     SCM str;
     SCM port;
#endif
{
  register int c;
  register int j = 0;
  register char *p;
  sizet len;
  ASSERT (NIMP (str) && STRINGP (str), str, ARG1, s_read_line_x);
  p = CHARS (str);
  len = LENGTH (str);
  if UNBNDP
    (port) port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG2, s_read_line_x);
  c = scm_gen_getc (port);
  if (EOF == c)
    return EOF_VAL;
  while (1)
    {
      switch (c)
	{
	case LINE_INCREMENTORS:
	case EOF:
	  return MAKINUM (j);
	default:
	  if (j >= len)
	    {
	      scm_gen_ungetc (c, port);
	      return BOOL_F;
	    }
	  p[j++] = c;
	  c = scm_gen_getc (port);
	}
    }
}



PROC (s_write_line, "write-line", 1, 1, 0, scm_write_line);
#ifdef __STDC__
SCM 
scm_write_line (SCM obj, SCM port)
#else
SCM 
scm_write_line (obj, port)
     SCM obj;
     SCM port;
#endif
{
  scm_display (obj, port);
  return scm_newline (port);
}



PROC (s_sys_ftell, "%ftell", 1, 0, 0, scm_sys_ftell);
#ifdef __STDC__
SCM 
scm_sys_ftell (SCM port)
#else
SCM 
scm_sys_ftell (port)
     SCM port;
#endif
{
  long pos;
  ASSERT (NIMP (port) && OPFPORTP (port), port, ARG1, s_sys_ftell);
  SYSCALL (pos = ftell (STREAM (port)));
  if (pos < 0)
    return BOOL_F;
  if (pos > 0 && CRDYP (port))
    pos--;
  return MAKINUM (pos);
}



PROC (s_sys_fseek, "%fseek", 3, 0, 0, scm_sys_fseek);
#ifdef __STDC__
SCM 
scm_sys_fseek (SCM port, SCM offset, SCM whence)
#else
SCM 
scm_sys_fseek (port, offset, whence)
     SCM port;
     SCM offset;
     SCM whence;
#endif
{
  int rv;
  ASSERT (NIMP (port) && OPFPORTP (port), port, ARG1, s_sys_fseek);
  ASSERT (INUMP (offset), offset, ARG2, s_sys_fseek);
  ASSERT (INUMP (whence) && (INUM (whence) < 3) && (INUM (whence) >= 0),
	  whence, ARG3, s_sys_fseek);
  CLRDY (port);			/* Clear ungetted char */
  /* Values of whence are interned in scm_init_ioext.  */
  rv = fseek (STREAM (port), INUM (offset), INUM (whence));
  return rv ? BOOL_F : BOOL_T;
}



PROC (s_sys_freopen, "%freopen", 3, 0, 0, scm_sys_freopen);
#ifdef __STDC__
SCM 
scm_sys_freopen (SCM filename, SCM modes, SCM port)
#else
SCM 
scm_sys_freopen (filename, modes, port)
     SCM filename;
     SCM modes;
     SCM port;
#endif
{
  FILE *f;
  ASSERT (NIMP (filename) && STRINGP (filename), filename, ARG1, s_sys_freopen);
  ASSERT (NIMP (modes) && STRINGP (modes), modes, ARG2, s_sys_freopen);
  DEFER_INTS;
  ASSERT (NIMP (port) && FPORTP (port) && CLOSEDP (port), port, ARG3, s_sys_freopen);
  SYSCALL (f = freopen (CHARS (filename), CHARS (modes), STREAM (port)));
  if (!f)
    {
      CAR (port) &= ~OPN;
      scm_remove_from_port_table (port);
      port = BOOL_F;
    }
  else
    {
      CAR (port) = tc16_fport | scm_mode_bits (CHARS (modes));
      SETSTREAM (port, f);
      if (BUF0 & (CAR (port) = tc16_fport | scm_mode_bits (CHARS (modes))))
	scm_setbuf0 (port);
    }
  ALLOW_INTS;
  return port;
}



PROC (s_sys_duplicate_port, "%duplicate-port", 2, 0, 0, scm_sys_duplicate_port);
#ifdef __STDC__
SCM 
scm_sys_duplicate_port (SCM oldpt, SCM modes)
#else
SCM 
scm_sys_duplicate_port (oldpt, modes)
     SCM oldpt;
     SCM modes;
#endif
{
  int oldfd;
  int newfd;
  FILE *f;
  SCM newpt;
  ASSERT (NIMP (oldpt) && OPPORTP (oldpt), oldpt, ARG1, s_sys_duplicate_port);
  ASSERT (NIMP (modes) && STRINGP (modes), modes, ARG2, s_sys_duplicate_port);
  NEWCELL (newpt);
  DEFER_INTS;
  oldfd = fileno ((FILE*) STREAM (oldpt));
  if (oldfd == -1)
    {
      ALLOW_INTS;
      return BOOL_F;
    };
  SYSCALL (newfd = dup (oldfd));
  if (newfd == -1)
    {
      ALLOW_INTS;
      return BOOL_F;
    };
  f = fdopen (newfd, CHARS (modes));
  if (!f)
    {
      SYSCALL (close (newfd));
      ALLOW_INTS;
      return BOOL_F;
    }
  SETSTREAM (newpt, f);
  if (BUF0 & (CAR (newpt) = tc16_fport | scm_mode_bits (CHARS (modes))))
    scm_setbuf0 (newpt);
  scm_add_to_port_table (newpt);
  ALLOW_INTS;
  return newpt;
}



PROC (s_sys_redirect_port, "%redirect-port", 2, 0, 0, scm_sys_redirect_port);
#ifdef __STDC__
SCM 
scm_sys_redirect_port (SCM into_pt, SCM from_pt)
#else
SCM 
scm_sys_redirect_port (into_pt, from_pt)
     SCM into_pt;
     SCM from_pt;
#endif
{
  int ans, oldfd, newfd;
  DEFER_INTS;
  ASSERT (NIMP (into_pt) && OPPORTP (into_pt), into_pt, ARG1, s_sys_redirect_port);
  ASSERT (NIMP (from_pt) && OPPORTP (from_pt), from_pt, ARG2, s_sys_redirect_port);
  oldfd = fileno ((FILE*) STREAM (into_pt));
  newfd = fileno ((FILE*) STREAM (from_pt));
  if (oldfd == -1 || newfd == -1)
    ans = -1;
  else
    SYSCALL (ans = dup2 (oldfd, newfd));
  ALLOW_INTS;
  return (ans == -1) ? BOOL_F : BOOL_T;
}


static long scm_tc16_dir;

PROC (s_sys_opendir, "%opendir", 1, 0, 0, scm_sys_opendir);
#ifdef __STDC__
SCM 
scm_sys_opendir (SCM dirname)
#else
SCM 
scm_sys_opendir (dirname)
     SCM dirname;
#endif
{
  DIR *ds;
  SCM dir;
  ASSERT (NIMP (dirname) && STRINGP (dirname), dirname, ARG1, s_sys_opendir);
  NEWCELL (dir);
  DEFER_INTS;
  SYSCALL (ds = opendir (CHARS (dirname)));
  if (!ds)
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  CAR (dir) = scm_tc16_dir | OPN;
  SETCDR (dir, ds);
  ALLOW_INTS;
  return dir;
}



PROC (s_sys_readdir, "%readdir", 1, 0, 0, scm_sys_readdir);
#ifdef __STDC__
SCM 
scm_sys_readdir (SCM port)
#else
SCM 
scm_sys_readdir (port)
     SCM port;
#endif
{
  struct dirent *rdent;
  DEFER_INTS;
  ASSERT (OPDIRP (port), port, ARG1, s_sys_readdir);
  errno = 0;
  SYSCALL (rdent = readdir ((DIR *) CDR (port)));
  ALLOW_INTS;
  return (rdent
	  ? scm_makfromstr (rdent->d_name, NAMLEN (rdent), 0)
	  : (errno ? BOOL_F : EOF_VAL));
}



PROC (s_rewinddir, "rewinddir", 1, 0, 0, scm_rewinddir);
#ifdef __STDC__
SCM 
scm_rewinddir (SCM port)
#else
SCM 
scm_rewinddir (port)
     SCM port;
#endif
{
  ASSERT (OPDIRP (port), port, ARG1, s_rewinddir);
  rewinddir ((DIR *) CDR (port));
  return UNSPECIFIED;
}



PROC (s_sys_closedir, "%closedir", 1, 0, 0, scm_sys_closedir);
#ifdef __STDC__
SCM 
scm_sys_closedir (SCM port)
#else
SCM 
scm_sys_closedir (port)
     SCM port;
#endif
{
  int sts;
  ASSERT (DIRP (port), port, ARG1, s_sys_closedir);
  DEFER_INTS;
  if (CLOSEDP (port))
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  SYSCALL (sts = closedir ((DIR *) CDR (port)));
  if (sts)
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  CAR (port) = scm_tc16_dir;
  ALLOW_INTS;
  return BOOL_T;
}



#ifdef __STDC__
static int 
scm_dir_print (SCM sexp, SCM port, int writing)
#else
static int 
scm_dir_print (sexp, port, writing)
     SCM sexp;
     SCM port;
     int writing;
#endif
{
  scm_prinport (sexp, port, "directory");
  return !0;
}



#ifdef __STDC__
static sizet 
scm_dir_free (SCM p)
#else
static sizet 
scm_dir_free (p)
     SCM p;
#endif
{
  if (OPENP (p))
    closedir ((DIR *) CDR (p));
  return 0;
}

static scm_smobfuns dir_smob = {scm_mark0, scm_dir_free, scm_dir_print, 0};


PROC (s_sys_mkdir, "%mkdir", 1, 1, 0, scm_sys_mkdir);
#ifdef __STDC__
SCM 
scm_sys_mkdir (SCM path, SCM mode)
#else
SCM 
scm_sys_mkdir (path, mode)
     SCM path;
     SCM mode;
#endif
{
#ifdef HAVE_MKDIR
  int rv;
  mode_t mask;
  ASSERT (NIMP (path) && STRINGP (path), path, ARG1, s_sys_mkdir);
  if (UNBNDP (mode))
    {
      mask = umask (0);
      umask (mask);
      SYSCALL (rv = mkdir (CHARS (path), 0777 ^ mask));
    }
  else
    {
      ASSERT (INUMP (mode), mode, ARG2, s_sys_mkdir);
      SYSCALL (rv = mkdir (CHARS (path), INUM (mode)));
    }
  return rv ? BOOL_F : BOOL_T;
#else
  return BOOL_F;
#endif
}


PROC (s_sys_rmdir, "%rmdir", 1, 0, 0, scm_sys_rmdir);
#ifdef __STDC__
SCM 
scm_sys_rmdir (SCM path)
#else
SCM 
scm_sys_rmdir (path)
     SCM path;
#endif
{
#ifdef HAVE_RMDIR
  int val;
  ASSERT (NIMP (path) && STRINGP (path), path, ARG1, s_sys_rmdir);
  SYSCALL (val = rmdir (CHARS (path)));
  return val ? BOOL_F : BOOL_T;
#else
  return BOOL_F;
#endif
}



PROC (s_sys_chdir, "%chdir", 1, 0, 0, scm_sys_chdir);
#ifdef __STDC__
SCM 
scm_sys_chdir (SCM str)
#else
SCM 
scm_sys_chdir (str)
     SCM str;
#endif
{
  int ans;
  ASSERT (NIMP (str) && STRINGP (str), str, ARG1, s_sys_chdir);
  SYSCALL (ans = chdir (CHARS (str)));
  return ans ? BOOL_F : BOOL_T;
}



PROC (s_sys_getcwd, "%getcwd", 0, 0, 0, scm_sys_getcwd);
#ifdef __STDC__
SCM 
scm_sys_getcwd (void)
#else
SCM 
scm_sys_getcwd ()
#endif
{
#ifdef HAVE_GETCWD
  char *rv;

  sizet size = 100;
  char *wd;
  SCM result = BOOL_F;

  DEFER_INTS;
  wd = scm_must_malloc (size, s_sys_getcwd);
  while ((rv = getcwd (wd, size)) == 0 && errno == ERANGE)
    {
      scm_must_free (wd);
      size *= 2;
      wd = scm_must_malloc (size, s_sys_getcwd);
    }
  if (rv != 0)
    result = scm_makfromstr (wd, strlen (wd), 0);
  scm_must_free (wd);
  ALLOW_INTS;
  return result;
#else
  return BOOL_F;
#endif
}



PROC (s_sys_chmod, "%chmod", 2, 0, 0, scm_sys_chmod);
#ifdef __STDC__
SCM 
scm_sys_chmod (SCM port_or_path, SCM mode)
#else
SCM 
scm_sys_chmod (port_or_path, mode)
     SCM port_or_path;
     SCM mode;
#endif
{
  int rv;
  ASSERT (INUMP (mode), mode, ARG2, s_sys_chmod);
  ASSERT (NIMP (port_or_path), port_or_path, ARG1, s_sys_chmod);
  if (STRINGP (port_or_path))
    SYSCALL (rv = chmod (CHARS (port_or_path), INUM (mode)));
  else
    {
      ASSERT (OPFPORTP (port_or_path), port_or_path, ARG1, s_sys_chmod);
      rv = fileno ((FILE*) STREAM (port_or_path));
      if (rv != -1)
	SYSCALL (rv = fchmod (rv, INUM (mode)));
    }
  return rv ? BOOL_F : BOOL_T;
}



#ifdef __EMX__
#include <sys/utime.h>
#else
#include <utime.h>
#endif

PROC (s_sys_utime, "%utime", 1, 2, 0, scm_sys_utime);
#ifdef __STDC__
SCM 
scm_sys_utime (SCM pathname, SCM actime, SCM modtime)
#else
SCM 
scm_sys_utime (pathname, actime, modtime)
     SCM pathname;
     SCM actime;
     SCM modtime;
#endif
{
  int rv;
  struct utimbuf utm_tmp;

  ASSERT (NIMP (pathname) && STRINGP (pathname), pathname, ARG1, s_sys_utime);

  if (UNBNDP (actime))
    SYSCALL (time (&utm_tmp.actime));
  else
    utm_tmp.actime = scm_num2ulong (actime, (char *) ARG2, s_sys_utime);

  if (UNBNDP (modtime))
    SYSCALL (time (&utm_tmp.modtime));
  else
    utm_tmp.modtime = scm_num2ulong (modtime, (char *) ARG3, s_sys_utime);

  SYSCALL (rv = utime (CHARS (pathname), &utm_tmp));
  return rv ? BOOL_F : BOOL_T;
}



PROC (s_umask, "umask", 0, 1, 0, scm_umask);
#ifdef __STDC__
SCM 
scm_umask (SCM mode)
#else
SCM 
scm_umask (mode)
     SCM mode;
#endif
{
  mode_t mask;
  if (UNBNDP (mode))
    {
      mask = umask (0);
      umask (mask);
    }
  else {
    ASSERT (INUMP (mode), mode, ARG1, s_umask);
    mask = umask (INUM (mode));
  }
  return MAKINUM (mask);
}



PROC (s_sys_rename, "%rename", 2, 0, 0, scm_sys_rename);
#ifdef __STDC__
SCM 
scm_sys_rename (SCM oldname, SCM newname)
#else
SCM 
scm_sys_rename (oldname, newname)
     SCM oldname;
     SCM newname;
#endif
{
  int rv;
  ASSERT (NIMP (oldname) && STRINGP (oldname), oldname, ARG1, s_sys_rename);
  ASSERT (NIMP (newname) && STRINGP (newname), newname, ARG2, s_sys_rename);
#ifdef STDC_HEADERS
  SYSCALL (rv = rename (CHARS (oldname), CHARS (newname)));
  return rv ? BOOL_F : BOOL_T;
#else
  DEFER_INTS;
  SYSCALL (rv = link (CHARS (oldname), CHARS (newname)));
  if (!rv)
    {
      SYSCALL (rv = unlink (CHARS (oldname)));;
      if (rv)
	/* unlink failed.  remove new name */
	SYSCALL (unlink (CHARS (newname))); 
    }
  ALLOW_INTS;
  return rv ? BOOL_F : BOOL_T;
#endif
}



PROC (s_sys_fileno, "%fileno", 1, 0, 0, scm_sys_fileno);
#ifdef __STDC__
SCM 
scm_sys_fileno (SCM port)
#else
SCM 
scm_sys_fileno (port)
     SCM port;
#endif
{
  int fd;
  ASSERT (NIMP (port) && OPFPORTP (port), port, ARG1, s_sys_fileno);
  fd = fileno ((FILE*) STREAM (port));
  return (fd == -1) ? BOOL_F : MAKINUM (fd);
}



PROC (s_sys_isatty, "%isatty", 1, 0, 0, scm_sys_isatty);
#ifdef __STDC__
SCM 
scm_sys_isatty (SCM port)
#else
SCM 
scm_sys_isatty (port)
     SCM port;
#endif
{
  int rv;
  ASSERT (NIMP (port) && OPFPORTP (port), port, ARG1, s_sys_isatty);
  rv = fileno ((FILE*) STREAM (port));
  if (rv == -1)
    return EOF_VAL;
  else
    {
      rv = isatty (rv);
      return  rv ? BOOL_T : BOOL_F;
    }
}



PROC (s_sys_fdopen, "%fdopen", 2, 0, 0, scm_sys_fdopen);
#ifdef __STDC__
SCM
scm_sys_fdopen (SCM fdes, SCM modes)
#else
SCM
scm_sys_fdopen (fdes, modes)
     SCM fdes;
     SCM modes;
#endif
{
  FILE *f;
  SCM port;

  ASSERT (INUMP (fdes), fdes, ARG1, s_sys_fdopen);
  ASSERT (NIMP (modes) && STRINGP (modes), modes, ARG2, s_sys_fdopen);
  DEFER_INTS;
  f = fdopen (INUM (fdes), CHARS (modes));
  if (f == NULL)
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  NEWCELL (port);
  CAR (port) = tc16_fport | scm_mode_bits (CHARS (modes));
  SETSTREAM (port,f);
  scm_add_to_port_table (port);
  ALLOW_INTS;
  return port;
}



/* Move a port's underlying file descriptor to a given value.
 * Returns: #f for error.
 *           0 if fdes is already the given value.
 *           1 if fdes moved. 
 * MOVE->FDES is implemented in Scheme and calls this primitive.
 */
PROC (s_sys_primitive_move_to_fdes, "%primitive-move->fdes", 2, 0, 0, scm_sys_primitive_move_to_fdes);
#ifdef __STDC__
SCM
scm_sys_primitive_move_to_fdes (SCM port, SCM fd)
#else
SCM
scm_sys_primitive_move_to_fdes (port, fd)
     SCM port;
     SCM fd;
#endif
{
  FILE *stream;
  int old_fd;
  int new_fd;
  int rv;

  ASSERT (NIMP (port) && OPFPORTP (port), port, ARG1, s_sys_primitive_move_to_fdes);
  ASSERT (INUMP (fd), fd, ARG2, s_sys_primitive_move_to_fdes);
  DEFER_INTS;
  stream = STREAM (port);
  old_fd = fileno (stream);
  new_fd = INUM (fd);
  if  (old_fd == new_fd)
    {
      ALLOW_INTS;
      return MAKINUM (0);
    }
  scm_evict_ports (new_fd);
  rv = dup2 (old_fd, new_fd);
  if (rv == -1)
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  scm_setfileno (stream, new_fd);
  SYSCALL (close (old_fd));  
  ALLOW_INTS;
  return MAKINUM (1);
}



PROC (s_sys_access, "%access", 2, 0, 0, scm_sys_access);
#ifdef __STDC__
SCM 
scm_sys_access (SCM path, SCM how)
#else
SCM 
scm_sys_access (path, how)
     SCM path;
     SCM how;
#endif
{
  int rv;
  int ihow;
  ASSERT (NIMP (path) && STRINGP (path), path, ARG1, s_sys_access);
  ASSERT (INUMP (how), how, ARG2, s_sys_access);
  /* "how" values are interned in scm_init_ioext.  */
  rv = access (CHARS (path), INUM (how));
  return rv ? BOOL_F : BOOL_T;
}



SCM scm_stat2scm P ((struct stat * stat_temp));
PROC (s_sys_stat, "%stat", 1, 0, 0, scm_sys_stat);
#ifdef __STDC__
SCM 
scm_sys_stat (SCM port_or_path)
#else
SCM 
scm_sys_stat (port_or_path)
     SCM port_or_path;
#endif
{
  int rv;
  struct stat stat_temp;
  ASSERT (NIMP (port_or_path), port_or_path, ARG1, s_sys_stat);
#ifdef MCH_AMIGA
  ASSERT (STRING (port_or_path), port_or_path, ARG1, s_sys_stat);
#endif
  if (STRINGP (port_or_path))
    SYSCALL (rv = stat (CHARS (port_or_path), &stat_temp));
#ifndef MCH_AMIGA
  else
    {
      ASSERT (OPFPORTP (port_or_path), port_or_path, ARG1, s_sys_stat);
      DEFER_INTS;
      rv = fileno ((FILE*) STREAM (port_or_path));
      ALLOW_INTS;
      if (rv != -1)
	SYSCALL (rv = fstat (rv, &stat_temp));
    }
#endif
  return rv ? BOOL_F : scm_stat2scm (&stat_temp);
}



#ifdef __STDC__
SCM 
scm_stat2scm (struct stat *stat_temp)
#else
SCM 
scm_stat2scm (stat_temp)
     struct stat *stat_temp;
#endif
{
  SCM ans = scm_make_vector (MAKINUM (13), UNSPECIFIED, SCM_UNDEFINED);
  SCM *ve = VELTS (ans);
  ve[0] = scm_ulong2num ((unsigned long) stat_temp->st_dev);
  ve[1] = scm_ulong2num ((unsigned long) stat_temp->st_ino);
  ve[2] = scm_ulong2num ((unsigned long) stat_temp->st_mode);
  ve[3] = scm_ulong2num ((unsigned long) stat_temp->st_nlink);
  ve[4] = scm_ulong2num ((unsigned long) stat_temp->st_uid);
  ve[5] = scm_ulong2num ((unsigned long) stat_temp->st_gid);
#ifdef HAVE_ST_RDEV
  ve[6] = scm_ulong2num ((unsigned long) stat_temp->st_rdev);
#else
  ve[6] = BOOL_F;
#endif
  ve[7] = scm_ulong2num ((unsigned long) stat_temp->st_size);
  ve[8] = scm_ulong2num ((unsigned long) stat_temp->st_atime);
  ve[9] = scm_ulong2num ((unsigned long) stat_temp->st_mtime);
  ve[10] = scm_ulong2num ((unsigned long) stat_temp->st_ctime);
#ifdef AC_STRUCT_ST_BLKSIZE
  ve[11] = scm_ulong2num ((unsigned long) stat_temp->st_blksize);
#else
  ve[11] = scm_ulong2num (4096L);
#endif
#ifdef AC_STRUCT_ST_BLOCKS
  ve[12] = scm_ulong2num ((unsigned long) stat_temp->st_blocks);
#else
  ve[12] = BOOL_F;
#endif

  return ans;
}



PROC (s_getpid, "getpid", 0, 0, 0, scm_getpid);
#ifdef __STDC__
SCM 
scm_getpid (void)
#else
SCM 
scm_getpid ()
#endif
{
  return MAKINUM ((unsigned long) getpid ());
}



PROC (s_sys_putenv, "%putenv", 1, 0, 0, scm_sys_putenv);
#ifdef __STDC__
SCM
scm_sys_putenv (SCM str)
#else
SCM
scm_sys_putenv (str)
     SCM str;
#endif
{
#ifdef HAVE_PUTENV
  ASSERT (NIMP (str) && STRINGP (str), str, ARG1, s_sys_putenv);
  return putenv (CHARS (str)) ? BOOL_F : BOOL_T;
#else
  return BOOL_F;
#endif
}



void 
scm_init_ioext ()
{
  /* fseek() symbols.  */
  scm_sysintern ("SEEK_SET", MAKINUM (SEEK_SET));
  scm_sysintern ("SEEK_CUR", MAKINUM (SEEK_CUR));
  scm_sysintern ("SEEK_END", MAKINUM (SEEK_END));

  /* access() symbols.  */
  scm_sysintern ("R_OK", MAKINUM (R_OK));
  scm_sysintern ("W_OK", MAKINUM (W_OK));
  scm_sysintern ("X_OK", MAKINUM (X_OK));
  scm_sysintern ("F_OK", MAKINUM (F_OK));

  /* File type/permission bits.  */
#ifdef S_IRUSR
  scm_sysintern ("S_IRUSR", MAKINUM (S_IRUSR));
#endif
#ifdef S_IWUSR
  scm_sysintern ("S_IWUSR", MAKINUM (S_IWUSR));
#endif
#ifdef S_IXUSR
  scm_sysintern ("S_IXUSR", MAKINUM (S_IXUSR));
#endif
#ifdef S_IRWXU
  scm_sysintern ("S_IRWXU", MAKINUM (S_IRWXU));
#endif

#ifdef S_IRGRP
  scm_sysintern ("S_IRGRP", MAKINUM (S_IRGRP));
#endif
#ifdef S_IWGRP
  scm_sysintern ("S_IWGRP", MAKINUM (S_IWGRP));
#endif
#ifdef S_IXGRP
  scm_sysintern ("S_IXGRP", MAKINUM (S_IXGRP));
#endif
#ifdef S_IRWXG
  scm_sysintern ("S_IRWXG", MAKINUM (S_IRWXG));
#endif

#ifdef S_IROTH
  scm_sysintern ("S_IROTH", MAKINUM (S_IROTH));
#endif
#ifdef S_IWOTH
  scm_sysintern ("S_IWOTH", MAKINUM (S_IWOTH));
#endif
#ifdef S_IXOTH
  scm_sysintern ("S_IXOTH", MAKINUM (S_IXOTH));
#endif
#ifdef S_IRWXO
  scm_sysintern ("S_IRWXO", MAKINUM (S_IRWXO));
#endif

#ifdef S_ISUID
  scm_sysintern ("S_ISUID", MAKINUM (S_ISUID));
#endif
#ifdef S_ISGID
  scm_sysintern ("S_ISGID", MAKINUM (S_ISGID));
#endif
#ifdef S_ISVTX
  scm_sysintern ("S_ISVTX", MAKINUM (S_ISVTX));
#endif

#ifdef S_IFMT
  scm_sysintern ("S_IFMT", MAKINUM (S_IFMT));
#endif
#ifdef S_IFDIR
  scm_sysintern ("S_IFDIR", MAKINUM (S_IFDIR));
#endif
#ifdef S_IFCHR
  scm_sysintern ("S_IFCHR", MAKINUM (S_IFCHR));
#endif
#ifdef S_IFBLK
  scm_sysintern ("S_IFBLK", MAKINUM (S_IFBLK));
#endif
#ifdef S_IFREG
  scm_sysintern ("S_IFREG", MAKINUM (S_IFREG));
#endif
#ifdef S_IFLNK
  scm_sysintern ("S_IFLNK", MAKINUM (S_IFLNK));
#endif
#ifdef S_IFSOCK
  scm_sysintern ("S_IFSOCK", MAKINUM (S_IFSOCK));
#endif
#ifdef S_IFIFO
  scm_sysintern ("S_IFIFO", MAKINUM (S_IFIFO));
#endif

  scm_add_feature ("i/o-extensions");
  scm_add_feature ("line-i/o");
#ifdef HAVE_PIPE
/*
  scm_ptobs[0x0ff & (tc16_pipe >> 8)].fclose = pclose;
  scm_ptobs[0x0ff & (tc16_pipe >> 8)].free = pclose;
  scm_ptobs[0x0ff & (tc16_pipe >> 8)].print = prinpipe;
  scm_add_feature (s_pipe);
*/
#endif

  scm_tc16_dir = scm_newsmob (&dir_smob);
#include "ioext.x"
}

