/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var objectKeys = require( './../../../keys' );
var ENV = require( '@stdlib/process/env' );


// FUNCTIONS //

/**
* Copies `process.env`.
*
* ## Notes
*
* -   This implementation accommodates `process.env` on older Node.js versions (<=v0.10.x), where `process.env` was object-like, but would show unexpected behavior when attempting to get own property descriptors.
*
* @private
* @returns {Object} copy of `process.env`
*/
function copy() {
	var keys;
	var env;
	var out;
	var i;

	keys = objectKeys( ENV );
	out = {};
	for ( i = 0; i < keys.length; i++ ) {
		env = keys[ i ];
		out[ env ] = ENV[ env ];
	}
	return out;
}


// MAIN //

/**
* Returns worker environment variables.
*
* @private
* @param {Options} opts - options
* @param {string} opts.cmd - executable file/command
* @param {boolean} opts.ordered - boolean indicating whether to preserve order of script output
* @param {(NonNegativeInteger|null)} opts.uid - process user identity
* @param {(NonNegativeInteger|null)} opts.gid - process group identity
* @param {string} opts.encoding - `stdio` encoding
* @param {NonNegativeInteger} opts.maxBuffer - max child process `stdio` buffer size
* @returns {Object} environment variables
*/
function env( opts ) {
	var out = copy();

	out.WORKER_CMD = opts.cmd;
	out.WORKER_ENCODING = opts.encoding;
	out.WORKER_MAX_BUFFER = opts.maxBuffer;

	if ( opts.ordered ) {
		out.WORKER_ORDERED = '1';
	}
	if ( opts.uid ) {
		out.WORKER_UID = opts.uid;
	}
	if ( opts.gid ) {
		out.WORKER_GID = opts.gid;
	}
	return out;
}


// EXPORTS //

module.exports = env;
