/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sejda.sambox.pdmodel.interactive.documentnavigation.destination;

import org.sejda.sambox.cos.COSArray;
import org.sejda.sambox.cos.COSName;
import org.sejda.sambox.util.Matrix;

/**
 * This represents a destination to a page and the page contents will be magnified to just fit on
 * the screen.
 *
 * @author Ben Litchfield
 */
public class PDPageFitDestination extends PDPageDestination
{
    /**
     * The type of this destination.
     */
    protected static final String TYPE = "Fit";
    /**
     * The type of this destination.
     */
    protected static final String TYPE_BOUNDED = "FitB";

    /**
     * Default constructor.
     */
    public PDPageFitDestination()
    {
        array.growToSize(2);
        array.set(1, COSName.getPDFName(TYPE));

    }

    /**
     * Constructor from an existing destination array.
     *
     * @param arr The destination array.
     */
    public PDPageFitDestination(COSArray arr)
    {
        super(arr);
    }

    /**
     * A flag indicating if this page destination should just fit bounding box of the PDF.
     *
     * @return true If the destination should fit just the bounding box.
     */
    public boolean fitBoundingBox()
    {
        return TYPE_BOUNDED.equals(array.getName(1));
    }

    /**
     * Set if this page destination should just fit the bounding box.  The default is false.
     *
     * @param fitBoundingBox A flag indicating if this should fit the bounding box.
     */
    public void setFitBoundingBox(boolean fitBoundingBox)
    {
        array.growToSize(2);
        if (fitBoundingBox)
        {
            array.set(1, COSName.getPDFName(TYPE_BOUNDED));
        }
        else
        {
            array.set(1, COSName.getPDFName(TYPE));
        }
    }

    @Override
    public void transform(Matrix transformation)
    {
        //nothing to do
    }
}
