/*
** ypserver.c			YP server routines.
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS YP Server.
**
** The NYS YP Server is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS YP Server is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public
** License along with the NYS YP Server; see the file COPYING.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include "system.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <gdbm.h>
#include <unistd.h>
#include <sys/stat.h>
#include "yp.h"


#ifndef _PATH_YP
#define _PATH_YP   "/var/yp"
#endif
char *path_ypdb = _PATH_YP;

int debug_flag = 1;


static void Gperror(const char *str)
{
    fprintf(stderr, "%s: GDBM Error Code #%d\n", str, gdbm_errno);
}


void *ypproc_null_2_svc(void *dummy,
			struct svc_req *rqstp)
{
    static int foo;

    
    if (debug_flag)
	fprintf(stderr, "ypproc_null()\n");

    return (void *) &foo;
}

/*
** Return 1 if the name is a valid domain name served by us, else 0.
*/
static int is_valid_domain(const char *domain)
{
    char buf[1025];
    struct stat sbuf;
    

    if (strcmp(domain, "binding") == 0 ||
	strcmp(domain, "..") == 0 ||
	strcmp(domain, ".") == 0 ||
	strchr(domain, '/'))
	return 0;
    
    sprintf(buf, "%s/%s", path_ypdb, domain);
    if (stat(buf, &sbuf) < 0 || !S_ISDIR(sbuf.st_mode))
	return 0;

    return 1;
}


bool_t *ypproc_domain_2_svc(domainname *name,
			    struct svc_req *rqstp)
{
    static bool_t result;


    if (debug_flag)
	fprintf(stderr, "ypproc_domain(\"%s\")\n", *name);

    if (is_valid_domain(*name))
	result = TRUE;
    else
	result = FALSE;
    
    if (debug_flag)
	fprintf(stderr, "\t-> %s.\n",
		(result == TRUE ? "Ok" : "Not served by us"));
    
    return &result;
}


bool_t *ypproc_domain_nonack_2_svc(domainname *name,
				   struct svc_req *rqstp)
{
    static bool_t result;

    if (debug_flag)
	fprintf(stderr, "ypproc_domain_nonack(\"%s\")\n", *name);
    
    if (!is_valid_domain(*name))
    {
	if (debug_flag)
	    fprintf(stderr, "\t-> Not a valid domain.\n");

	/* Bail out and don't return any RPC value */
	return NULL;
    }
    
    if (debug_flag)
	fprintf(stderr, "\t-> OK.\n");
    
    result = TRUE;
    return &result;
}


/*
** Open a GDBM database
*/
static GDBM_FILE open_database(const char *domain,
			       const char *map)
{
    GDBM_FILE dbp;
    char buf[1025];


    /* Better be safe than sorry... */
    if (strcmp(domain, "binding") == 0 ||
	strcmp(domain, "..") == 0 ||
	strcmp(domain, ".") == 0 ||
	strchr(domain, '/'))
	return 0;

    if (map[0] == '.' || strchr(map, '/'))
	return 0;

    sprintf(buf, "%s/%s/%s", path_ypdb, domain, map);
    
    dbp = gdbm_open(buf, 0, O_RDONLY, 0, NULL);

    if (debug_flag && dbp == NULL)
	Gperror("gdbm_open()");
    
    return dbp;
}


/*
** Get a record from a GDBM database.
*/
int read_database(GDBM_FILE dbp,
		  const datum *ikey,
		  datum *okey,
		  datum *dval,
		  int all_flag)
{
    int first_flag = 0;
    datum nkey, ckey;
    
    
    if (ikey == NULL || ikey->dptr == NULL)
    {
	ckey = gdbm_firstkey(dbp);
	first_flag = 1;
    }
    else
	ckey = *ikey;
    
    if (ckey.dptr == NULL)
	return YP_NOKEY;

    while (1)
    {
	*dval = gdbm_fetch(dbp, ckey);
	if (dval->dptr == NULL)
	{
	    /* Free key, unless it comes from the caller! */
	    if (ikey == NULL || ckey.dptr != ikey->dptr)
		free(ckey.dptr);

	    if (ikey && ikey->dptr != NULL)
		return YP_NOKEY;
	    else
		if (first_flag)
		    return YP_BADDB;
		else
		    return YP_FALSE;
	}
	
	if (all_flag || strncmp(dval->dptr, "YP_", 3) != 0)
	{
	    if (okey)
		*okey = ckey;
	    else if (ikey == NULL || ikey->dptr != ckey.dptr)
		free(ckey.dptr);
	    
	    return YP_TRUE;
	}

	/* Free old value */
	free(dval->dptr);
	
	nkey = gdbm_nextkey(dbp, ckey);

	/* Free old key, unless it comes from the caller! */
	if (ikey == NULL || ckey.dptr != ikey->dptr)
	    free(ckey.dptr);

	if (ckey.dptr == NULL)
	    return YP_NOMORE;

	ckey = nkey;
    }
}


/*
** Get the DateTimeModified value for a certain map database
*/
static unsigned long get_dtm(const char *domain,
			     const char *map)
{
    struct stat sbuf;
    char buf[1025];

    sprintf(buf, "%s/%s/%s", path_ypdb, domain, map);
    if (stat(buf, &sbuf) < 0)
	return 0;
    else
	return (unsigned long) sbuf.st_mtime;
}


/*
** YP function "MATCH" implementation
*/
ypresp_val *ypproc_match_2_svc(ypreq_key *key,
			       struct svc_req *rqstp)
{
    static ypresp_val result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_match():\n");
	fprintf(stderr, "\t\tdomainname = \"%s\"\n",
		key->domain);
	fprintf(stderr, "\t\tmapname = \"%s\"\n",
		key->map);
	fprintf(stderr, "\t\tkeydat = \"%.*s\"\n",
		(int) key->key.keydat_len,
		key->key.keydat_val);
    }

    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }
    
    if (key->domain[0] == '\0' || key->map[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum rdat, qdat;
	
	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    qdat.dsize = key->key.keydat_len;
	    qdat.dptr = key->key.keydat_val;

	    result.stat = read_database(dbp, &qdat, NULL, &rdat, 1);

	    if (result.stat == YP_TRUE)
	    {
		result.val.valdat_len = rdat.dsize;
		result.val.valdat_val = rdat.dptr;
	    }

	    gdbm_close(dbp);
	}
    }

    if (debug_flag)
    {
	if (result.stat == YP_TRUE)
	    fprintf(stderr, "\t-> Value = \"%.*s\"\n",
		    (int) result.val.valdat_len,
		    result.val.valdat_val);
	else
	    fprintf(stderr, "\t-> Error #%d\n", result.stat);
    }
    
    return &result;
}



ypresp_key_val *ypproc_first_2_svc(ypreq_key *key,
				   struct svc_req *rqstp)
{
    static ypresp_key_val result;


    if (debug_flag)
    {
	fprintf(stderr, "ypproc_first():\n");
	fprintf(stderr, "\tdomainname = \"%s\"\n", key->domain);
	fprintf(stderr, "\tmapname = \"%s\"\n", key->map);
	fprintf(stderr, "\tkeydat = \"%.*s\"\n",
		(int) key->key.keydat_len,
		key->key.keydat_val);
    }

    result.key.keydat_len = 0;
    if (result.key.keydat_val)
    {
	free(result.key.keydat_val);
	result.key.keydat_val = NULL;
    }
    
    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }

    if (key->map[0] == '\0' || key->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum dkey, dval;
	
	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    result.stat = read_database(dbp, NULL, &dkey, &dval, 0);

	    if (result.stat == YP_TRUE)
	    {
		result.key.keydat_len = dkey.dsize;
		result.key.keydat_val = dkey.dptr;
		
		result.val.valdat_len = dval.dsize;
		result.val.valdat_val = dval.dptr;
	    }
	    
	    gdbm_close(dbp);
	}
    }
    
    if (debug_flag)
    {
	if (result.stat == YP_TRUE)
	    fprintf(stderr, "\t-> Key = \"%.*s\", Value = \"%.*s\"\n",
		    (int) result.key.keydat_len,
		    result.key.keydat_val,
		    (int) result.val.valdat_len,
		    result.val.valdat_val);
		    
	else
	    fprintf(stderr, "\t-> Error #%d\n", result.stat);
    }
    
    return &result;
}


ypresp_key_val *ypproc_next_2_svc(ypreq_key *key,
				  struct svc_req *rqstp)
{
    static ypresp_key_val result;

    
    if (debug_flag)
    {
	fprintf(stderr, "ypproc_match():\n");
	fprintf(stderr, "\tdomainname = \"%s\"\n", key->domain);
	fprintf(stderr, "\tmapname = \"%s\"\n", key->map);
	fprintf(stderr, "\tkeydat = \"%.*s\"\n",
		(int) key->key.keydat_len,
		key->key.keydat_val);
    }

    result.key.keydat_len = 0;
    if (result.key.keydat_val)
    {
	free(result.key.keydat_val);
	result.key.keydat_val = NULL;
    }
    
    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }
    
    if (key->map[0] == '\0' || key->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum dkey, dval, okey;


	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    dkey.dsize = key->key.keydat_len;
	    dkey.dptr  = key->key.keydat_val;
	    
	    result.stat = read_database(dbp, &dkey, &okey, &dval, 0);

	    if (result.stat == YP_TRUE)
	    {
		result.key.keydat_len = okey.dsize;
		result.key.keydat_val = okey.dptr;
		
		result.val.valdat_len = dval.dsize;
		result.val.valdat_val = dval.dptr;
	    }
	    gdbm_close(dbp);
	}
    }

    if (debug_flag)
    {
	if (result.stat == YP_TRUE)
	    fprintf(stderr, "\t-> Key = \"%.*s\", Value = \"%.*s\"\n",
		    (int) result.key.keydat_len,
		    result.key.keydat_val,
		    (int) result.val.valdat_len,
		    result.val.valdat_val);
	else
	    fprintf(stderr, "\t-> Error #%d\n", result.stat);
    }

    return &result;
}

static void print_ypmap_parms(const struct ypmap_parms *pp)
{
    fprintf(stderr, "\t\tdomain   = \"%s\"\n", pp->domain);
    fprintf(stderr, "\t\tmap      = \"%s\"\n", pp->map);
    fprintf(stderr, "\t\tordernum = %u\n", pp->ordernum);
    fprintf(stderr, "\t\tpeer     = \"%s\"\n", pp->peer);
}


/*
** XXXX Implement Me!
*/
ypresp_xfr *ypproc_xfr_2_svc(ypreq_xfr *xfr,
			     struct svc_req *rqstp)
{
    static ypresp_xfr result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_xfr_2_svc():\n\tmap_parms:\n");
	print_ypmap_parms(&xfr->map_parms);
	fprintf(stderr, "\t\ttransid = %u\n", xfr->transid);
	fprintf(stderr, "\t\tprog = %u\n", xfr->prog);
	fprintf(stderr, "\t\tport = %u\n", xfr->port);
	fprintf(stderr, "\t-> Request denied.\n");
    }

    result.transid = xfr->transid;
    result.xfrstat = YPXFR_REFUSED;
    
    return &result;
}


void *ypproc_clear_2_svc(void *dummy,
			 struct svc_req *rqstp)
{
    static int foo;

    if (debug_flag)
    	fprintf(stderr, "ypproc_clear_2_svc()\n");

    return (void *) &foo;
}


static int ypall_close(void *data)
{
    if (debug_flag)
	if (data == NULL)
	    fprintf(stderr, "ypall_close() called with NULL pointer.\n");
	else
	    fprintf(stderr, "ypall_close() called.\n");
	    
    gdbm_close(*(GDBM_FILE *) data);
    return 0;
}


static int ypall_encode(ypresp_key_val *val,
			void *data)
{
    datum dkey, dval, okey;
    int status;

    if (debug_flag)
	fprintf(stderr, "ypall_encode() called with key \"%.*s\".\n",
		(int) val->key.keydat_len,
		val->key.keydat_val);
    
    dkey.dptr = val->key.keydat_val;
    dkey.dsize = val->key.keydat_len;
    
    status = read_database(*(GDBM_FILE *) data, &dkey, &okey, &dval, 0);

    if (status == YP_TRUE)
    {
	val->key.keydat_val = okey.dptr;
	val->key.keydat_len = okey.dsize;

	val->val.valdat_val = dval.dptr;
	val->val.valdat_len = dval.dsize;
    }
    
    return status;
}


ypresp_all *ypproc_all_2_svc(ypreq_nokey *nokey,
			     struct svc_req *rqstp)
{
    static ypresp_all result;
    extern xdr_ypall_cb_t xdr_ypall_cb;

    
    if (debug_flag)
    {
	fprintf(stderr, "ypproc_all_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
    }

    xdr_ypall_cb.u.encode = NULL;
    xdr_ypall_cb.u.close  = NULL;
    xdr_ypall_cb.data = NULL;
    
    result.more = TRUE;
    
    if (nokey->map[0] == '\0' || nokey->domain[0] == '\0')
	result.ypresp_all_u.val.stat = YP_BADARGS;
    else if (!is_valid_domain(nokey->domain))
	result.ypresp_all_u.val.stat = YP_NODOM;
    else
    {
	datum dkey, dval;
	
	GDBM_FILE dbp = open_database(nokey->domain, nokey->map);
	if (dbp == NULL)
	    result.ypresp_all_u.val.stat = YP_NOMAP;
	else
	{
	    result.ypresp_all_u.val.stat = read_database(dbp,
							 NULL,
							 &dkey,
							 &dval,
							 0);

	    if (result.ypresp_all_u.val.stat == YP_TRUE)
	    {
		result.ypresp_all_u.val.key.keydat_len = dkey.dsize;
		result.ypresp_all_u.val.key.keydat_val = dkey.dptr;
		
		result.ypresp_all_u.val.val.valdat_len = dval.dsize;
		result.ypresp_all_u.val.val.valdat_val = dval.dptr;

		xdr_ypall_cb.u.encode = ypall_encode;
		xdr_ypall_cb.u.close  = ypall_close;
		xdr_ypall_cb.data = (void *) &dbp;

		if (debug_flag)
		    fprintf(stderr, "Returning because YP_TRUE\n");
		
		return &result;
	    }

	    if (debug_flag)
		fprintf(stderr, "Closing ypall GDBM file.\n");
	    
	    gdbm_close(dbp);
	}
    }
    
    return &result;
}


ypresp_master *ypproc_master_2_svc(ypreq_nokey *nokey,
				   struct svc_req *rqstp)
{
    static ypresp_master result;

    
    if (debug_flag)
    {
	fprintf(stderr, "ypproc_master_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
    }

    if (result.peer)
    {
	free(result.peer);
	result.peer = NULL;
    }

    if (nokey->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(nokey->domain))
	result.stat = YP_NODOM;
    else
    {
	GDBM_FILE dbp = open_database(nokey->domain, nokey->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    datum key, val;

	    key.dsize = sizeof("YP_MASTER_NAME")-1;
	    key.dptr = "YP_MASTER_NAME";
	    
	    val = gdbm_fetch(dbp, key);
	    if (val.dptr == NULL)
	    {
		/* No YP_MASTER_NAME record in map? Assume we are Master */
		static char hostbuf[1025];

		gethostname(hostbuf, sizeof(hostbuf)-1);
		result.peer = strdup(hostbuf);
	    }
	    else
		result.peer = val.dptr;

	    result.stat = YP_TRUE;
	    gdbm_close(dbp);
	}
    }

    if (result.peer == NULL)
	result.peer = strdup("");

    if (debug_flag)
	fprintf(stderr, "\t-> Peer = \"%s\"\n", result.peer);
    
    return &result;
}


ypresp_order *ypproc_order_2_svc(ypreq_nokey *nokey,
				 struct svc_req *rqstp)
{
    static ypresp_order result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_order_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
    }

    result.ordernum = 0;

    if (nokey->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(nokey->domain))
	result.stat = YP_NODOM;
    else
    {
	GDBM_FILE dbp = open_database(nokey->domain, nokey->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    datum key, val;

	    key.dsize = sizeof("YP_LAST_MODIFIED")-1;
	    key.dptr = "YP_LAST_MODIFIED";
	    
	    val = gdbm_fetch(dbp, key);
	    if (val.dptr == NULL)
	    {
		/* No YP_LAST_MODIFIED record in map? Use DTM timestamp.. */
		result.ordernum = get_dtm(nokey->domain, nokey->map);
	    }
	    else
	    {
		result.ordernum = atoi(val.dptr);
		free(val.dptr);
	    }

	    result.stat = YP_TRUE;
	    gdbm_close(dbp);
	}
    }

    if (debug_flag)
	fprintf(stderr, "-> Order # %d\n", result.ordernum);
    
    return &result;
}


static void free_maplist(ypmaplist *mlp)
{
    ypmaplist *next;

    while (mlp != NULL)
    {
	next = mlp->next;
	free(mlp->map);
	free(mlp);
	mlp = next;
    }
}

static int add_maplist(ypmaplist **mlhp,
		       char *map)
{
    ypmaplist *mlp;

    mlp = malloc(sizeof(*mlp));
    if (mlp == NULL)
	return -1;

    mlp->map = strdup(map);
    if (mlp->map == NULL)
    {
	free(mlp);
	return -1;
    }

    mlp->next = *mlhp;
    *mlhp = mlp;

    return 0;
}


ypresp_maplist *ypproc_maplist_2_svc(domainname *name,
				     struct svc_req *rqstp)
{
    static ypresp_maplist result;
    

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_maplist_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", *name);
    }

    if (result.maps)
	free_maplist(result.maps);
    
    result.maps = NULL;

    if ((*name)[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(*name))
	result.stat = YP_NODOM;
    else
    {
	DIR *dp;

	dp = opendir(path_ypdb);
	if (dp == NULL)
	{
	    if (debug_flag)
		perror("opendir");
	    
	    result.stat = YP_BADDB;
	}
	else
	{
	    struct dirent *dep;
	    
	    while ((dep = readdir(dp)) != NULL)
		if (add_maplist(&result.maps, dep->d_name) < 0)
		{
		    result.stat = YP_YPERR;
		    break;
		}
	    
	    closedir(dp);
	}
    }

    if (debug_flag)
    {
	if (result.stat == YP_TRUE)
	{
	    ypmaplist *p;

	    p = result.maps;
	    fprintf(stderr, "-> ");
	    while (p)
	    {
		fprintf(stderr, "%s", p->map);
		if (p->next)
		    fprintf(stderr, ", ");
	    }
	    putc('\n', stderr);
	}
	else
	    fprintf(stderr, "\t-> Error #%d\n", result.stat);
    }
    return &result;
}
