/*
** yp_passwd.c           NIS Version 2 Passwd map access routines
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include "config.h"

#ifdef ENABLE_YP


#include <stdio.h>
#include <errno.h>
#include "yp_passwd.h"
#include "rpcsvc/ypclnt.h"


static char *xcopy(char **cp, char *str)
{
    char *start = *cp;
    
    if (str != NULL)
	while (*str)
	    *(*cp)++ = *str++;

    *(*cp)++ = '\0';
    return start;
}


static char *xstrtok(char *cp, int delim)
{
    static char *str = NULL;


    if (cp)
	str = cp;

    if (*str == '\0')
	return NULL;

    cp = str;
    while (*str && *str != delim)
	str++;

    if (*str)
	*str++ = '\0';

    return cp;
}


static struct passwd *pwent_parse(char *str, int len)
{
    static struct passwd pwd;
    extern char *strtok();
    static char buf[2048];
    

    /* Can't handle this -> let's just forget it... */
    if (len > 2040)
	return NULL;

    strncpy(buf, str, len);
    buf[len] = '\0';
    
    pwd.pw_name   = xstrtok(buf, ':');
    pwd.pw_passwd = xstrtok(NULL, ':');
    pwd.pw_uid    = atoi(xstrtok(NULL, ':'));
    pwd.pw_gid    = atoi(xstrtok(NULL, ':'));
    pwd.pw_gecos  = xstrtok(NULL, ':');
    pwd.pw_dir    = xstrtok(NULL, ':');
    pwd.pw_shell  = xstrtok(NULL, ':');

    return &pwd;
}


static int rewind_flag = 1;
static char *savekey = NULL;
static int savekeylen = 0;


void yp_setpwent(void)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


void yp_endpwent(void)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


struct passwd *yp_getpwent(void)
{
    struct passwd *pw;
    char *domain;
    char *result;
    int len;
    char *outkey;
    int keylen;

    
    if (yp_get_default_domain(&domain))
	return NULL;
    
    if (rewind_flag)
    {
	if (yp_first(domain, "passwd.byname",
		     &outkey, &keylen,
		     &result, &len))
	    return NULL;
	
	rewind_flag = 0;
	savekey = outkey;
	savekeylen = keylen;
    }
    else
    {
	if (yp_next(domain, "passwd.byname",
		    savekey, savekeylen, &outkey, &keylen,
		    &result, &len))
	    return NULL;
	
	free(savekey);
	savekey = outkey;
	savekeylen = keylen;
    }

    pw = pwent_parse(result, len);

    free(domain);
    free(result);
    
    return pw;
}


struct passwd *yp_getpwuid(uid_t uid)
{
    struct passwd *pw;
    char *domain;
    char *result;
    int len;
    char buf[256];


    sprintf(buf, "%d", uid);
    
    if (yp_get_default_domain(&domain))
	return NULL;
    
    if (yp_match(domain, "passwd.byuid", buf, strlen(buf), &result, &len))
	return NULL;

    pw = pwent_parse(result, len);

    free(domain);
    free(result);
    
    return pw;
}


struct passwd *yp_getpwnam(char *name)
{
    struct passwd *pw;
    char *domain;
    char *result;
    int len;

    if (yp_get_default_domain(&domain))
	return NULL;
    
    if (yp_match(domain, "passwd.byname", name, strlen(name), &result, &len))
	return NULL;

    pw = pwent_parse(result, len);
    
    free(domain);
    free(result);    
    return pw;
}


#endif /* ENABLE_YP */
