/*
** yp_hosts.c           NIS Version 2 Hosts map access routines
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Authors: Peter Eriksson <pen@signum.se>
**          Michael A. Griffith <grif@cs.ucr.edu>
*/

#include "config.h"

#ifdef ENABLE_YP


#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "yp_misc.h"
#include "yp_hosts.h"
#include "rpcsvc/ypclnt.h"


static struct hostent *hostent_parse(char *str, int len)
{
    static struct hostent host;
    extern char *strtok();
    static char buf[2048];
    static unsigned long addr;
#ifdef MULTIHOMED
    static char *addr_list[YP_MAX_HOST_ADDR];
#else 
    static char *addr_list[1];
#endif
    static char *alias_list[YP_MAX_HOST_ALIASES];
    static int i;
                        
    
    /* Can't handle this -> let's just forget it... */
    if (len > 2040)
	return NULL;

    strncpy(buf, str, len);
    buf[len] = '\0';

#ifdef LSB
    addr =  atoi(_yp_xstrtok(buf,  '.')) << 24;
    addr |= atoi(_yp_xstrtok(NULL, '.')) << 16;
    addr |= atoi(_yp_xstrtok(NULL, '.')) << 8;
    addr |= atoi(_yp_xstrtok(NULL, ' '));
#else
#error No MSB support yet.
#endif

#ifdef MULTIHOMED
#error No multihomed hosts yet.
#else
    host.h_length = 1;
    host.h_addr_list = (char **) &addr_list;
    host.h_addr_list[0] = (char *) htonl(addr);
#endif

    host.h_addrtype = AF_INET;
    host.h_aliases = alias_list;

    while ((host.h_name = _yp_strip_names(_yp_xstrtok(NULL, ' '))))
    {
       if (strlen(host.h_name) > 0)
	  break;
    }

    i = 0;
    while ((host.h_aliases[i] = _yp_strip_names(_yp_xstrtok(NULL, ' '))))
    {
       if (strlen(host.h_aliases[i]) > 0)
	   ++i;
    }
    host.h_aliases[i] = 0;
        
    return &host;
}


static int rewind_flag = 1;
static char *savekey = NULL;
static int savekeylen = 0;


void yp_sethostent(int stayopen, void *opts)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


void yp_endhostent(void *opts)
{
    rewind_flag = 1;
    if (savekey)
	free(savekey);
}


struct hostent *yp_gethostent(void *opts)
{
    struct hostent *host;
    char *map;
    char *domain;
    char *result;
    int len;
    char *outkey;
    int keylen;


    map = _ypopts_getmd(opts, "hosts", ".byname", &domain);
    if (map == NULL)
	return NULL;

    host = NULL;
    if (rewind_flag)
    {
	if (yp_first(domain, map,
		     &outkey, &keylen,
		     &result, &len))
	    goto error;
	
	rewind_flag = 0;
	savekey = outkey;
	savekeylen = keylen;
    }
    else
    {
	if (yp_next(domain, map,
		    savekey, savekeylen, &outkey, &keylen,
		    &result, &len))
	    goto error;
	
	free(savekey);
	savekey = outkey;
	savekeylen = keylen;
    }

    host = hostent_parse(result, len);
    free(result);

  error:
    free(map);
    free(domain);
    
    return host;
}


struct hostent *yp_gethostbyaddr(char *addr, int alen, int type, void *opts)
{
    struct hostent *host;
    char *map;
    char *domain;
    char *result;
    int len;
    char buf[256];

 
    map = _ypopts_getmd(opts, "hosts", ".byaddr", &domain);
    if (map == NULL)
	return NULL;

    strcpy(buf, _yp_printaddr(ntohl((*(unsigned long *) addr))));
                
    host = NULL;
    
    if (yp_match(domain, map, buf, strlen(buf), &result, &len) == 0)
    {
	host = hostent_parse(result, len);
	free(result);
    }

    free(map);
    free(domain);
    
    return host;
}


struct hostent *yp_gethostbyname(char *name, void *opts)
{
    struct hostent *host;
    char *map;
    char *domain;
    char *result;
    int len;

    map = _ypopts_getmd(opts, "hosts", ".byname", &domain);
    if (map == NULL)
	return NULL;

    host = NULL;
    if (yp_match(domain, map, name, strlen(name), &result, &len) == 0)
    {
	host = hostent_parse(result, len);
	free(result);
    }

    free(map);
    free(domain);
    return host;
}

#endif /* ENABLE_YP */


