/*
** nis_if.c                Cover routines for NIS+
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include "config.h"

#ifdef ENABLE_NISEMU

#include <stdio.h>
#include <rpc/rpc.h>
#include "rpcsvc/nis.h"
#include "nis_alloc.h"
#include "nis_conf.h"
#include "xalloc.h"


extern char *strstr();


static int nissetup_flag = 0;


int nis_setup(void)
{
    if (nissetup_flag)
	return 0;
    
    if (nisconf_set())
	return -1;

    nissetup_flag = 1;
    return 0;
}



nis_name nis_splitname(nis_name name, int *len, nis_attr **val)
{
    nis_name rname;
    char *cp;
    int size;


    if (name == NULL)
	return NULL;
    
    if (len)
	*len = 0;
    if (val)
	*val = NULL;

    size = 0;

    
    /* Not of "[key=value,key=value,...],foo.." format? */
    if  (name[0] != '[')
	return name;

    ++name;
    cp = strchr(name, ']');
    if (cp == NULL)
	return NULL;
    
    if (cp[1] == ',')
	rname = cp+2;
    else
	rname = cp+1;

    if (len == NULL || val == NULL)
	return rname;
    
    *cp = '\0';

    cp = strtok(name, ",");
    while (cp)
    {
	name = cp;
	cp = strchr(name, '=');
	if (!cp)
	    return NULL;
	*cp++ = '\0';

	if ((*len) + 1 >= size)
	{
	    size += 10;
	    if (Xalloc(val, size) == NULL)
		return NULL;
	}

	(*val)[*len].zattr_ndx = xstrdup(name);
	if (((*val)[*len].zattr_ndx) == NULL)
	    return NULL;
	    
	(*val)[*len].zattr_val.zattr_val_len = strlen(cp)+1;
	
	(*val)[*len].zattr_val.zattr_val_val = xstrdup(cp);
	if ((*val)[*len].zattr_val.zattr_val_val == NULL)
	    return NULL;
	
	(*len)++;
	
	cp = strtok(NULL, ",");
    }

    return rname;
}




nis_result *
  _nis_call(struct nis_result * (*cfh)(void *req, CLIENT *clh),
	    void *req)
			     
{
    struct nis_result *res;
    int nresp_flag = 0;
    CLIENT *clh;
    char *address;
    
    
    if (nis_setup())
	return NULL;

    clh = nisconf_domain2client(nis_local_directory(), NULL, &address);
    if (clh == NULL)
	return NULL;

    while ((res = (*cfh)(req, clh)) == NULL)
    {
	nresp_flag = 1;
	fprintf(stderr,
    "NIS: server \"%s\" not responding, still trying...\n",
		address);
    }

    if (nresp_flag == 1)
	fprintf(stderr, "NIS: server \"%s\" OK.\n",
		address);

    return res;
}


nis_result *nis_lookup(nis_name name, u_long flags)
{
    nis_result *res;
    struct ns_request nsreq;
    nis_name *names;
    int i;

    
    if (nis_setup())
	return NULL;

    if (flags & EXPAND_NAME)
    {
        names = nis_getnames(name);
        if (names == NULL)
            return NULL;

        res = NULL;
	i = 0;
	while (names[i] != NULL && (res == NULL || (res->status >= 2)))
        {
            nsreq.ns_name = names[i];
      
            nsreq.ns_object.ns_object_len = 0;
            nsreq.ns_object.ns_object_val = NULL;
	
            res = _nis_call(&nis_lookup_3, &nsreq);

	    i++;
        }

        if (res && res->status == NIS_NOT_ME)
            res->status = NIS_NOTFOUND;

    }
    else
    {
        nsreq.ns_name = name;
      
	nsreq.ns_object.ns_object_len = 0;
	nsreq.ns_object.ns_object_val = NULL;
	
	res = _nis_call(&nis_lookup_3, &nsreq);
    }

    return nis_dupresult(res);
}


struct ib_request *_nis_build_ib_request(nis_name name, 
					 struct ib_request *ibreq)
{
    ibreq->ibr_name = nis_splitname(name,
			 &ibreq->ibr_srch.ibr_srch_len,
			 &ibreq->ibr_srch.ibr_srch_val);
    
    if (ibreq->ibr_name == NULL)
      return NULL;

    ibreq->ibr_flags = 0;
    
    ibreq->ibr_obj.ibr_obj_len = 0;
    ibreq->ibr_obj.ibr_obj_val = NULL;
    
    ibreq->ibr_cbhost.ibr_cbhost_len = 0;
    ibreq->ibr_cbhost.ibr_cbhost_val = NULL;

    ibreq->ibr_bufsize = 0;

    ibreq->ibr_cookie.n_len = 0;
    ibreq->ibr_cookie.n_bytes = NULL;
    
    return ibreq;
}


nis_result *nis_first_entry(nis_name name)
{
    struct ib_request ibreq;
    nis_result *res;

    
    if (nis_setup() == -1)
	return NULL;

    if (_nis_build_ib_request(name, &ibreq) == NULL)
      return NULL;

    res = _nis_call(&nis_ibfirst_3, &ibreq);

    if (ibreq.ibr_srch.ibr_srch_val)
	nis_freeattrs(ibreq.ibr_srch.ibr_srch_val,
		      ibreq.ibr_srch.ibr_srch_len);

    return nis_dupresult(res);
}




nis_result *nis_next_entry(nis_name name, netobj *cookie)
{
    nis_result *res;
    struct ib_request ibreq;


    if (nis_setup() == -1)
	return NULL;
    
    if (_nis_build_ib_request(name, &ibreq) == NULL)
      return NULL;

    if (cookie)
      ibreq.ibr_cookie = *cookie;

    res = _nis_call(&nis_ibnext_3, &ibreq);

    if (ibreq.ibr_srch.ibr_srch_val)
	nis_freeattrs(ibreq.ibr_srch.ibr_srch_val,
		      ibreq.ibr_srch.ibr_srch_len);

    return nis_dupresult(res);
}

#endif /* ENABLE_NISEMU */
