/*  Copyright (c) 1995 John E. Davis (davis@space.mit.edu)
 *  All rights reserved.
 */
#include <config.h>
#include "features.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>


#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <slang.h>

#include "clientlib.h"
#include "nntp.h"
#include "misc.h"
#include "slrn.h"

int Slrn_Can_Post;
int Slrn_Query_Reconnect = 1;
int Slrn_NNTP_Has_XOver = 0;
static int Has_XPat = -1;	       /* -1 means probe needs performed */
static int Has_Xhdr = -1;
static int Has_Xgtitle = -1;

static int Reconnect_Ok;
static char Last_Reconnect_Command[256];

/* This function does not attempt a reconnection.  A reconnection can be
 * achieved only in a definite context.  
 */
int slrn_put_server_cmd (char *line, char *buf, unsigned int len)
{
   Reconnect_Ok = 0;
   if (-1 == client_put_server (line))
     {
	slrn_exit_error ("Server closed connection on %s command.", line);
     }
   *buf = 0;
   slrn_get_server (buf, len, 1);
   return atoi (buf);
}


int slrn_get_server (char *line, unsigned int len, int fatal)
{
   int ret;
   *line = 0;
   while (1)
     {
	errno = 0;
	ret = client_get_server (line, (int) len);
	if (ret == -1)
	  {
#ifdef EAGAIN
	     if (errno == EAGAIN)
	       {
		  sleep (1);
		  continue;
	       }
#endif
#ifdef EINTR
	     if (errno == EINTR)
	       {
		  slrn_error ("read aborted.");
		  return -1;
	       }
#endif
	     if (fatal) slrn_exit_error ("Read failed from server. (%d)", errno);
	     return -1;
	  }
	else return ret;
     }
}

static int nntp_attempt_reconnect (void)
{
   char buf[256];
   
   while (1)
     {
	if (Slrn_Query_Reconnect 
	    && (slrn_get_yesno (1, "NNTP connection dropped -- reconnect") <= 0))
	  return -1;
	
	if (nntp_initialize_server (Slrn_Server_Name) != 0)
	  continue;
	
	if ((NULL != Slrn_Current_Group) && ('\0' != *Slrn_Current_Group))
	  {
	     sprintf (buf, "GROUP %s", Slrn_Current_Group);
	     if ((-1 == client_put_server (buf))
		 || (-1 == slrn_get_server (buf, sizeof (buf) - 1, 0)))
	       continue;
	  }
	break;
     }
   
   return 0;
}

static int put_server_with_reconnect (char *cmd, char *buf, unsigned int len)
{
   Reconnect_Ok = 1;
   strcpy (Last_Reconnect_Command, cmd);
   
   while (1)
     {
	*buf = 0;
	if ((-1 == client_put_server (Last_Reconnect_Command))
	    || (-1 == slrn_get_server (buf, len - 1, 0))
	    || (ERR_FAULT == atoi (buf)))
	  {
	     if (-1 == nntp_attempt_reconnect ())
	       {
		  slrn_exit_error ("Server connection closed. (%s)", Last_Reconnect_Command);
	       }
	  }
	else break;
     }
   
   return atoi(buf);
}


int nntp_select_group (char *name, int *min, int *max)
{
   char group[256];
   int estim;
   int code;
   
   sprintf (group, "GROUP %s", name);
   
   code = put_server_with_reconnect (group, group, sizeof (group));
   
   /* responses 211 and 411 as well as 502 */
   if (code != OK_GROUP) 
     {
	if (code == ERR_ACCESS)
	  {
	     *min = *max = 0;
	     return 0;
	  }
	return -1;
     }
   
   if (3 != sscanf(group + 4, "%d %d %d", &estim, min, max)) return -1;

   return 0;
}

static int Server_Inited = 0;
void nntp_close_server (void)
{
   client_close_server ();
   Server_Inited = 0;
}

static int nntp_authorization (void)
{
   char line[NNTP_STRLEN];
   
   sprintf (line, "authinfo user %s", Slrn_User_Info.nnrpname);
   
   if (NEED_AUTHDATA == slrn_put_server_cmd (line, line, sizeof (line)))
     {
	sprintf (line, "authinfo pass %s", Slrn_User_Info.nnrppass);
	
	switch (slrn_put_server_cmd (line, line, sizeof (line)))
	  {
	   case ERR_ACCESS:
	     return -1;
	   case OK_AUTH:
	     Slrn_Can_Post = 1;
	     return 0;
	  }
     }
   return 0;
}

static int probe_server (char *cmd)
{
   char line[256];
   
   if (ERR_COMMAND == put_server_with_reconnect (cmd, line, sizeof (line)))
     return 0;
   return 1;
}

#define PROBE_XCMD(var, cmd) (((var) != -1) ? (var) : ((var) = probe_server(cmd)))

int nntp_has_cmd (char *cmd)
{
   if (!strcmp (cmd, "XHDR"))
     return PROBE_XCMD(Has_Xhdr, cmd);
   
   if (!strcmp (cmd, "XPAT"))
     return PROBE_XCMD(Has_XPat, cmd);
   
   if (!strcmp (cmd, "XGTITLE"))
     return PROBE_XCMD(Has_Xgtitle, cmd);
       
   return probe_server (cmd);
}


int nntp_initialize_server (char *host)
{
   char line[256];
   int code;
   char *str;
   char msg1[] = "Connecting to server at %s ...";
   char msg2[] = "Unable to initialize server: %s";
   char msg4[] = "Authenticating %s...";
   int tt_inited = (Slrn_TT_Initialized & 1);
   
   if (Server_Inited) nntp_close_server ();
   
   if (tt_inited == 0)
     slrn_tty_message (0, msg1, host);
   else
     {
	slrn_message ("Connecting to server at %s ...", host);
	slrn_smg_refresh ();
     }
   
   if ((code = client_server_init (host, line, sizeof(line))) == -1)
     {
	if (tt_inited == 0)
	  slrn_tty_message (2, "failed!");
	else
	  {
	     slrn_message ("Connecting to server at %s ... failed!", host);
	     slrn_smg_refresh ();
	  }
	return -1;
     }
   Server_Inited = 1;
   
   switch (code)
     {
      case OK_CANPOST:
	Slrn_Can_Post = 1;
	str = "Posting ok.";
	break;
      case OK_NOPOST:
	Slrn_Can_Post = 0;
	str = "Posting NOT ok.";
	break;
      default:
	if (tt_inited)
	  {
	     slrn_message (msg2, line);
	     slrn_smg_refresh ();
	  }
	else slrn_tty_message (1 | 2, msg2, line);
	return -1;
     }
   
   if (tt_inited)
     {
	slrn_message (str);
	slrn_smg_refresh ();
     }
   else slrn_tty_message (1 | 2, str);
   
   /* Apparantly mode reader should occur before the authorization code.  The
    * reason is that after receiving mode reader, the server will fork a
    * process that actually performs the role of server.  It is what needs
    * authorization.
    */
   
   if (ERR_ACCESS == slrn_put_server_cmd ("mode reader", line, sizeof (line)))
     {
	char msg3[] = "NNTP: 'mode reader' failed: %s";
	if (tt_inited)
	  {
	     slrn_message (msg3, line);
	     slrn_smg_refresh ();
	  }
	else slrn_tty_message (2, msg3);
 	return (-1);
     }

   if ((Slrn_User_Info.nnrpname != NULL)
       && (*Slrn_User_Info.nnrpname != 0))
     {
	if (tt_inited == 0)
	  slrn_tty_message (0, msg4, Slrn_User_Info.nnrpname);
	else
	  {
	     slrn_message (msg4, host);
	     slrn_smg_refresh ();
	  }
	
	/* Authenticate ourselves. If we fail the server will disconnect us
	 so we need to reconnect if we were disconnected */
	if (nntp_authorization() < 0)
	  {
	     /* The server probably hung up on us.  Check and see.
 	      * If it did, then we will login again as a default user.
 	      * Send the date command to see if the server is still alive.
	      */
	     if ((-1 == client_put_server("date"))
		 || (-1 == slrn_get_server(line,sizeof(line),0)))
 	       {
		  /* Yup it hung up on us. */
		  if (tt_inited)
		    {
		       slrn_message ("Failed!! Using Default.");
		       slrn_smg_refresh ();
		    }
		  else
		    {
		       slrn_tty_message (0, "Failed!! Using Default.\r\n");
		    }
		  line[0] = '\0';
		  if ( -1 == client_server_init (host, line, sizeof(line) ))
		    {
		       return -1;
		    }
 	       }
	  }
	else
	  {
	     if (tt_inited)
	       {
		  slrn_message ("Succeeded");
		  slrn_smg_refresh ();
	       }
	     else slrn_tty_message (0, "Succeeded");
	  }
     }

   
   if (ERR_COMMAND == slrn_put_server_cmd ("XOVER", line, sizeof (line)))
     {
	char msg5[] = "Server %s does not implement the XOVER command.";
	Slrn_NNTP_Has_XOver = 0;
	if (tt_inited)
	  {
	     slrn_message (msg5, host);
	     slrn_smg_refresh();
	  }
	else slrn_tty_message (2, msg5, host);
     }
   else Slrn_NNTP_Has_XOver = 1;
   
   return 0;
}

char *nntp_read_line (char *line, unsigned int len)
{
   /* This needs to be modified if Reconnection_Ok is non-zero.  Somehow,
    * the calling routine needs to be told to re-issue the NNTP command 
    * and restart.
    */
   slrn_get_server (line, len, 1);
   
   if ((*line == '.') && (*(line + 1) == 0)) return NULL;
   return line;
}

int nntp_xpat_cmd (char *hdr, int rmin, int rmax, char *pat)
{
   char buf[512];
   
   if (0 == PROBE_XCMD(Has_XPat, "XPAT"))
     return -1;
   
   sprintf (buf, "XPAT %s %d-%d *%s*", hdr, rmin, rmax, pat);
   if (OK_HEAD != put_server_with_reconnect (buf, buf, sizeof (buf)))
     return -1;
   return 0;
}

int nntp_xgtitle_cmd (char *pattern)
{
   char buf[512];
   
   /* XGTITLE appears broken on some servers.  So, do not probe for it. */
   if (Has_Xgtitle == 0) return -1;
   
   if (Has_Xgtitle == -1)
     {
	slrn_message ("Sending Query to Server...");
	slrn_smg_refresh ();
     }
   
   sprintf (buf, "XGTITLE %s", pattern);
      
   if (OK_XGTITLE != put_server_with_reconnect (buf, buf, sizeof (buf)))
     {
	slrn_message ("Server does not support XGTITLE command.");
	Has_Xgtitle = 0;
	return -1;
     }
   
   Has_Xgtitle = 1;
   return 0;
}

int nntp_select_article (int n, char *msgid)
{
   char cmd[256];
   
   if (n != -1)
     sprintf (cmd, "ARTICLE %d", n);
   else
     sprintf (cmd, "ARTICLE %s", msgid);
   
   if (OK_ARTICLE == put_server_with_reconnect (cmd, cmd, sizeof (cmd)))
     return 0;
   else return -1;
}

static int XOver_Done;
static int XOver_Min;
static int XOver_Max;
static int XOver_Next;

static int Suspend_XOver_For_Kill = 0;

/* We cannot reconnect on this because of the context may be too complex. */
int nntp_open_xover (int min, int max)
{
   char buf[512];
   if (Slrn_NNTP_Has_XOver && !Suspend_XOver_For_Kill)
     {
	sprintf (buf, "XOVER %d-%d", min, max);
	
	if (OK_XOVER != slrn_put_server_cmd (buf, buf, sizeof (buf)))
	  {
	     XOver_Done = 1;
	     slrn_error ("XOVER failed.");
	     return -1;
	  }
     }
   XOver_Next = XOver_Min = min;
   XOver_Max = max;
   XOver_Done = 0;
   return 0;
}

/* RFC850: The  required  headers  are
 * Relay-Version,  Posting-Version,  From,  Date, Newsgroups,
 * Subject,  Message-ID,  Path. */

#define MAX_XOVER_SIZE 1024
typedef struct
{
   char name[25];
   unsigned int colon;
   char buf[MAX_XOVER_SIZE + 16];
}
Header_Xover_Type;

/* These MUST be arranged in XOver format:
 * id|subj|from|date|msgid|refs|bytes|line|misc stuff
 *
 * !!!DO NOT REARRANGE OR ADD ANYTHING WITHOUT UPDATING THE ARRAY REFERENCES!!!
 */

#define NUM_XOVER_HEADERS 7
Header_Xover_Type Xover_Headers[NUM_XOVER_HEADERS] =
{
     {"Subject:", 8, ""},
     {"From:", 5, ""},
     {"Date:", 5, ""},
     {"Message-ID:", 11, ""},
     {"References:", 11, ""},
     /* {"Bytes:", 6, ""}, */
     {"Lines:", 6, ""},
     {"Xref:", 5, ""}
};

/* The rest of these are for scoring/killing */
#define MAX_EXTRA_XOVER_HEADERS 20
Header_Xover_Type *Extra_Xover_Headers[MAX_EXTRA_XOVER_HEADERS];
static unsigned int Next_Extra_Xover_Header;

void nntp_open_suspend_xover (void)
{
   Suspend_XOver_For_Kill = 1;
   Next_Extra_Xover_Header = 0;
}

void nntp_close_suspend_xover (void)
{
   unsigned int i;
   
   if (Suspend_XOver_For_Kill == 0) return;
   Suspend_XOver_For_Kill = 0;
   for (i = 0; i < MAX_EXTRA_XOVER_HEADERS; i++)
     {
	if (Extra_Xover_Headers[i] != NULL)
	  SLFREE (Extra_Xover_Headers[i]);
	Extra_Xover_Headers[i] = NULL;
     }
   Next_Extra_Xover_Header = 0;
}

char *nntp_get_extra_xover_header (char *hdr)
{
   unsigned int i;
   unsigned int colon = strlen (hdr) + 1;
   for (i = 0; i < Next_Extra_Xover_Header; i++)
     {
	Header_Xover_Type *xov = Extra_Xover_Headers[i];
	if ((colon == xov->colon)
	    && (!slrn_case_strcmp ((unsigned char *)xov->name,
				   (unsigned char *)hdr)))
	  return xov->buf;
     }
   return NULL;
}



static char *read_create_xover_line (int id, char *the_buf)
{
   char *hbuf, *b;
   char buf[8 * MAX_XOVER_SIZE];
   unsigned int colon = 0;
   char *p, ch;
   unsigned int group_len;
   int i, headers_not_found;
   Header_Xover_Type *xov = NULL;
   
   for (i = 0; i < NUM_XOVER_HEADERS; i++)
     {
	*Xover_Headers[i].buf = 0;
     }
   Next_Extra_Xover_Header = 0;
   
   headers_not_found = NUM_XOVER_HEADERS;
   
   hbuf = NULL;
   
   while (NULL != nntp_read_line (buf, sizeof(buf)))
     {
	unsigned int maxlen;
	ch = *buf;
	
	if ((ch == ' ') || (ch == '\t'))
	  {
	     if (hbuf == NULL) continue;
	     if (hbuf - xov->buf >= MAX_XOVER_SIZE - 2)
	       {
		  hbuf = NULL;
		  continue;
	       }
	     
	     *hbuf++ = ' ';	       /* one blank to separate */
	     *hbuf = 0;
	     colon = 0;
	     /* Drop to add continuation */
	  }
	else
	  {
	     if (headers_not_found) i = 0;
	     else i = NUM_XOVER_HEADERS;
	     
	     while (i < NUM_XOVER_HEADERS)
	       {
		  if ((ch == *Xover_Headers[i].name)
		      && !slrn_case_strncmp ((unsigned char *)Xover_Headers[i].name,
					     (unsigned char *)buf, Xover_Headers[i].colon))
		    {
		       headers_not_found--;
		       
		       xov = Xover_Headers + i;
		       colon = xov->colon;
		       hbuf = xov->buf;
		       break;
		    }
		  i++;
	       }
	     
	     if (i == NUM_XOVER_HEADERS)
	       {
		  if ((!Suspend_XOver_For_Kill)
		      || (Next_Extra_Xover_Header == MAX_EXTRA_XOVER_HEADERS))
		    {
		       hbuf = NULL;
		       continue;
		    }
		  
		  xov = Extra_Xover_Headers[Next_Extra_Xover_Header];
		  if (xov == NULL)
		    {
		       xov = (Header_Xover_Type *) SLMALLOC (sizeof(Header_Xover_Type));
		       if (xov == NULL)
			 {
			    hbuf = NULL;
			    continue;
			 }
		       Extra_Xover_Headers[Next_Extra_Xover_Header] = xov;
		    }
		  
		  b = buf;
		  while (*b && (*b != ':')) b++;
		  if (*b != ':')
		    {
		       hbuf = NULL;
		       continue;
		    }
		  *b++ = 0;
		  colon = (b - buf);
		  
		  strncpy (xov->name, buf, sizeof (xov->name) - 1);
		  xov->name[sizeof(xov->name) - 1] = 0;
		  xov->colon = colon;
		  
		  Next_Extra_Xover_Header += 1;
		  hbuf = xov->buf;
	       }
	  }
	
	
	
	/* trim trailing whitespace */
	(void) slrn_trim_string (buf);
	
	/* trim leading whitespace */
	b = slrn_skip_whitespace (buf + colon);
	
	maxlen = (MAX_XOVER_SIZE - (unsigned int)(hbuf - xov->buf)) - 1;
	strncpy (hbuf, b, maxlen);
	hbuf[maxlen] = 0;
	
	while (*hbuf)
	  {
	     if (*hbuf == '\t') *hbuf = ' ';
	     hbuf++;
	  }
     }
   
   /* we should try to extract the id from the Xref header if possible */
   if (id == -1)
     {
	char *xref = Xover_Headers[6].buf;
	if (*xref != 0)
	  {
	     group_len = strlen (Slrn_Current_Group);
	     p = xref;
	     while ((ch = *p) != 0)
	       {
		  if (ch == ' ')
		    {
		       p++;
		       continue;
		    }
		  if (!strncmp (p, Slrn_Current_Group, group_len))
		    {
		       p += group_len;
		       if (*p == ':')
			 {
			    p++;
			    id = atoi (p);
			    break;
			 }
		    }
		  /* skip to next space */
		  while (((ch = *p) != 0) && (ch != ' ')) p++;
	       }
	  }
	
	if ((id == -1) && PROBE_XCMD(Has_XPat, "XPAT"))
	  {
	     char xpatbuf[256];
	     char *msgid = Xover_Headers[3].buf;
	     if (-1 != nntp_xpat_cmd ("Message-Id", Slrn_Server_Min, Slrn_Server_Max, msgid))
	       {
		  /* This should only loop once. */
		  while (nntp_read_line (xpatbuf, sizeof (xpatbuf) - 1) != NULL)
		    {
		       id = atoi (xpatbuf);
		    }
	       }
	  }
     }
   
   
   /* put in standard XOVER format:
    * id|subj|from|date|msgid|refs|bytes|line|misc stuff
    */
   
   sprintf (the_buf, "%d\t%s\t%s\t%s\t%s\t%s\t%s\t%s\tXref: %s",
	    id,
	    Xover_Headers[0].buf,      /* subject */
	    Xover_Headers[1].buf,      /* from */
	    Xover_Headers[2].buf,      /* date */
	    Xover_Headers[3].buf,      /* msgid */
	    Xover_Headers[4].buf,      /* references */
	    "",			       /* bytes */
	    Xover_Headers[5].buf,      /* lines */
	    Xover_Headers[6].buf       /* xref */
	    );
   return the_buf;
}

char *nntp_read_xover (char *the_buf, unsigned int len)
{
   char buf[512];
   int id;
   int already_tried = 0;
   
   if (Slrn_NNTP_Has_XOver && !Suspend_XOver_For_Kill)
     {
	return nntp_read_line (the_buf, len);
     }
   
   if (XOver_Next > XOver_Max) return NULL;
   
   while (XOver_Done == 0)
     {
	sprintf (buf, "HEAD %d", XOver_Next);
	if (OK_HEAD != slrn_put_server_cmd (buf, buf, sizeof(buf)))
	  {
	     already_tried = 0;
	     /* This is ugly */
	     
	     /* If a head fails even though server says it has article, jump
	      * back and go on.
	      */
	     server_is_messed_up_head_failed:
	     
	     do
	       {
		  if (OK_NOTEXT != slrn_put_server_cmd ("NEXT", buf, sizeof (buf)))
		    {
		       if (atoi(buf) == ERR_NONEXT)
			 {
			    XOver_Done = 1;
			    return NULL;
			 }
		       
		       slrn_exit_error ("Server failed NEXT request.");
		    }
		  
		  id = atoi (buf + 4);
		  /* Try going back 5 articles */
		  if ((already_tried == 0) && (id < XOver_Next - 5))
		    {
		       
		       sprintf (buf, "HEAD %d", XOver_Next - 5);
		       
		       if (OK_HEAD == slrn_put_server_cmd (buf, buf, sizeof(buf)))
			 {
			    /* read all header */
			    while (nntp_read_line (buf, sizeof(buf)) != NULL)  ;
			 }
		       already_tried = 1;
		    }
	       }
	     while (id < XOver_Next);
	     
	     
	     if (OK_HEAD != slrn_put_server_cmd ("HEAD", buf, sizeof(buf)))
	       {
		  sprintf (buf, "HEAD %d", id);
		  
		  if (OK_HEAD != slrn_put_server_cmd (buf, buf, sizeof(buf)))
		    {
		       already_tried++;
		       if (already_tried > 10)
			 {
			    slrn_exit_error ("Server Error: Head failed.");
			 }
		       
		       goto server_is_messed_up_head_failed;
		    }
	       }
	  }
	/* extract article number */
	id = atoi(buf + 4);
	if (id >= XOver_Min) break;
	
	/* we just did a head.  Read it and do next until range is ok */
	while ((0 == slrn_get_server (buf, sizeof(buf), 1)) && (*buf != '.'))
	  {
	     ;
	  }
	
	while (id < XOver_Min)
	  {
	     if (OK_NOTEXT == slrn_put_server_cmd ("NEXT", buf, sizeof (buf)))
	       {
		  id = atoi(buf + 4);
		  continue;
	       }
	     XOver_Done = 1;
	     break;
	  }
     }
   
   if (XOver_Done) return NULL;
   if (id > XOver_Max)
     {
	/* clear out header text */
	XOver_Done = 1;
	while (nntp_read_line (buf, sizeof(buf)) != NULL)  ;
	return NULL;
     }
   
   read_create_xover_line (id, the_buf);
   
   if (id > XOver_Max)
     {
	XOver_Done = 1;
	return NULL;
     }
   
   
   if (OK_NOTEXT == slrn_put_server_cmd ("NEXT", buf, sizeof (buf)))
     {
	XOver_Next = atoi(buf + 4);
	if (XOver_Next > XOver_Max)
	  {
	     XOver_Done = 1;
	  }
     }
   else XOver_Done = 1;
   return the_buf;
}

void nntp_close_xover ()
{
   XOver_Done = 1;
}


char *nntp_head_from_msgid (char *msgid, char *buf, unsigned int len)
{
   int id;
   
   if ((msgid == NULL) || (*msgid == 0)) return NULL;
   
   sprintf (buf, "HEAD %s", msgid);
   if (OK_HEAD != put_server_with_reconnect (buf, buf, len))
     return NULL;
   
   id = atoi(buf + 4);
   if (id == 0) id = -1;
   return read_create_xover_line (id, buf);
}

int nntp_start_post (void)
{
   char line[256];
   
   if (CONT_POST != put_server_with_reconnect ("POST", line, sizeof (line)))
     {
	slrn_error ("Posting not allowed.");
	return -1;
     }
   return 0;
}

int nntp_end_post (void)
{
   char line[256];
   
#ifdef VMS
   netwrite (SERVER_WRITE_FP, "\r\n.\r\n", 5);
#else
   fputs("\r\n.\r\n", SERVER_WRITE_FP);
   fflush (SERVER_WRITE_FP);
#endif
   slrn_get_server (line, 255, 1);
   if (atoi(line) != 240)
     {
	slrn_error ("Article rejected: %s", line);
	return -1;
     }
   return 0;
}

int nntp_printf (char *fmt, ...)
{
   va_list ap;
#ifdef VMS
   char buf[512];
   va_start (ap, fmt);
   vsprintf (buf, fmt, ap);
   netwrite (SERVER_WRITE_FP, buf, strlen (buf));
#else
   va_start (ap, fmt);
   vfprintf (SERVER_WRITE_FP, fmt, ap);
#endif
   return 0;
}

int nntp_puts (char *s)
{
#ifndef VMS
   fputs (s, SERVER_WRITE_FP);
#else
   netwrite (SERVER_WRITE_FP, s, strlen (s));
#endif
   return 0;
}

int nntp_xhdr_command (char *hdr, int num, char *buf, unsigned int buflen)
{
   char cmd[256], tmpbuf[1024];
   int found;
   unsigned int colon;
   
   if (PROBE_XCMD(Has_Xhdr, "XHDR"))
     {
	char *b, ch;
	sprintf (cmd, "XHDR %s %d", hdr, num);
	if (OK_HEAD != put_server_with_reconnect (cmd, buf, buflen))
	  return -1;
	
	if (NULL == nntp_read_line (tmpbuf, sizeof(tmpbuf))) return -1;
	
	/* skip past article number */
	b = tmpbuf;
	while (((ch = *b++) >= '0') && (ch <= '9'))
	  ;
	strncpy (buf, b, buflen - 1);
	buf[buflen - 1] = 0;
	
	/* I should handle multi-line returns but I doubt that there will be
	 * any for our use of xhdr
	 */
	while (nntp_read_line(tmpbuf, sizeof (tmpbuf)) != NULL)
	  ;
	return 0;
     }
   
   sprintf (cmd, "HEAD %d", num);
   if (OK_HEAD != put_server_with_reconnect (cmd, buf, buflen))
     return -1;
   
   found = 0;
   colon = strlen (hdr);
   
   while (NULL != nntp_read_line (tmpbuf, sizeof (tmpbuf)))
     {
	char *b;
	if (found
	    || slrn_case_strncmp ((unsigned char *) tmpbuf, (unsigned char *) hdr, colon)
	    || (tmpbuf[colon] != ':'))
	  continue;
	
	found = 1;
	
	b = tmpbuf + colon;
	if (*b == ' ') b++;
	strncpy (buf, b, buflen - 1);
	buf[buflen - 1] = 0;
     }
   return 0;
}


int nntp_list_newsgroups (void)
{
   char buf[256];
   
   if (OK_GROUPS != slrn_put_server_cmd ("LIST NEWSGROUPS", buf, sizeof (buf)))
     {
	slrn_exit_error ("Server failed to return proper response to LIST NEWGROUPS");
     }
   return 0;
}

int nntp_list_active (void)
{
   char buf[256];

   if (OK_GROUPS != slrn_put_server_cmd ("LIST", buf, sizeof (buf)))
     {
	slrn_exit_error("List failed: %s", buf);
     }
   return 0;
}

