/*
    GRAPH.I
    Declarations of Yorick graphics functions.

    $Id: graph.i,v 1.1 1993/08/27 18:32:09 munro Exp $
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/*--------------------------------------------------------------------------*/
/* Control functions */

extern window;
/* DOCUMENT window, n, display="host:server.screen", dpi=100/75, wait=0/1,
                       private=0/1, hcp="hcp_filename", dump=0/1,
                       legends=1/0, style="style_sheet_filename"
     select window N as the current graphics output window.  N may
     range from 0 to 7, inclusive.  Each graphics window corresponds to
     an X window, and optionally has its own associated hardcopy file.
     If N is omitted, it defaults to the current coordinate system.

     The X window will appear on your default display at 75 dpi, unless
     you specify the display and/or dpi keywords.  A dpi=100 X window
     is larger than a dpi=75 X window; both represent the same thing
     on paper.  Use display="" to create a graphics window which has
     no associated X window (you should do this if you want to make
     plots in a non-interactive batch mode).

     By default, an X window will attempt to use shared colors, which
     permits several Yorick graphics windows (including windows from
     multiple instances of Yorick) to use a common palette.  You can
     force an X window to post its own colormap (set its colormap
     attribute) with the private=1 keyword.  You will most likely have
     to fiddle with your window manager to understand how it handles
     colormap focus if you do this.  Use private=0 to return to shared
     colors.

     By default, Yorick will not wait for the X window to become visible;
     code which creates a new window, then plots a series of frames to
     that window should use wait=1 to assure that all frames are actually
     plotted.

     By default, a graphics window does NOT have a hardcopy file
     of its own -- any request for hardcopy are directed to the
     default hardcopy file, so hardcopy output from any window goes
     to a single file.  By specifying the hcp keyword, however, a
     hardcopy file unique to this window will be created.  If the
     "hcp_filename" ends in ".ps", the hardcopy file will be a PostScript
     file; otherwise, hardcopy files are in binary CGM format.  Use
     hcp="" to revert to the default hardcopy file (closing the window
     specific file, if any).  The legends keyword, if present, controls
     whether the curve legends are (legends=1, the default) or are not
     (legends=0) dumped to the hardcopy file.  The dump keyword, if
     present, controls whether all colors are converted to a gray scale
     (dump=0, the default), or the current palette is dumped at the
     beginning of each page of hardcopy output.  (The legends keyword
     applies to all pictures dumped to hardcopy from this graphics
     window.  The dump keyword applies only to the specific hardcopy
     file defined using the hcp keyword -- use the dump keyword in the
     hcp_file command to get the same effect in the default hardcopy
     file.)

     If both display="" and hcp="", the graphics window will be
     entirely eliminated.

     The style keyword, if present, specifies the name of a Gist style
     sheet file; the default is "work.gs".  The style sheet determines
     the number and location of coordinate systems, tick and label styles,
     and the like.  Other choices include "axes.gs", "boxed.gs",
     "work2.gs", and "boxed2.gs".

     If invoked as a function, window(...) returns the current
     window number.
   SEE ALSO: plsys, hcp_file, fma, hcp, redraw, palette, animate, plg,
             winkill, gridxy
 */

func winkill(n)
/* DOCUMENT winkill
         or winkill, n
     deletes the current graphics window, or graphics window N (0-7).
   SEE ALSO: window
 */
{
  window, n, display="", hcp="";
}

extern current_window;
/* DOCUMENT n= current_window()
     returns the number of the current graphics window, or -1 if none.
 */

extern hcp_file;
/* DOCUMENT hcp_file, filename, dump=0/1
     sets the default hardcopy file to FILENAME.  If FILENAME ends with
     ".ps", the file will be a PostScript file, otherwise it will be a
     binary CGM file.  By default, the hardcopy file name will be
     "Aa00.cgm", or "Ab00.cgm" if that exists, or "Ac00.cgm" if both
     exist, and so on.  The default hardcopy file gets hardcopy from all
     graphics windows which do not have their own specific hardcopy file
     (see the window command).  If the dump keyword is present and non-zero,
     the current palette will be dumped at the beginning of each frame
     of the default hardcopy file.  With dump=0, the default behavior of
     converting all colors to a gray scale is restored.
   SEE ALSO: window, fma, hcp, plg
 */

extern hcp_finish;
/* DOCUMENT filename= hcp_finish()
         or filename= hcp_finish(n)
     closes the current hardcopy file and returns the filename.
     If N is specified, closes the hcp file associated with window N
     and returns its name; use hcp_finish(-1) to close the default
     hardcopy file.
   SEE ALSO: window, fma, hcp, hcp_out, plg
 */

func hcp_out(n,keep=)
/* DOCUMENT hcp_out
         or hcp_out, n
     finishes the current hardcopy file and sends it to the printer.
     If N is specified, prints the hcp file associated with window N;
     use hcp_out,-1 to print the default hardcopy file.
     Unless the KEEP keyword is supplied and non-zero, the file will
     be deleted after it is processed by gist and sent to lpr.
   SEE ALSO: window, fma, hcp, hcp_finish, plg
 */
{
  filename= hcp_finish();
  if (filename) {
    if (strpart(filename,-2:0)==".ps")
      system, swrite(format=LPR_FORMAT, filename);
    else
      system, swrite(format=GIST_FORMAT, filename);
    if (!keep) remove, filename;
  }
}

func eps(name)
/* DOCUMENT eps, name
     writes the picture in the current graphics window to the Encapsulated
     PostScript file NAME+".epsi" (i.e.- the suffix .epsi is added to NAME).
     The eps function requires the ps2epsi utility which comes with the
     project GNU Ghostscript program.  Any hardcopy file associated with
     the current window is first closed, but the default hardcopy file is
     unaffected.  As a side effect, legends are turned off and color table
     dumping is turned on for the current window.
     The external variable PS2EPSI_FORMAT contains the format for the
     command to start the ps2epsi program.
   SEE ALSO: window, fma, hcp, hcp_finish, plg
 */
{
  name+= ".ps";
  window, hcp=name, dump=1, legends=0;
  hcp;
  window, hcp="";
  system, swrite(format=PS2EPSI_FORMAT, name);
  remove, name;
}
if (is_void(PS2EPSI_FORMAT)) PS2EPSI_FORMAT= "ps2epsi %s";

extern fma;
/* DOCUMENT fma
     frame advance the current graphics window.  The current picture
     remains displayed in the associated X window until the next element
     is actually plotted.
   SEE ALSO: window, hcp, animate, plg
 */

extern hcp;
extern hcpon;
extern hcpoff;
/* DOCUMENT hcp
            hcpon
	    hcpoff
     The hcp command sends the picture displayed in the current graphics
     window to the hardcopy file.  (The name of the default hardcopy file
     can be specified using hcp_file; each individual graphics window may
     have its own hardcopy file as specified by the window command.)
     The hcpon command causes every fma (frame advance) command to do
     and implicit hcp, so that every frame is sent to the hardcopy file.
     The hcpoff command reverts to the default "demand only" mode.
   SEE ALSO: window, fma, plg
 */

extern redraw;
/* DOCUMENT redraw
     redraws the X window associated with the current graphics window.
   SEE ALSO: window, fma, hcp, plg
 */

extern palette;
/* DOCUMENT palette, filename
         or palette, source_window_number
         or palette, red, green, blue, ntsc=1/0
         or palette, red, green, blue, gray
	 or palette, red, green, blue, query=1
	 or palette, red, green, blue, gray, query=1
     sets (or retrieves with query=1) the palette for the current
     graphics window.  The FILENAME is the name of a Gist palette file;
     the standard palettes are "earth.gp", "stern.gp", "rainbow.gp",
     "heat.gp", "gray.gp", and "yarg.gp".  Use the maxcolors keyword
     in the pldefault command to put an upper limit on the number of
     colors which will be read from the palette in FILENAME.

     In the second form, the palette for the current window is copied
     from the SOURCE_WINDOW_NUMBER.  If the X colormap for the window is
     private, there will still be two separate X colormaps for the two
     windows, but they will have the same color values.

     In the third form, RED, GREEN, and BLUE are 1-D arrays of the same
     length specifying the palette you wish to install; the values
     should vary between 0 and 255, and your palette should have no
     more than 240 colors.  If ntsc=0, monochrome devices (such as most
     laser printers) will use the average brightness to translate your
     colors into gray; otherwise, the NTSC (television) averaging will
     be used (.30*RED+.59*GREEN+.11*BLUE).  Alternatively, you can specify
     GRAY explicitly.

     Ordinarily, the palette is not dumped to a hardcopy file
     (color hardcopy is still rare and expensive), but you can
     force the palette to dump using the window or hcp_file commands.
   SEE ALSO: window, fma, hcp, pldefault, plg
 */

extern animate;
/* DOCUMENT animate
         or animate, 0/1
     without any arguments, toggles animation mode; with argument 0,
     turns off animation mode, with argument 1 turns on animation mode.
     In animation mode, the X window associated with a graphics window
     is actually an offscreen pixmap which is bit-blitted onscreen
     when an fma command is issued.  This is confusing unless you are
     actually trying to make a movie, but results in smoother animation
     if you are.  Generally, you should turn animation on, run your movie,
     then turn it off.
   SEE ALSO: window, fma, plg
 */

extern plsys;
/* DOCUMENT plsys, n
     sets the current coordinate system to number N in the current
     graphics window.  If N equals 0, subsequent elements will be
     plotted in absolute NDC coordinates outside of any coordinate
     system.  The default style sheet "work.gs" defines only a single
     coordinate system, so the only other choice is N equal 1.  You
     can make up your own style sheet (using a text editor) which
     defines mulitple coordinate systems.  You need to do this if
     you want to display four plots side by side on a single page,
     for example.  The standard style sheets "work2.gs" and "boxed2.gs"
     define two overlayed coordinate systems with the first labeled
     to the right of the plot and the second labeled to the left of
     the plot.  When using overlayed coordinate systems, it is your
     responsibility to ensure that the x-axis limits in the two
     systems are identical.
   SEE ALSO: window, limits, plg
 */

/*--------------------------------------------------------------------------*/
/* Plotting functions (output primitives) */

extern plg;
/* DOCUMENT plg, y, x
         or plg, y
     plots a graph of Y versus X.  Y and X must be 1-D arrays of equal
     length; if X is omitted, it defaults to [1, 2, ..., numberof(Y)].
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     type, width, color, closed, smooth
             marks, marker, mspace, mphase
	     rays, arrowl, arroww, rspace, rphase
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj
             limits, logxy, range, fma, hcp
 */

extern plm;
/* DOCUMENT plm, y, x, boundary=0/1, inhibit=0/1/2
         or plm, y, x, ireg, boundary=0/1, inhibit=0/1/2
	 or plm, boundary=0/1, inhibit=0/1/2
     plots a mesh of Y versus X.  Y and X must be 2-D arrays with equal
     dimensions.  If present, IREG must be a 2-D region number array
     for the mesh, with the same dimensions as X and Y.  The values of
     IREG should be positive region numbers, and zero for zones which do
     not exist.  The first row and column of IREG never correspond to any
     zone, and should always be zero.  The default IREG is 1 everywhere
     else.  If present, the BOUNDARY keyword determines whether the
     entire mesh is to be plotted (boundary=0, the default), or just the
     boundary of the selected region (boundary=1).  If present, the
     INHIBIT keyword causes the (X(,j),Y(,j)) lines to not be plotted
     (inhibit=1), or the (X(i,),Y(i,)) lines to not be plotted (inhibit=2).
     By default (inhibit=0), mesh lines in both logical directions are
     plotted.
     The Y, X, and IREG arguments may all be omitted to default to the
     mesh set by the most recent plmesh call.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     type, width, color
	     region
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj, plmesh
             limits, logxy, range, fma, hcp
 */

extern plmesh;
/* DOCUMENT plmesh, y, x, ireg, triangle=tri_array
         or plmesh
     sets the default mesh for subsequent plm, plc, plv, and plf calls.
     In the second form, deletes the default mesh (until you do this,
     or switch to a new default mesh, the default mesh arrays persist and
     take up space in memory).  The Y, X, and IREG arrays should all be
     the same shape; Y and X will be converted to double, and IREG will
     be converted to int.  If IREG is omitted, it defaults to IREG(1,)=
     IREG(,1)= 0, IREG(2:,2:)=1; that is, region number 1 is the whole
     mesh.  The triangulation array TRI_ARRAY is used by plc; the
     correspondence between TRI_ARRAY indices and zone indices is the
     same as for IREG, and its default value is all zero.
     The IREG or TRI_ARRAY arguments may be supplied without Y and X
     to change the region numbering or triangulation for a given set of
     mesh coordinates.  However, a default Y and X must already have been
     defined if you do this.
     If Y is supplied, X must be supplied, and vice-versa.
   SEE ALSO: plm, plc, plv, plf
 */

extern plc;
/* DOCUMENT plc, z, y, x, levs=z_values
         or plc, z, y, x, ireg, levs=z_values
	 or plc, z, levs=z_values
     plots a contours of Z on the mesh Y versus X.  Y, X, and IREG are
     as for plm.  The Z array must have the same shape as Y and X.
     The function being contoured takes the value Z at each point
     (X,Y) -- that is, the Z array is presumed to be point-centered.
     The Y, X, and IREG arguments may all be omitted to default to the
     mesh set by the most recent plmesh call.
     The LEVS keyword is a list of the values of Z at which you want
     contour curves.  The default is eight contours spanning the
     range of Z.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     type, width, color, smooth
             marks, marker, mspace, mphase
	     smooth, triangle, region
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj, plmesh
             limits, logxy, range, fma, hcp
 */

extern plv;
/* DOCUMENT plv, v, u, y, x, scale=dt
         or plv, v, u, y, x, ireg, scale=dt
	 or plv, v, u, scale=dt
     plots a vector field (U,V) on the mesh (X,Y).  Y, X, and IREG are
     as for plm.  The V and U arrays must have the same shape as Y and X.
     The Y, X, and IREG arguments may all be omitted to default to the
     mesh set by the most recent plmesh call.
     The SCALE keyword is the conversion factor from the units of
     (U,V) to the units of (X,Y) -- a time interval if (U,V) is a velocity
     and (X,Y) is a position -- which determines the length of the
     vector "darts" plotted at the (X,Y) points.  If omitted, SCALE is
     chosen so that the longest ray arrows have a length comparable
     to a "typical" zone size.
     You can use the scalem keyword in pledit to make adjustments to the
     SCALE factor computed by default.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     type, width, color, smooth
             marks, marker, mspace, mphase
	     triangle, region
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj, plmesh, pledit,
             limits, logxy, range, fma, hcp
 */

extern plf;
/* DOCUMENT plf, z, y, x
         or plf, z, y, x, ireg
	 or plf, z
     plots a filled mesh Y versus X.  Y, X, and IREG are as for plm.
     The Z array must have the same shape as Y and X, or one smaller
     in both dimensions.  If Z is of type char, it is used "as is",
     otherwise it is linearly scaled to fill the current palette, as
     with the bytscl function.
     (See the bytscl function for explanation of top, cmin, cmax.)
     The mesh is drawn with each zone in the color derived from the Z
     function and the current palette; thus Z is interpreted as a
     zone-centered array.
     The Y, X, and IREG arguments may all be omitted to default to the
     mesh set by the most recent plmesh call.
     A solid edge can optionally be drawn around each zone by setting
     the EDGES keyword non-zero.  ECOLOR and EWIDTH determine the edge
     color and width.  The mesh is drawn zone by zone in order from
     IREG(2+imax) to IREG(jmax*imax) (the latter is IREG(imax,jmax)),
     so you can achieve 3D effects by arranging for this order to
     coincide with back-to-front order.  If Z is nil, the mesh zones
     are filled with the background color, which you can use to
     produce 3D wire frames.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     region, top, cmin, cmax, edges, ecolor, ewidth
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj, plmesh,
             limits, logxy, range, fma, hcp, palette, bytscl, histeq_scale
 */

extern pli;
/* DOCUMENT pli, z
         or pli, z, x1, y1
         or pli, z, x0, y0, x1, y1
     plots the image Z as a cell array -- an array of equal rectangular
     cells colored according to the 2-D array Z.  The first dimension
     of Z is plotted along x, the second dimension is along y.
     If Z is of type char, it is used "as is", otherwise it is linearly
     scaled to fill the current palette, as with the bytscl function.
     (See the bytscl function for explanation of top, cmin, cmax.)
     If X1 and Y1 are given, they represent the coordinates of the
     upper right corner of the image.  If X0, and Y0 are given, they
     represent the coordinates of the lower left corner, which is at
     (0,0) by default.  If only the Z array is given, each cell will be
     a 1x1 unit square, with the lower left corner of the image at (0,0).
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide, top, cmin, cmax
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj,
             limits, logxy, range, fma, hcp, palette, bytscl, histeq_scale
 */

extern pldj;
/* DOCUMENT pldj, x0, y0, x1, y1
     plots disjoint lines from (X0,Y0) to (X1,Y1).  X0, Y0, X1, and Y1
     may have any dimensionality, but all must have the same number of
     elements.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     type, width, color
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj
             limits, logxy, range, fma, hcp
 */

extern plt;
/* DOCUMENT plt, text, x, y, tosys=0/1
     plots TEXT (a string) at the point (X,Y).  The exact relationship
     between the point (X,Y) and the TEXT is determined by the
     justify keyword.  TEXT may contain newline ("\n") characters
     to output multiple lines of text with a single call.  The
     coordinates (X,Y) are NDC coordinates (outside of any coordinate
     system) unless the tosys keyword is present and non-zero, in
     which case the TEXT will be placed in the current coordinate
     system.  However, the character height is NEVER affected by the
     scale of the coordinate system to which the text belongs.
     Note that the pledit command takes dx and/or dy keywords to
     adjust the position of existing text elements.
     The following keywords are legal (each has a separate help entry):
   KEYWORDS: legend, hide
	     color, font, height, opaque, path, justify
   SEE ALSO: plg, plm, plc, plv, plf, pli, plt, pldj, pledit
             limits, range, fma, hcp, pltitle
 */

func pltitle(title)
/* DOCUMENT pltitle, title
     Plot TITLE centered above the coordinate system for any of the
     standard Gist styles.  You will need to customize this for other
     plot styles.
 */
{
  plt, title, 0.395, 0.870, font="helvetica", justify="CA", height=18;
}

/*--------------------------------------------------------------------------*/
/* Plot limits and log/linear scaling */

e= "e";         /* for use with limits and range functions */

extern limits;
/* DOCUMENT limits
         or limits, xmin, xmax, ymin, ymax,
	            square=0/1, nice=0/1, restrict=0/1
	 or old_limits= limits()
	 or limits, old_limits

     In the first form, restores all four plot limits to extreme values.

     In the second form, sets the plot limits in the current coordinate
     system to XMIN, XMAX, YMIN, YMAX, which may be nil or omitted to
     leave the corresponding limit unchanged, a number to fix the
     corresponding limit to a specified value, or the string "e" to
     make the corresponding limit take on the extreme value of the
     currently displayed data.

     If present, the square keyword determines whether limits marked
     as extreme values will be adjusted to force the x and y scales
     to be equal (square=1) or not (square=0, the default).
     If present, the nice keyword determines whether limits will be
     adjusted to nice values (nice=1) or not (nice=0, the default).
     There is a subtlety in the meaning of "extreme value" when one
     or both of the limits on the OPPOSITE axis have fixed values --
     does the "extreme value" of the data include points which
     will not be plotted because their other coordinate lies outside
     the fixed limit on the opposite axis (restrict=0, the default),
     or not (restrict=1)?

     If called as a function, limits returns an array of 5 doubles;
     OLD_LIMITS(1:4) are the current xmin, xmax, ymin, and ymax,
     and int(OLD_LIMITS(5)) is a set of flags indicating extreme
     values and the square, nice, restrict, and log flags.

     In the fourth form, OLD_LIMITS is as returned by a previous
     limits call, to restore the limits to a previous state.

     In an X window, the limits may also be adjusted interactively
     with the mouse.  Drag left to zoom in and pan (click left to zoom
     in on a point without moving it), drag middle to pan, and click
     (and drag) right to zoom out (and pan).  If you click just above
     or below the plot, these operations will be restricted to the
     x-axis; if you click just to the left or right, the operations
     are restricted to the y-axis.  A shift-left click, drag, and
     release will expand the box you dragged over to fill the plot
     (other popular software zooms with this paradigm).  If the
     rubber band box is not visible with shift-left zooming, try
     shift-middle or shift-right for alternate XOR masks.  Such
     mouse-set limits are equivalent to a limits command specifying
     all four limits EXCEPT that the unzoom command can revert to
     the limits before a series of mouse zooms and pans.

     The limits you set using the limits or range functions carry over
     to the next plot -- that is, an fma operation does NOT reset the
     limits to extreme values.

   SEE ALSO: plsys, range, logxy, zoom_factor, unzoom, plg
 */

func range(ymin, ymax) { limits,,, ymin, ymax; }
/* DOCUMENT range, ymin, ymax
     sets the y-axis plot limits in the current coordinate system to
     YMIN, YMAX, which may be nil or omitted to leave the corresponding
     limit unchanged, a number to fix the corresponding limit to a
     specified value, or the string "e" to make the corresponding limit
     take on the extreme value of the currently displayed data.
     Use    limits, xmin, xmin
     to accomplish the same function for the x-axis plot limits.
   SEE ALSO: plsys, limits, logxy, plg
 */

extern logxy;
/* DOCUMENT logxy, xflag, yflag
     sets the linear/log axis scaling flags for the current coordinate
     system.  XFLAG and YFLAG may be nil or omitted to leave the
     corresponding axis scaling unchanged, 0 to select linear scaling,
     or 1 to select log scaling.
   SEE ALSO: plsys, limits, range, plg, gridxy
 */

extern gridxy;
/* DOCUMENT gridxy, flag
         or gridxy, xflag, yflag
     Turns on or off grid lines according to FLAG.  In the first form, both
     the x and y axes are affected.  In the second form, XFLAG and YFLAG
     may differ to have different grid options for the two axes.  In either
     case, a FLAG value of 0 means no grid lines (the default), a value of
     1 means grid lines at all major ticks (the level of ticks which get
     grid lines can be set in the style sheet), and a FLAG value of 2 means
     that the coordinate origin only will get a grid line.  In styles with
     multiple coordinate systems, only the current coordinate system is
     affected.
     The keywords can be used to affect the style of the grid lines.
   KEYWORDS: color, type, width
   SEE ALSO: window, plsys, limits, range, logxy
 */

extern zoom_factor;
/* DOCUMENT zoom_factor, factor
     sets the zoom factor for mouse-click zoom in and zoom out operations.
     The default FACTOR is 1.5; FACTOR should always be greater than 1.0.
   SEE ALSO: limits, range, unzoom, plg
 */

extern unzoom;
/* DOCUMENT unzoom
     restores limits to their values before zoom and pan operations
     performed interactively using the mouse.
     Use    old_limits=  limits()
            ...
	    limits, old_limits
     to save and restore plot limits generally.
   SEE ALSO: limits, range, zoom_factor, plg
 */

/*--------------------------------------------------------------------------*/
/* Keywords for plotting functions */

local legend;
/* DOCUMENT legend=   plotting keyword
     sets the legend for a plot.  The default legend is a concatentation
     of the strings used in the original plotting command (plg, plm, etc.),
     except for the plt command, which has no default legend.
     Legends are never plotted to the X window; use the plq command to
     see them interactively.  Legends will appear in hardcopy output
     unless they have been explicitly turned off.
   PLOTTING COMMANDS: plg, plm, plc, plv, plf, pli, plt, pldj
   SEE ALSO: hide
 */

local hide;
/* DOCUMENT hide=   plotting keyword
     sets the visibility of a plotted element.  The default is hide=0,
     which means that the element will be visible.  Use hide=1 to remove
     the element from the plot (but not from the display list).
   PLOTTING COMMANDS: plg, plm, plc, plv, plf, pli, plt, pldj
   SEE ALSO: legend
 */

local type;
/* DOCUMENT type=   plotting keyword
     selects line type.  Valid values are the strings "solid", "dash",
     "dot", "dashdot", "dashdotdot", and "none".  The "none" value
     causes the line to be plotted as a polymarker.
     The type value may also be a number; 0 is "none", 1 is "solid",
     2 is "dash", 3 is "dot", 4 is "dashdot", and 5 is "dashdotdot".
   PLOTTING COMMANDS: plg, plm, plc, pldj
   SEE ALSO: width, color, marks, marker, rays, closed, smooth
 */

local width;
/* DOCUMENT width=   plotting keyword
     selects line width.  Valid values are positive floating point numbers
     giving the line thickness relative to the default line width of one
     half point, width= 1.0.
   PLOTTING COMMANDS: plg, plm, plc, pldj, plv (only if hollow=1)
   SEE ALSO: type, color, marks, marker, rays, closed, smooth
 */

local color;
/* DOCUMENT color=   plotting keyword
     selects line or text color.  Valid values are the strings "bg", "fg",
     "black", "white", "red", "green", "blue", "cyan", "magenta", "yellow",
     or a 0-origin index into the current palette.  The default is "fg".
     Negative numbers may be used instead of the strings: -1 is bg
     (background), -2 is fg (foreground), -3 is black, -4 is white,
     -5 is red, -6 is green, -7 is blue, -8 is cyan, -9 is magenta, and
     -10 is yellow.
   PLOTTING COMMANDS: plg, plm, plc, pldj, plt
   SEE ALSO: type, width, marks, marker, mcolor, rays, closed, smooth
 */

local marks;
/* DOCUMENT marks=   plotting keyword
     selects unadorned lines (marks=0), or lines with occasional markers
     (marks=1).  Ignored if type is "none" (indicating polymarkers instead
     of occasional markers).  The spacing and phase of the occasional
     markers can be altered using the mspace and mphase keywords; the
     character used to make the mark can be altered using the marker
     keyword.
   PLOTTING COMMANDS: plg, plc
   SEE ALSO: type, width, color, marker, rays, mspace, mphase, msize, mcolor
 */

local marker;
/* DOCUMENT marker=   plotting keyword
     selects the character used for occasional markers along a polyline,
     or for the polymarker if type is "none".  The special values
     '\1', '\2', '\3', '\4', and '\5' stand for point, plus, asterisk,
     circle, and cross, which are prettier than text characters on output
     to some devices.  The default marker is the next available capital
     letter, 'A', 'B', ..., 'Z'.
   PLOTTING COMMANDS: plg, plc
   SEE ALSO: type, width, color, marks, rays, mspace, mphase, msize, mcolor
 */

local mspace, mphase, msize, mcolor;
/* DOCUMENT mspace=   plotting keyword
         or mphase=   plotting keyword
         or msize=    plotting keyword
         or mcolor=   plotting keyword
     selects the spacing, phase, and size of occasional markers placed
     along polylines.  The msize also selects polymarker size if type
     is "none".  The spacing and phase are in NDC units (0.0013 NDC
     equals 1.0 point); the default mspace is 0.16, and the default
     mphase is 0.14, but mphase is automatically incremented for
     successive curves on a single plot.  The msize is in relative
     units, with the default msize of 1.0 representing 10 points.
     The mcolor keyword is the same as the color keyword, but controls
     the marker color instead of the line color.  Setting the color
     automatically sets the mcolor to the same value, so you only
     need to use mcolor if you want the markers for a curve to be a
     different color than the curve itself.
   PLOTTING COMMANDS: plg, plc
   SEE ALSO: type, width, color, marks, marker, rays
 */

local rays;
/* DOCUMENT rays=   plotting keyword
     selects unadorned lines (rays=0), or lines with occasional ray
     arrows (rays=1).  Ignored if type is "none".  The spacing and phase
     of the occasional arrows can be altered using the rspace and rphase
     keywords; the shape of the arrowhead can be modified using the
     arroww and arrowl keywords.
   PLOTTING COMMANDS: plg, plc
   SEE ALSO: type, width, color, marker, marks, rspace, rphase
             arroww, arrowl
 */

local rspace, rphase, arroww, arrowl;
/* DOCUMENT rspace=   plotting keyword
         or rphase=   plotting keyword
         or arroww=   plotting keyword
         or arrowl=   plotting keyword
     selects the spacing, phase, and size of occasional ray arrows
     placed along polylines.  The spacing and phase are in NDC units
     (0.0013 NDC equals 1.0 point); the default rspace is 0.13, and
     the default rphase is 0.11375, but rphase is automatically
     incremented for successive curves on a single plot.
     The arrowhead width, arroww, and arrowhead length, arrowl are
     in relative units, defaulting to 1.0, which translates to an
     arrowhead 10 points long and 4 points in half-width.
   PLOTTING COMMANDS: plg
   SEE ALSO: type, width, color, marks, marker, rays
 */

local closed, smooth;
/* DOCUMENT closed=   plotting keyword
         or smooth=   plotting keyword
     selects closed curves (closed=1) or default open curves (closed=0),
     or Bezier smoothing (smooth>0) or default piecewise linear curves
     (smooth=0).  The value of smooth can be 1, 2, 3, or 4 to get
     successively more smoothing.  Only the Bezier control points are
     plotted to an X window; the actual Bezier curves will show up in
     PostScript hardcopy files.  Closed curves join correctly, which
     becomes more noticeable for wide lines; non-solid closed curves
     may look bad because the dashing pattern may be incommensurate
     with the length of the curve.
   PLOTTING COMMANDS: plg, plc (smooth only)
   SEE ALSO: type, width, color, marks, marker, rays
 */

local font, height, opaque, path, justify;
/* DOCUMENT font=     plotting keyword
         or height=   plotting keyword
         or opaque=   plotting keyword
         or path=     plotting keyword
         or justify=  plotting keyword
     selects text properties.  The font can be any of the strings
     "courier", "times", "helvetica" (the default), "symbol", or
     "schoolbook".  Append "B" for boldface and "I" for italic, so
     "courierB" is boldface Courier, "timesI" is Times italic, and
     "helveticaBI" is bold italic (oblique) Helvetica.  Your X server
     should have the Adobe fonts (available free from the MIT X
     distribution tapes) for all these fonts, preferably at both 75
     and 100 dpi.  Occasionally, a PostScript printer will not be
     equipped for some fonts; often New Century Schoolbook is missing.
     The font keyword may also be an integer: 0 is Courier, 4 is Times,
     8 is Helvetica, 12 is Symbol, 16 is New Century Schoolbook, and
     you add 1 to get boldface and/or 2 to get italic (or oblique).

     The height is the font size in points; 14.0 is the default.
     X windows only has 8, 10, 12, 14, 18, and 24 point fonts, so
     don't stray from these sizes if you want what you see on the
     screen to be a reasonably close match to what will be printed.

     By default, opaque=0 and text is transparent.  Set opaque=1 to
     white-out a box before drawing the text.  The default path
     (path=0) is left-to-right text; set path=1 for top-to-bottom text.

     The default text justification, justify="NN" is normal is both
     the horizontal and vertical directions.  Other possibilities
     are "L", "C", or "R" for the first character, meaning left,
     center, and right horizontal justification, and "T", "C", "H",
     "A", or "B", meaning top, capline, half, baseline, and bottom
     vertical justification.  The normal justification "NN" is equivalent
     to "LA" if path=0, and to "CT" if path=1.  Common values are
     "LA", "CA", and "RA" for garden variety left, center, and right
     justified text, with the y coordinate at the baseline of the
     last line in the string presented to plt.  The characters labeling
     the right axis of a plot are "RH", so that the y value of the
     text will match the y value of the corresponding tick.  Similarly,
     the characters labeling the bottom axis of a plot are "CT".
     The justification may also be a number, horizontal+vertical,
     where horizontal is 0 for "N", 1 for "L", 2 for "C", or 3 for "R",
     and vertical is 0 for "N", 4 for "T", 8 for "C", 12 for "H",
     16 for "A", or 20 for "B".

   PLOTTING COMMANDS: plt
   SEE ALSO: color
 */

local region;
/* DOCUMENT region=   plotting keyword
     selects the part of mesh to consider.  The region should match one
     of the numbers in the IREG array.  Putting region=0 (the default)
     means to plot the entire mesh, that is, everything EXCEPT region
     zero (non-existent zones).  Any other number means to plot only
     the specified region number; region=3 would plot region 3 only.
   PLOTTING COMMANDS: plm, plc, plv, plf
 */

local triangle;
/* DOCUMENT triangle=   plotting keyword
     sets the triangulation array for a contour plot.  The triangulation
     array must be the same shape as the IREG (region number) array, and
     the correspondence between mesh zones and indices is the same as
     for IREG.  The triangulation array is used to resolve the ambiguity
     in saddle zones, in which the function Z being contoured has two
     diagonally opposite corners high, and the other two corners low.
     The triangulation array element for a zone is 0 if the algorithm is
     to choose a triangulation, based on the curvature of the first
     contour to enter the zone.  If zone (i,j) is to be triangulated
     from point (i-1,j-1) to point (i,j), then TRIANGLE(i,j)=1,
     while if it is to be triangulated from (i-1,j) to (i,j-1), then
     TRIANGLE(i,j)=-1.  Contours will never cross this "triangulation
     line".
     You should rarely need to fiddle with the traingulation array;
     it is a hedge for dealing with pathological cases.
   PLOTTING COMMANDS: plc
 */

local hollow, aspect;
/* DOCUMENT hollow=   plotting keyword
         or aspect=   plotting keyword
     set the appearance of the "darts" of a vector field plot.  The
     default darts, hollow=0, are filled; use hollow=1 to get just the
     dart outlines.  The default is aspect=0.125; aspect is the ratio
     of the half-width to the length of the darts.  Use the color
     keyword to control the color of the darts.
   PLOTTING COMMANDS: plv
   SEE ALSO: color
 */

local edges, ecolor, ewidth;
/* DOCUMENT edges=   plotting keyword
         or ecolor=   plotting keyword
         or ewidth=   plotting keyword
     set the appearance of the zone edges in a filled mesh plot (plf).
     By default, edges=0, and the zone edges are not plotted.  If
     edges=1, a solid line is drawn around each zone after it is
     filled; the edge color and width are given by ecolor and ewidth,
     which are "fg" and 1.0 by default.
   PLOTTING COMMANDS: plf
   SEE ALSO: color, width
 */

/*--------------------------------------------------------------------------*/
/* Inquiry and editing functions */

extern plq;
/* DOCUMENT plq
         or plq, n_element
	 or plq, n_element, n_contour
	 or legend_list= plq()
	 or properties= plq(n_element, n_contour)
     Called as a subroutine, prints the list of legends for the current
     coordinate system (with an "(H)" to mark hidden elements), or prints
     a list of current properties of element N_ELEMENT (such as line type,
     width, font, etc.), or of contour number N_CONTOUR of element number
     N_ELEMENT (which must be contours generated using the plc command).
     Called as a function, returns either the list of legend strings, or a
     list of pointers to the values of the various element properties.
     Elements and contours are both numbered starting with one; hidden
     elements or contours are included in this numbering.

     The PROPERTIES list returned by plq is a list of pointers to the
     relevent properties for the specified graphical element.  Each
     possible property has a particular index in the returned PROPERTIES
     list as follows:
     *PROPERTIES(1)   int([element type (0 for none, 1 for plg, 2 for pldj,
                                         3 for plt, 4 for plm, 5 for plf,
					 6 for plv, 7 for plc, 8 for pli),
                           hide flag])
     *PROPERTIES(2)   string(legend)
     *PROPERTIES(3)   int array, depends on type (names match keywords):
       1 plg:  [color, type, marks, mcolor, marker, rays, closed, smooth]
       2 pldj: [color, type]
       3 plt:  [color, font, path, justify, opaque]
       4 plm:  [color, type, region, boundary, inhibit]
       5 plf:  [region, edges, ecolor]
       6 plv:  [region, color, hollow]
       7 plc:  [region, color, type, marks, mcolor, marker, smooth]
       8 pli:  nil
     *PROPERTIES(4)   double array, depends on type (names match keywords):
       1 plg:  [width, msize, mspace, mphase, rspace, rphase, arrowl, arroww]
       2 pldj: [width]
       3 plt:  [height, x, y]
       4 plm:  [width]
       5 plf:  [ewidth]
       6 plv:  [width, aspect, scale]
       7 plc:  [width, msize, mspace, mphase]
       8 pli:  [x0, x1, y0, y1]
     *PROPERTIES(5)   long array, depends on type (names match arguments):
       1 plg:  [npoints, &x, &y]
       2 pldj: [npoints, &x0, &y0, &x1, &y1]
       3 plt:  [nchars, &text]
       4 plm:  [imax, jmax, &x, &y, &ireg]
       5 plf:  [imax, jmax, &x, &y, &ireg, &colors]
       6 plv:  [imax, jmax, &x, &y, &ireg, &u, &v]
       7 plc:  [imax, jmax, &x, &y, &ireg, &z, &triangle, nlevs, &levs]
       8 pli:  [imax, jmax, &colors]
     You can use the reshape function to peek at the data at the addresses
     returned in PROPERTIES(5) as longs.  The appropriate data types are:
     char for text, int for ireg, short for triangle, char for colors, and
     double for everything else.  In a plf, colors is (imax-1)-by-(jmax-1).
     Although PROPERTIES(5) returns pointers to the data plotted, attempting
     to poke new values into this data will not produce immediate changes
     to your plot, since the graphics package does not realize that anything
     has changed.  Use pledit to make changes to plotted elements.

     The plq function always operates on the current coordinate system
     in the current graphics window; use window and plsys to change these.
   SEE ALSO: window, plsys, pledit, pldefault, plg
 */

extern pledit;
/* DOCUMENT pledit, key1=value1, key2=value2, ...
         or pledit, n_element, key1=value1, key2=value2, ...
         or pledit, n_element, n_contour, key1=value1, key2=value2, ...
     changes some property of element number N_ELEMENT (and contour
     number N_CONTOUR of that element).  If N_ELEMENT and N_CONTOUR are
     omitted, the default is the most recently added element, or the
     element specified in the most recent plq query command.

     The keywords can be any of the keywords that apply to the current
     element.  These are:
       plg:  color, type, width,
             marks, mcolor, marker, msize, mspace, mphase,
	     rays, rspace, rphase, arrowl, arroww,
	     closed, smooth
       pldj: color, type, width
       plt:  color, font, height, path, justify, opaque
       plm:  region, boundary, inhibit, color, type, width
       plf:  region
       plv:  region, color, hollow, width, aspect, scale
       plc:  region, color, type, width,
             marks, mcolor, marker, msize, mspace, mphase
	     smooth, levs
     (For contours, if you aren't talking about a particular N_CONTOUR,
      any changes will affect ALL the contours.)

     A plv (vector field) element can also take the scalem
     keyword to multiply all vector lengths by a specified factor.

     A plt (text) element can also take the dx and/or dy
     keywords to adjust the text position by (dx,dy).

   SEE ALSO: window, plsys, plq, pldefault, plg
 */

extern pldefault;
/* DOCUMENT pldefault, key1=value1, key2=value2, ...
     sets default values for the various properties of graphical elements.

     The keywords can be most of the keywords that can be passed to the
     plotting commands:
       plg:  color, type, width,
             marks, mcolor, msize, mspace, mphase,
	     rays, rspace, rphase, arrowl, arroww
       pldj: color, type, width
       plt:  color, font, height, path, justify, opaque
       plm:  color, type, width
       plv:  color, hollow, width, aspect
       plc:  color, type, width,
             marks, mcolor, marker, msize, mspace, mphase
       plf:  edges, ecolor, ewidth

     The initial default values are:
       color="fg", type="solid", width=1.0 (1/2 point),
       marks=1, mcolor="fg", msize=1.0 (10 points),
          mspace=0.16, mphase=0.14,
       rays=0, arrowl=1.0 (10 points), arroww=1.0 (4 points),
          rspace=0.13, rphase=0.11375,
       font="helvetica", height=12.0, path=0, justify="NN", opaque=0,
       hollow= 0, aspect=0.125,
       edges=0, ecolor="fg", ewidth=1.0 (1/2 point)

     Additional default keywords are:
       dpi, style, legends  (see window command)
       palette              (to set default filename as in palette command)
       maxcolors            (default 200)

   SEE ALSO: window, plsys, plq, pledit, plg
 */

/*--------------------------------------------------------------------------*/
/* Miscellany */

extern bytscl;
/* DOCUMENT bytscl(z)
         or bytscl(z, top=max_byte, cmin=lower_cutoff, cmax=upper_cutoff)
     returns a char array of the same shape as Z, with values linearly
     scaled to the range 0 to one less than the current palette size.
     If MAX_BYTE is specified, the scaled values will run from 0 to
     MAX_BYTE instead.
     If LOWER_CUTOFF and/or UPPER_CUTOFF are specified, Z values outside
     this range are mapped to the cutoff value; otherwise the linear
     scaling maps the extreme values of Z to 0 and MAX_BYTE.
   SEE ALSO: plf, pli, histeq_scale
 */

extern mesh_loc;
/* DOCUMENT mesh_loc(y0, x0)
         or mesh_loc(y0, x0, y, x)
         or mesh_loc(y0, x0, y, x, ireg)
     returns the zone index (=i+imax*(j-1)) of the zone of the mesh
     (X,Y) (with optional region number array IREG) containing the
     point (X0,Y0).  If (X0,Y0) lies outside the mesh, returns 0.
     Thus, eg- ireg(mesh_loc(x0, y0, y, x, ireg)) is the region number of
     the region containing (x0,y0).  If no mesh specified, uses default.
     X0 and Y0 may be arrays as long as they are conformable.
   SEE ALSO: plmesh, moush, mouse
 */

extern mouse;
/* DOCUMENT result= mouse(system, style, prompt)
     displays a PROMPT, then waits for a mouse button to be pressed,
     then released.  Returns array of eleven doubles:
       result= [x_pressed, y_pressed, x_released, y_released,
                xndc_pressed, yndc_pressed, xndc_released, yndc_released,
	        system, button, modifiers]

     If SYSTEM>=0, the first four coordinate values will be relative to
     that coordinate system.
     For SYSTEM<0, the first four coordinate values will be relative to
     the coordinate system under the mouse when the button was pressed.
     The second four coordinates are always normalized device coordinates,
     which start at (0,0) in the lower left corner of the 8.5x11 sheet of
     paper the picture will be printed on, with 0.0013 NDC unit being
     1/72.27 inch (1.0 point).  Look in the style sheet for the location
     of the viewport in NDC coordinates (see the style keyword).

     If STYLE is 0, there will be no visual cues that the mouse
     command has been called; this is intended for a simple click.
     If STYLE is 1, a rubber band box will be drawn; if STYLE is 2,
     a rubber band line will be drawn.  These disappear when the
     button is released.

     Clicking a second button before releasing the first cancels the
     mouse function, which will then return nil.
     Ordinary text input also cancels the mouse function, which again
     returns nil.

     The left button reverses forground for background (by XOR) in
     order to draw the rubber band (if any).  The middle and right
     buttons use other masks, in case the rubber band is not visible
     with the left button.

     long(result(9)) is the coordinate system in which the first four
     coordinates are to be interpreted.
     long(result(10)) is the button which was pressed, 1 for left, 2
     for middle, and 3 for right (4 and 5 are also possible).
     long(result(11)) is a mask representing the modifier keys which
     were pressed during the operation: 1 for shift, 2 for shift lock,
     4 for control, 8 for mod1 (alt or meta), 16 for mod2, 32 for mod3,
     64 for mod4, and 128 for mod5.

   SEE ALSO: moush
 */

func moush(y, x, ireg)
/* DOCUMENT moush()
         or moush(y, x, ireg)
     returns the 1-origin zone index for the point clicked in
     for the default mesh, or for the mesh (X,Y) (region array IREG).
 */
{
  xy= mouse(-1, 0, "<Click mouse in mesh>");
  if (is_void(xy)) return [];
  else return mesh_loc(xy(2), xy(1), y, x, ireg);
}

extern pause;
/* DOCUMENT pause, milliseconds
     pause for the specified number of milliseconds of wall clock
     time, or until input arrives from the keyboard.
     This is intended for use in creating animated sequences.
 */

/*--------------------------------------------------------------------------*/

func histeq_scale(z, top=, cmin=, cmax=)
/* DOCUMENT histeq_scale(z, top=top_value, cmin=cmin, cmax=cmax)
     returns a byte-scaled version of the array Z having the property
     that each byte occurs with equal frequency (Z is histogram
     equalized).  The result bytes range from 0 to TOP_VALUE, which
     defaults to one less than the size of the current palette (or
     255 if no pli, plf, or palette command has yet been issued).

     If non-nil CMIN and/or CMAX is supplied, values of Z beyond these
     cutoffs are not included in the frequency counts.

   SEE ALSO: bytscl, plf, pli
 */
{
  if (is_void(top)) top= bytscl([0.,1.])(2);  /* palette size - 1 */
  top= long(top);
  if (top<0 | top>255) error, "top value out of range 0-255";
  y= z(*);
  if (!is_void(cmin)) y= y(where(y>=cmin));
  if (!is_void(cmax)) y= y(where(y<=cmax));
  y= y(sort(y));
  x= span(0.,1., numberof(y));
  xp= span(0.,1., top+2);
  bins= interp(y, x, xp);
  list= where(bins(dif)<=0.0);
  if (numberof(list)) {
    /* some value (or values) of z are repeated many times --
       try to handle this by adding a small slope to the sorted y */
    dy= y(0)-y(1);
    if (!dy) dy= 1.0;
    for (eps=1.e-10 ; eps<1000.1 ; eps*=10.) {
      bins= interp(y+eps*dy*x, x, xp);
      list= where(bins(dif)<=0.0);
      if (!numberof(list)) break;
    }
    if (eps>1000.) error, "impossible error??";
  }
  return char(max(min(digitize(z,bins)-2,top),0));
}

/*--------------------------------------------------------------------------*/

extern _pl_init;
/* xxDOCUMENT _pl_init
     initializes the Gist graphics package -- DON'T EVER CALL THIS.
 */
_pl_init, GISTPATH; /* ...except right here (see paths.i) */
