;;;
;;; F2C.EL
;;;
;;; Convert a buffer of Fortran source into "equivalent" C source.
;;; This is NOT an F77 compiler -- idea is to try to get the sense,
;;; and you will still have a lot of cleanup by hand (but not as much
;;; as without this).
;;;
;;; LIMITATIONS:
;;; (1) No attempt to deal with Fortran I/O.  You should repair this
;;;     comprehensively anyway.
;;; (2) No attempt to deal with Fortran character variables (sheesh).
;;; (3) Cannot correctly figure out whether an indexed array in a
;;;     subroutine or function argument is intended to be passed by
;;;     address.  Your ANSI C compiler will detect these errors, however.
;;; (4) No attempt is made to handle common blocks.
;;;
;;; WHAT IS DONE:
;;; (1) The buffer is put into C mode, and the file it is visiting is
;;;     changed to *.c.
;;; (2) All non-comments converted to lower case, trailing blanks stripped.
;;; (3) Numbers of form 1.234d-56 changed to 1.234e-56.
;;; (4) Fortran data type names converted to C (e.g.-  integer --> long).
;;; (5) Fortran .op. operators converted to C equivalents.
;;; (6) Fortran DO loops converted to C for loops.
;;; (7) Fortran block IF statements converted to C punctuation.
;;; (8) Fortran comments converted to C.
;;; (9) Fortran ** operators converted to fake pow_xx(base,power) function
;;;     call.  You need to go through these by hand and decide on the
;;;     appropriate action.  The ANSI C pow() function is usually wrong.
;;; (10) Fortran function/subroutine declarations and bodies identified
;;;      and converted to C.  A prototype is also generated.
;;; (11) Fortran labels and GOTOs are converted to C.  An attempt is made
;;;      to handle two and three branch arithmetic if statements, but
;;;      these should usually be checked by hand.  The labels in a
;;;      computed goto are corrected, but it is not converted to a
;;;      C switch statement -- you must do this by hand.  Fortran ASSIGN
;;;      statements are not handled -- usually you can put together an
;;;      equivalent switch statement as for computed GOTOs.
;;; (12) Function names are added to RETURN statements as required.
;;; (13) Semicolons are added at the end of all statements.
;;; (14) Fortran dimension and type declarations are recognized and used
;;;      to generate function parameter prototypes and to generate macros
;;;      for indexing arrays (see (3) below).  Any function parameter
;;;      which is dimensioned as an array gets a prototype like:
;;;           type varname[]
;;;      Any function parameter which appears as the left hand side of an
;;;      assignment in the function body gets a prototype like:
;;;           type *varname
;;; (15) Variables which are implicitly declared in Fortran are given actual
;;;      C declaration statements.  (Some few can be missed.)
;;; (16) Most Fortran intrinsic functions are recognized, and a first
;;;      attempt at conversion is made.  Many of these should be checked
;;;      by hand -- especially macros which reference their arguments
;;;      multiple times (be sure these arguments are not function calls).
;;;
;;; SYMBOLS CHANGED OR MODIFIED:
;;; (1) Fortran function return values become
;;;         funcname_R
;;; (2) Fortran statement label 1234 becomes
;;;         L_1234
;;; (3) A 1D array varname is indexed via a macro:
;;;         varname_1
;;;     while 2D, 3D, etc. array indexing occurs through macros
;;;         varname_2, varname_3, etc.
;;;
;;; BUGS:
;;; (1) EXTERNAL handler broken
;;; (2) varname --> varname_1 etc. should not be done in comments
;;; (3) n== tricks gizmo that recognizes n as an output parameter (n=)
;;; (4) L_1234: needs following semicolon if it was a CONTINUE statement
;;; (5) DATA statement handler doesn't always work
;;; (6) All #define generated in one function should have matching
;;;     #undef at end of function body
;;; (7) Many occurences of output function parameters are not dereferenced
;;;
;;; TO DO:
;;; (1) Generate switch statement for computed goto.
;;; (2) Generate static type modifier for Fortran SAVE statements.
;;; (3) Convert entire f2c to run in "compiler style", building a
;;;     symbol table as it parses the file.  This would make many
;;;     more repairs possible...



(defvar f2c-called-list nil)

(defun f2c ()
  "*Make a stab at converting the current buffer from Fortran source code
into C source code.  This is not intended to be a Fortran compiler; instead
the intent is to capture the sense of the Fortran."
  (interactive)
  (c-mode)
  (if (not (string-match "^\\.c" (substring buffer-file-name -2)))
      (set-visited-file-name
       (concat (substring buffer-file-name 0 -2) ".c")))
  (let ((beg-body (make-marker))
        (end-body (make-marker))
	f2c-called-list)
    (message "Fixing lexical differences")
    (f2c-zap-trailing) ; remove trailing whitespace, ensure final newline
    (f2c-downcase) ; C is case-sensitive, Fortran is not -comments excepted
    (f2c-fix-doubles) ; get rid of any 3.4d-5 style numbers
    (f2c-split-data) ; simplify data statement syntax for f2c-handle-data
    (message "Fixing types")
    (f2c-fix-types) ; try to convert Fortran data type names to C
    (message "Fixing operators")
    (f2c-fix-ops) ; convert logical operators, e.g.- .eq. --> ==
    (message "Fixing do loops")
    (f2c-fix-dos) ; convert Fortran do loops to C for loops
    (message "Fixing block ifs")
    (f2c-fix-ifs) ; convert Fortran if-then-else to C
    (message "Fixing comments")
    (f2c-fix-comments) ; convert Fortran comments to C comments
    (f2c-fix-powers) ; SQ, CUBE, etc. macros
    ;; remaining repairs are all within each function or subroutine body
    (goto-char (point-min))
    (while (< (point) (point-max))
      (while (or (= (following-char) ?\n) (not (f2c-not-comment)))
        (forward-line 1)) ; skip blank or comment lines
      (insert "\n\n\n")
      (let ((beg-function (point))
            (eol (f2c-eol)))
        (goto-char eol)
        (insert "\n{")
        (set-marker beg-body (point))
        (re-search-forward "^[ \t]+end")
        (replace-match "      }") ; leading blanks for f2c-not-comment
        (backward-char 1)
        (set-marker end-body (point))
	(goto-char beg-function)
        (f2c-discontinue)
	(goto-char beg-function)
	(skip-chars-forward " \t")
	(delete-region beg-function (point))
	(let ((func-type (f2c-next-token eol))
	      (func-name (f2c-next-token eol "("))
              (func-args (f2c-arglist eol)))
          (if (not func-args) (insert "(void)"))
          (if (not (string= func-type "void"))
              ;; functions need result declared, returns repaired
              (let (beg)
                (goto-char beg-body)
                (forward-line 1)
                (while  (or (= (following-char) ?\n)
                            (not (f2c-not-comment))
                            (looking-at "[ \t]+implicit\\b"))
                  (forward-line 1))
                (insert "      " func-type " " func-name "_R\n")
                (setq beg (point))
                (while (re-search-forward "\\breturn\\b"
                                          (marker-position end-body) t)
                  (if (f2c-not-comment) (insert " " func-name)))
                (goto-char beg)
                (while (re-search-forward (concat "\\b" func-name "\\b")
                                          (marker-position end-body) t)
                  (if (f2c-not-comment) (insert "_R")))))
          (message (concat "Processing " func-name ": labels"))
          (f2c-fix-labels beg-body end-body)
          (message (concat "Processing " func-name ": semicolons"))
          (f2c-fix-semicolons beg-body end-body)
          (f2c-fix-declarations func-args beg-body end-body func-name)
          (goto-char beg-body)
          (backward-char)
          (indent-c-exp)
          (goto-char end-body)
          (forward-line 1))))
    (message "Cleaning arithmetic ifs")
    (f2c-cleanup-gotos)
    (message "Repairing intrinsics")
    (f2c-cleanup-intrinsics)
    (f2c-collect-prototypes)))

;;------------------------------------------------------------------------

(defun f2c-fix-powers ()
  (goto-char (point-min))
  (while (re-search-forward "\\*\\*" nil t)
    (if (f2c-not-comment)
	(progn
	  (replace-match ",")
	  (backward-sexp)
	  (insert "pow_xx(")
	  (forward-sexp 2)
	  (insert ")")))))

(defun f2c-collect-prototypes ()
  (goto-char (point-min))
  (while (not (= (following-char) ?\n)) (forward-line 1))
  (insert "\n\n\n/*-----prototypes of functions defined here-----*/\n")
  (let ((top (point))
	(body (make-marker))
	proto)
    (insert "/*-----end of prototypes-----*/\n")
    (while (search-forward "{" nil t)
      (beginning-of-line)
      (set-marker body (point))
      (backward-sexp)
      (beginning-of-line)
      (setq proto (buffer-substring (point) body))
      (goto-char top)
      (insert "extern " proto)
      (backward-char)
      (insert ";")
      (goto-char body)
      (forward-sexp)))
  (goto-char (point-min)))

;;------------------------------------------------------------------------

(defun f2c-cleanup-intrinsics ()
  "Clean up a selection of common Fortran intrinsic functions."
  (goto-char (point-min))
  (while (not (= (following-char) ?\n)) (forward-line 1))
  (insert "\n\n\n/*-----Fortran intrinsics converted-----*/\n")
  (f2c-declare-intrinsic "pow_xx" nil
			 "extern double pow_xx(examine,by_hand);\n")
  (f2c-declare-intrinsic "abs" nil "#define abs(x) ((x)>=0?(x):-(x))\n")
  (f2c-declare-intrinsic "iabs" nil "#define iabs(x) ((x)>=0?(x):-(x))\n")
  (f2c-declare-intrinsic "dabs" nil
			 "#define dabs(x) ((double)((x)>=0?(x):-(x)))\n")
  (f2c-declare-intrinsic "d?sqrt" "sqrt" "extern double sqrt(double);\n")
  (f2c-declare-intrinsic "d?exp" "exp" "extern double exp(double);\n")
  (f2c-declare-intrinsic "[ad]log" "log" "extern double log(double);\n")
  (f2c-declare-intrinsic "[ad]log10" "log10" "extern double log10(double);\n")
  (f2c-declare-intrinsic "d?sin" "sin" "extern double sin(double);\n")
  (f2c-declare-intrinsic "d?cos" "cos" "extern double cos(double);\n")
  (f2c-declare-intrinsic "d?tan" "tan" "extern double tan(double);\n")
  (f2c-declare-intrinsic "d?asin" "asin" "extern double asin(double);\n")
  (f2c-declare-intrinsic "d?acos" "acos" "extern double acos(double);\n")
  (f2c-declare-intrinsic "d?atan" "atan" "extern double atan(double);\n")
  (f2c-declare-intrinsic "d?atan2" "atan2" "extern double atan2(double);\n")
  (f2c-declare-intrinsic "d?sinh" "sinh" "extern double sinh(double);\n")
  (f2c-declare-intrinsic "d?cosh" "cosh" "extern double cosh(double);\n")
  (f2c-declare-intrinsic "d?tanh" "tanh" "extern double tanh(double);\n")
  (f2c-declare-intrinsic "[ad]mod" "fmod" "extern double fmod(double);\n")
  (f2c-declare-intrinsic "mod" nil "#define mod(x,y) ((x)%(y))\n")
  ;(f2c-declare-intrinsic "float" nil "#define float(x) ((float)x)\n")
  (f2c-declare-intrinsic "dble" nil "#define dble(x) ((double)x)\n")
  (f2c-declare-intrinsic "ifix" nil "#define ifix(x) ((long)x)\n")
  (f2c-declare-intrinsic "nint" nil
			 "#define nint(x) ((long)((x)>=0?(x)+.5:(x)-.5))\n")
  (f2c-declare-intrinsic "idnint" nil
			 "#define nint(x) ((long)((x)>=0?(x)+.5:(x)-.5))\n")
  ;(f2c-declare-intrinsic "int" nil "#define int(x) ((long)x)\n")
  (f2c-declare-intrinsic "idint" nil "#define idint(x) ((long)x)\n")
  ;; this aint macro is not really safe!
  (f2c-declare-intrinsic "aint" nil "#define aint(x) ((double)((long)(x)))\n")
  ;; Note -- these might be very bad if y argument involves a function call
  (f2c-declare-intrinsic "isign" nil
			 "#define isign(x,y) ((((x)<0)!=((y)<0))?-(x):(x))\n")
  (f2c-declare-intrinsic "sign" nil
			 "#define sign(x,y) ((((x)<0)!=((y)<0))?-(x):(x))\n")
  (f2c-declare-intrinsic "dsign" nil
			 "#define dsign(x,y) ((((x)<0)!=((y)<0))?-(x):(x))\n")
  ;; these fail if more than two arguments...
  (f2c-declare-intrinsic "min" nil "#define min(x,y) ((x)<(y)?(x):(y))\n")
  (f2c-declare-intrinsic "max" nil "#define max(x,y) ((x)>(y)?(x):(y))\n")
  (insert "/*-----end of Fortran intrinsics-----*/\n"))

;; in Fortran math library (not libm.a?)
;;extern double pow_di(doublereal *x, integer *i);         x**n
;;extern double pow_dd(doublereal *x, doublereal *s);      x**x  (pow)
;;extern double d_lg10(doublereal *x);                     log10?
;;extern double d_sign(doublereal *c, doublereal *x);      sign
;;extern integer i_dnnt(doublereal *x);                    nint

(defun f2c-declare-intrinsic (from to at-top)
  (let ((pattern (concat "\\b" from "\\b"))
	(top (point)))
    (if (re-search-forward pattern nil t)
	(progn
	  (if to (progn
		   (replace-match to)
		   (while (re-search-forward pattern nil t)
		     (replace-match to))))
	  (goto-char top)
	  (insert at-top)
	  t))))

;;------------------------------------------------------------------------

(defun f2c-cleanup-gotos ()
  "Clean up 2 and 3 branch Fortran if-gotos."
  (goto-char (point-min))
  (while (re-search-forward "#goto" nil t)
    (let* ((beg (point))
	   (end (save-excursion (skip-chars-forward "^;") (point)))
	   (labels (f2c-label-list nil end)))
      (goto-char beg)
      (cond ((= (length labels) 2)
	     (backward-char 5)
	     (delete-region (point) end)
	     (insert " goto " (car labels)
		     "; else goto " (car (cdr labels))))
	    ((= (length labels) 3)
	     (backward-char 5)
	     (delete-region (point) (1+ end))
	     (backward-sexp)
	     (setq end (point))
	     (forward-word -1)
	     (delete-region (point) end)
	     (insert "{double go_cnd=")
	     (forward-sexp)
	     (insert "; if (go_cnd<0) goto " (car labels)
		     "; else if (go_cnd==0) goto " (car (cdr labels))
		     "; else goto " (car (cdr (cdr labels))) ";}"))))))

(defun f2c-label-list (list limit)
  (let ((next (f2c-next-label limit)))
    (if next
	(f2c-label-list (append list (list next)) limit)
      list)))

(defun f2c-next-label (limit)
  (let ((vend (progn (forward-sexp) (point))))
    (if (> (point) limit)
	(progn (goto-char limit)
	       nil)
      (backward-sexp)
      (let ((result (buffer-substring (point) vend)))
	(goto-char vend)
	result))))

;;------------------------------------------------------------------------

(defun f2c-arglist (eol)
  "Scan through a function argument list, returning it as a list of strings."
  (let (arglist arg)
    (while (setq arg (f2c-next-token eol ","))
      (setq arglist (cons arg arglist)))
    arglist))

(defun f2c-next-keyword (limit)
  "Scan to the next non-comment line and return the first word on it."
  (beginning-of-line)
  (while (or (= (following-char) ?\n) (not (f2c-not-comment)))
      (forward-line 1)) ; skip blank or comment lines
  (let ((end (progn (forward-word 1) (point))) result)
    (forward-word -1)
    (setq result (buffer-substring (point) end))
    (goto-char end)
    (if (< (point) limit) result)))

(defvar f2c-scalar-list nil)

(defun f2c-fix-declarations (func-args beg-body end-body func-name)
  "Repair Fortran declarations, for some programming styles."
  (let ((beg (marker-position beg-body))
        (implicit-type "float")
	(orig-func-args (copy-sequence func-args))
        keyword f2c-scalar-list body)
    (goto-char beg)

    ;; Try to handle existing Fortran declarations
    (message (concat "Processing " func-name ": explicit declarations"))
    (while (progn
             (setq keyword (f2c-next-keyword (marker-position end-body)))
             (cond ((string= keyword "dimension")
                    (f2c-handle-dimension func-args implicit-type end-body)
                    t)
                   ((or (string= keyword "double")
                        (string= keyword "long")
                        (string= keyword "float")
                        (string= keyword "int")
                        (string= keyword "short")
                        (string= keyword "char"))
                    (f2c-handle-type keyword func-args end-body)
                    t)
                   ((string= keyword "implicit")
                    (setq implicit-type (f2c-next-token
                                         (marker-position end-body)))
                    (f2c-comment-out)
                    t)
                   ((string= keyword "parameter")
                    (f2c-handle-parameter)
                    t)
                   ((string= keyword "data")
                    (f2c-handle-data beg implicit-type)
                    t)
                   ((string= keyword "external")
                    (f2c-handle-external beg implicit-type)
                    t)
                   ((string= keyword "intrinsic")
                    (f2c-handle-intrinsic)
                    t))))

    (beginning-of-line)
    (setq body (point)) ; end of declarations
    (insert "      /*-----end-of-declarations-----*/\n")
    (message (concat "Processing " func-name ": implicit declarations"))
    (while (re-search-forward "\\b\\sw+[ \t]*=[^=]"
                              (marker-position end-body) t)
      (let* ((name (buffer-substring (match-beginning 0)
                                     (progn (backward-char 2)
                                            (skip-chars-backward " \t")
                                            (point))))
             (mem (member name orig-func-args)))
        (cond (mem
	       ;; if a parameter's value is set, it must be a pointer
	       (forward-word -1)
               (insert "*")
	       (forward-word 1)
	       (let ((na (length mem)))
		 (if (setq mem (member name func-args))
		     (setcar mem (concat
				  (if (member (aref name 0)
					      '(?i ?j ?k ?l ?m ?n))
				      "long" implicit-type)
				  " *" name))
		   (setq mem (nthcdr (- (length func-args) na) func-args))
		   (if (and (string-match " " (car mem))
			    (not (= (aref (car mem) (match-end 0)) ?*)))
		       (setcar mem (concat
				    (substring (car mem) 0 (match-end 0))
				    "*" name))))))
              ((not (member name f2c-scalar-list))
	       ;; otherwise, implicitly declared local variable
               (save-excursion
                 (goto-char body)
                 (insert "      "
                         (if (member (aref name 0) '(?i ?j ?k ?l ?m ?n))
                             "long" implicit-type)
                         " " name ";\n")
                 (setq f2c-scalar-list (cons name f2c-scalar-list)))))))
    (goto-char body)
    (insert "      /*-----implicit-declarations-----*/\n")
    ;; Replace argument list with prototypes
    (goto-char beg)
    (forward-line -1)
    (while (not (search-forward "(" beg t)) (forward-line -1))
    (skip-chars-forward " \t")
    (setq func-args (reverse func-args))
    (let ((start (point)))
      (while (not (= (following-char) ?\)))
        (forward-word 1)
	(delete-region start (point))
	(if (string-match " " (car func-args))
	    (insert (car func-args))
	  (insert (if (member (aref (car func-args) 0) '(?i ?j ?k ?l ?m ?n))
		      "long"
		    implicit-type)
		  " " (car func-args)))
        (setq func-args (cdr func-args))
	(skip-chars-forward " \t,")
	(setq start (point))))))

;;------------------------------------------------------------------------

(defun f2c-handle-dimension (func-args implicit-type end-body)
  (let ((type (consp (f2c-is-integer)))
        (beg (point))
	alternates)
    (forward-word -1)
    (delete-region (point) beg)
    (insert (if type "long" implicit-type))
    (f2c-skip-declarator)
    (while (progn (skip-chars-forward "^;a-z")
                  (not (= (following-char) ?\;)))
      (if (eq (consp (f2c-is-integer)) type)
          (f2c-skip-declarator)
        (setq alternates
              (concat (if alternates
                          (concat alternates ", ")
                        (concat "      " (if type implicit-type "long") " "))
                      (f2c-remove-declarator)))))
    (goto-char beg)
    (f2c-handle-type (if type "long" implicit-type) func-args end-body)
    (if alternates
        (progn
          (setq beg (point))
          (insert alternates ";\n")
          (goto-char beg)
          (forward-word 1)
          (f2c-handle-type (if type implicit-type "long")
			   func-args end-body)))))

(defun f2c-skip-declarator ()
  (forward-word 1) ; skip name, including preceding punctuation
  (skip-chars-forward " \t")
  (if (= (following-char) ?\() (forward-sexp)))

(defun f2c-remove-declarator ()
  (skip-chars-backward " \t\n")
  (let ((zapped (= (preceding-char) ?,)))
    (if zapped (delete-char -1) (skip-chars-forward " \t\n"))
    (if (= (following-char) ?\n) (skip-chars-forward " \t\n"))
    (let ((beg (point)) result)
      (forward-word 1)
      (skip-chars-forward " \t")
      (if (= (following-char) ?\() (forward-sexp))
      (setq result (buffer-substring beg (point)))
      (if (not zapped)
          (progn
            (skip-chars-forward " \t\n")
            (if (= (following-char) ?,) (forward-char 1))))
      (delete-region beg (point))
      result)))

(defun f2c-is-integer ()
  (save-excursion
    (skip-chars-forward " \t")
    (member (following-char) '(?i ?j ?k ?l ?m ?n))))

;;------------------------------------------------------------------------

(defun f2c-handle-type (keyword func-args end-body)
  (let ((beg (point))
        (nuke-line t)
	dimensioned name dims)
    (while (progn (skip-chars-forward "^;a-z")
                  (not (= (following-char) ?\;)))
      (let* ((start (point))
             (name (buffer-substring start (progn (forward-word 1) (point))))
             (dstart (progn (skip-chars-forward " \t\n") (point)))
             (dims (if (= (following-char) ?\()
                       (buffer-substring dstart
                                         (progn (forward-sexp) (point))))))
        (let ((mem (member name func-args)))
          (if (not mem)
              (progn
                (setq nuke-line nil)
                (if (not dims)
                    (setq f2c-scalar-list (cons name f2c-scalar-list))))
            (goto-char start)
            (f2c-remove-declarator))
          (if (not dims)
              (if mem (setcar mem (concat keyword " " (car mem))))
            (if mem (setcar mem (concat keyword " " (car mem) "[]"))
              (setq dims (f2c-c-dims dstart))) ; list if local
            (setq dimensioned
                  (cons (cons name dims) dimensioned))))))
    (forward-line 1)
    (if nuke-line
        (let ((end (point)))
          (goto-char beg)
          (beginning-of-line)
          (delete-region (point) end)))
    (while dimensioned
      (setq name (car (car dimensioned)))
      (setq dims (cdr (car dimensioned)))
      (setq dimensioned (cdr dimensioned))
      (let (suffix start)
	(if (consp dims)
            ;; local array
	    (let ((ndims (length dims)))
	      (setq suffix (concat "_" (number-to-string ndims)))
	      (f2c-insert-define name ndims)
              (while dims
                (insert "[a" (number-to-string (length dims)) "-1]")
                (setq dims (cdr dims))))
          ;; array parameter
	  (let* ((dim-list (f2c-convert-dimlist dims))
		 (ndims (length dim-list)))
	    (setq suffix (concat "_" (number-to-string ndims)))
            (f2c-insert-define name ndims)
            (insert name "[")
            (insert (f2c-index-macro-body
                     (concat "a" (number-to-string ndims) "-1")
                     (cdr dim-list)))
            (insert "]")))
	(insert "\n")
	;; Now fix up all references to the variable to use this macro
        (setq start (point))
	(while (re-search-forward (concat "\\b" name "\\b")
				  (marker-position end-body) t)
	  (if (progn (skip-chars-forward " \t") (= (following-char) ?\())
	      (let ((beg (match-beginning 0))
		    (end (match-end 0)))
		(goto-char end)
		(insert suffix)
		(forward-sexp)
		(skip-chars-forward " \t\n")
		(if (member (following-char) '(?\) ?,))
		    (progn
		      (goto-char beg)
		      (skip-chars-backward " \t\n")
		      (if (member (preceding-char) '(?\( ?,))
			  (progn
			    (goto-char beg)
			    ;; may require address -- need prototype to tell
			    ;(insert "(void *)&")
			    (forward-word 1))))))))
        (goto-char start)))))

(defun f2c-convert-dimlist (dims)
  (let (dim-list
        (beg 1))
    (if (string-match "\\*" dims) (aset dims (match-beginning 0) ?1))
    (while (string-match "\\b\\sw+\\b" dims beg)
      (setq beg (match-end 0))
      (setq dim-list (cons (substring dims (match-beginning 0) beg)
			   dim-list)))
    dim-list))

(defun f2c-index-macro-body (body dims)
  (if (not dims) body
    (let ((ndims (length dims)))
      (f2c-index-macro-body
       (concat "a" (number-to-string ndims) "-1+" (car dims) "*(" body ")")
       (cdr dims)))))

(defun f2c-insert-define (name ndims)
  (insert "#undef " name "_" (number-to-string ndims) "\n")
  (insert "#define " name "_" (number-to-string ndims) "(")
  (let ((i 1))
    (while (<= i ndims)
      (insert "a" (number-to-string i))
      (if (< i ndims) (insert ","))
      (setq i (1+ i))))
  (insert ") "))

(defun f2c-c-dims (start)
  (let ((end (point)) dim-list result)
    (goto-char start)
    (while (progn (skip-chars-forward " \t(,")
                  (not (= (following-char) ?\))))
      (let ((beg (point)))
        (setq dim-list
              (cons (buffer-substring beg (progn (forward-word 1) (point)))
                    dim-list))))
    (delete-region start end)
    (setq result dim-list)
    (while dim-list
      (insert "[" (car dim-list) "]")
      (setq dim-list (cdr dim-list)))
    result))

;;------------------------------------------------------------------------

(defun f2c-split-data ()
  "Split certain data statements into multiple data statements."
  (goto-char (point-min))
  (let ((eol-mark (make-marker)) end)
    (while (re-search-forward "^[ \t]+data\\b" nil t)
      (set-marker eol-mark (f2c-eol))
      (while (re-search-forward "/[ \t]*," (marker-position eol-mark) t)
        (setq end (point))
        (skip-chars-backward "^/")
        (delete-region (point) end)
        (insert "\n      data")))))

(defun f2c-handle-data (beg implicit-type)
  (let* ((name (f2c-next-word (save-excursion (end-of-line) (point))))
         (decl (or (progn (forward-word -2) (f2c-steal-decl name beg))
		   (concat (if (member (aref name 0) '(?i ?j ?k ?l ?m ?n))
			       "long" implicit-type) " " name)))
	 (dstart (point))
	 (nstart (progn (forward-word 1)
			(skip-chars-forward " \t\n") (point))))
    (skip-chars-forward "^;/")
    (if (and decl (= (following-char) ?/))
        (if (string-match "\\[" decl)
	    (progn
	      (delete-region dstart (1+ (point)))
	      (insert "static " decl "={")
	      (skip-chars-forward "^;/")
	      (if (= (following-char) ?/) (delete-char 1))
	      (insert "}"))
	  (let (name-list val-list
		(eon (point))
		(eov (save-excursion (skip-chars-forward "^;") (point))))
	    (setq val-list (f2c-word-list nil eov))
	    (goto-char nstart)
	    (forward-word 1)
	    (setq name-list (f2c-word-list (list decl) eon))
	    (if (= (length name-list) (length val-list))
		(progn
		  (goto-char dstart)
		  (beginning-of-line)
		  (setq dstart (point))
		  (goto-char eov)
		  (forward-line 1)
		  (delete-region dstart (point))
		  (while name-list
		    (insert "      static "
			    (or decl
				(f2c-steal-decl (car name-list) beg)
				(if (member (aref (car name-list) 0)
					    '(?i ?j ?k ?l ?m ?n))
				    "long" implicit-type))
			    "= " (car val-list) ";\n")
		    (setq name-list (cdr name-list))
		    (setq val-list (cdr val-list)))
		  (backward-char 2))))))
    (skip-chars-forward "^;")
    (forward-line 1)))

(defun f2c-word-list (list limit)
  (let ((next (f2c-next-word limit)))
    (if next
	(f2c-word-list (append list (list next)) limit)
      list)))

(defun f2c-next-word (limit)
  (let ((vend (progn (forward-word 1) (point))))
    (if (> (point) limit)
	(progn (goto-char limit)
	       nil)
      (forward-word -1)
      (let ((result (buffer-substring (point) vend)))
	(goto-char vend)
	result))))

(defun f2c-steal-decl (name beg)
  (save-excursion
    (let (decl)
      (while
          (and
           (re-search-backward (concat "\\b" name "\\b[^_]") beg t)
           (or (not (f2c-not-comment))
               (let ((start (point)) end fin)
                 (forward-word 1)
                 (skip-chars-forward "^;,")
                 (if (= (following-char) ?,)
                     (setq end (1+ (point)))
                   (setq fin (point)))
                 (skip-chars-backward " \t\n")
                 (setq decl (buffer-substring start (point)))
                 (goto-char start)
                 (skip-chars-backward " \t\n")
                 (if end
                     (delete-region (point) end)
                   (if (= (preceding-char) ?,)
                       (delete-region (1- (point)) fin)
                     (setq fin nil)))
                 (while (= (preceding-char) ?,)
                   (backward-char)
                   (while (progn
                            (skip-chars-backward " \t\n")
                            (= (preceding-char) ?\]))
                     (backward-sexp))
                   (forward-word -1)
                   (skip-chars-backward " \t\n"))
                 (setq start (point))
                 (beginning-of-line)
                 (setq decl
                       (concat (buffer-substring (point) start) " " decl))
                 (if (not (or end fin))
                     (progn
                       (setq start (point))
                       (skip-chars-forward "^;")
                       (forward-line 1)
                       (delete-region start (point))))))))
      decl)))

;;------------------------------------------------------------------------

(defun f2c-handle-parameter ()
  (let ((beg (save-excursion (beginning-of-line) (point)))
        (count 0)
        param-list)
    (while (progn (skip-chars-forward " \t,()")
                  (not (= (following-char) ?\;)))
      (let* ((start (point))
             (name (buffer-substring
                    start (progn (forward-word 1) (point))))
             (vstart (progn (skip-chars-forward " \t=") (point)))
             (value (buffer-substring
                     vstart (progn (skip-chars-forward "^,)")
                                   (skip-chars-backward " \t") (point)))))
        (setq param-list (cons (cons name value) param-list))))
    (forward-line 1)
    (delete-region beg (point))
    ;; In LAPACK previous line will be type declaration -- zap it
    (f2c-maybe-nuke-prev beg (car (car param-list)))
    (setq beg (point))
    (while param-list
      (insert "#undef " (car (car param-list)) "\n"
              "#define " (car (car param-list))
	      " " (cdr (car param-list)) "\n")
      (setq param-list (cdr param-list))
      (+ count 2)
      (goto-char beg))
    (forward-line count)))

(defun f2c-maybe-nuke-prev (beg what)
  (forward-line -1)
  (let ((prev (point)))
    (if (save-excursion
          (re-search-forward (concat "\\b" what "\\b")
                             beg t))
        (delete-region prev beg))))

;;------------------------------------------------------------------------

(defun f2c-handle-external (beg implicit-type)
  (let* ((bol (point))
         (eol (progn (skip-chars-forward "^;") (forward-line 1) (point)))
         (mrk (set-marker (make-marker) eol)))
    (goto-char bol)
    (while (progn (skip-chars-forward " \t\n")
                  (not (= (following-char) ?\;)))
      (let* ((name (f2c-next-token eol ","))
             (decl (or (save-excursion (goto-char bol)
				       (f2c-steal-decl name beg))
		       (concat
			(cond ((member name f2c-called-list)
			       "void")
			      ((member (aref name 0) '(?i ?j ?k ?l ?m ?n))
			       "long")
			      (t
			       implicit-type))
			" " name))))
        (save-excursion
          (goto-char mrk)
          (insert-before-markers
	   (concat "      extern " decl "();\n")))))
    (goto-char bol)
    (beginning-of-line)
    (delete-region (point) eol)
    (goto-char mrk)))

(defun f2c-handle-intrinsic ()
  (f2c-comment-out))

(defun f2c-comment-out ()
  (beginning-of-line)
  (insert "/*")
  (re-search-forward ";")
  (insert "*/")
  (forward-char 1))

;;------------------------------------------------------------------------

(defun f2c-fix-semicolons (beg-body end-body)
  "Put semi-colons at the end of all statements and remove continuation
markers in column 6."
  (goto-char beg-body)
  (while (< (point) (marker-position end-body))
    (while (or (= (following-char) ?\n) (not (f2c-not-comment)))
      (forward-line 1)) ; skip blank or comment lines
    (f2c-discontinue)
    (if (not (member (preceding-char) '(?: ?{ ?}))) (insert ";"))
    (forward-char 1)))

(defun f2c-discontinue ()
  "Remove continuation characters from following lines, leaving point
at end of final continuation line."
  (while (progn
           (forward-line 1)
             (skip-chars-forward " \t")
             (= (current-column) 5))
      (delete-char 1)
      (insert " "))
  (beginning-of-line)
  (backward-char 1))

;;------------------------------------------------------------------------

(setq f2c-label-list nil)

(defun f2c-fix-labels (beg-body end-body)
  "Repair statement labels by prefixing ones which are referenced by L_.
Unreferenced labels are replaced by blanks."
  (goto-char beg-body)
  (let (f2c-label-list)
    ;; First, build a list of labels actually referenced in gotos.
    ;; (Assume do references have already been eliminated in favor of for.)
    (while (re-search-forward "\\bgo[ \t]*to\\b"
                              (marker-position end-body) t)
      (replace-match "goto")
      (skip-chars-forward " \t")
      (if (= (following-char) ?\() (forward-char 1))
      (while (f2c-add-label)
        (save-excursion (skip-chars-backward "0-9") (insert "L_"))))
    ;; Next, repair or remove the labels themselves
    (goto-char beg-body)
    (forward-char 1)
    (while (< (point) (marker-position end-body))
      (while (and
	      (or (= (following-char) ?\n) (not (consp (f2c-not-comment))))
	      (< (point) (marker-position end-body)))
        (forward-line 1))
      (if (< (point) (marker-position end-body))
          (progn
            (skip-chars-forward " \t")
            (let ((beg (point)))
              (skip-chars-forward "0-9")
              (if (member (buffer-substring beg (point)) f2c-label-list)
                  (progn (goto-char beg) (insert "      L_")
                         (skip-chars-forward "0-9")
                         (insert ":")
                         (setq beg (point))
                         (and (looking-at "[ \t]*continue$")
                              (delete-region beg
                                             (progn (end-of-line) (point)))))
                (delete-region beg (point))
                (insert "      ")))
            (forward-line 1)))))
  ;; Finally, remove any remaining continue statements
  (goto-char beg-body)
  (while (re-search-forward "\\bcontinue\\b" (marker-position end-body) t)
    (if (f2c-not-comment)
        (delete-region (progn (beginning-of-line) (point))
                       (progn (forward-line 1) (point))))))

(defun f2c-add-label ()
  "Find next label of goto and add to f2c-label-list, return nil if none."
  (let ((limit (f2c-eol)))
    (skip-chars-forward " \t," limit)
    (while (and (< (point) limit) (= (following-char) ?\n))
      (forward-char 1)
      (move-to-column 6)
      (skip-chars-forward " \t," limit))
    (let ((beg (point)) end)
      (if (< beg limit)
	  (progn
	    (skip-chars-forward "0-9" limit)
	    (if (> (point) beg)
                (let ((result (buffer-substring beg (point))))
                  (if (not (member result f2c-label-list))
                      (setq f2c-label-list (cons result f2c-label-list)))
                  result)))))))

;;------------------------------------------------------------------------

(defun f2c-fix-comments ()
  "Replace Fortran comment lines by C comments.  f2c-not-comment will
still correctly identify all comment lines.  Empty Fortran comment lines
will become blank lines, however."
  (goto-char (point-min))
  (while (< (point) (point-max))
    (while (f2c-not-comment)
      (forward-line 1))
    (if (< (point) (point-max))
        (let (blank (prev-was-blank (f2c-blank-comment)))
          (insert "/*")
          (forward-line 1)
          (while (and (< (point) (point-max))
                      (not (f2c-not-comment))
                      (or (not (setq blank (f2c-blank-comment)))
                          (not prev-was-blank)))
            (setq prev-was-blank blank)
            (forward-line 1))
          (backward-char 1)
          (insert "*/")
          (forward-line 1))))
  (goto-char (point-min))
  (while (re-search-forward "/\\*.\\*/" nil t)
    (replace-match "")))

(defun f2c-blank-comment ()
  "nil if this comment line has more than 1 non-blank character."
  (save-excursion
    (skip-chars-forward " \t")
    (or (= (following-char) ?\n)
        (progn (forward-char 1) (= (following-char) ?\n)))))

;;------------------------------------------------------------------------

(defun f2c-fix-ifs ()
  "Replace all Fortran if-s with C if-s.  The word #goto is inserted before
two and three branch if-s to aid f2c-fix-labels."
  (let ((case-fold-search nil))
    (f2c-replace "\\bthen\\b" "{")
    (f2c-replace "\\belse\\b" "} else {")
    (f2c-replace "\\belseif\\b" "} else if")
    (f2c-replace "} else {[ \t]*if" "} else if")
    (f2c-replace "\\bend[ \t]*if\\b" "}")
    (goto-char (point-min))
    (while (re-search-forward "^[ \t]+if[ \t]*(" nil t)
      (backward-char 1)
      (forward-sexp)
      (skip-chars-forward " \t")
      (and (member (following-char) '(?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0 ?,))
           (insert "#goto ")))))

;;------------------------------------------------------------------------

(defun f2c-fix-dos ()
  "Replace all Fortran do-s with C for-s."
  (goto-char (point-min))
  (let ((case-fold-search nil))
    (while (re-search-forward "\\bdo\\b" (point-max) t)
      (f2c-do-convert)
      (forward-line 1))))

(defun f2c-do-convert ()
  "Convert do loop to more C-like form."
  (beginning-of-line)
  (save-excursion
    (let ((limit (f2c-eol)))
      (if (re-search-forward "\\bdo\\b" limit t)
          (let* ((beg (match-beginning 0))
                 (label (f2c-next-token limit "\\=[ \t]*,?"))
		 (variable (f2c-next-token limit))
                 (init-expr (f2c-next-expr limit "="))
                 (last-expr (f2c-next-expr limit ","))
                 (step-expr (or (f2c-next-expr limit ",") "1")))
            (if (and last-expr (f2c-beyond-label label))
                (let ((init (concat "for (" variable "=" init-expr " ; "))
                      test
		      (step (concat " ; " variable "+=" step-expr ")")))
                  (insert "      }\n")
                  (goto-char beg)
                  (delete-region beg limit)
		  (cond ((string-match "^[0-9]+$" step-expr)
                         (setq test (concat variable "<=" last-expr)))
                        ((string-match "^-[0-9]+$" step-expr)
                         (setq test (concat variable ">=" last-expr)))
                        (t
                         (setq test
                               (concat step-expr ">0?"
                                       (concat variable "<=" last-expr) ":"
                                       (concat variable ">=" last-expr)))))
                  (insert init test step " {"))))))))

(defun f2c-next-token (limit &optional following)
  "Advance past and return next token (a number or word)."
  (skip-chars-forward " \t" limit)
  (while (and (< (point) limit) (= (following-char) ?\n))
    (forward-char 1)
    (move-to-column 6)
    (skip-chars-forward " \t" limit))
  (let ((beg (point)))
    (if (< beg limit)
	(progn
          (forward-word 1)
          (if (and (> (point) beg) (<= (point) limit))
              (let ((result (buffer-substring beg (point))))
                (if following (re-search-forward following limit t))
                result))))))

(defun f2c-next-expr (limit &optional leading)
  "Advance past and return next expression (up to comma or closed paren),
optionally skipping a leading punctuation string (not regexp)."
  (and
   (or (not leading) (search-forward leading limit t))
   (progn
     (skip-chars-forward " \t" limit)
     (while (and (< (point) limit) (= (following-char) ?\n))
       (forward-char 1)
       (move-to-column 6)
       (skip-chars-forward " \t" limit))
     (let ((beg (point)) (end (point)))
       (while (and (< end limit)
                   (not (looking-at "[ \t]*[,)]")))
         ;; ********************************* WRONG if continuation line!!!
         (forward-sexp)
         (setq end (point)))
       (if (> end beg) (buffer-substring beg end))))))

(defun f2c-beyond-label (label &optional limit)
  "Move to line just beyond label (a string), assuming you are before it."
  (if (re-search-forward (concat "^[ \t]*" label "\\b") limit t)
      (progn (forward-line 1) t)))

;;------------------------------------------------------------------------

(defun f2c-zap-trailing ()
  "Remove trailing whitespace from all lines."
  (goto-char (point-min))
  (while (< (point) (point-max))
    (end-of-line)
    (let ((end (point)))
      (skip-chars-backward " \t")
      (delete-region (point) end))
    (forward-line 1))
  (if (not (= (preceding-char) ?\n)) (insert "\n")))

(defun f2c-downcase ()
  "Downcase all non-comment Fortran source lines."
  (goto-char (point-min))
  (while (< (point) (point-max))
    (while (or (= (following-char) ?\n) (not (f2c-not-comment)))
      (forward-line 1)) ; skip blank or comment lines
    (let ((beg (point)))
      (forward-line 1)
      (while (f2c-not-comment)
        (forward-line 1))
      (downcase-region beg (point)))))

(defun f2c-fix-doubles ()
  "Get rid of any 3.4e-5 style numbers, in favor of 3.4e-5 style."
  (f2c-replace "\\b\\([0-9]*.?[0-9]*\\)[dD]\\([-+]?[0-9]+\\)\\b"
               "\\1e\\2"))

(defun f2c-fix-ops ()
  "Convert Fortran logical operators into C."
  (f2c-replace "\\.eq\\." "==")
  (f2c-replace "\\.ne\\." "!=")
  (f2c-replace "\\.and\\." "&&")
  (f2c-replace "\\.or\\." "||")
  (f2c-replace "\\.not\\." "!")
  (f2c-replace "\\.true\\." "1")
  (f2c-replace "\\.false\\." "0")
  (f2c-replace "\\.gt\\." ">")
  (f2c-replace "\\.ge\\." ">=")
  (f2c-replace "\\.lt\\." "<")
  (f2c-replace "\\.le\\." "<="))

(defun f2c-fix-types ()
  "Convert Fortran data type names into C (also function, subroutine, call)."
  (f2c-replace "\\bprecision\\b" "")
  (f2c-replace "\\breal[ \t]*\*[ \t]*8\\b" "double")
  (f2c-replace "\\breal[ \t]*\*[ \t]*4\\b" "float")
  (f2c-replace "\\breal\\b" "float")
  (f2c-replace "\\binteger[ \t]*\*[ \t]*4\\b" "long")
  (f2c-replace "\\binteger[ \t]*\*[ \t]*2\\b" "short")
  (f2c-replace "\\binteger\\b" "long")
  (f2c-replace "\\blogical\\([ \t]*\*[ \t]*[24]\\)?\\b" "int")
  (f2c-replace "\\bcharacter\\([ \t]*\*[ \t]*1\\)?\\b" "char")
  (f2c-replace "\\bfunction\\b" "")
  (goto-char (point-min))
  (while (re-search-forward "\\b\\(subroutine\\|call\\)\\b" nil t)
    (if (f2c-not-comment)
        (let ((prev (preceding-char)))
          (replace-match (if (= prev ?e) "void" ""))
	  (if (= prev ?e)
	      (progn
		(forward-word 1)
		(skip-chars-forward " \t"))
	    (f2c-add-called (f2c-next-word (save-excursion
					     (forward-line 1) (point)))))
          (if (= (following-char) ?\n)
              (insert (if (= prev ?e) "" "()")))))))

(defun f2c-add-called (name)
  (if (not (member name f2c-called-list))
      (setq f2c-called-list (cons name f2c-called-list))))

(defun f2c-eol ()
  "Return value of point at end of (possibly continued) Fortran line."
  (save-excursion
    (forward-line 1)
    (while (save-excursion (skip-chars-forward " \t")
                           (= (current-column) 5))
      (forward-line 1))
    (1- (point))))

(defun f2c-replace (from to &optional limit)
  "Replace all occurences of regexp FROM by TO on non-comment lines."
  (goto-char (point-min))
  (while (re-search-forward from limit t)
    (if (f2c-not-comment) (replace-match to))))

(defun f2c-not-comment ()
  "nil if point is on a comment line."
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (or (> (current-column) 4)
        (memq (following-char) '(?0 ?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?\n)))))

;;------------------------------------------------------------------------
