/*
 * Copyright (c) 1994, 1995.  Netscape Communications Corporation.  All
 * rights reserved.
 * 
 * Use of this software is governed by the terms of the license agreement for
 * the Netscape FastTrack or Netscape Enterprise Server between the
 * parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * objset.h: Handles object sets
 * 
 * Each object is produced by reading a config file of some form. See the
 * server documentation for descriptions of the directives that are 
 * recognized, what they do, and how they are parsed.
 * 
 * This module requires the pblock and buffer modules from the base library.
 *
 * Rob McCool
 */


#ifndef OBJSET_H
#define OBJSET_H

#include "netsite.h"
#include "base/pblock.h"
#include "base/buffer.h"
#include "frame/object.h"



/* ------------------------------ Constants ------------------------------- */


/* 
 * The default number of objects to leave room for in an object set,
 * and the number of new entries by which to increase the size when that 
 * room is filled.
 */

#define OBJSET_INCSIZE 8

/*
 * When parsing config. files, httpd will put a limit on how long
 * the parameters to a directive can be (in characters).
 * 
 * Currently set to 10 lines (80 chars/line).
 */

#ifdef MCC_PROXY
#define MAX_DIRECTIVE_LEN 100000
#else
#define MAX_DIRECTIVE_LEN 800
#endif /* MCC_PROXY */

/*
 * The size of the hash tables that store a directive's parameters
 */

#define PARAMETER_HASH_SIZE 3


/* ------------------------------ Structures ------------------------------ */


/*
 * httpd_objset is a container for a bunch of objects. obj is a 
 * NULL-terminated array of objects. pos points to the entry after the last
 * one in the array. You should not mess with pos, but can read it to find
 * the last entry.
 *
 * The initfns array is a NULL-terminated array of the Init functions 
 * associated with this object set. If there are no Init functions associated
 * with this object set, initfns can be NULL. Each pblock specifies the
 * parameters which are passed to the function when it's executed.
 */

typedef struct {
    int pos;
    httpd_object **obj;

    pblock **initfns;
} httpd_objset;


/* ------------------------------ Prototypes ------------------------------ */


/*
 * objset_scan_buffer will scan through buffer, looking for object 
 * configuration information, and adding them to the object set os if it 
 * finds any. If os is NULL it will allocate a new object set.
 *
 * If any error occurs (syntax error, premature EOF) this function will
 * free os, print an error message into errstr, and return NULL.
 * This is because a config. file error is viewed as a catastrophic error
 * from which httpd should not try to recover. If httpd were to continue
 * after an error, it would not behave as the admin. expected and he/she
 * may not notice until it's too late.
 *
 * Upon EOF the file will not be closed.
 */

NSAPI_PUBLIC httpd_objset *objset_scan_buffer(filebuffer *buf, char *errstr, 
                                              httpd_objset *os);

/*
 * objset_create creates a new object set and returns a pointer to it.
 */

NSAPI_PUBLIC httpd_objset *objset_create(void);

/*
 * objset_free will free an object set, any associated objects, and any
 * associated Init functions.
 */

NSAPI_PUBLIC void objset_free(httpd_objset *os);

/*
 * objset_free_setonly frees only the object set, and not the associated
 * objects or init functions.
 */
NSAPI_PUBLIC void objset_free_setonly(httpd_objset *os);

/*
 * objset_new_object will add a new object to objset with the specified
 * name. It returns a pointer to the new object (which may be anywhere in 
 * the objset).
 */

NSAPI_PUBLIC httpd_object *objset_new_object(pblock *name, httpd_objset *os);

/*
 * objset_add_object will add the existing object to os.
 */

NSAPI_PUBLIC void objset_add_object(httpd_object *obj, httpd_objset *os);


/*
 * objset_add_init will add the initialization function specified by 
 * initfn to the given object set. Modifies os->initfns.
 */
NSAPI_PUBLIC void objset_add_init(pblock *initfn, httpd_objset *os);

/*
 * objset_findbyname will find the object in objset having the given name,
 * and return the object if found, and NULL otherwise.
 * ign is a set of objects to ignore.
 */

NSAPI_PUBLIC httpd_object *objset_findbyname(char *name, httpd_objset *ign, 
                                             httpd_objset *os);

/*
 * objset_findbyppath will find the object in objset having the given 
 * partial path entry. Returns object if found, NULL otherwise.
 * ign is a set of objects to ignore.
 */

NSAPI_PUBLIC httpd_object *objset_findbyppath(char *ppath, httpd_objset *ign, 
                                              httpd_objset *os);

/*
 * objset_copydirs takes a given objset and copies all of the 
 * directive in directive set <dir> into a new objset, dst_os.  If 
 * dst_os is NULL, it creates a new objset.
 * The newly created objset only has one object in it.
 */
httpd_objset *objset_copydirs(httpd_objset *dst_os, httpd_objset *os, int dir);

#ifdef XP_WIN32

/* Read the registry configuration parameters into memory */

NSAPI_PUBLIC 
httpd_objset *objset_scan_registry(char *errstr, httpd_objset *os, 
                                   char *product_key);

/* Recursively scan the object key */

NSAPI_PUBLIC VOID 
ProcessObject(
	httpd_objset *nos, HKEY hObjectKey,
	CHAR *Object, DWORD ObjectIndex
	);

/* Recursively scan the client key */

NSAPI_PUBLIC BOOLEAN
ProcessClient(
	httpd_object *cobj, HKEY hObjectKey,
	CHAR *Client, DWORD ObjectIndex
	);

/* Recursively scan the directive key */

NSAPI_PUBLIC BOOLEAN
ProcessDirective (
	httpd_object *cobj, pblock *param, pblock *client, HKEY hObjectKey,
	CHAR *Directive, DWORD ObjectIndex, DWORD DirectiveIndex, DWORD *clrefs,
	BOOLEAN IsObject
	);

/* Recursively scan the function key */

NSAPI_PUBLIC BOOLEAN
ProcessFunction(
	pblock **param, HKEY hDirectiveKey, CHAR *Function,
	DWORD ObjectIndex, DWORD DirectiveIndex, DWORD FunctionIndex);

/* Verify the name value pair */

NSAPI_PUBLIC int _ntverify_parameter(char *str);

/* Enter name value pair into pblock */
NSAPI_PUBLIC
int ntpblock_str2pblock(int ParameterIndex, char *name, char *value, 
	int namelength, int valuelength, pblock *pb);

/* Get name value pair from a pblock entry */
NSAPI_PUBLIC
int ntpblock_pblock2str(struct pb_entry *p, PCHAR *ParameterName, 
	PCHAR *ParameterValue);

/* Given a key and its parent, delete the key recursively */
NSAPI_PUBLIC
BOOLEAN ClearConfigurationParameters(
	HKEY hParentKey, 
	PCHAR CompleteKeyName, 
	PCHAR KeyName);

/* Look at the hidden file in the server's directory structure to 
 * figure out the name of this server */
NSAPI_PUBLIC
char *GetProductKey();

#endif /* XP_WIN32 */


#endif
