/*
 * Copyright (c) 1994, 1995.  Netscape Communications Corporation.  All
 * rights reserved.
 * 
 * Use of this software is governed by the terms of the license agreement for
 * the Netscape FastTrack or Netscape Enterprise Server between the
 * parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * file.h: system specific functions for reading/writing files
 * 
 * Rob McCool
 */


#ifndef FILE_H
#define FILE_H

#include "netsite.h"
#include "systems.h"


/*
 * I cheat: These are set up such that system_read can be a macro for read
 * under UNIX. IO_OKAY is anything positive.
 */

#define IO_OKAY 1
#define IO_ERROR -1
#define IO_EOF 0


#if defined(FILE_UNIX)
#include <sys/types.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#endif


/* -------------------------- File related defs --------------------------- */


/* The disk page size on this machine. */
#define FILE_BUFFERSIZE 4096


/*
 * The fd data type for this system.
 */

#if defined(FILE_UNIX)
typedef int SYS_FILE;
#define SYS_ERROR_FD -1
#define SYS_STDERR STDERR_FILENO

#elif defined(FILE_WIN32)
#include "sem.h"

typedef struct {
    HANDLE fh;
	char *fname;
    SEMAPHORE flsem;
} file_s;
typedef file_s* SYS_FILE;

#define SYS_ERROR_FD NULL
#define SYS_STDERR NULL
typedef HANDLE pid_t;
#define ERROR_PIPE \
	(ERROR_BROKEN_PIPE | ERROR_BAD_PIPE |\
	 ERROR_PIPE_BUSY | ERROR_PIPE_LISTENING | ERROR_PIPE_NOT_CONNECTED)

#define CONVERT_TO_PRINTABLE_FORMAT(Filename) 	\
{									   		\
	register char *s;					   	\
	if (Filename)				   			\
	for (s = Filename; *s; s++) 	   		\
		if ( *s	== '\\')				   	\
			*s = '/';				   		\
}
									 
#define CONVERT_TO_NATIVE_FS(Filename) 	   \
{									   	   \
	register char *s;				   	   \
	if (Filename)						   \
		for (s = Filename; *s; s++) 	   \
			if ( *s	== '/')				   \
				*s = '\\';				   \
}									 

#define rtfile_notfound() ((errno == ENOENT) || (errno == ESRCH))

#else
#ifndef __EDG__
#error "undefined file typing for current system"
#endif
#endif

#ifdef XP_UNIX
#define FILE_PATHSEP '/'
#define FILE_PARENT "../"

#define system_chdir chdir
#elif defined XP_WIN32
#define FILE_PATHSEP '/'
#define system_chdir SetCurrentDirectory
#define FILE_PARENT "..\\"
#endif


/*
 * Wrapper for stat system call
 */
#include <sys/stat.h>
NSAPI_PUBLIC int system_stat(char *name, struct stat *finfo);

/*
 * Wrapper for rename system call
 */
#define system_rename(old, new) rename(old, new)

/*
 * Wrapper to remove file
 */
#ifdef FILE_UNIX
#define system_unlink(path) unlink(path)
#else
#define system_unlink(path) (DeleteFile(path) ? 0 : -1)
#endif

/*
 * system_fread reads sz bytes from fd into to buf, return number of bytes
 * read, or IO_EOF if EOF, or IO_ERROR if error. 
 */

#if defined(FILE_UNIX)
#define system_fread(fd,buf,sz) read(fd,buf,sz)

#elif defined (FILE_WIN32)
NSAPI_PUBLIC int system_fread(SYS_FILE fd, char *buf, int sz);
NSAPI_PUBLIC int system_pread(SYS_FILE fd, char *buf, int sz);
#endif

/*
 * system_fopenRO opens a given file for reading only
 * system_fopenWA opens a given file for writing, appending new output
 */

#if defined(FILE_UNIX)
#define system_fopenRO(path) open(path, O_RDONLY)
#define system_fopenWA(path) \
               open(path, O_RDWR | O_CREAT | O_APPEND, 0644)
#define system_fopenRW(path) \
               open(path, O_RDWR | O_CREAT, 0644)
#define system_fopenWT(path) \
               open(path, O_RDWR | O_CREAT | O_TRUNC, 0644)
#elif defined(FILE_WIN32)
NSAPI_PUBLIC SYS_FILE system_fopenRO(char *path);
NSAPI_PUBLIC SYS_FILE system_fopenWA(char *path);
NSAPI_PUBLIC SYS_FILE system_fopenRW(char *path);
NSAPI_PUBLIC SYS_FILE system_fopenWT(char *path);
#endif


/*
 * system_fclose closes the file fd
 */

#if defined(FILE_UNIX)
#define system_fclose(fd) close(fd)

#elif defined(FILE_WIN32)
NSAPI_PUBLIC void system_fclose(SYS_FILE fd);
#endif


/*
 * This call stops core dumps in a portable way. Returns -1 on error.
 */

NSAPI_PUBLIC int system_nocoredumps(void);

#ifdef XP_WIN32
NSAPI_PUBLIC char *system_winsockerr(void);
NSAPI_PUBLIC char *system_winerr(void);
#endif

#if defined(FILE_UNIX)
#define system_lseek lseek

#elif defined(FILE_WIN32)
#define system_lseek(fd, off, wh) \
  SetFilePointer(fd->fh, (long) off, NULL, wh)
#endif

/*
 * system_write writes sz bytes from buf to fd. The handler function should
 * handle partial writes and anything else like that. Returns IO_*
 */

NSAPI_PUBLIC int system_fwrite(SYS_FILE fd,char *buf,int sz);

/*
 * system_fwrite_atomic locks the given fd before writing to it. This avoids
 * interference between simultaneous writes. Returns IO_*
 */

NSAPI_PUBLIC int system_fwrite_atomic(SYS_FILE fd, char *buf, int sz);

/*
 * system_errmsg returns the last error that occured while processing file
 * descriptor fd. fd does not have to be specified (if the error is a global
 * such as in UNIX systems). PPS: Rob is a halfwit. This parameter is useless.
 */

#ifdef FILE_WIN32
NSAPI_PUBLIC int file_notfound(void);
#else
#include <errno.h>
#define file_notfound() (errno == ENOENT)
#define rtfile_notfound() (errno == ENOENT)
#endif

#define system_errmsg(unused) system_errmsg_fn()
NSAPI_PUBLIC char *system_errmsg_fn(void);


/*
 * flock locks a file against interference from other processes
 * ulock unlocks it.
 */
#ifdef BSD_FLOCK
#include <sys/file.h>
#define system_initlock(fd) (0)
#define system_flock(fd) flock(fd, LOCK_EX)
#define system_ulock(fd) flock(fd, LOCK_UN)
#define system_tlock(fd) flock(fd, LOCK_EX | LOCK_NB)

#elif defined(XP_UNIX)  /* !BSD_FLOCK */
#include <unistd.h>
#define system_initlock(fd) (0)
#define system_flock(fd) lockf(fd, F_LOCK, 0)
#define system_ulock(fd) lockf(fd, F_ULOCK, 0)
#define system_tlock(fd) lockf(fd, F_TLOCK, 0)

#elif defined(FILE_WIN32)
NSAPI_PUBLIC int system_initlock(SYS_FILE fd);
#define system_flock(fd) sem_grab(fd->flsem)
#define system_ulock(fd) sem_release(fd->flsem)
#endif

/*
 * setinherit sets whether the file descriptor fd can be inherited by child 
 * processes. A non-zero value means they can. Return of -1 means error.
 */
NSAPI_PUBLIC int file_setinherit(SYS_FILE fd, int value);


/*
 * unix2local converts a unix-style pathname to a local one
 */

#ifdef XP_UNIX
#define file_unix2local(path,p2) strcpy(p2,path)
#elif defined XP_WIN32
NSAPI_PUBLIC void file_unix2local(char *path, char *p2);
#endif /* XP_WIN32 */

/* -------------------------- Dir related defs ---------------------------- */


#ifdef XP_UNIX
#include <dirent.h>
typedef DIR* SYS_DIR;
typedef struct dirent SYS_DIRENT;
#define dir_open opendir
#define dir_read readdir
#define dir_close closedir
#define dir_create(path) mkdir(path, 0755)
#define dir_remove rmdir

#elif defined XP_WIN32

typedef struct {
    char *d_name;
} dirent_s;

typedef struct {
    HANDLE dp;
    WIN32_FIND_DATA fdata;
    dirent_s de;
} dir_s;

typedef dir_s* SYS_DIR;
typedef dirent_s SYS_DIRENT;

NSAPI_PUBLIC SYS_DIR dir_open(char *path);
NSAPI_PUBLIC SYS_DIRENT *dir_read(SYS_DIR ds);
NSAPI_PUBLIC void dir_close(SYS_DIR ds);
#define dir_create _mkdir
#define dir_remove _rmdir

#endif /* XP_WIN32 */

/*
 * create a directory and any of its parents
 */
NSAPI_PUBLIC int dir_create_all(char *dir);


/*
 * Thread-safe variants of localtime and gmtime
 */
#ifdef HAVE_TIME_R
#ifdef OSF1 /* grumble */
#define system_localtime(curtime, ret) (localtime_r(curtime, ret), ret)
#define system_gmtime(curtime, ret) (gmtime_r(curtime, ret), ret)
#else
#define system_localtime(curtime, ret) localtime_r(curtime, ret)
#define system_gmtime(curtime, ret) gmtime_r(curtime, ret)
#endif

#else

#ifdef THREAD_ANY

NSAPI_PUBLIC struct tm *
system_localtime_r(const time_t *timer, struct tm *result);
NSAPI_PUBLIC struct tm *
system_gmtime_r(const time_t *timer, struct tm *result);

#define system_localtime(curtime, ret) system_localtime_r(curtime, ret)
#define system_gmtime(curtime, ret) system_gmtime_r(curtime, ret)
#else
#define system_localtime(curtime, ret) localtime(curtime)
#define system_gmtime(curtime, ret) gmtime(curtime)
#endif

#endif

#endif
