/*
   OSMEM.C
   Platform dependent part of memory management.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* ------------------------------------------------------------------------ */

/* These routines are wrappers for the ANSI C routines, which are
   guaranteed to have the following semantics:
   (1) os_malloc(0) gives a non-0 pointer, same as os_malloc(1)
   (2) os_free(0) is a no-op
   (3) os_realloc(0, size) is same as os_malloc(size)
 */
extern void *os_malloc(unsigned long nbytes);
extern void os_free(void *p);
extern void *os_realloc(void *p, unsigned long nbytes);

extern long os_malloc_count;  /* simplest bug detector (see below) */
long os_malloc_count= 0;

/* ------------------------------------------------------------------------ */

#undef OS_MALLOC
#undef OS_FREE
#undef OS_REALLOC

/* Compile with OS_DEBUG to get a very primitive debugging memory manager.
   This checks for overruns and underruns.  It also initializes the entire
   array to -1, which is normally illegal if interpreted as a floating
   point value.  Set breakpoint at DebugError to catch the overrun or
   underrun errors.

   The count os_malloc_count is always available; it is the difference
   between calls to malloc (or realloc with p==0) and free.
 */
#ifdef OS_DEBUG
static void *DebugMalloc(unsigned long nbytes);
static void DebugFree(void *p);
static void *DebugRealloc(void *p, unsigned long new);

#define OS_MALLOC DebugMalloc
#define OS_FREE DebugFree
#define OS_REALLOC DebugRealloc

static int DebugMCheck(void *p);    /* call anytime to check over/underruns */
static void DebugError(char *text); /* can set debugger breakpoint here */
extern long os_mem_errors;
long os_mem_errors= 0;

#else
/* Don't bother to try to include <stdlib.h> -- it might not exist,
   and it requires the nasty size_t type.  */
extern void *malloc(unsigned long nbytes);
extern void free(void *p);
extern void *realloc(void *p, unsigned long nbytes);

#define OS_MALLOC malloc
#define OS_FREE free
#define OS_REALLOC realloc
#endif

/* ------------------------------------------------------------------------ */

void *os_malloc(unsigned long nbytes)
{
  void *p= OS_MALLOC(nbytes? nbytes : 1L);
  os_malloc_count++;
  return p;
}

void os_free(void *p)
{
  if (p) {
    OS_FREE(p);
    os_malloc_count--;
  }
}

void *os_realloc(void *p, unsigned long nbytes)
{
  if (!nbytes) nbytes= 1;
  if (p) {
    p= OS_REALLOC(p, nbytes);
  } else {
    p= OS_MALLOC(nbytes);
    if (p) os_malloc_count++;
  }
  return p;
}

/* ------------------------------------------------------------------------ */

#ifdef OS_DEBUG
static void *DebugMalloc(unsigned long nbytes)
{
  void *p;
  long *fill, *fmax, orig;
  orig= nbytes;
  nbytes= (((nbytes-1)>>3) + 4)<<3;
  p= malloc(nbytes);
  if (p) {
    fill= p;
    fmax= (long *)((char *)p + nbytes);
    *fill++= orig;
    while (fill<fmax) *(fill++)= -1;
    p= (char *)p + 16;
  }
  return p;
}

static void *DebugRealloc(void *p, unsigned long new)
{
  long *fill= (long *)((char *)p - 16);
  long nbytes= ((((*fill)-1)>>3) + 4)<<3;
  long orig;
  DebugMCheck(p);
  p= fill;
  orig= new;
  new= (((new-1)>>3) + 4)<<3;
  p= realloc(p, new);
  if (p) {
    long *fmax= (long *)((char *)p + new);
    fill= p;
    *fill= orig;
    if (new > nbytes) {
      fill= (long *)((char *)p + nbytes);
      while (fill<fmax) *(fill++)= -1;
    } else {
      char *past= (char *)p + 16 + orig;
      while (past<(char *)fmax) *(past++)= '\xff';
    }
    p= (char *)p + 16;
  }
  return p;
}

static void DebugFree(void *p)
{
  DebugMCheck(p);
  free((char *)p - 16);
}

static int DebugMCheck(void *p)
{
  if (p) {
    long nbytes, orig, *fill;
    char *past, *pmax;
    past= (char *)p - 16;
    fill= (long *)past;
    orig= *fill++;
    nbytes= (((orig-1)>>3) + 4)<<3;
    while (fill<(long *)p) {
      if (*fill++ != -1) {
	DebugError("underrun");
	return -1;
      }
    }
    p= past;
    pmax= past + nbytes;
    past+= 16+orig;
    while (past<pmax) {
      if (*past++ != '\xff') {
	DebugError("overrun");
	return 1;
      }
    }
  }
  return 0;
}

static void DebugError(char *text)
{
  os_mem_errors++;
}

#endif

/* ------------------------------------------------------------------------ */
