/*
 * write the master list to the ~/.schedule file.
 *
 *	write_database(list, path, what)	Write appointments and config
 *						info to a database file.
 */

#include <stdio.h>
#include <time.h>
#include <Xm/Xm.h>
#ifdef MIPS
#include <sys/types.h>
#include <sys/fcntl.h>
#else
#include <unistd.h>
#include <fcntl.h>
#endif
#include <sys/stat.h>
#include "cal.h"

extern struct tm	*time_to_tm();
extern char		*resolve_tilde();	/* evaluates path with ~ */
extern int		errno;
extern char		*progname;		/* argv[0] */
extern struct config	config;			/* global configuration data */
extern Widget		mainwindow;		/* for error popup */
extern struct user	*user;			/* user list for week view */
extern int		nusers;			/* # of users in user list */
extern char		Print_Spooler[100];	/* print spooling command */


/*
 * write a list to a file. The <what> bitmap determines what is written:
 * WR_CONFIG writes configuration info, WR_PUBLIC writes public appointments,
 * and WR_PRIVATE writes private appointments. Specifying WR_PRIVATE has the
 * side effect of making the file read-only. After calling this routine,
 * call resynchronize_daemon() to tell the daemon to re-read the databases.
 * Returns FALSE if the file could not be written.
 */

write_database(list, path, what)
	struct list		*list;		/* list to write */
	char			*path;		/* file to write list to */
	int			what;		/* bitmap of WR_* macros */
{
	register struct config	*c = &config;
	FILE			*fp;		/* open file */
	char			line[1024];	/* line buffer */
	register char		*p, *q;		/* string copy pointers */
	register struct entry	*ep;		/* list entry being written */
	int			i;		/* entry counter */
	int			perm;		/* file permissions */
	struct tm		*tm;		/* time to m/d/y h:m:s conv */

	list->locked = TRUE;
	path = resolve_tilde(path);

	/* Backup Database... */
	sprintf(line, "%s.bak", path);
	link(path, line); unlink(path);

	if (!(fp = fopen(path, "w"))) {
		create_error_popup(mainwindow, errno,
			"Failed to create appointment database %s", path);
		list->locked = FALSE;
		return(FALSE);
	}

	perm = what & WR_PRIVATE ? 0600 : 0644;
#if 0					/* SCO has no fchmod */
	if (fchmod(fileno(fp), perm))
#else
	if (chmod(path, perm))
#endif
		create_error_popup(mainwindow, errno,
		      "Cannot chmod 0%o appointment database %s.", perm, path);
	lockfile(fp, TRUE);
	if (what & WR_CONFIG) {
		fprintf(fp, "o\t%c%c%c%c%c%c%c%c%c%c%c-- %d %d %d %d %d\n",
					c->sunday_first ? 's' : '-',
					c->ampm         ? 'a' : '-',
					c->mmddyy       ? 'm' : '-',
					c->autodel      ? 'd' : '-',
					c->julian       ? 'j' : '-',
					c->weeknum      ? 'w' : '-',
					c->nopast       ? 'n' : '-',
					c->bigwarning   ? 'b' : '-',
					c->weekwarn     ? 'w' : '-',
					c->weekuser     ? 'u' : '-',
					c->smallmonth   ? 's' : '-',
					c->early_time,
					c->late_time,
					(int)c->wintimeout,
					c->week_minhour,
					c->week_maxhour);

		fprintf(fp, "t\t%d %d %d %d %d\n",
					(int)c->adjust_time,
					(int)c->raw_tzone,
					c->dst_flag,
					c->dst_begin,
					c->dst_end);

		fprintf(fp, "e\t%c%c%c %s\n",
					c->ewarn_window ? 'w' : '-',
					c->ewarn_mail   ? 'm' : '-',
					c->ewarn_exec   ? 'x' : '-',
					c->ewarn_prog   ? c->ewarn_prog
							    : "");
		fprintf(fp, "l\t%c%c%c %s\n",
					c->lwarn_window ? 'w' : '-',
					c->lwarn_mail   ? 'm' : '-',
					c->lwarn_exec   ? 'x' : '-',
					c->lwarn_prog   ? c->lwarn_prog
							    : "");
		fprintf(fp, "a\t%c%c%c %s\n",
					c->alarm_window ? 'w' : '-',
					c->alarm_mail   ? 'm' : '-',
					c->alarm_exec   ? 'x' : '-',
					c->alarm_prog   ? c->alarm_prog
							    : "");
		if (*Print_Spooler)
			fprintf(fp, "p\t%s\n", Print_Spooler);

		if (c->mailer)
			fprintf(fp, "m\t%s\n", c->mailer);

		fprintf(fp, "U\t%d\n", nusers);
		for (i=0; i < nusers; i++)
			fprintf(fp, "u\t%s %s %d %d\n",
					user[i].name,
					user[i].home,
					user[i].suspended,
					user[i].color);
	}
	for (ep=list->entry, i=0; i < list->nentries; i++, ep++) {
		if (ep->private && !(what & WR_PRIVATE) ||
		   !ep->private && !(what & WR_PUBLIC))
			continue;
		tm = time_to_tm(ep->time);
		fprintf(fp,
		  "%d/%d/%d  %d:%d:%d  %d:%d:%d  %d:%d:%d  %d:%d:%d  %c%c%c\n",
			(int)tm->tm_mon+1,
			(int)tm->tm_mday,
			(int)tm->tm_year + 1900,
			(int)(ep->notime ? 99 : tm->tm_hour),
			(int)(ep->notime ? 99 : tm->tm_min),
			(int)(ep->notime ? 99 : tm->tm_sec),
			(int)ep->length     / 3600,
			(int)ep->length     / 60   % 60,
			(int)ep->length            % 60,
			(int)ep->early_warn / 3600,
			(int)ep->early_warn / 60   % 60,
			(int)ep->early_warn        % 60,
			(int)ep->late_warn  / 3600,
			(int)ep->late_warn  / 60   % 60,
			(int)ep->late_warn         % 60,
			(int)ep->suspended ? 'S' : '-',
			(int)ep->private   ? 'P' : '-',
			(int)ep->noalarm   ? 'N' : '-');

		if (ep->rep_every || ep->rep_weekdays || ep->rep_days
						      || ep->rep_yearly)
			fprintf(fp, "R\t%d %d %d %d %d\n",
				(int)ep->rep_every,
				(int)ep->rep_last,
				(int)ep->rep_weekdays,
				(int)ep->rep_days,
				(int)ep->rep_yearly);

		if (ep->note)
			for (p=ep->note; *p; ) {
				for (q=line; *p && *p != '\n'; )
					*q++ = *p++;
				*q = 0;
				fprintf(fp, "N\t%s\n", line);
				if (*p) p++;
			}

		if (ep->message)
			for (p=ep->message; *p; ) {
				for (q=line; *p && *p != '\n'; )
					*q++ = *p++;
				*q = 0;
				fprintf(fp, "M\t%s\n", line);
				if (*p) p++;
			}

		if (ep->script)
			for (p=ep->script; *p; ) {
				for (q=line; *p && *p != '\n'; )
					*q++ = *p++;
				*q = 0;
				fprintf(fp, "S\t%s\n", line);
				if (*p) p++;
			}

		if (ep->meeting)
			for (p=ep->meeting; *p; ) {
				for (q=line; *p && *p != '\n'; )
					*q++ = *p++;
				*q = 0;
				fprintf(fp, "G\t%s\n", line);
				if (*p) p++;
			}
	}
	lockfile(fp, FALSE);
	fclose(fp);
	list->modified = FALSE;
	list->locked   = FALSE;
	return(TRUE);
}
