#!/usr/staff/bin/perl
#
# nhl-build.pl: Create the schedule[] matrix from input file(s).
#
# George Ferguson, ferguson@cs.rochester.edu, 4 Aug 1993.
#
# Assumes input files (stdin if none given) are of the form:
#	MON DAY HOME AWAY SITE
# with one line per game, no order necessary. The SITE field is optional.
# Outputs a list of strings to stdout representing the schedule[] array.
# Errors and conflicts are reported to stderr.
#
# You'll need to add empty entries for dates corresponding to the
# special_dates[] array.
#

# Check the "bynum" comparison function used to sort dates. You will need
# to change it depending on whether your season straddles the start of the
# year or not.

# The length of each month, Jan==1. *Adjust for seasons ending in leap years!*
# For 1993-94, Feb has 28 days.
@monthlen = (0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

# This list must be in the same order as the teams[] array
@teams = ("ANA",
          "BOS",
          "BUF",
          "CGY",
          "CHI",
          "DAL",
          "DET",
          "EDM",
	  "FLA",
          "HFD",
          "LOS",
          "MTL",
	  "NJD",
          "NYI",
          "NYR",
          "OTT",
          "PHL",
          "PIT",
          "QUE",
          "STL",
          "SJS",
          "TMP",
          "TOR",
          "VAN",
          "WSH",
          "WPG",
);

# This list must be in the same order as the extra_info[] array
@extras = ("Cleveland",
	    "Halifax",
	    "Hamilton",
	    "Minneapolis",
	    "Orlando",
	    "Phoenix",
	    "Sacramento",
	    "Saskatoon",
);

# Characters used in the schedule[] array
$teamcodes  = 'abcdefghijklmnopqrstuvwxyz';
$extracodes = 'ABCDEFGH';
$idlecode   = '-';
$athomecode = '+';

##########################################
# That should be it to change.

# Create assoc arrays with indices as values:
for ($i=0; $i <= $#teams; $i++) {
    $teamindex{$teams[$i]} = $i;
    $teamcode{$teams[$i]} = substr($teamcodes,$i,1);
}
for ($i=0; $i <= $#extras; $i++) {
    $extracode{$extras[$i]} = substr($extracodes,$i,1);
}

while (<>) {
    #print STDERR;
    # Remove trailing newline
    chop($_);
    # Removeleading whitespace
    s/^ +//;
    # Split into fields
    ($month,$day,$home,$away,$extra) = split(' ');
    # Convert HOME to index, complain if bogus
    if (($homeindex = $teamindex{$home}) eq "") {
	printf STDERR ("bad home team: %s(%d): %s\n",$ARGV,$.,$home);
	exit(1);
    }
    $homecode = $teamcode{$home};
    # Convert AWAY to index, complain if bogus
    if (($awayindex = $teamindex{$away}) eq "") {
	printf STDERR ("bad away team: %s(%d): %s\n",$ARGV,$.,$away);
	exit(1);
    }
    $awaycode = $teamcode{$away};
    # Convert EXTRA to code if present, complain if bogus
    if ($extra) {
	if (($extracode = $extracode{$extra}) eq "") {
	    printf STDERR ("bad extra info: %s(%d): %s\n",$ARGV,$.,$extra);
	    exit(1);
	}
    }
    # Entries are stored in an array keyed by mon/day
    $key = "$month/$day";
    if (!$schedule{$key}) {
	$schedule{$key} = $idlecode x ($#teams+1);
    }
    # Mark away team entry
    $code = substr($schedule{$key},$awayindex,1);
    if ($code ne $idlecode && $code ne $homecode) {
	&error("away conflict",$awayindex,$code);
    } else {
	substr($schedule{$key},$awayindex,1) = $homecode;
    }
    # Mark home team entry
    $code = substr($schedule{$key},$homeindex,1);
    if ($extra) {		# Extra info game
	if ($code ne $idlecode && $code ne $extracode) {
	    &error("extra conflict",$homeindex,$code);
	} else {
	    substr($schedule{$key},$homeindex,1) = $extracode;
	}
    } else {			# Regular home game
	if ($code ne $idlecode && $code ne $athomecode) {
	    &error("home conflict",$homeindex,$code);
	} else {
	    substr($schedule{$key},$homeindex,1) = $athomecode;
	}
    }
    # Reset line numbers at eof of each file
    if (eof) {
	close(ARGV);
    }
}
		       
# Sort keys to extract first and last dates
@keys = keys %schedule;
@keys = sort bynum @keys;
($startmonth,$startday) = split('/',$keys[0]);
($endmonth,$endday) = split('/',$keys[$#keys]);
#print STDERR "start: $startmonth/$startday\n";
#print STDERR "end: $endmonth/$endday\n";

# Walk through the schedule, watch out for empty days
for ($month=$startmonth;
     $month != $endmonth+1; $month=(($month==12)?1:($month+1))){
    for ($day=($month==$startmonth)?$startday:1;
	 $day <= (($month==$endmonth)?$endday:$monthlen[$month]); $day++) {
	$key = "$month/$day";
	#print STDERR "$key\n";
	if ($schedule{$key}) {
	    printf("    \"%s\", /* %2d/%-2d */\n",$schedule{$key},$month,$day);
	} else {
	    printf("    \"%s\", /* %2d/%-2d */\n","-"x($#teams+1),$month,$day);
	}
    }
}
exit(0);

#
# Subroutines
#

# Return -1, 0, or 1 after comparing $a and $b as MM/DD dates (used by sort)
sub bynum {
    local($amon,$aday) = split('/',$a);
    local($bmon,$bday) = split('/',$b);
# Uncomment and adjust these if your season straddles the start of the year.
# Or comment them out if it doesn't.
    $amon += 20 if ($amon < 10);
    $bmon += 20 if ($bmon < 10);
    ($amon != $bmon) ? $amon <=> $bmon : $aday <=> $bday;
}

# Print an error message
sub error {
    local($str,$index,$code) = @_;
    local($team) = $teams[$index];
    local($label);
    if ($code =~ /[a-z]/) {
	$label = sprintf("at %s",$teams[ord($code)-ord(substr($teamcodes,0,1))]);
    } elsif ($code =~ /[A-Z]/) {
	$label = sprintf("-> %s",$extra[ord($code)-ord(substr($extracodes,0,1))]);
    } elsif ($code eq $athomecode) {
	$label = "home";
    } else {
	$label = "???";
    }
    printf STDERR ("%s: %s(%2d): %2d/%-2d %s %s: entry %2d (%s) = `%s' (%s)\n",
		   $str,$ARGV,$.,$month,$day,$home,$away,$index,$team,$code,$label);
}
