#include <qcursor.h>
#include <qpainter.h>

#include <kapplication.h>
#include <kdebug.h>
#include <krootpixmap.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kpixmapio.h>

#include "kasbar.h"

#include "kasitem.h"
#include "kasbar.moc"

const int SMALL_EXTENT = 36;
const int MEDIUM_EXTENT = 52;
const int LARGE_EXTENT = 68;

KasBar::KasBar( Orientation o, QWidget *parent, const char *name, WFlags f )
   : QWidget( parent, name, f ),
     items(),
     orient( o ),
     itemUnderMouse_( 0 ),
     maxBoxes_( 100 ), // Temp value
     itemSize_( Medium ),
     itemExtent_( MEDIUM_EXTENT ),
     oldR( 0 ),
     oldC( 0 ),
     oldO( Horizontal ),
     transparent_( false ),
     enableTint_( false ),
     tintAmount_( 0.1 ),
     tintColour_( colorGroup().mid() ),
     actBg( 0 ), inactBg( 0 ),
     activePenColor_( Qt::black ), inactivePenColor_( Qt::black )
{
    setBackgroundMode( NoBackground );
    items.setAutoDelete( true );
    setMouseTracking( true );
    setMaxBoxes( 0 );
}

KasBar::~KasBar()
{
   delete actBg;
   delete inactBg;
}

void KasBar::setItemSize( int size )
{
   if ( size == itemSize_ )
      return;
   itemSize_ = size;

   switch( size ) {
   case Small:
     itemExtent_ = SMALL_EXTENT;
     break;
   case Medium:
     itemExtent_ = MEDIUM_EXTENT;
     break;
   case Large:
     itemExtent_ = LARGE_EXTENT;
     break;
   default:
     itemExtent_ = MEDIUM_EXTENT;
   }

   delete actBg;
   delete inactBg;
   actBg = 0;
   inactBg = 0;

   emit itemSizeChanged( size );
   emit configChanged();

   updateLayout();
}

void KasBar::setTransparent( bool enable )
{
   if ( transparent_ == enable )
      return;

   transparent_ = enable;

   if ( transparent_ ) {
       kdDebug(1345) << "KasBar: Enabling transparency" << endl;

      rootPix = new KRootPixmap( this );
      connect( rootPix, SIGNAL( backgroundUpdated(const QPixmap &) ),
	       this, SLOT( setBackground(const QPixmap &) ) );

      rootPix->setCustomPainting( true );

      if ( enableTint_ )
	 rootPix->setFadeEffect( tintAmount_, tintColour_ );

      rootPix->start();
   }
   else {
       kdDebug(1345) << "KasBar: Disabling transparency" << endl;

       rootPix->stop();
       delete rootPix;
       rootPix = 0;
   }

   emit configChanged();
   repaint();
}

void KasBar::setTint( bool enable )
{
   if ( enableTint_ == enable )
      return;

   enableTint_ = enable;

   if ( transparent_ && rootPix ) {
      if ( enableTint_ ) {
	 rootPix->setFadeEffect( tintAmount_, tintColour_ );
      }
      else {
	 rootPix->setFadeEffect( 0.0, tintColour_ );
      }

      emit configChanged();
      repaint( true );
   }
}

void KasBar::setTint( double amount, QColor color )
{
   tintAmount_ = amount;
   tintColour_ = color;

   if ( transparent_ && enableTint_ ) {
      rootPix->setFadeEffect( tintAmount_, tintColour_ );
      emit configChanged();

      if ( rootPix->isAvailable() )
	rootPix->repaint( true );
   }
}

void KasBar::setTintColor( const QColor &c )
{
   setTint( tintAmount_, c );
}

void KasBar::setTintAmount( int percent )
{
   double amt = (double) percent / 100.0;
   setTint( amt, tintColour_ );
}

void KasBar::setMaxBoxes( int count )
{
  if ( count != maxBoxes_ ) {
    if ( count == 0 )
      count = 15; // XXX Hacked

    maxBoxes_ = count;
    emit configChanged();
    updateLayout();
  }
}

QSize KasBar::sizeHint( Orientation o,  QSize )
{
   QSize s;

   unsigned int r=0, c=0;
   if( items.count() > (unsigned int) maxBoxes_ ) {
      r = items.count()/maxBoxes_;
      c = maxBoxes_;
   }
   else {
      r = 1;
      c = items.count();
   }

   if( r*c < items.count() ) // remainders
      ++r;

   if( o == Horizontal ) {
      s.setWidth( c*itemExtent() );
      s.setHeight( r*itemExtent() );
   }
   else {
      s.setWidth( r*itemExtent() );
      s.setHeight( c*itemExtent() );
   }

   return s;
}

void KasBar::updateLayout()
{
//    kdDebug(1345) << "KasBar: updateLayout(), count is " << items.count() << endl;

   // Work out the number of rows and columns
   unsigned int r=0, c=0;
   if( items.count() > (unsigned int) maxBoxes_ ) {
      r = items.count()/maxBoxes_;
      c = maxBoxes_;
   }
   else{
      r = 1;
      c = items.count();
   }

   if( r*c < items.count() ) // remainders
      ++r;

   if ( oldR != r || oldC != c || oldO != orient ) {
      if ( orient == Horizontal )
	 resize( c * itemExtent(), r * itemExtent() );
      else
	 resize( r * itemExtent(), c * itemExtent() );
      oldR = r;
      oldC = c;
      oldO = orient;

      update();
   }

}

void KasBar::append( KasItem *i )
{
   if ( !i )
      return;

   items.append( i );
   updateLayout();
}

void KasBar::insert( int index, KasItem *i )
{
   if ( (!i) || (index < 0) )
      return;

   items.insert( index, i );
   updateLayout();
}

void KasBar::remove( KasItem *i )
{
   items.remove( i );

   if ( i == itemUnderMouse_ )
      itemUnderMouse_ = 0;
   updateLayout();
}

void KasBar::clear()
{
   items.clear();
   itemUnderMouse_ = 0;
   updateLayout();
}

void KasBar::mousePressEvent(QMouseEvent *ev)
{
   KasItem *i = itemAt( ev->pos() );
   if ( i )
      i->mousePressEvent( ev );
}

void KasBar::updateMouseOver()
{
    updateMouseOver( mapFromGlobal( QCursor::pos() ) );
}

void KasBar::updateMouseOver( QPoint pos )
{
   KasItem *i = itemAt(pos);
   if ( itemUnderMouse_ != i ) {
      if ( itemUnderMouse_ )
	 itemUnderMouse_->mouseLeave();
      if ( i )
	 i->mouseEnter();
      itemUnderMouse_ = i;
   }
}

void KasBar::mouseMoveEvent(QMouseEvent *ev)
{
    updateMouseOver( ev->pos() );
}

void KasBar::dragMoveEvent ( QDragMoveEvent *ev )
{
   KasItem *i = itemAt( ev->pos() );
   if ( itemUnderMouse_ != i ) {
      if ( itemUnderMouse_ )
	 itemUnderMouse_->dragLeave();
      if ( i )
	 i->dragEnter();
      itemUnderMouse_ = i;
   }
}

void KasBar::paintEvent(QPaintEvent *ev)
{
//    kdDebug(1345) << "KasBar: paintEvent()" << endl;

    QPainter p(this);

    // If we're empty
    if ( !items.count() ) {
	erase( ev->rect() );
	paintBackground( &p, ev->rect() );
    }

   QRect cr;
   KasItem *i;
   int r=0, c=0;

   if ( orientation() == Horizontal ) {
       int totalcols = width()/itemExtent();

       for ( i = items.first(); i; i = items.next() ) {
	   if ( c >= totalcols ) {
	       c = 0; ++r;
	   }
	   cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	   if ( ev->rect().intersects(cr) )
	       i->paint( &p, c*itemExtent(), r*itemExtent() );
	   ++c;
       }
   }
   else {
       int totalrows = height()/itemExtent();

       for ( i = items.first(); i; i = items.next() ) {
	   if ( r >= totalrows ) {
	       r = 0; ++c;
	   }
	   cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	   if ( ev->rect().intersects(cr) )
	       i->paint( &p, c*itemExtent(), r*itemExtent() );
	   ++r;
       }
   }
}

void KasBar::resizeEvent(QResizeEvent *ev)
{
//    kdDebug(1345) << "KasBar: resizeEvent()" << endl;

    QWidget::resizeEvent(ev);
    emit layoutChanged();
}

void KasBar::repaintItem(KasItem *i, bool erase )
{
//    kdDebug(1345) << "KasBar: repaintItem()" << endl;
   QPoint p = itemPos( i );
   repaint( QRect( p, QSize( itemExtent(), itemExtent()   ) ), transparent_ || erase );
}

KasItem* KasBar::itemAt(const QPoint &p)
{
   QRect cr;
   KasItem *i;
   int r=0, c=0;
   int totalcols = width()/itemExtent();
   int totalrows = height()/itemExtent();

   if(orient == Horizontal){
      for (i = items.first(); i; i = items.next()){
	 if(c >= totalcols){
	    c = 0;
	    ++r;
	 }
	 cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	 if(cr.contains(p))
	    return(i);
	 ++c;
      }
   }
   else {
      for (i = items.first(); i; i = items.next()) {
	 if(r >= totalrows){
	    r = 0; ++c;
	 }
	 cr.setRect(c*itemExtent(), r*itemExtent(), itemExtent(), itemExtent());
	 if(cr.contains(p))
	    return(i);
	 ++r;
      }
   }

   return 0;
}

QPoint KasBar::itemPos( KasItem *i )
{
   int x;
   int y;
   int totalcols = width()/itemExtent();
   int totalrows = height()/itemExtent();
   int index = items.find( i );
   if ( index == -1 ) {
      x = y = -1;
      return QPoint( x, y );
   }

   int r = 0;
   int c = 0;

   if ( ( orient == Horizontal ) && totalcols ) {
      r = index / totalcols;
      c = index % totalcols;
   }
   else if ( ( orient == Vertical ) && totalrows ) {
      c = index / totalrows;
      r = index % totalrows;
   }

   x = c * itemExtent();
   y = r * itemExtent();
   return QPoint( x+1, y+1 );  // +1,+1 prevents #50654
}

KPixmap *KasBar::activeBg()
{
   if ( !actBg ) {
      actBg = new KPixmap;

      actBg->resize( itemExtent()-4, itemExtent()-13-4 );
      KPixmapEffect::gradient( *actBg,
			       colorGroup().light(), colorGroup().mid(),
			       KPixmapEffect::DiagonalGradient );
   }

   return actBg;
}

KPixmap *KasBar::inactiveBg()
{
   if ( !inactBg ) {
      inactBg = new KPixmap;

      inactBg->resize( itemExtent()-4, itemExtent()-13-4 );
      KPixmapEffect::gradient( *inactBg,
			       colorGroup().mid(), colorGroup().dark(),
			       KPixmapEffect::DiagonalGradient );
   }

   return inactBg;
}

void KasBar::setBackground( const QPixmap &newBg )
{
    bg = newBg;
    update();
}

void KasBar::paintBackground( QPainter *p, const QRect &r )
{
    // If we're transparent and know the bg
    if ( transparent_ && (!bg.isNull()) ) {
	p->drawPixmap( QPoint(0, 0), bg, r );
	return;
    }

    // We're opaque
    p->fillRect( rect(), colorGroup().brush(QColorGroup::Mid) );
}

