/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * log.c - assorted utilities for maintaining the logs
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#if HAVE_DIRENT_H
#	include <dirent.h>
#	define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#	define dirent direct
#	define NAMLEN(dirent) (dirent)->d_namlen
#	if HAVE_SYS_NDIR_H
#		include <sys/ndir.h>
#	endif
#	if HAVE_SYS_DIR_H
#		include <sys/dir.h>
#	endif
#	if HAVE_NDIR_H
#		include <ndir.h>
#	endif
#endif

#include "callbacks_clist.h"
#include "utils.h"
#include "types.h"
#include "log.h"
#include "support.h"

extern GtkWidget *mainnotebook;
extern statetype state;
extern preferencestype preferences;
extern gchar **qso;

/* extracted name of the log from filename
   returned string should be free'd */
gchar *
logname (gchar * filename)
{
  gchar *logname;
  gchar **split = g_strsplit (filename, ".", 1);
  logname = g_strdup (split[0]);
  g_strfreev (split);
  return (logname);
}

/* prepend qsos to the log */
gint
fillin_clist (LOGDB * handle, qso_t q[], gpointer arg)
{
  logtype *logw = (logtype *) arg;
  gint newrow;

  state.qsos++;
  logw->qsos++;
  q[NR] = g_strdup_printf ("%d", logw->qsos);
  newrow = gtk_clist_prepend (GTK_CLIST (logw->clist), q);
  if (preferences.logcolor == 1)
    setcallstyle (logw->clist, newrow, preferences.themecolor,
		  preferences.themefont);
  return 0;
}

/* create a new struct for a log */
static logtype *
new_logwindow (void)
{
  logtype *newlog;
  gint i;

  newlog = g_new0 (struct logtype, 1);
  newlog->scrolledwindow = NULL;
  newlog->clist = NULL;
  newlog->label = NULL;
  newlog->logname = NULL;
  newlog->logchanged = FALSE;
  newlog->qsos = 0;
  newlog->columns = 0;
  for (i = 0; i < QSO_FIELDS; i++)
    newlog->logfields[i] = -1;
  return (newlog);
}

/* open a log and return a struct */
logtype *
openlog (LOGDB * lp, gchar * name, gint page)
{
  logtype *logwindow;
  gint i, j;
  GtkWidget *columnlabel;
  gchar *logn;

  logwindow = new_logwindow ();
  logwindow->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (logwindow->scrolledwindow);
  gtk_container_add (GTK_CONTAINER (mainnotebook), logwindow->scrolledwindow);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW
				  (logwindow->scrolledwindow),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  logwindow->clist = gtk_clist_new (QSO_FIELDS);
  gtk_widget_show (logwindow->clist);
  gtk_signal_connect (GTK_OBJECT (logwindow->clist), "select_row",
		      GTK_SIGNAL_FUNC (on_log_select_row), NULL);
  gtk_container_add (GTK_CONTAINER (logwindow->scrolledwindow),
		     logwindow->clist);

  logwindow->columns = lp->column_nr;
  for (j = 0; j < QSO_FIELDS; j++)
    {
      gtk_clist_set_column_auto_resize (GTK_CLIST (logwindow->clist), j,
					TRUE);
      logwindow->logfields[j] = lp->column_fields[j];
    }
  gtk_clist_column_titles_show (GTK_CLIST (logwindow->clist));

  columnlabel = gtk_label_new ("NR");
  gtk_clist_set_column_widget (GTK_CLIST (logwindow->clist), NR, columnlabel);
  gtk_clist_column_title_passive (GTK_CLIST (logwindow->clist), NR);

  /* see which fields are in the log and add a column label */
  for (j = 1; j < QSO_FIELDS; j++)
    {
      for (i = 0; i < lp->column_nr; i++)
	{
	  if (j == lp->column_fields[i])
	    break;
	}
      if (j == U1)
	columnlabel = gtk_label_new (preferences.freefield1);
      else if (j == U2)
	columnlabel = gtk_label_new (preferences.freefield2);
      else
	columnlabel = gtk_label_new (strfield (j));
      gtk_clist_set_column_widget (GTK_CLIST (logwindow->clist), j,
				   columnlabel);
      gtk_clist_column_title_passive (GTK_CLIST (logwindow->clist), j);
      if (i == lp->column_nr)	/* field not found, hide this column */
	gtk_clist_set_column_visibility (GTK_CLIST (logwindow->clist), j,
					 FALSE);
    }

  logn = logname (name);
  logwindow->label = gtk_label_new (logn);
  gtk_widget_show (logwindow->label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (mainnotebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (mainnotebook),
							 page),
			      logwindow->label);
  gtk_misc_set_padding (GTK_MISC (logwindow->label), 10, 0);
  setlabelstyle (logwindow->label, preferences.themecolor,
		 preferences.themefont);
  logwindow->logname = g_strdup (logn);

  g_free (logn);
  return (logwindow);
}

/* saving of the log */
void
savelog (gpointer arg, gchar * logfile, gint type)
{
  LOGDB *lp;
  gint i, j;
  GtkWidget *widget;
  gchar *label, **item, *tmpu1, *tmpu2;
  gint fields[QSO_FIELDS], widths[QSO_FIELDS];
  logtype *logw = (logtype *) arg;
  FILE *ifp, *ofp;
  gchar buf[1024];


  gtk_clist_freeze (GTK_CLIST (logw->clist));
  item = g_new0 (gchar *, QSO_FIELDS);

  /* how many columns do we have and what are the labels, skip first field */
  /* ----> unknown fields have a fixed label */
  for (i = 0; i < logw->columns; i++)
    {
      widget =
	gtk_clist_get_column_widget (GTK_CLIST (logw->clist),
				     logw->logfields[i]);
      gtk_label_get (GTK_LABEL (widget), &label);
      if (logw->logfields[i] == U1)
	fields[i] = U1;
      else if (logw->logfields[i] == U2)
	fields[i] = U2;
      else
	fields[i] = parse_field_name (label);
      widths[i] = parse_field_width (fields[i]);
    }

  if (type == TYPE_ADIF)
    lp =
      log_file_create ("/tmp/xlog.adi", type, logw->columns, fields, widths);
  else
    lp = log_file_create (logfile, type, logw->columns, fields, widths);

  if (lp)
    {
      for (i = GTK_CLIST (logw->clist)->rows - 1; i >= 0; i--)
	{
	  for (j = 0; j < logw->columns; j++)
	    gtk_clist_get_text (GTK_CLIST (logw->clist), i,
				logw->logfields[j], &item[fields[j]]);
	  log_file_qso_append (lp, item);
	}
      log_file_close (lp);
      if (type == TYPE_ADIF)	/* search and replace U1 and U2 when using ADIF */
	{
	  ifp = fopen ("/tmp/xlog.adi", "r");
	  ofp = fopen (logfile, "w");
	  if (ifp && ofp)
	    {
	      while (fgets (buf, 1024, ifp))
		{
		  tmpu1 = strreplace (buf, "U1", preferences.freefield1);
		  tmpu2 = strreplace (tmpu1, "U2", preferences.freefield2);
		  fputs (tmpu2, ofp);
		}
	      fclose (ifp);
	      fclose (ofp);
	    }
	}
    }
  else
    {
      update_statusbar (_("Creation of logfile failed"));
    }
  gtk_clist_thaw (GTK_CLIST (logw->clist));
  g_free (item);
}


/* look for logs in dir */
GString *
getlogs (gchar * path, gint filetype)
{
  DIR *directory;
  struct dirent *dirEntry;
  gchar *filen;
  GString *logs = g_string_new ("");

  directory = opendir (path);
  while (NULL != (dirEntry = readdir (directory)))
    {
      filen = g_strdup (dirEntry->d_name);
      if (dirEntry->d_type == DT_REG)	/* only regular files */
	{
	  if (filetype == 0)	/* type xlog */
	    {
	      g_strreverse (filen);
	      if (g_strncasecmp (filen, "golx.", 5) == 0)
		{
		  g_string_append (logs, dirEntry->d_name);
		  g_string_append_c (logs, '\n');
		}
	    }
	  else			/* other types */
	    {
	      if ((g_strcasecmp (filen, "..") != 0)
		  && (g_strcasecmp (filen, ".") != 0))
		{
		  g_string_append (logs, dirEntry->d_name);
		  g_string_append_c (logs, '\n');
		}
	    }
	}
    }
  return (logs);
}
