#ifndef COMMON_AI_H
#define COMMON_AI_H

#include <qtimer.h>
#include <qptrvector.h>

#include <kdialogbase.h>
#include <knuminput.h>
#include <krandomsequence.h>
#include "gsettings.h"


class Board;
class Piece;

//-----------------------------------------------------------------------------
class AIElement
{
 public:
    AIElement(const char *key, const char *description,
              double cdef, double cmin, double cmax,
              int tdef, int tmin, int tmax,
              int (*function)(const Board &, const Board &));

    QString triggerConfigKey() const;
    QString coeffConfigKey() const;
    bool triggered() const;
    double points(const Board &main, const Board &current) const;

    const char   *_description;
    const int     _tdef, _tmin, _tmax;
    int           _trigger;
    const double  _cdef, _cmin, _cmax;
    double        _coeff;

 private:
    const char *_key;
    int (*_function)(const Board &, const Board &);
};

//-----------------------------------------------------------------------------
class AIPiece
{
 public:
	AIPiece();
    ~AIPiece();

    void init(const Piece *p, Board *b);
	bool place();
	bool increment();

    int dec() const  { return curDec; }
    uint rot() const { return curRot; }

 private:
	uint    nbPos, nbRot, curPos, curRot;
	int     curDec;
	const Piece *_piece;
    Piece       *_p;
    Board       *_board;

    void reset();
};

//-----------------------------------------------------------------------------
class AI : public QObject
{
 Q_OBJECT
 public:
    AI(uint thinkTime, uint orderTime);
	virtual ~AI();

    KSettingWidget *createSettingWidget();

	void launch(Board *main);
	void stop();
	void start();

 private slots:
	void timeout();
    void settingsChanged();

 protected:
    void addElement(AIElement *);

	virtual void initThink();

	static int nbOccupiedLines(const Board &, const Board &);
	static int nbHoles(const Board &, const Board &);
	static int nbSpaces(const Board &, const Board &);
	static int peakToPeak(const Board &, const Board &);
	static int mean(const Board &, const Board &);
	static int nbRemoved(const Board &, const Board &);

 private:
	bool think();
	void startTimer();
	bool emitOrder();
	double points() const;
    void resizePieces(uint size);

	QTimer                   timer;
	enum ThinkState { Thinking, GivingOrders };
	ThinkState               state;
	uint                     thinkTime, orderTime;
	bool                     stopped;
	QMemArray<AIPiece *>     pieces;
    QPtrVector<AIElement>    elements;
	Board                   *main, *board;
    KRandomSequence          random;

    bool   hasBestPoints;
    double bestPoints;
    int    bestDec;
    uint   bestRot;
};

//-----------------------------------------------------------------------------
class AISettingWidget : public KSettingWidget
{
 Q_OBJECT
 public:
    AISettingWidget(const QPtrVector<AIElement> &);

    static uint readThinkingDepth();
    static void readElement(AIElement *);

 private:
    void addElement(const AIElement *element, QGrid *grid);
};

#endif
