/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 * Special thanks go to Bruce Momjian (candle!root@ls.com) for his
 * suggestions and code.
 *
 */

#include <mbase.h>
#include <parse.h>

/*
 * DEFINITIONS ----------------------------------------------------------------
 *
 */

#ifndef TABSTOP
#define TABSTOP 5
#endif

#define MAX_WIDTH  80
#define FULL_WIDTH "%-79.79s"

#define NONE   0
#define BEFORE 1
#define DURING 2
#define AFTER  4

#define CREDIT "MetalBase 5.1 Report Writer"


/*
 * PARSING TABLES -------------------------------------------------------------
 *
 */

static char *aTokens[] =
   {
   "print",    "skip",     "top",       "bottom",    "left",    "right",
   "page",     "newpage",  "header",    "last",      "on",      "footer",
   "size",     "keep",     ";",         "columns",   "rows",    "continued",
   "line",     "lines",    "centered",  "to",        ":",       "column",
   ",",        "time",     "date",      "using",     "format",  "<=",
   "=<",       "<",        ">=",        "=>",        ">",       "==",
   "=",        "reverse",  "data",      "margin",    "+",       "end",
   "queue",    ""
   };

#define tokFIRST tokPRINT

typedef enum
   {
   tokPRINT,   tokSKIP,    tokTOP,      tokBOTTOM,   tokLEFT,   tokRIGHT,
   tokPAGE,    tokNEWPAGE, tokHEADER,   tokLAST,     tokON,     tokFOOTER,
   tokSIZE,    tokKEEP,    tokSEMI,     tokCOLUMNS,  tokROWS,   tokCONTINUED,
   tokLINE,    tokLINES,   tokCENTERED, tokTO,       tokCOLON,  tokCOLUMN,
   tokCOMMA,   tokTIME,    tokDATE,     tokUSING,    tokFORMAT, tokLTEQ,
   tokEQLT,    tokLT,      tokGTEQ,     tokEQGT,     tokGT,     tokEQUAL,
   tokEQ,      tokREVERSE, tokDATA,     tokMARGIN,   tokPLUS,   tokEND,
   tokQUEUE,   tokUNKNOWN
   } token;


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

   FParse *parseArgs  XARGS( (int, char **) );
   char   *pchNext    XARGS( (FParse *) );
   bool    tokSkip    XARGS( (FParse *, token) );
   int     numLines   XARGS( (FParse *, long, int) );
   void    endOfLine  XARGS( (FParse *) );
   token   tokCheck   XARGS( (FParse *, char *) );
   void    syntaxErr  XARGS( (char *) );
   void    makeSpaces XARGS( (char *, int) );
   void    fillData   XARGS( (dataptr, int, char *) );

   bool    sectionSize    XARGS( (FParse *) );
   bool    sectionOn      XARGS( (FParse *) );
   void    sectionPrint   XARGS( (FParse *, long, int, bool) );
   void    commandPrint   XARGS( (FParse *, bool) );

   void  skipl        XARGS( (int) );


/*
 * VARIABLES ------------------------------------------------------------------
 *
 */

relation *rel;             /* Relation pointer (from mb_inc())           */
char     *key = NULL;      /* Encryption key (NULL if none)              */
dataptr   rec, buf;        /* Record buffers (dynamic allocation)        */

int   lastLine = 0;        /* Last line number                           */

long  posLast   = 0L;      /* Position in template for Last section      */
long  posHeader = 0L;      /* Position in template for Header section    */
long  posFooter = 0L;      /* Position in template for Footer section    */
long  posOther  = 0L;      /* Temporary position recorder                */

int   lineLast   = 0;
int   lineHeader = 0;
int   lineFooter = 0;
int   lineOther  = 0;

bool  fKeepLast   = FALSE; /* Keep Last section for new search?          */
bool  fKeepHeader = FALSE; /* Keep Header section for new search?        */
bool  fKeepFooter = FALSE; /* Keep Footer section for new search?        */

int   num_col = 80;        /* 80-column (standard pagesize) paper        */
int   num_row = 66;        /* 66-line (standard pagesize) paper          */

int   marginTop    =  4;   /* 2/3" top margin by default                 */
int   marginBottom =  6;   /* 1" bottom margin by default                */
int   marginLeft   = 10;   /* 1" left margin by default                  */
int   marginRight  = 10;   /* 1" right margin by default                 */

int   cbWidth,  width;     /* Width of template inside margins           */
int   cbHeight, height;    /* Height of template inside margins          */
int   pageno  =  1;        /* Start with page 1, obviously               */
int   left;


/*
 * MAIN CODE ------------------------------------------------------------------
 *
 */

void
main  (argc, argv)
int    argc;
char **argv;
{
   FParse    *fp;
   char      *pWord, temp[128];
   token      tok;
   bool       fSkip;
   bool       fInData;
   int        rc = 1;


/*
 * This function will exit if it cannot open the specified report template,
 * or return FALSE if there is a syntax error on the command line.
 *
 */

   if (! (fp = parseArgs (argc, argv)))
      {
      fprintf (stderr, "format: report [-k encryption_key] template\n");
      mb_exit (1);
      }


/*
 * Now that it has been opened, try to determine which relation(s) it
 * specifies.  If we can open it, malloc a pair of record-length buffers,
 * so we'll be able to perform queries.
 *
 */

   if ((tok = tokCheck (fp, pchNext (fp))) != tokDATA)
      {
      syntaxErr ("data");
      }

   fInData = TRUE;

   strcpy (temp, pchNext (fp));
   if (! strcmp (pWord = &temp[strlen(temp)-4], ".rel"))
      *pWord = 0;

   if ((rel = mb_inc (temp, key)) == RNULL)
      {
      fprintf (stderr, "Cannot open relation '%s' : %s\n", temp, mb_error);
      goto lblERROR;
      }
   if ((rec = (dataptr)malloc (2 + rel->cbRecord)) == NULL)
      {
      fprintf (stderr, "Out of memory\n");
      goto lblERROR;
      }
   if ((buf = (dataptr)malloc (2 + rel->cbRecord)) == NULL)
      {
      fprintf (stderr, "Out of memory\n");
      free    (rec);
      goto lblERROR;
      }


/*
 * Great.  We've now opened the relations and are ready to parse the template.
 * Reports are generated on the fly, so this is a little more complex than
 * it need be:
 *
 */

   for ( ; !fpEOF(fp); )
      {
      if ((tok = tokCheck (fp, pWord = pchNext (fp))) == tokEND)
         break;

      switch (tok)
         {
         case tokSIZE:      if (! sectionSize (fp))
                               goto lblERROR_2;
                            fSkip = FALSE;
                           break;

         case tokON:        if (! sectionOn (fp))
                               goto lblERROR_2;
                            fSkip = TRUE;
                           break;

         case tokNEWPAGE:   printf ("\f");
                            endOfLine (fp);
                            fSkip = FALSE;
                           break;

         case tokPAGE:      pageno = atoi (pchNext (fp));
                            endOfLine (fp);
                            fSkip = FALSE;
                           break;

         case tokHEADER:    fKeepHeader = FALSE;
                            switch (tok = tokCheck (fp, pWord = pchNext (fp)))
                               {
                               case tokKEEP: fKeepHeader = TRUE;
                                            break;
                               default:      fpBackup (fp);
                                            break;
                               }
                            fSkip = TRUE;
                            posHeader  = fpGetPos (fp);
                            lineHeader = fpLineno (fp);
                           break;

         case tokLAST:      fKeepLast = FALSE;
                            switch (tok = tokCheck (fp, pWord = pchNext (fp)))
                               {
                               case tokKEEP: fKeepLast = TRUE;
                                            break;
                               default:      fpBackup (fp);
                                            break;
                               }
                            fSkip = TRUE;
                            posLast  = fpGetPos (fp);
                            lineLast = fpLineno (fp);
                           break;

         case tokFOOTER:    fKeepFooter = FALSE;
                            switch (tok = tokCheck (fp, pWord = pchNext (fp)))
                               {
                               case tokKEEP: fKeepFooter = TRUE;
                                            break;
                               default:      fpBackup (fp);
                                            break;
                               }
                            fSkip = TRUE;
                            posFooter  = fpGetPos (fp);
                            lineFooter = fpLineno (fp);
                           break;

         case tokSEMI:     break;

         default:           if (! fSkip)
                               {
                               if (fInData)
                                  {
                                  fprintf (stderr,
                                    "Multiple relations are not supported.\n");
                                  goto lblERROR_2;
                                  }
                               fprintf (stderr,
                                    "Unrecognized token '%s', line %d.\n",
                                     pWord, lastLine);
                               goto lblERROR_2;
                               }
                           break;
         }

      fInData = FALSE;
      }

   rc = 0;

lblERROR_2:
   if (rec != NULL)  free (rec);
   if (buf != NULL)  free (buf);

lblERROR:
   fpClose (fp);
   mb_exit (rc);
}


/*
 * UTILITIES ------------------------------------------------------------------
 *
 */

FParse *
parseArgs (argc, argv)
int        argc;
char           **argv;
{
   char   *pch, name[256];
   FParse *fp;

   for (--argc,++argv; argc; argc--,argv++)
      {
      if (*(pch = *argv) != '-')
         break;

      switch (*(++pch))
         {
         case 'k':  if (*(1+pch))
                       pch ++;
                    else
                       {
                       argc--;  argv++;
                       if (! argc)
                          {
                          fprintf (stderr, "encryption key must follow -k\n");
                          return NULL;
                          }
                       pch = *argv;
                       }
                     key = pch;
                   break;

         default:   fprintf (stderr, "unrecognized option '%c'\n", *pch);
                    return NULL;
                   break;
         }
      }

   if (! argc)
      {
      return FALSE;
      }

   strcpy (name, pch);
   if (strcmp (&name[strlen(name)-4], ".rpt"))
      strcat (name, ".rpt");

   if ((fp = fpInit (name, TRUE)) == NULL)
      {
      fprintf (stderr, "cannot open report template '%s'\n", name);
      exit (2);
      }

   return fp;
}

/*
 * fillData() is used to set up a comparison record.
 *
 */

void
fillData (ptr, idx, str)
dataptr   ptr;
int            idx;
char               *str;
{
   char    temp[128];
   dataptr x;
   long    tlong;
   int     i, j, k, f;

   i = 0;                   /* Left edge of word in str */

   for (k = 0; ; k++)       /* k = idxFld */
      {
      if (! str[i])
         break;

      f = rel->idxFld[idx][k];
      x = (dataptr)((char *)ptr + rel->cbStart[f]);  /* Position of field */

      for (j = i; str[j] && str[j] != ','; j++)
         ;
      strcpy (temp, &str[i]);
      temp[j-i] = 0;
      i = j;

      switch (rel->fldType[f])
         {
         case T_SHORT:   *(short   *)x = (short) atoi (temp);  break;
         case T_USHORT:  *(ushort  *)x = (ushort)atoi (temp);  break;
         case T_LONG:    *(long    *)x = (long)  atol (temp);  break;
         case T_ULONG:   *(ulong   *)x = (ulong) atol (temp);  break;
         case T_FLOAT:   *(float   *)x = (float) atof (temp);  break;
         case T_DOUBLE:  *(double  *)x = (double)atof (temp);  break;
         case T_MONEY:   *(double  *)x = (double)atof (temp);  break;
         case T_TIME:    *(mb_time *)x = scn_time (temp);      break;
         case T_DATE:    *(mb_date *)x = scn_date (temp);      break;
         case T_SERIAL:  *(long    *)x = (long)  atol (temp);  break;
         case T_PHONE:   scn_phone ((mb_phone *)x, temp);      break;
         case T_BYTE:    strtohex (x, temp, rel->cbLen[f]);    break;
         default:        strzcpy (x, temp, rel->cbLen[f]);     break;
         }

      if (rel->fldType[f] == T_MONEY)
         {
         tlong = (long)(100.0 * (double)atof (temp));
         *(double *)x = (double)tlong / 100.0;
         }
      }
}

/*
 * numLines() reads from position 'pos' until it doesn't see a print or
 * skip command--it keeps track of the number of lines used in the section,
 * and returns it.  It's used to make headers and footers work well with
 * top and bottom margins.
 *
 */

int
numLines (fp, pos, line)
FParse   *fp;
long          pos;
int                line;
{
   char   *pWord;
   long    posOld;
   int     num, lineno;
   token   tok;

/*
 * First, remember where we are now, and move to the new position...
 *
 */

   posOld = fpGetPos (fp);
   lineno = fpLineno (fp);

   fpSetPos (fp, pos, line);

   for (num = 0; !fpEOF(fp); )
      {
      switch (tokCheck (fp, pWord = pchNext (fp)))
         {
         case tokPRINT:  tok = tokCheck (fp, pWord = pchNext (fp));

                         if (tok != tokCONTINUED)
                            num++;
                        break;

         case tokSKIP:   num += atoi (pchNext (fp));

                         (void)tokSkip (fp, tokLINES);
                         (void)tokSkip (fp, tokLINE);
                        break;

         default:        goto lblDONE;
                        break;
         }

      while (* (pchNext(fp)) != ';')
         {
         if (fpEOF (fp))
            goto lblDONE;
         }
      }

lblDONE:
   fpSetPos (fp, posOld, lineno);

   return num;
}


/*
 * SECTION PARSING CODE -------------------------------------------------------
 *
 *
 * sectionPrint() actually performs the printings for any given section.  :)
 *
 */

void
sectionPrint (fp, pos, line, fReally)
FParse       *fp;
long              pos;
int                    line;
bool                         fReally;
{
   token   tok;
   long    posOld;
   int     n, lineno;

   if (! pos)
      return;

   posOld = fpGetPos (fp);
   lineno = fpLineno (fp);

   fpSetPos (fp, pos, line);


   width = cbWidth;
   left  = marginLeft;

   for ( ; !fpEOF(fp); )
      {
      switch (tok = tokCheck (fp, pchNext (fp)))
         {
         case tokPRINT:    commandPrint (fp, fReally);
                          break;

         case tokSKIP:     n = atoi (pchNext (fp));

                           (void)tokSkip (fp, tokLINES);
                           (void)tokSkip (fp, tokLINE);

                           endOfLine (fp);

                           if (fReally)
                              skipl (n);
                          break;

         default:          goto lblDONE;
                          break;
         }
      }

lblDONE:
   fpSetPos (fp, posOld, lineno);
}


/*
 * commandPrint() processes 1 PRINT command.  Huge, eh?
 *
 */

#define justLEFT   0
#define justCENTER 1
#define justRIGHT  2

void
commandPrint (fp, fReally)
FParse       *fp;
bool              fReally;
{
   mb_phone tphone;
   int      Justify;
   bool     fContinue;
   token    tok;
   long     tlong;    /* typ == 1, 3, 4 */
   double   tdouble;  /* typ == 2       */
   char    *pWord;
   char     line[128], word[128], modif[128], temp[128];
   int      typ, fmt;
   int      n;


/*
 * First, find out if we have any modifiers on the print command:
 *
 */

   fContinue = FALSE;
   Justify   = justLEFT;

   while ((tok = tokCheck (fp, pchNext (fp))) != tokCOLON)
      {
      switch (tok)
         {
         case tokCONTINUED:   fContinue = TRUE;      break;
         case tokCENTERED:    Justify = justCENTER;  break;
         case tokRIGHT:       Justify = justRIGHT;   break;
         case tokLEFT:        Justify = justLEFT;    break;
         default:             syntaxErr (":");       break;
         }
      }


/*
 * Then process the print command itself:
 *
 */

   for (line[0] = 0; (tok = tokCheck (fp, pWord = pchNext(fp))) != tokSEMI; )
      {
      typ = 0;

/*
 * First, check for a hard-coded string.  Those will be quoted:
 *
 */

      if (fpQuoted (fp))
         {
         strcpy (word, pWord);
         goto lblFINISH;
         }

/*
 * Then check to see if they've put in a constant...
 *
 */

      if (isdigit (*pWord) || *pWord == '.')
         {
         if (strchr (pWord, '.'))
            {
            typ = 2;
            tdouble = (double)atof (pWord);
            goto lblFINISH;
            }

         typ=1;
         tlong = atol (pWord);
         goto lblFINISH;
         }

/*
 * Before we decide that it's a variable name, make sure it's not a column
 * command or field separator (interperated as a space-tab - enough spaces
 * to get to the next increment of TABSTOP characters)
 *
 */

      if (tok == tokCOLUMN)
         {
         n = atoi (pchNext (fp)) -left -strlen(line);
         makeSpaces (word, n);
         goto lblFINISH;
         }

      if (tok == tokPLUS)
         {
         n = TABSTOP- (strlen (line) % TABSTOP);
         makeSpaces (word, n);
         goto lblFINISH;
         }

      if (tok == tokCOMMA)
         {
         word[0] = ' ';
         word[1] = 0;
         goto lblFINISH;
         }


/*
 * It's a variable of some sort; make 'word' the name and 'modif' any modifer.
 * If they use the relation name prefix, remove it.
 *
 */

      fmt = -1;
      modif[0] = 0;

      strcpy (word, pWord);

      if ((pWord = strchr (word, '!')) != NULL)
         {
         strcpy (modif, 1+pWord);
         *pWord = 0;
         }

      if (pWord = strchr (word, '.'))
         {
         strzcpy (word, 1+pWord, strlen(1+pWord));
         }

/*
 * First check if it's a system! variable:
 *
 */

      tok = tokCheck (fp, modif);

      if (! strcmp (word, "system"))
         {
         switch (tok)
            {
            case tokPAGE:  tlong = (long)pageno;     typ = 1;  break;
            case tokTIME:  tlong = (long)curtime();  typ = 3;  break;
            case tokDATE:  tlong = (long)curdate();  typ = 4;  break;
            default:       strcpy (word, CREDIT);    typ = 0;  break;
            }

         goto lblFINISH;
         }


/*
 * That done, figure out what type it should be and assign it.
 *
 */

      if ((n = fldnum (rel, word)) == -1)
         {
         strcat (word, "?");
         goto lblFINISH;
         }

      pWord = (char *)rec + rel->cbStart[n];

      switch (rel->fldType[n])
         {
         case T_SHORT:   tlong   = (long)   *(short   *)pWord;  typ = 1; break;
         case T_USHORT:  tlong   = (long)   *(ushort  *)pWord;  typ = 1; break;
         case T_LONG:    tlong   = (long)   *(long    *)pWord;  typ = 1; break;
         case T_SERIAL:  tlong   = (long)   *(long    *)pWord;  typ = 1; break;
         case T_ULONG:   tlong   = (long)   *(ulong   *)pWord;  typ = 1; break;
         case T_FLOAT:   tdouble = (double) *(float   *)pWord;  typ = 2; break;
         case T_DOUBLE:  tdouble = (double) *(double  *)pWord;  typ = 2; break;
         case T_MONEY:   tdouble = (double) *(double  *)pWord;  typ = 2; break;
         case T_TIME:    tlong   = (long)   *(mb_time *)pWord;  typ = 3; break;
         case T_DATE:    tlong   = (long)   *(mb_date *)pWord;  typ = 4; break;
         case T_PHONE:   phncpy (&tphone, (mb_phone *)pWord);   typ = 5; break;
         case T_BYTE:    hextostr (word, pWord, rel->cbLen[n]);          break;
         default:        strzcpy (word, pWord, rel->cbLen[n]);           break;
         }


lblFINISH:

      if (tokSkip (fp, tokUSING) || tokSkip (fp, tokFORMAT))
         {
         pWord = pchNext (fp);
         if (typ == 3 || typ == 4 || typ == 5)
            {
            fmt = atoi (pWord);
            }
         else
            {
            strcpy (modif, pWord);
            switch (typ)
               {
               case 0:  strcpy  (temp, word);
                        sprintf (word, modif, temp);
                       break;
               case 1:  sprintf (word, modif, tlong);
                       break;
               case 2:  sprintf (word, modif, tdouble);
                       break;
               }
            typ = 0;
            }
         }

      if (fmt == -1)
         {
         if (typ == 3)  fmt = 1;  /* Time  format 1 == '5:12pm'            */
         if (typ == 4)  fmt = 4;  /* Date  format 4 == 'Wed Nov 18 1992'   */
         if (typ == 5)  fmt = 1;  /* Phone format 1 == '(615) 574-7474 x15 */
         }

      switch (typ)
         {
         case 1:  sprintf (word, "%ld", tlong);
                 break;
         case 2:  sprintf (word, "%lg", tdouble);
                 break;
         case 3:  strcpy  (word, fmt_time (tlong, fmt));
                 break;
         case 4:  strcpy  (word, fmt_date (tlong, fmt));
                 break;
         case 5:  strcpy  (word, fmt_phone (&tphone, fmt));
                 break;
         }

      if (tokSkip (fp, tokTO))
         {
         n = atoi (pchNext (fp)) -left -strlen(word) - strlen(line);
         makeSpaces (temp, n);
         strcat (word, temp);
         }

      strcat (line, word);
      }

/*
 * and actually print it.
 *
 */

   if (fReally)
      {
      switch (Justify)
         {
         case justLEFT:    n = 0;                             break;
         case justCENTER:  n = ((width - strlen(line)) / 2);  break;
         case justRIGHT:   n = (width - strlen (temp));       break;
         }

      if (left == marginLeft)
         n += marginLeft;

      makeSpaces (temp, n);
      printf ("%s%s", temp, line);

      if (fContinue)
         {
         width -= strlen(line);
         left  += strlen(line);
         }
      else
         {
         width = cbWidth;
         left  = marginLeft;
         printf ("\n");
         }
      }
}

bool
sectionSize (fp)
FParse      *fp;
{
   token  tok;
   char  *pWord;

   for ( ; !fpEOF(fp); )
      {
      tok = tokCheck (fp, pWord = pchNext (fp));

      switch (tok)
         {
         case tokCOLUMNS:   num_col = atoi (pchNext (fp));
                           break;
         case tokROWS:      num_row = atoi (pchNext (fp));
                           break;

         case tokTOP:       tokSkip (fp, tokMARGIN);
                            marginTop = atoi (pchNext (fp));
                           break;
         case tokBOTTOM:    tokSkip (fp, tokMARGIN);
                            marginBottom = atoi (pchNext (fp));
                           break;
         case tokLEFT:      tokSkip (fp, tokMARGIN);
                            marginLeft = atoi (pchNext (fp));
                           break;
         case tokRIGHT:     tokSkip (fp, tokMARGIN);
                            marginRight = atoi (pchNext (fp));
                           break;

         case tokPAGE:
         case tokNEWPAGE:
         case tokHEADER:
         case tokLAST:
         case tokON:
         case tokFOOTER:    fpBackup (fp);
                            goto lblDONE;
                           break;

         default:       fprintf (stderr, "token '%s' unrecognized\n", pWord);
                        return FALSE;
                       break;
         }

      if (* (pWord = pchNext (fp)) != ';')
         {
         syntaxErr (";");
         }
      }

lblDONE:
   return TRUE;
}

bool
sectionOn (fp)
FParse    *fp;
{
   token      tok;
   char      *pWord;
   int        stop;
   int        idx;
   int        i;
   int        nLines;
   int        pageLines;
   int        usedLines;
   mb_action  dir, act;
   bool       fDoneOne;
   bool       fBadRec;


   cbWidth  = num_col -marginRight -marginLeft;
   cbHeight = num_row -marginTop   -marginBottom;

/*
 * Perform ON clause.  We are currently sitting right after the keyword On,
 * and the following are conditionals on the operation of this clause:
 *
 *   posHeader:  0L==no header - else, location of first print/skip command
 *   posLast:    0L==no last   - else, location of first print/skip command
 *   posFooter:  0L==no footer - else, location of first print/skip command
 *
 */

   act  = FIRST;
   stop = NONE;
   dir  = NEXT;

   pWord = pchNext (fp);
   if ((idx = idxnum (rel, pWord)) < 0)
      {
      if (tokCheck (fp, pWord) == tokQUEUE)
         {
         act = FIRST_Q;
         idx = 0;        /* Abitrary; unused */
         }
      else
         {
         fprintf (stderr, "invalid index '%s' referenced\n", pWord);
         return FALSE;
         }
      }

   if (act != FIRST_Q)
      {
      switch (tok = tokCheck (fp, pWord = pchNext (fp)))
         {
         case tokLTEQ:   act = FIRST; stop = AFTER;         break;
         case tokEQLT:   act = FIRST; stop = AFTER;         break;
         case tokLT:     act = FIRST; stop = DURING|AFTER;  break;
         case tokGTEQ:   act = GTEQ;  stop = NONE;          break;
         case tokEQGT:   act = GTEQ;  stop = NONE;          break;
         case tokGT:     act = GTHAN; stop = NONE;          break;
         case tokEQUAL:  act = EQUAL; stop = AFTER;         break;
         case tokEQ:     act = EQUAL; stop = AFTER;         break;
         default:        fpBackup (fp);                     break;
         }
      }

   if ((act != FIRST && act != FIRST_Q) || stop != NONE)
      {
      if (tokCheck (fp, pWord = pchNext (fp)) != tokUNKNOWN)
         {
         fprintf (stderr, "query requires comparison value\n");
         return FALSE;
         }

      fillData (rec, idx, pWord);
      fillData (buf, idx, pWord);
      }

   if (tokSkip (fp, tokREVERSE))
      {
      dir = PREV;
      switch (act)
         {
         case FIRST_Q:  act = LAST_Q;
                       break;
         case FIRST:    if (stop == NONE)    act = LAST;
                        if (stop == AFTER)   act = LTEQ,  stop = NONE;
                        if (stop &  DURING)  act = LTHAN, stop = NONE;
                       break;
         case GTEQ:     if (stop == NONE)    act = LAST,  stop = BEFORE;
                       break;
         case GTHAN:    if (stop == NONE)    act = LAST,  stop = DURING|BEFORE;
                       break;
         }
      }

   posOther  = fpGetPos (fp);
   lineOther = fpLineno (fp);
   nLines    = numLines (fp, posOther, lineOther);
   usedLines = 0;

   pageLines  = cbHeight;
   pageLines -= numLines (fp, posHeader, lineHeader);
   pageLines -= numLines (fp, posFooter, lineFooter);

/*
 * Now actually retrieve the necessary records...
 *
 */

   fDoneOne = FALSE;

   for (;;)
      {
      fBadRec = FALSE;
      if (mb_sel (rel, idx, rec, act, buf) != MB_OKAY)
         {
         if (fDoneOne || mb_errno != MB_NO_SUCH)
            break;
         else
            fBadRec = TRUE;
         }
      fDoneOne = TRUE;

      if (! fBadRec && stop != NONE)
         {
         i = compare (rel, rec, buf, idx);

         if ( (i <  0 && (stop & BEFORE)) ||
              (i == 0 && (stop & DURING)) ||
              (i >  0 && (stop & AFTER)))
            {
            break;
            }
         }

      fpSetPos (fp, posOther, lineOther);

      if (usedLines +nLines > pageLines)     /* Should advance to next page? */
         {
         skipl (pageLines -usedLines);
         sectionPrint (fp, posFooter, lineFooter, TRUE);
         usedLines = 0;
         skipl (marginBottom);
         pageno++;
         }

      if (usedLines == 0)                    /* Starting a new page (header) */
         {
         skipl (marginTop);
         sectionPrint (fp, posHeader, lineHeader, TRUE);
         }

      sectionPrint (fp, posOther, lineOther, (bool)(!fBadRec));

      if (fBadRec)
         break;

      usedLines += nLines;
      act = dir;
      }

   if (mb_errno != MB_NO_SUCH && mb_errno != MB_OKAY)
      {
      fprintf (stderr, "aborted -- %s\n", mb_error);
      return FALSE;
      }

   posOther  = fpGetPos (fp);
   lineOther = fpLineno (fp);

   skipl (pageLines -usedLines);

   if (posLast)
      sectionPrint (fp, posLast, lineLast, TRUE);
   else
      sectionPrint (fp, posFooter, lineFooter, TRUE);

   fpSetPos (fp, posOther, lineOther);
   skipl (marginBottom);

   pageno++;

/*
 * Done with this clause.  Erase current header/footer/last positions--they
 * don't carry from one clause to another.  That is, unless they had 'keep'
 * keywords...
 *
 */

   if (! fKeepLast)    posLast   = 0L;
   if (! fKeepFooter)  posFooter = 0L;
   if (! fKeepHeader)  posHeader = 0L;

   return TRUE;
}


/*
 * SERVICE ROUTINES -----------------------------------------------------------
 *
 */

void
syntaxErr (x)     /* COMPLAIN ABOUT SYNTAX AND EXIT ------------------------ */
char      *x;
{
   fprintf (stderr, "report: expected keyword '%s', line %d\n", x, lastLine);
   mb_exit (3);
}

void
endOfLine (fp)
FParse    *fp;
{
   if (tokCheck (fp, pchNext (fp)) != tokSEMI)
      {
      syntaxErr (";");
      }
}

char *
pchNext (fp)      /* RETURN NEXT WORD, SKIPPING ANY COMMENTS --------------- */
FParse  *fp;
{
   char  *pWord;

   lastLine = fpLineno (fp);

   while (*(pWord = fpWord (fp)) == '#')
      {
      (void)fpLine (fp);
      }

   if (fpEOF (fp))
      {
      fprintf (stderr, "expected END before end-of-file, line %d\n", lastLine);
      mb_exit (1);
      }

   return pWord;
}

token
tokCheck (fp, pWord)  /* CHECK IF A WORD IS A TOKEN ------------------------ */
FParse   *fp;
char         *pWord;
{
   token  tok;

   if (fpQuoted (fp))
      return tokUNKNOWN;

   for (tok = tokFIRST; tok < tokUNKNOWN; tok = (token)(((int)tok)+1))
      {
      if (! strcmp (pWord, aTokens[(int)tok]))
         break;
      }

   return tok;
}

bool
tokSkip (fp, tok) /* SKIP A TOKEN IF IT COMES NEXT ------------------------- */
FParse  *fp;
token        tok;
{
   if (tokCheck (fp, pchNext (fp)) != tok)
      {
      fpBackup (fp);
      return FALSE;
      }

   return TRUE;
}

void
skipl (n)         /* PRINT n BLANK LINES ----------------------------------- */
int    n;
{
   for ( ; n > 0; n--)
      printf ("\n");
}

void
makeSpaces (buf, n)   /* FILL A BUFFER WITH n SPACES ----------------------- */
char       *buf;
int              n;
{
   n = max (n, 0);

   buf[n] = 0;
   for (n--; n >= 0; n--)
      buf[n] = ' ';
}

