#! /bin/sh
#
# gen-lock-obj.sh - Build tool to construct the lock object.
#
# Copyright (C) 2020, 2021  g10 Code GmbH
#
# This file is part of libgpg-error.
#
# libgpg-error is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2.1 of
# the License, or (at your option) any later version.
#
# libgpg-error is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, see <https://www.gnu.org/licenses/>.
#

#
# Following variables should be defined to invoke this script
#
#   CC
#   OBJDUMP
#   AWK
#   ac_ext
#   ac_object
#   host
#   LOCK_ABI_VERSION
#
# An example:
#
# LOCK_ABI_VERSION=1 host=x86_64-pc-linux-gnu host_alias=x86_64-linux-gnu \
#     CC=$host_alias-gcc OBJDUMP=$host_alias-objdump ac_ext=c ac_objext=o \
#     AWK=gawk ./gen-lock-obj.sh
#

if test -n "`echo -n`"; then
    ECHO_C='\c'
    ECHO_N=''
else
    ECHO_C=''
    ECHO_N='-n'
fi

if test "$1" = --disable-threads; then
    cat <<EOF
## lock-obj-pub.$host.h - NO LOCK SUPPORT
## File created by gen-lock-obj.sh - DO NOT EDIT
## To be included by mkheader into gpg-error.h

/* Dummy object - no locking available.  */
typedef struct
{
  long _vers;
} gpgrt_lock_t;

#define GPGRT_LOCK_INITIALIZER {-1}
EOF
else
AWK_VERSION_OUTPUT=$($AWK 'BEGIN { print PROCINFO["version"] }')
if test -n "$AWK_VERSION_OUTPUT"; then
    # It's GNU awk, which supports PROCINFO.
    AWK_OPTION=--non-decimal-data
fi

cat <<'EOF' >conftest.$ac_ext
#include <pthread.h>
pthread_mutex_t mtx = PTHREAD_MUTEX_INITIALIZER;
EOF

if $CC -c conftest.$ac_ext; then :
  ac_mtx_size=$($OBJDUMP -j .bss -t conftest.$ac_objext \
         | $AWK $AWK_OPTION '
/mtx$/ { mtx_size = int("0x" $5) }
END { print mtx_size }')
else
    echo "Can't determine mutex size"
    exit 1
fi
rm -f core conftest.err conftest.$ac_objext conftest.$ac_ext

cat <<EOF
## lock-obj-pub.$host.h
## File created by gen-lock-obj.sh - DO NOT EDIT
## To be included by mkheader into gpg-error.h

typedef struct
{
  long _vers;
  union {
    volatile char _priv[$ac_mtx_size];
    long _x_align;
    long *_xp_align;
  } u;
} gpgrt_lock_t;

EOF

# FIXME: Support different alignment conditions of:
#
#     USE_16BYTE_ALIGNMENT
#     USE_DOUBLE_FOR_ALIGNMENT
#     USE_LONG_DOUBLE_FOR_ALIGNMENT
#

echo ${ECHO_N} "#define GPGRT_LOCK_INITIALIZER {$LOCK_ABI_VERSION,{{${ECHO_C}"

i=0
while test "$i" -lt $ac_mtx_size; do
    if test "$i" -ne 0 -a "$(( $i % 8 ))" -eq 0; then
        echo ' \'
        echo ${ECHO_N} "                                    ${ECHO_C}"
    fi
    echo ${ECHO_N} "0${ECHO_C}"
    if test "$i" -lt $(($ac_mtx_size - 1)); then
        echo ${ECHO_N} ",${ECHO_C}"
    fi
    i=$(( i + 1 ))
done
echo '}}}'
fi

cat <<'EOF'
##
## Local Variables:
## mode: c
## buffer-read-only: t
## End:
##
EOF

exit 0
