/*****************************************************************

Copyright (c) 2000 Bill Nagel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include "quickbutton.h"
#include "quickaddappsmenu.h"

#include <qpainter.h>
#include <qdrawutil.h>
#include <qpopupmenu.h>
#include <qtooltip.h>

#include <kmimetype.h>
#include <klocale.h>
#include <kdesktopfile.h>
#include <krun.h>
#include <kiconeffect.h>
#include <kglobalsettings.h>
#include <kcursor.h>
#include <kapplication.h>
#include <kipc.h>
#include <kiconloader.h>
#include <kurldrag.h>
#include <kstandarddirs.h>

QuickButton :: QuickButton(const QString &u, QWidget *parent, const char *name)
   : QButton(parent, name)
{
   if (parent && ! parent->parent())
   {
     setBackgroundMode(X11ParentRelative);
   }
   setBackgroundOrigin( AncestorOrigin );
   setMouseTracking(true);
   highlight = false;
   oldCursor = cursor();
   id = u;
   if (id.startsWith("/"))
   {
     // Absolute path
     url.setPath(id);
     
     if (id.endsWith(".desktop"))
     {
       // Strip path
       QString s = id;
       s = s.mid(s.findRev('/')+1);
       s = s.left(s.length()-8);
       service = KService::serviceByDesktopName(s);
       if (!service)
       {
         service = new KService(id);
         if (!service->isValid())
         {
           // service is a KShared pointer, don't try to delete it!
           service = 0;
         }
       }
     }
   }
   else if (!KURL::isRelativeURL(id))
   {
     // Full URL
     url = id;
   }
   else
   {
     // menu-id
     service = KService::serviceByMenuId(id);
   }

   QString comment;
   if (service)
   {
     url.setPath(locate("apps", service->desktopEntryPath()));
     if (!service->menuId().isEmpty())
       id = service->menuId();
   
     comment = service->comment();
     if (comment.isEmpty())
       comment = service->genericName();
     if (comment.isEmpty())
       comment = service->name();
     else
       comment = service->name() + " - " + comment;
   }
   else 
   {
     comment = url.prettyURL();
   }
   QToolTip::add(this, comment);   

   icon = KMimeType::pixmapForURL(url, 0, KIcon::Panel, KIcon::SizeSmall, KIcon::DefaultState);
   iconh = KMimeType::pixmapForURL(url, 0, KIcon::Panel, KIcon::SizeSmall, KIcon::ActiveState);

   QBrush bgbrush( colorGroup().brush( QColorGroup::Background ) );
   if( bgbrush.pixmap() )
   {
      setBackgroundOrigin( AncestorOrigin );
      setPaletteBackgroundPixmap( *bgbrush.pixmap() );
   }

   resize(20, 20);

   QuickAddAppsMenu *addAppsMenu = new QuickAddAppsMenu(parent, this);
   popup = new QPopupMenu(this);
   popup->insertItem(i18n("Add Application"), addAppsMenu);
   popup->insertSeparator();
   popup->insertItem(SmallIcon("remove"), i18n("Remove"), this, SLOT(slotRemove()));

   slotSettingsChanged(KApplication::SETTINGS_MOUSE);
   connect(kapp, SIGNAL(settingsChanged(int)), SLOT(slotSettingsChanged(int)));
   connect(kapp, SIGNAL(iconChanged(int)), SLOT(slotIconChanged(int)));
   connect(this, SIGNAL(clicked()), SLOT(slotExec()));
   connect(this, SIGNAL(removeIcon(QuickButton *)), parent, SLOT(removeIcon(QuickButton *)));
   kapp->addKipcEventMask(KIPC::SettingsChanged);
   kapp->addKipcEventMask(KIPC::IconChanged);
}

QuickButton :: ~QuickButton()
{
}

QString QuickButton :: getURL()
{
   return url.url();
}

QString QuickButton :: getId()
{
   return id;
}

void QuickButton :: paintEvent( QPaintEvent * )
{
   QPainter p( this );
   drawButton( &p );
}

void QuickButton :: drawButton(QPainter *p)
{
   if (isDown() || isOn()) {
      p->fillRect(rect(), colorGroup().brush(QColorGroup::Mid));
      qDrawWinButton(p, 0, 0, width(), height(), colorGroup(), true);
   }

   drawButtonLabel(p);
}

void QuickButton :: drawButtonLabel(QPainter *p)
{
   QPixmap *pix = &icon;
   if (highlight) pix = &iconh;

   int d = 0;
   if (isDown() || isOn()) d = 1;

   if (width() == 24)
      p->drawPixmap(4 + d, 4 + d, *pix);
   else
      p->drawPixmap(2 + d, 2 + d, *pix);
}

void QuickButton :: enterEvent(QEvent*)
{
   if (changeCursorOverItem)
      setCursor(KCursor().handCursor());

   highlight = true;
   repaint();
}

void QuickButton :: leaveEvent(QEvent*)
{
   if (changeCursorOverItem)
      setCursor(oldCursor);

   highlight = false;
   repaint();
}

void QuickButton :: mousePressEvent(QMouseEvent *e)
{
   if (e->button() == RightButton)
      popup->popup(e->globalPos());
   else if ( e->button() == LeftButton) {
      dragPos = e->pos();
      QButton::mousePressEvent(e);
   }
}

void QuickButton :: mouseMoveEvent(QMouseEvent *e)
{
   if ((e->state() & LeftButton) == 0) return;
   QPoint p(e->pos() - dragPos);
   if (p.manhattanLength() <= KGlobalSettings::dndEventDelay())
      return;
   
   setDown(false);
   KURL::List uris;
   uris.append(url);
   KURLDrag *dd = new KURLDrag( uris, this );
   dd->setPixmap(icon);

   dd->drag();
}

void QuickButton :: slotSettingsChanged(int category)
{
   if (category != KApplication::SETTINGS_MOUSE) return;

   changeCursorOverItem = KGlobalSettings::changeCursorOverIcon();

   if(!changeCursorOverItem)
      setCursor(oldCursor);
}

void QuickButton :: slotIconChanged(int)
{
   icon = KMimeType::pixmapForURL(url, 0, KIcon::Panel, KIcon::SizeSmall, KIcon::DefaultState);
   iconh = KMimeType::pixmapForURL(url, 0, KIcon::Panel, KIcon::SizeSmall, KIcon::ActiveState);
   repaint();
}

void QuickButton :: slotExec()
{
    setDown(false);
    repaint();
    KIconEffect::visualActivate(this, rect());
    kapp->propagateSessionManager();   // is this needed?
    if (service)
      KRun::run(*service, KURL::List());
    else
      new KRun(url, 0, url.isLocalFile());
}

void QuickButton :: slotRemove()
{
    emit removeIcon(this);
}
#include "quickbutton.moc"
