/*
 * Copyright (c) 2002-2005 DevTeam Ltd. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * The complete licence text can be found at
 * http://www.jniwrapper.com/pages/jexcel/license
 */
package com.jniwrapper.win32.jexcel.samples.demo;

import com.jniwrapper.util.Logger;
import com.jniwrapper.win32.jexcel.*;
import com.jniwrapper.win32.jexcel.ui.*;
import com.jniwrapper.win32.ui.Wnd;
import com.jniwrapper.win32.ui.MessageBox;
import com.jniwrapper.win32.ui.WindowTools;
import com.jniwrapper.win32.ui.dialogs.OpenSaveFileDialog;
import com.jniwrapper.win32.HResult;
import com.jniwrapper.win32.automation.OleMessageLoop;
import com.sun.java.swing.plaf.windows.WindowsMenuBarUI;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.lang.reflect.InvocationTargetException;

/**
 * @author Vladimir Kondrashchenko
 * @author Dmitriy Milashenko
 */

public class JExcelDemo extends JFrame
{
    private static final Logger LOG = Logger.getInstance(JExcelDemo.class);
    private static final String FILTER = "Excel Workbooks (*.xls) | *.xls";

    private JWorkbook _workbook;
    private JMenuBar _menuBar;
    private JTextArea _eventLog;
    private AutomationBarAction _automationBarAction;
    private EventLogAction _eventLogAction;
    private StaticViewAction _staticViewAction;
    private TitledPane _eventLogPane;
    private TitledPane _automationBarPane;
    private JSplitPane _splitPane;
    private JSplitPane _subSplitPane;
    private WorksheetEventLogger _worksheetEventLogger;
    private WorkbookEventLogger _workbookEventLogger;
    private JScrollPane _eventLogScrollPane;
    private AutomationPanel _automationPanel;
    private JMenuItem _miSaveAs;
    private JMenuItem _miPrintPreview;
    private JMenuItem _miPrint;
    private JMenuItem _miNew;
    private JMenuItem _miOpen;
    private JCheckBoxMenuItem _miStaticView;

    public JExcelDemo() throws ExcelException
    {
        super("JExcel Demo");

        setSize(900, 700);
        setLocationRelativeTo(null);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        getContentPane().setLayout(new BorderLayout());

        _worksheetEventLogger = new WorksheetEventLogger();
        _workbookEventLogger = new WorkbookEventLogger();
        _workbook = new JWorkbook();
        if (!_workbook.isWorkbookCreated())
        {
            throw new IllegalStateException("Failed to obtain a workbook");
        }
        List worksheets = _workbook.getWorksheets();
        for (int i = 0; i < worksheets.size(); i++)
        {
            Worksheet worksheet = (Worksheet)worksheets.get(i);
            worksheet.addWorksheetEventListener(_worksheetEventLogger);
        }
        _workbook.addWorkbookEventListener(_workbookEventLogger);
        _workbook.addJWorkbookEventListener(new JWorkbookListener());

        _workbook.setMinimumSize(new Dimension(200, 200));

        _eventLogAction = new EventLogAction();
        _automationBarAction = new AutomationBarAction();
        _staticViewAction = new StaticViewAction();

        _eventLog = new JTextArea();
        _eventLog.setFont(new Font("Courier New", Font.PLAIN, 11));
        _eventLog.setEditable(false);
        _eventLog.setBorder(null);
        _eventLogScrollPane = new JScrollPane(_eventLog);
        _eventLogScrollPane.setBorder(null);
        JPanel logPanel = new JPanel(new BorderLayout());
        logPanel.add(_eventLogScrollPane, BorderLayout.CENTER);
        logPanel.setMinimumSize(new Dimension(200, 80));
        _automationPanel = new AutomationPanel(_workbook);
        _automationPanel.setMinimumSize(new Dimension(244, 473));
        _automationPanel.setMaximumSize(new Dimension(244, 0));
        _automationBarPane = new TitledPane("Automation Bar", _automationPanel, null, _automationBarAction);
        _subSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true);
        _subSplitPane.setLeftComponent(_workbook);
        _subSplitPane.setRightComponent(_automationBarPane);
        _subSplitPane.setBorder(null);
        _subSplitPane.setDividerSize(2);
        _subSplitPane.addComponentListener(new ComponentAdapter()
        {
            public void componentResized(ComponentEvent e)
            {
                _subSplitPane.setDividerLocation(_subSplitPane.getWidth() - 246);
            }
        });
        _subSplitPane.addPropertyChangeListener(JSplitPane.DIVIDER_LOCATION_PROPERTY, new PropertyChangeListener()
        {
            public void propertyChange(PropertyChangeEvent evt)
            {
                _subSplitPane.setDividerLocation(_subSplitPane.getWidth() - 246);
            }
        });
        JPopupMenu popupMenu = new JPopupMenu();
        popupMenu.add(new JMenuItem(new AbstractAction("Clear")
        {
            public void actionPerformed(ActionEvent e)
            {
                _eventLog.setText("");
            }
        }));
        _eventLogPane = new TitledPane("Event Log", logPanel, popupMenu, _eventLogAction);
        _splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true);
        _splitPane.setTopComponent(_subSplitPane);
        _splitPane.setBottomComponent(_eventLogPane);
        _splitPane.setBorder(null);
        _splitPane.setDividerSize(2);
        _splitPane.setResizeWeight(0.8);
        getContentPane().add(_splitPane, BorderLayout.CENTER);

        _menuBar = createMenuTree();
        updateRecentFiles();

        setJMenuBar(_menuBar);
        _automationPanel.updateRecentFiles();
    }

    private void enableMenuItems()
    {
        _miPrint.setEnabled(true);
        _miPrintPreview.setEnabled(true);
        _miSaveAs.setEnabled(true);
        _miStaticView.setEnabled(true);
    }

    private void printLog(String s)
    {
        _eventLog.append(s + '\n');
        _eventLogScrollPane.getVerticalScrollBar().setValue(_eventLogScrollPane.getVerticalScrollBar().getMaximum());
        _eventLogScrollPane.repaint();
    }

    private JMenuBar createMenuTree()
    {
        JMenuBar menuBar = new JMenuBar();
        menuBar.setUI(new WindowsMenuBarUI());

        JMenu fileMenu = new JMenu("File");
        fileMenu.setMnemonic(KeyEvent.VK_F);

        _miNew = new JMenuItem(new AbstractAction("New...")
        {
            public void actionPerformed(ActionEvent e)
            {
                _workbook.newWorkbook();
            }
        });
        _miNew.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.CTRL_MASK));

        _miOpen = new JMenuItem(new AbstractAction("Open...")
        {
            public void actionPerformed(ActionEvent e)
            {
                OpenSaveFileDialog dialog = new OpenSaveFileDialog(JExcelDemo.this);
                dialog.setFilter(FILTER);

                if (dialog.getOpenFileName())
                {
                    try
                    {
                        _workbook.openWorkbook(new File(dialog.getFileName()));
                    }
                    catch (FileNotFoundException e1)
                    {
                        JOptionPane.showMessageDialog(SwingUtilities.windowForComponent(JExcelDemo.this),
                                "Cannot find file " + dialog.getFileName() + ".", "File not found",
                                JOptionPane.ERROR_MESSAGE);
                    }
                }
            }
        });
        _miOpen.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.CTRL_MASK));

        JMenuItem miClose = new JMenuItem(new AbstractAction("Close")
        {
            public void actionPerformed(ActionEvent e)
            {
                _workbook.close();
            }
        });

        _miSaveAs = new JMenuItem(new AbstractAction("Save As...")
        {
            public void actionPerformed(ActionEvent e)
            {
                OpenSaveFileDialog dialog = new OpenSaveFileDialog(JExcelDemo.this);
                dialog.setFilter(FILTER);

                if (dialog.getSaveFileName())
                {
                    try
                    {
                        _workbook.saveCopyAs(new File(dialog.getFileName()));
                    }
                    catch (IOException el)
                    {
                        JOptionPane.showMessageDialog(JExcelDemo.this,
                                "Unable to save the workbook to the specified file.",
                                "JExcel Demo",
                                JOptionPane.ERROR_MESSAGE);
                    }
                }
            }
        });

        _miPrintPreview = new JMenuItem(new AbstractAction("Print Preview")
        {
            public void actionPerformed(ActionEvent e)
            {
                _workbook.setPrintPreview(true);
            }
        });

        _miPrint = new JMenuItem(new AbstractAction("Print...")
        {
            public void actionPerformed(ActionEvent e)
            {
                _workbook.showPrintDialog();
            }
        });
        _miPrint.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, ActionEvent.CTRL_MASK));

        JMenuItem miExit = new JMenuItem(new AbstractAction("Exit")
        {
            public void actionPerformed(ActionEvent e)
            {
                processWindowEvent(new WindowEvent(JExcelDemo.this, WindowEvent.WINDOW_CLOSING));
            }
        });

        fileMenu.add(_miNew);
        fileMenu.add(_miOpen);
        fileMenu.add(miClose);
        fileMenu.addSeparator();
        fileMenu.add(_miSaveAs);
        fileMenu.addSeparator();
        fileMenu.add(_miPrintPreview);
        fileMenu.add(_miPrint);
        fileMenu.addSeparator();
        fileMenu.addSeparator();
        fileMenu.add(miExit);
        fileMenu.addItemListener(new ItemListener()
        {
            public void itemStateChanged(ItemEvent e)
            {
                if (e.getStateChange() == ItemEvent.SELECTED)
                {
                    updateRecentFiles();
                }
            }
        });

        JMenu viewMenu = new JMenu("View");
        viewMenu.setMnemonic(KeyEvent.VK_V);

        JCheckBoxMenuItem miShowAutomationBar = new JCheckBoxMenuItem(_automationBarAction);
        _automationBarAction.setMenuItem(miShowAutomationBar);
        miShowAutomationBar.setSelected(true);
        JCheckBoxMenuItem miShowEventLog = new JCheckBoxMenuItem(_eventLogAction);
        _eventLogAction.setMenuItem(miShowEventLog);
        miShowEventLog.setSelected(true);

        _miStaticView = new JCheckBoxMenuItem(_staticViewAction);
        _staticViewAction.setMenuItem(_miStaticView);

        viewMenu.add(miShowAutomationBar);
        viewMenu.add(miShowEventLog);
        viewMenu.addSeparator();
        viewMenu.add(_miStaticView);

        JMenu helpMenu = new JMenu("Help");
        JMenuItem about = new JMenuItem(new ShowAboutAction(this));
        helpMenu.add(about);

        menuBar.add(fileMenu);
        menuBar.add(viewMenu);
        menuBar.add(helpMenu);

        return menuBar;
    }

    private void updateRecentFiles()
    {
        List files;
        try
        {
            files = _workbook.getApplication().getRecentFiles();
        }
        catch (Exception e)
        {
            return;
        }
        if (!_workbook.isWorkbookCreated())
        {
            return;
        }
        JMenu fileMenu = _menuBar.getMenu(0);
        while (fileMenu.getItemCount() > 11)
        {
            fileMenu.remove(9);
        }

        JMenuItem recentFile;
        final int max_length = 40;

        for (int i = 0; i < files.size(); i++)
        {
            final File file = (File)files.get(i);
            String path = file.getAbsolutePath();
            if (path.length() > max_length)
            {
                int pos = path.lastIndexOf('\\');
                if (pos != -1 && (max_length - path.length() + pos) > 0)
                {
                    String subPath = path.substring(0, max_length - path.length() + pos);
                    path = path.substring(0, subPath.lastIndexOf('\\') + 1) + "..." + path.substring(pos, path.length());
                }
            }
            recentFile = new JMenuItem(new AbstractAction(path)
            {
                public void actionPerformed(ActionEvent e)
                {
                    try
                    {
                        _workbook.openWorkbook(file);
                    }
                    catch (FileNotFoundException ex)
                    {
                        JOptionPane.showMessageDialog(SwingUtilities.windowForComponent(JExcelDemo.this),
                                "Cannot find file " + file.getAbsolutePath() + ".", "File not found",
                                JOptionPane.ERROR_MESSAGE);
                    }
                }
            });
            fileMenu.insert(recentFile, 9 + i);
        }
    }

    private class JWorkbookListener extends JWorkbookEventAdapter
    {
        public void newWorkbook(JWorkbookEventObject eventObject)
        {
            _staticViewAction.setSelected(false);
            List worksheets = _workbook.getWorksheets();
            for (int i = 0; i < worksheets.size(); i++)
            {
                Worksheet worksheet = (Worksheet)worksheets.get(i);
                worksheet.addWorksheetEventListener(_worksheetEventLogger);
            }
            _workbook.addWorkbookEventListener(_workbookEventLogger);
            printLog("New workbook: " + eventObject.getWorkbook().getWorkbookName() + ".");
            _automationPanel.updateRecentFiles();
            enableMenuItems();
        }

        public void workbookOpened(JWorkbookEventObject eventObject)
        {
            _staticViewAction.setSelected(false);
            List worksheets = _workbook.getWorksheets();
            for (int i = 0; i < worksheets.size(); i++)
            {
                Worksheet worksheet = (Worksheet)worksheets.get(i);
                worksheet.addWorksheetEventListener(_worksheetEventLogger);
            }
            _workbook.addWorkbookEventListener(_workbookEventLogger);
            printLog("Opened workbook: " + eventObject.getWorkbook().getWorkbookName() + ".");
            _automationPanel.updateRecentFiles();
            enableMenuItems();
        }

        public void beforeWorkbookClose(JWorkbookEventObject eventObject) throws JWorkbookInterruptException
        {
            if (!_workbook.isSaved())
            {
                int msgBoxresult = MessageBox.show(new Wnd(_workbook),
                            "Excel confirmation",
                            "Do you want to save changes you made to '"+_workbook.getWorkbookName()+"'",
                            MessageBox.YESNOCANCEL);
                if (msgBoxresult==MessageBox.IDYES)
                {
                        java.awt.Window parent = SwingUtilities.getWindowAncestor(_workbook);
                        OpenSaveFileDialog saveDialog = new OpenSaveFileDialog(parent);
                        saveDialog.setFilter(FILTER);
                        boolean result = saveDialog.getSaveFileName();
                        if (result)
                        {
                            String newFileName = saveDialog.getFileName();
                            File newFile = new File(newFileName);
                            try
                            {
                                if (_workbook.saveCopyAs(newFile).getValue()!= HResult.S_OK)
                                {
                                    throw new JWorkbookInterruptException("Workbook closing canceled by user");
                                }
                            } catch (IOException e)
                            {
                                throw new JWorkbookInterruptException("Input/Output error",e);
                            }
                        }else
                        {
                            throw new JWorkbookInterruptException("Workbook closing canceled by user");
                        }
                }else if (msgBoxresult==MessageBox.IDCANCEL)
                {
                    throw new JWorkbookInterruptException("Workbook closing canceled by user");
                }
            }
            List worksheets = _workbook.getWorksheets();
            for (int i = 0; i < worksheets.size(); i++)
            {
                Worksheet worksheet = (Worksheet)worksheets.get(i);
                worksheet.removeWorksheetEventListener(_worksheetEventLogger);
            }
            _workbook.removeWorkbookEventListener(_workbookEventLogger);
            printLog("Workbook \"" + eventObject.getWorkbook().getWorkbookName() + "\" closed.");

            _staticViewAction.setSelected(false);
            _miPrint.setEnabled(false);
            _miPrintPreview.setEnabled(false);
            _miSaveAs.setEnabled(false);
            _miStaticView.setEnabled(false);

            _automationPanel.setSelection(null);
        }

        public void printPreviewStateChanged(JWorkbookEventObject eventObject)
        {
            boolean isPrintPreview = eventObject.getJWorkbook().isPrintPreview();
            if (isPrintPreview)
            {
                printLog("Workbook \"" + eventObject.getWorkbook().getWorkbookName() + "\" is in the print preview mode.");
            }
            else
            {
                printLog("The print preview mode is closed.");
            }

            JMenu fileMenu = _menuBar.getMenu(0);
            for (int i = 0; i < fileMenu.getItemCount() - 1; i++)
            {
                if (fileMenu.getItem(i) != null)
                {
                    fileMenu.getItem(i).setEnabled(!isPrintPreview);
                }
            }
            fileMenu = _menuBar.getMenu(1);
            fileMenu.getItem(3).setEnabled(!isPrintPreview);
        }
    }

    class EventLogAction extends AbstractAction
    {
        private boolean _selected;
        private JCheckBoxMenuItem _menuItem;

        public EventLogAction()
        {
            super("Show Event Log");
            _selected = true;
        }

        public void setMenuItem(JCheckBoxMenuItem menuItem)
        {
            _menuItem = menuItem;
        }

        public void actionPerformed(ActionEvent e)
        {
            _selected = !_selected;
            _eventLogPane.setVisible(_selected);
            if (_selected)
            {
                _splitPane.setDividerLocation(0.8);
            }
            _menuItem.setSelected(_selected);
        }
    }

    class AutomationBarAction extends AbstractAction
    {
        private boolean _selected;
        private JCheckBoxMenuItem _menuItem;

        public AutomationBarAction()
        {
            super("Show Automation Bar");
            _selected = true;
        }

        public void setMenuItem(JCheckBoxMenuItem menuItem)
        {
            _menuItem = menuItem;
        }

        public void actionPerformed(ActionEvent e)
        {
            _selected = !_selected;
            _automationBarPane.setVisible(_selected);
            if (_selected)
            {
                _subSplitPane.setDividerLocation(_subSplitPane.getWidth() - 246);
            }
            _menuItem.setSelected(_selected);
        }
    }

    class StaticViewAction extends AbstractAction
    {
        private boolean _selected;
        private JCheckBoxMenuItem _menuItem;

        public StaticViewAction()
        {
            super("Static view");
            _selected = false;
        }

        public void setMenuItem(JCheckBoxMenuItem menuItem)
        {
            _menuItem = menuItem;
        }

        public void actionPerformed(ActionEvent e)
        {
            _selected = !_selected;
            _menuItem.setSelected(_selected);
            _workbook.setStaticMode(_selected);
            enableMenuItems();
        }

        public void setSelected(boolean value)
        {
            _selected = value;
            _menuItem.setSelected(value);
            enableMenuItems();
        }

        private void enableMenuItems()
        {
            _miPrint.setEnabled(!_selected);
            _miPrintPreview.setEnabled(!_selected);
            _miSaveAs.setEnabled(!_selected);
            _miNew.setEnabled(!_selected);
            _miOpen.setEnabled(!_selected);
        }
    }

    class ShowAboutAction extends AbstractAction
    {
        java.awt.Window _owner;

        ShowAboutAction(java.awt.Window owner)
        {
            super("About JExcel");
            _owner = owner;
        }

        public void actionPerformed(ActionEvent e)
        {
            AboutDialog aboutDialog = new AboutDialog((Frame)_owner,
                    "About JExcel Demo",
                    "<html><H2>JExcel Demo</H2><H3></H3>" +
                    "Copyright  2000-2006 TeamDev Ltd.",
                    new ImageIcon(getClass().getResource("res/logo.jpg")));
            aboutDialog.setVisible(true);
        }
    }


    class WorksheetEventLogger implements WorksheetEventListener
    {
        public void activated(WorksheetEventObject eventObject)
        {
            printLog(eventObject.getWorksheet().getName() + " activated.");
        }

        public void beforeDoubleClick(WorksheetEventObject eventObject)
        {
            printLog("Double click on " + eventObject.getCell().getAddress() + ".");
        }

        public void beforeRightClick(WorksheetEventObject eventObject)
        {
            printLog("Right click on " + eventObject.getRange().getAddress() + ".");
        }

        public void changed(WorksheetEventObject eventObject)
        {
            printLog(eventObject.getRange().getAddress() + " is changed.");
        }

        public void deactivated(WorksheetEventObject eventObject)
        {
            printLog(eventObject.getWorksheet().getName() + " deactivated.");
        }

        public void selectionChanged(WorksheetEventObject eventObject)
        {
            Range range = eventObject.getRange();
            _automationPanel.setSelection(range);
            printLog(range.getAddress() + " selected.");
        }
    }

    class WorkbookEventLogger extends WorkbookEventAdapter
    {
        public void newSheet(WorkbookEventObject eventObject)
        {
            Worksheet worksheet = eventObject.getWorksheet();
            printLog("\"" + worksheet.getName() + "\" sheet is added to \"" +
                    eventObject.getWorkbook().getWorkbookName() + "\" workbook.");
            worksheet.addWorksheetEventListener(_worksheetEventLogger);
        }
    }

    private static void setupFrameIcon(java.awt.Window owner)
    {
        try
        {
            final Wnd winWnd = new Wnd(owner);
            final com.jniwrapper.win32.gdi.Icon bigIcon = new com.jniwrapper.win32.gdi.Icon(JExcelDemo.class.getResourceAsStream("res/jexcel.ico"), new Dimension(32, 32));
            final com.jniwrapper.win32.gdi.Icon smallIcon = new com.jniwrapper.win32.gdi.Icon(JExcelDemo.class.getResourceAsStream("res/jexcel.ico"), new Dimension(16, 16));
            winWnd.setWindowIcon(smallIcon, com.jniwrapper.win32.gdi.Icon.IconType.SMALL);
            winWnd.setWindowIcon(bigIcon, com.jniwrapper.win32.gdi.Icon.IconType.BIG);
        }
        catch (Exception e)
        {
            LOG.error("", e);
        }
    }


    private static void setupLF()
    {
        try
        {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());

            UIDefaults defaults = UIManager.getDefaults();
            Font tahoma = new Font("Tahoma", 0, 11);
            defaults.put("MenuItem.font", tahoma);
            defaults.put("TabbedPane.font", tahoma);
            defaults.put("TextField.font", tahoma);
            defaults.put("Label.font", tahoma);
            defaults.put("Button.font", tahoma);
            defaults.put("ComboBox.font", tahoma);
            defaults.put("Panel.font", tahoma);
        }
        catch (Exception e)
        {
            LOG.error("", e);
        }
    }

    static
    {
        setupLF();
        JPopupMenu.setDefaultLightWeightPopupEnabled(false);
        JFrame.setDefaultLookAndFeelDecorated(true);
    }

    public static void main(String[] args) throws ExcelException
    {
        final JExcelDemo excelDemo = new JExcelDemo();
        excelDemo.addWindowListener(new WindowAdapter()
        {
            public void windowOpened(WindowEvent e)
            {
                setupFrameIcon(excelDemo);
            }
        });
        excelDemo.setExtendedState(JFrame.MAXIMIZED_BOTH);
        excelDemo.setVisible(true);
    }
}
