create or replace package SVCO.repository authid current_user is
  /*
  * "Schema Version Control for Oracle" (SVCO)
  *      http://www.sumsoftsolutions.com
  * PLEASE READ LICENSE AGREEMENT CAREFULLY BEFORE USING THIS SOFTWARE
  *
  * PURPOSE
  * This package is the user interface to a basic repository functions.
  *
  * SECURITY
  * This package is granted to SVCO_REP_OPERATOR and SVCO_REP_ADMIN roles.
  */

  --
  -- DESCRIPTION
  -- This procedure creates a new schema revision
  --
  -- PARAMETERS
  -- Revision - Return new revision number if succeeded otherwise 
  --            last revision number if no difference found else 
  --            null if SchemaName not found or no supported objects found.
  --
  -- RevisionCreated - Return TRUE if a new revision created, otherwise - FALSE.
  --
  -- SchemaName - Schema name (user name) for which you create a new revision,
  --              if not given - sys_context('USERENV', 'CURRENT_SCHEMA').
  --              Case insensitive.
  --
  -- Version - Your own identifier for a new revision, max size 20 characters
  --
  -- Comments - Your own comments for a new revision, max size 2KByte
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- This procedure performs a transaction.
  --
  procedure CreateRevision(Revision        out number
                          ,RevisionCreated out varchar2
                          ,SchemaName      in varchar2 default null
                          ,Version         in varchar2 default null
                          ,Comments        in varchar2 default null);

  --
  -- DESCRIPTION
  -- This record type is needed for cursor type refDifference.
  --
  type Difference_curtype is record(
     object_type   version_objects.object_type%type
    ,object_name   version_objects.object_name%type
    ,object_status varchar2(10));

  --
  -- DESCRIPTION
  -- This cursor type required for parameter in procedure GetRevisionsDifference.
  --
  type refDifference is ref cursor return Difference_curtype;

  --
  -- DESCRIPTION
  -- This procedure returns a cursor with objects that are changed in both revisions
  -- inside of one schema.
  --
  -- PARAMETERS
  -- SchemaName - Schema name (user name),
  --              if not given - sys_context('USERENV', 'CURRENT_SCHEMA').
  --              Case insensitive.
  --
  -- Revision1 - Source revision number, if not given - last revision.
  --
  -- Version1 - Source user identifier for revision, if not given - last revision.
  --
  -- Revision2 - Destination revision number, if not given - last revision.
  --
  -- Version2 - Destination user identifier for revision, if not given - last revision.
  --
  -- Difference - Result cursor with changed objects - their names and their types.
  --              Empty cursor if no difference found.
  --
  -- EXCEPTIONS
  -- exceptions_.exc_no_rights
  -- exceptions_.exc_no_data_found
  -- exceptions_.exc_too_many_rows
  -- exceptions_.exc_unhandled
  --
  -- TRANSACTION
  -- No.
  --
  procedure GetRevisionsDifference(SchemaName in varchar2 default null
                                  ,Revision1  in number default null
                                  ,Version1   in varchar2 default null
                                  ,Revision2  in number default null
                                  ,Version2   in varchar2 default null
                                  ,Difference out refDifference);

end repository;
/
