/*
 *   @(#) MPEG-I Video Decoder 1.0 Demicron (demicron@demicron.com)
 *
 *   AD_LayerIDecoder.java   2002-08-20
 *
 *   Copyright (C) 2002  Demicron
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */





/**
 * Implements decoding of MPEG Audio Layer I frames.
 */

class AD_LayerIDecoder implements AD_FrameDecoder
{
    protected BufferedBitStream	 		stream;
    protected AD_Header 			header;
    protected AD_SynthesisFilter 	filter1, filter2;
    protected AD_Obuffer 			buffer;
    protected int 				which_channels;
    protected int				mode;

    protected int				num_subbands;
    protected Subband[]			subbands = new Subband[32];
    protected AD_Crc16[]			crc = new AD_Crc16[1];

    public AD_LayerIDecoder() {
    }

    public void create(BufferedBitStream stream0, AD_Header header0,
		       AD_SynthesisFilter filtera, AD_SynthesisFilter filterb,
		       AD_Obuffer buffer0, int which_ch0) {
		stream         = stream0;
		header         = header0;
		filter1        = filtera;
		filter2        = filterb;
		buffer         = buffer0;
		which_channels = which_ch0;

		num_subbands = header.number_of_subbands();
		mode = header.mode();
		createSubbands();
    }


    public void decodeFrame() throws InterruptedException {
		resetSubbands();
		readAllocation();
		readScaleFactorSelection();

		if ((crc != null) || header.checksum_ok()) {
			readScaleFactors();
			readSampleData();
		}
    }


    protected void resetSubbands() {
		for (int i = 0; i < num_subbands; i++) {
			subbands[i].reset();
		}
    }

    protected void createSubbands() {
		int i;
		if (mode == AD_Header.SINGLE_CHANNEL) {

			for (i = 0; i < num_subbands; ++i)
				subbands[i] = new SubbandLayer1(i);

		} else if (mode == AD_Header.JOINT_STEREO) {

			for (i = 0; i < header.intensity_stereo_bound(); ++i) {
				subbands[i] = new SubbandLayer1Stereo(i);
			}
			for (; i < num_subbands; ++i) {
				subbands[i] = new SubbandLayer1IntensityStereo(i);
			}

		} else {

			for (i = 0; i < num_subbands; ++i)
				subbands[i] = new SubbandLayer1Stereo(i);
		}
    }

    protected void readAllocation() throws InterruptedException {

		// start to read audio data:
		for (int i = 0; i < num_subbands; ++i) {
			// subbands[i].read_allocation(stream, header, null);
			// BYOFix: use this to turn crc off
			subbands[i].read_allocation(stream, header, crc[0]);
		}
    }

    protected void readScaleFactorSelection() throws InterruptedException {
		// scale factor selection not present for layer I.
    }

    protected void readScaleFactors() throws InterruptedException {
		for (int i = 0; i < num_subbands; ++i)
			subbands[i].read_scalefactor(stream, header);
    }

    protected void readSampleData() throws InterruptedException {
		boolean read_ready = false;
		boolean write_ready = false;
		int mode = header.mode();
		int i;
		do {
			for (i = 0; i < num_subbands; ++i) {
				// System.out.println("i: " + i + " ns: " + num_subbands);
				read_ready = subbands[i].read_sampledata(stream);
			}
			do {
				for (i = 0; i < num_subbands; ++i)
					write_ready = subbands[i].put_next_sample(which_channels, filter1, filter2);

				filter1.calculate_pcm_samples(buffer);
				if ((which_channels == AD_OutputChannels.BOTH_CHANNELS) && (mode != AD_Header.SINGLE_CHANNEL))
					filter2.calculate_pcm_samples(buffer);
			} while (!write_ready);
		} while (!read_ready);
    }

    /**
     * Abstract base class for subband classes of layer I and II
     */
    static abstract class Subband
    {
		/*
		 *  Changes from version 1.1 to 1.2:
		 *    - array size increased by one, although a scalefactor with index 63
		 *      is illegal (to prevent segmentation faults)
		 */
		// Scalefactors for layer I and II, Annex 3-B.1 in ISO/IEC DIS 11172:
		public static final float scalefactors[] =
		{
			2.00000000000000f, 1.58740105196820f, 1.25992104989487f, 1.00000000000000f,
			0.79370052598410f, 0.62996052494744f, 0.50000000000000f, 0.39685026299205f,
			0.31498026247372f, 0.25000000000000f, 0.19842513149602f, 0.15749013123686f,
			0.12500000000000f, 0.09921256574801f, 0.07874506561843f, 0.06250000000000f,
			0.04960628287401f, 0.03937253280921f, 0.03125000000000f, 0.02480314143700f,
			0.01968626640461f, 0.01562500000000f, 0.01240157071850f, 0.00984313320230f,
			0.00781250000000f, 0.00620078535925f, 0.00492156660115f, 0.00390625000000f,
			0.00310039267963f, 0.00246078330058f, 0.00195312500000f, 0.00155019633981f,
			0.00123039165029f, 0.00097656250000f, 0.00077509816991f, 0.00061519582514f,
			0.00048828125000f, 0.00038754908495f, 0.00030759791257f, 0.00024414062500f,
			0.00019377454248f, 0.00015379895629f, 0.00012207031250f, 0.00009688727124f,
			0.00007689947814f, 0.00006103515625f, 0.00004844363562f, 0.00003844973907f,
			0.00003051757813f, 0.00002422181781f, 0.00001922486954f, 0.00001525878906f,
			0.00001211090890f, 0.00000961243477f, 0.00000762939453f, 0.00000605545445f,
			0.00000480621738f, 0.00000381469727f, 0.00000302772723f, 0.00000240310869f,
			0.00000190734863f, 0.00000151386361f, 0.00000120155435f, 0.00000000000000f /* illegal scalefactor */
		};

		public abstract void read_allocation (BufferedBitStream stream, AD_Header header, AD_Crc16 crc) throws InterruptedException;
		public abstract void read_scalefactor (BufferedBitStream stream, AD_Header header) throws InterruptedException;
		public abstract boolean read_sampledata (BufferedBitStream stream) throws InterruptedException;
		public abstract boolean put_next_sample (int channels, AD_SynthesisFilter filter1, AD_SynthesisFilter filter2);

		public abstract void reset();

    };

    /**
     * Class for layer I subbands in single channel mode.
     * Used for single channel mode
     * and in derived class for intensity stereo mode
     */
    static class SubbandLayer1 extends Subband {

		// Factors and offsets for sample requantization
		public static final float table_factor[] = {
			0.0f, (1.0f/2.0f) * (4.0f/3.0f), (1.0f/4.0f) * (8.0f/7.0f), (1.0f/8.0f) * (16.0f/15.0f),
			(1.0f/16.0f) * (32.0f/31.0f), (1.0f/32.0f) * (64.0f/63.0f), (1.0f/64.0f) * (128.0f/127.0f),
			(1.0f/128.0f) * (256.0f/255.0f), (1.0f/256.0f) * (512.0f/511.0f),
			(1.0f/512.0f) * (1024.0f/1023.0f), (1.0f/1024.0f) * (2048.0f/2047.0f),
			(1.0f/2048.0f) * (4096.0f/4095.0f), (1.0f/4096.0f) * (8192.0f/8191.0f),
			(1.0f/8192.0f) * (16384.0f/16383.0f), (1.0f/16384.0f) * (32768.0f/32767.0f)
		};

		public static final float table_offset[] = {
			0.0f, ((1.0f/2.0f)-1.0f) * (4.0f/3.0f), ((1.0f/4.0f)-1.0f) * (8.0f/7.0f), ((1.0f/8.0f)-1.0f) * (16.0f/15.0f),
			((1.0f/16.0f)-1.0f) * (32.0f/31.0f), ((1.0f/32.0f)-1.0f) * (64.0f/63.0f), ((1.0f/64.0f)-1.0f) * (128.0f/127.0f),
			((1.0f/128.0f)-1.0f) * (256.0f/255.0f), ((1.0f/256.0f)-1.0f) * (512.0f/511.0f),
			((1.0f/512.0f)-1.0f) * (1024.0f/1023.0f), ((1.0f/1024.0f)-1.0f) * (2048.0f/2047.0f),
			((1.0f/2048.0f)-1.0f) * (4096.0f/4095.0f), ((1.0f/4096.0f)-1.0f) * (8192.0f/8191.0f),
			((1.0f/8192.0f)-1.0f) * (16384.0f/16383.0f), ((1.0f/16384.0f)-1.0f) * (32768.0f/32767.0f)
		};

		protected int		 subbandnumber;
		protected int		 samplenumber;
		protected int		 allocation;
		protected float		 scalefactor;
		protected int		 samplelength;
		protected float		 sample;
		protected float		 factor, offset;

		/**
		 * Construtor.
		 */
		public SubbandLayer1(int subbandnumber) {
			this.subbandnumber = subbandnumber;
			samplenumber = 0;
		}

		public void reset() {
			samplenumber = 0;
			allocation = 0;
			scalefactor = 0f;
			samplelength = 0;
			sample = 0f;
			factor = 0f;
			offset = 0f;
		}

		/**
		 *
		 */
		public void read_allocation(BufferedBitStream stream, AD_Header header, AD_Crc16 crc) throws InterruptedException {
			if ((allocation = stream.getBits (4)) == 15) ;
			//	 cerr << "WARNING: stream contains an illegal allocation!\n";
			// MPEG-stream is corrupted!
			if (crc != null)
				crc.add_bits (allocation, 4);
			if (allocation != 0) {
				samplelength = allocation + 1;
				factor = table_factor[allocation];
				offset = table_offset[allocation];
			}
		}

		/**
		 *
		 */
		public void read_scalefactor(BufferedBitStream stream, AD_Header header) throws InterruptedException {
			if (allocation != 0) scalefactor = scalefactors[stream.getBits(6)];
		}

		/**
		 *
		 */
		public boolean read_sampledata(BufferedBitStream stream) throws InterruptedException {
			if (allocation != 0) {
				sample = (float) (stream.getBits(samplelength));
			}
			if (++samplenumber == 12) {
				samplenumber = 0;
				return true;
			}
			return false;
		}

		/**
		 *
		 */
		public boolean put_next_sample(int channels, AD_SynthesisFilter filter1, AD_SynthesisFilter filter2) {
			if ((allocation !=0) && (channels != AD_OutputChannels.RIGHT_CHANNEL)) {
				float scaled_sample = (sample * factor + offset) * scalefactor;
				filter1.input_sample (scaled_sample, subbandnumber);
			}
			return true;
		}
    };

    /**
     * Class for layer I subbands in joint stereo mode.
     */
    static class SubbandLayer1IntensityStereo extends SubbandLayer1 {
		protected float 		channel2_scalefactor;

		/**
		 * Constructor
		 */
		public SubbandLayer1IntensityStereo(int subbandnumber) {
			super(subbandnumber);
		}

		/**
		 *
		 */
		public void read_allocation(BufferedBitStream stream, AD_Header header, AD_Crc16 crc) throws InterruptedException {
			super.read_allocation (stream, header, crc);
		}

		/**
		 *
		 */
		public void read_scalefactor (BufferedBitStream stream, AD_Header header) throws InterruptedException {
			if (allocation != 0) {
				scalefactor = scalefactors[stream.getBits(6)];
				channel2_scalefactor = scalefactors[stream.getBits(6)];
			}
		}

		/**
		 *
		 */
		public boolean read_sampledata(BufferedBitStream stream) throws InterruptedException {
			return super.read_sampledata (stream);
		}

		/**
		 *
		 */
		public boolean put_next_sample (int channels, AD_SynthesisFilter filter1, AD_SynthesisFilter filter2) {
			if (allocation !=0 ) {
				sample = sample * factor + offset;		// requantization
				if (channels == AD_OutputChannels.BOTH_CHANNELS) {
					float sample1 = sample * scalefactor,
						sample2 = sample * channel2_scalefactor;
					filter1.input_sample(sample1, subbandnumber);
					filter2.input_sample(sample2, subbandnumber);
				}
				else if (channels == AD_OutputChannels.LEFT_CHANNEL) {
					float sample1 = sample * scalefactor;
					filter1.input_sample(sample1, subbandnumber);
				} else {
					float sample2 = sample * channel2_scalefactor;
					filter1.input_sample(sample2, subbandnumber);
				}
			}
			return true;
		}
    };

    /**
     * Class for layer I subbands in stereo mode.
     */
    static class SubbandLayer1Stereo extends SubbandLayer1 {
		protected int 		channel2_allocation;
		protected float		channel2_scalefactor;
		protected int 		channel2_samplelength;
		protected float	 	channel2_sample;
		protected float  	channel2_factor, channel2_offset;


		/**
		 * Constructor
		 */
		public SubbandLayer1Stereo(int subbandnumber) {
			super(subbandnumber);
		}

		/**
		 *
		 */
		public void read_allocation (BufferedBitStream stream, AD_Header header, AD_Crc16 crc) throws InterruptedException {
			allocation = stream.getBits(4);
			channel2_allocation = stream.getBits(4);
			if (crc != null) {
				crc.add_bits(allocation, 4);
				crc.add_bits(channel2_allocation, 4);
			}
			if (allocation != 0) {
				samplelength = allocation + 1;
				factor = table_factor[allocation];
				offset = table_offset[allocation];
			}
			if (channel2_allocation != 0) {
				channel2_samplelength = channel2_allocation + 1;
				channel2_factor = table_factor[channel2_allocation];
				channel2_offset = table_offset[channel2_allocation];
			}
		}

		/**
		 *
		 */
		public void read_scalefactor(BufferedBitStream stream, AD_Header header) throws InterruptedException {
			if (allocation != 0) scalefactor = scalefactors[stream.getBits(6)];
			if (channel2_allocation != 0) channel2_scalefactor = scalefactors[stream.getBits(6)];
		}

		/**
		 *
		 */
		public boolean read_sampledata (BufferedBitStream stream) throws InterruptedException {
			boolean returnvalue = super.read_sampledata(stream);
			if (channel2_allocation != 0) {
				channel2_sample = (float) (stream.getBits(channel2_samplelength));
			}
			return(returnvalue);
		}

		/**
		 *
		 */
		public boolean put_next_sample(int channels, AD_SynthesisFilter filter1, AD_SynthesisFilter filter2) {
			super.put_next_sample (channels, filter1, filter2);
			if ((channel2_allocation != 0) && (channels != AD_OutputChannels.LEFT_CHANNEL)) {
				float sample2 = (channel2_sample * channel2_factor + channel2_offset) *
					channel2_scalefactor;
				if (channels == AD_OutputChannels.BOTH_CHANNELS)
					filter2.input_sample (sample2, subbandnumber);
				else
					filter1.input_sample (sample2, subbandnumber);
			}
			return true;
		}
    };
}
