// HexEditView.h : interface of the CHexEditView class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

/////////////////////////////////////////////////////////////////////////////

#include "ScrView.h"
#include "Partition.h"

#include <vector>
using namespace std;

class CHexEditDoc;

enum undo_type
{
    undo_unknown = '0', // Help detect errors
    undo_change  = 'C', // Document changed (actual undo done in doc)
    undo_move    = 'M', // Caret moved
    undo_sel     = '+', // Selection (when combined with undo_move)
    undo_swap    = 'S', // Swap between hex and char areas
    undo_disp    = 'D', // Toggle display of chars (on right of window)
    undo_ebcdic  = 'E', // Toggle EBCDIC display
    undo_font    = 'F', // Font changed
    undo_autofit = 'A', // Autofit toggled
    undo_setmark = 'X', // Mark set - keep track of previous mark
    undo_markchar= 'x', // Mark set - keep track of hex/char area
    undo_overtype= 'o', // Overtype toggled
    undo_readonly= 'R', // Read only mode toggled
    undo_rowsize = 'r', // Row size (display columns) changed
    undo_group_by= 'g', // Column grouping of display changed
    undo_offset  = 'f', // Offset of display changed
    undo_graphic = 'G', // Display graphic chars toggled
    undo_oem     = 'O', // Display IBM/OEM chars toggled
};

#pragma pack(push, 1)
    struct view_undo
    {
        unsigned char utype:7;
        unsigned char previous_too:1; // Used to group several undo's together
        union
        {
            long address;       // Where moved from (undo_move) OR
                                // selection start (undo_move) and end (undo_sel)
            LOGFONT *plf;       // Describes previous font
            long rowsize;       // no of columns before autofit turned on
            BOOL flag;          // swap, disp, markchar etc flag
        };
#ifndef NDEBUG
        int index;              // Document undo index (undo_change)
#endif

        // General constructor + default constructor (required by vector)
        view_undo(enum undo_type u = undo_unknown, BOOL p = FALSE)
            { utype = u; plf = NULL; previous_too = p; }
        // Copy constructor
        view_undo(const view_undo &from)
        {
            ASSERT(from.utype != undo_unknown);
            utype = from.utype; previous_too = from.previous_too;
            address = from.address;     // Naughty way to copy union
#ifndef NDEBUG
            index = from.index;
#endif
            if (from.utype == undo_font && from.plf != NULL)
            {
                plf = new LOGFONT;
                *plf = *from.plf;
            }
        }
        // Assignment copy operator
        view_undo &operator=(const view_undo &from)
        {
            if (&from != this)
            {
                ASSERT(from.utype != undo_unknown);
                // First destroy existin object if nec.
                if (utype == undo_font && plf != NULL)
                    delete plf;

                // Recreate this object from assigned value
                utype = from.utype; previous_too = from.previous_too;
                if (from.utype == undo_font && from.plf != NULL)
                {
                    // Since its from free store we need to keep a separate copy
                    plf = new LOGFONT;
                    *plf = *from.plf;
                }
                else
                    address = from.address;     // Naughty way to copy union
#ifndef NDEBUG
                index = from.index;
#endif
            }
            return *this;
        }
        ~view_undo()
        {
            ASSERT(utype != undo_unknown);
            // Delete any memory allocations specific to undo types
            if (utype == undo_font && plf != NULL)
                delete plf;
        }
        // These are required by vector<> under VC++ 5 even though not used
        operator==(const view_undo &) const { return false; }
        operator<(const view_undo &) const { return false; }
    };
#pragma pack(pop)

// Different types of undo's handle by the view
class CHexEditView : public CScrView
{
    friend class CHexEditApp;

protected: // create from serialization only
        CHexEditView();
        DECLARE_DYNCREATE(CHexEditView)

// Attributes
public:
    enum { max_buf = 1024 };
    CHexEditDoc* GetDocument();
    long GetMarkOffset() const  // offset of caret from marked position
    {
        return pos2addr(GetCaret()) - mark_;
    }
    long GetPos() const         // Address in file of caret position
    {
        return pos2addr(GetCaret());
    }
    BOOL GetSelAddr(long &start_addr, long &end_addr)
    {
        CPoint start, end;
        BOOL retval = GetSel(start, end);
        start_addr = pos2addr(start);
        end_addr   = pos2addr(end);
        return retval;
    }
    long GetMark() { return mark_; }
    void SetMark(long new_mark);
    BOOL ReadOnly() const { return readonly_; }
    BOOL OverType() const { return overtype_; }
    BOOL EbcdicMode() const { return ebcdic_; }

// Operations
    void recalc_display();      // Recalculate display parameters
    void do_mouse(CPoint dev_down, CSize doc_dist) ;
    void do_shift_mouse(CPoint dev_down, CSize doc_dist) ;
    void do_autofit(int state = -1);
    void do_chartoggle(int state = -1) ;
    void allow_mods();
    void do_insert();
    void do_char(UINT nChar);
    void do_read(CString file_name);
    void do_font(LOGFONT *plf);
    BOOL Search(const char *ss, int tt = 1, BOOL forward = TRUE);
    long GoAddress(long start_addr, long end_addr = -1);
    void SaveMove();
    void MoveToAddress(long start_addr, long end_addr = -1, long prev_start = -1, long prev_end = -1);
    void ToggleInsert() { OnInsert(); }
    void AllowMods() { OnAllowMods(); }
    bool CopyToClipboard();
    virtual BOOL MovePos(UINT nChar, UINT nRepCnt, BOOL, BOOL, BOOL);
    void StoreOptions();

public:

// Overrides
    virtual void DisplayCaret();
    CFont *SetFont(CFont *ff);

        // ClassWizard generated virtual function overrides
        //{{AFX_VIRTUAL(CHexEditView)
	public:
        virtual void OnDraw(CDC* pDC);  // overridden to draw this view
        virtual void OnInitialUpdate();
        virtual void OnPrepareDC(CDC* pDC, CPrintInfo* pInfo = NULL);
	protected:
        virtual BOOL OnPreparePrinting(CPrintInfo* pInfo);
        virtual void OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo);
        virtual void OnEndPrinting(CDC* pDC, CPrintInfo* pInfo);
        virtual void OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint);
        virtual void OnPrint(CDC* pDC, CPrintInfo* pInfo);
        virtual void OnEndPrintPreview(CDC* pDC, CPrintInfo* pInfo, POINT point, CPreviewView* pView);
	virtual void OnActivateView(BOOL bActivate, CView* pActivateView, CView* pDeactiveView);
	virtual BOOL PreCreateWindow(CREATESTRUCT& cs);
	//}}AFX_VIRTUAL

// Implementation
public:
        virtual ~CHexEditView();
#ifdef _DEBUG
        virtual void AssertValid() const;
        virtual void Dump(CDumpContext& dc) const;
#endif

protected:
    virtual void ValidateCaret(CPoint &pos, BOOL inside=TRUE);
    virtual void ValidateScroll(CPoint &pos, BOOL strict=FALSE);
    virtual void InvalidateRange(CPoint start, CPoint end);
    virtual void DoInvalidate();
    virtual void DoInvalidateRect(LPCRECT lpRect);
    virtual void DoInvalidateRgn(CRgn* pRgn);
    virtual void DoScrollWindow(int xx, int yy);
    virtual void DoUpdateWindow();
    virtual void DoHScroll(int total, int page, int pos);
    virtual void DoVScroll(int total, int page, int pos);
    void DoUpdate();

// Generated message map functions
// Make these public so they can be accessed by CHexEditApp::macro_play
// protected:
public:
        //{{AFX_MSG(CHexEditView)
        afx_msg void OnSize(UINT nType, int cx, int cy);
        afx_msg void OnAddrToggle();
        afx_msg void OnUpdateAddrToggle(CCmdUI* pCmdUI);
        afx_msg void OnGraphicToggle();
        afx_msg void OnUpdateGraphicToggle(CCmdUI* pCmdUI);
        afx_msg void OnCharToggle();
        afx_msg void OnUpdateCharToggle(CCmdUI* pCmdUI);
        afx_msg void OnFont();
        afx_msg void OnAutoFit();
        afx_msg void OnUpdateAutofit(CCmdUI* pCmdUI);
        afx_msg void OnAscEbc();
        afx_msg void OnUpdateAscEbc(CCmdUI* pCmdUI);
        afx_msg void OnControl();
        afx_msg void OnUpdateControl(CCmdUI* pCmdUI);
        afx_msg void OnChar(UINT nChar, UINT nRepCnt, UINT nFlags);
        afx_msg void OnSetFocus(CWnd* pOldWnd);
        afx_msg void OnKillFocus(CWnd* pNewWnd);
        afx_msg void OnDestroy();
        afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
        afx_msg void OnMark();
        afx_msg void OnGotoMark();
        afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
        afx_msg void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
        afx_msg void OnEditUndo();
        afx_msg void OnSearchHex();
        afx_msg void OnSearchAscii();
        afx_msg void OnUpdateEditUndo(CCmdUI* pCmdUI);
        afx_msg void OnSearchForw();
        afx_msg void OnSearchBack();
        afx_msg void OnAllowMods();
        afx_msg void OnUpdateAllowMods(CCmdUI* pCmdUI);
        afx_msg void OnControlToggle();
        afx_msg void OnInsert();
        afx_msg void OnUpdateInsert(CCmdUI* pCmdUI);
        afx_msg void OnSearchIcase();
        afx_msg void OnUpdateSearch(CCmdUI* pCmdUI);
        afx_msg void OnEditCompare();
        afx_msg void OnWindowNext();
        afx_msg void OnUpdateEditCompare(CCmdUI* pCmdUI);
        afx_msg void OnContextMenu(CWnd* pWnd, CPoint point);
        afx_msg void OnRButtonDown(UINT nFlags, CPoint point);
        afx_msg void OnIncByte();
        afx_msg void OnInc16bit();
        afx_msg void OnInc32bit();
        afx_msg void OnInc64bit();
        afx_msg void OnDecByte();
        afx_msg void OnDec16bit();
        afx_msg void OnDec32bit();
        afx_msg void OnDec64bit();
        afx_msg void OnFlip16bit();
        afx_msg void OnFlip32bit();
        afx_msg void OnFlip64bit();
        afx_msg void OnUpdateByte(CCmdUI* pCmdUI);
        afx_msg void OnUpdate16bit(CCmdUI* pCmdUI);
        afx_msg void OnUpdate32bit(CCmdUI* pCmdUI);
        afx_msg void OnUpdate64bit(CCmdUI* pCmdUI);
        afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
        afx_msg void OnSelectAll();
        afx_msg void OnEditCopy();
        afx_msg void OnEditCut();
        afx_msg void OnEditPaste();
        afx_msg void OnUpdateTextPaste(CCmdUI* pCmdUI);
        afx_msg void OnUpdateClipboard(CCmdUI* pCmdUI);
        afx_msg void OnSearchSel();
        afx_msg void OnUpdateUnicodePaste(CCmdUI* pCmdUI);
        afx_msg BOOL OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message);
        afx_msg void OnFontDec();
        afx_msg void OnFontInc();
        afx_msg void OnUpdateFontDec(CCmdUI* pCmdUI);
        afx_msg void OnUpdateFontInc(CCmdUI* pCmdUI);
        afx_msg void OnUpdateEditCut(CCmdUI* pCmdUI);
        afx_msg void OnPasteAscii();
        afx_msg void OnPasteEbcdic();
        afx_msg void OnPasteUnicode();
        afx_msg void OnCopyCchar();
        afx_msg void OnCopyHex();
        afx_msg void OnEditWriteFile();
        afx_msg void OnUpdateReadFile(CCmdUI* pCmdUI);
        afx_msg void OnReadFile();
        afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
        afx_msg void OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
        afx_msg void OnExtendToMark();
        afx_msg void OnSwapMark();
        afx_msg void OnRedraw();
        afx_msg void OnScrollDown();
        afx_msg void OnScrollUp();
        afx_msg void OnSwap();
        afx_msg void OnStartLine();
        afx_msg void OnDel();
        afx_msg void OnUpdateSwap(CCmdUI* pCmdUI);
	afx_msg void OnOemToggle();
	afx_msg void OnUpdateOemToggle(CCmdUI* pCmdUI);
	afx_msg BOOL OnMouseWheel(UINT nFlags, short zDelta, CPoint pt);
	//}}AFX_MSG
        afx_msg void OnFilePrint();
        afx_msg void OnFilePrintPreview();
        afx_msg void OnJumpHexAddr();
        DECLARE_MESSAGE_MAP()

private:
    enum { max_font_size = 100 };
#ifdef _DEBUG
    // Use smaller buffers to make it more likely to catch certain types of bugs
    enum { search_buf_len = 128, compare_buf_len = 64, clipboard_buf_len = 1024 };
#else
    enum { search_buf_len = 32768, compare_buf_len = 4096, clipboard_buf_len = 1024 };
#endif
    int hex_pos(int column, int width=0) const // get X coord of hex display column
    {
        ASSERT(column <= rowsize_);
        if (width == 0) width = text_width_;
        return (addr_width_ + column*3)*width + (column/group_by_)*width;
    }
    int char_pos(int column, int width=0) const // get X coord of ASCII/EBCDIC display column
    {
        ASSERT(column <= rowsize_);
        if (width == 0) width = text_width_;
        return (addr_width_ + rowsize_*3)*width + (rowsize_/group_by_)*width +
            column*width /* + ((column/4)*width)/2 */;
    }
    int pos_hex(int, BOOL inside = FALSE) const;  // Closest hex display col given X
    int pos_char(int, BOOL inside = FALSE) const; // Closest char area col given X
    CPoint addr2pos(long address) const; // Convert byte address in doc to display position
    long pos2addr(CPoint pos, BOOL inside = TRUE) const; // Convert a display position to closest address
    void show_prop(long address = -1); // Show current properties (modeless dlg)
    void show_pos(long address = -1); // Show position (dlg bar) given doc address
    void show_calc();           // Show view info in state of calculator buttons
    void invalidate_addr_range(long, long); // Invalidate display for address range
    BOOL do_undo();             // Undo one from top of undo stack
    BOOL check_ro(const char *); // Check if file is read only when changes attempted
    void load_bitmaps();        // Redraw buttons for this view
    BOOL do_search(BOOL forward = TRUE);
    BOOL search_forw(const unsigned char *ss, size_t, BOOL, int);
    BOOL search_back(const unsigned char *ss, size_t, BOOL, int);
    void view_context(CPoint point) ; // Display context menu
    CMDIChildWnd *comp_window(); // Compare windows
    void move_dlgs();           // Move modeless dialogs if they cover the caret
    void change_rowsize(int rowsize);
    void change_group_by(int group_by);
    void change_offset(int offset);
    void set_colours(const vector<partn> &c);

    CSize win_size_;            // Current window dimensions (device coords)
    CFont *pfont_;              // Current display font
    CBrush *pbrush_;            // Current brush (used with PatBlt)
    LOGFONT lf_, oem_lf_;       // Normal font and font for IBM/OEM graphics chars
    unsigned char buf_[max_buf];// Holds bytes for current line being displayed
    int rowsize_;               // Number of bytes per row
    int offset_;                // offset of start of row (0 to rowsize_ - 1)
    int real_offset_;           // Stores actual last set offset whihc may be different (larger)
                                // than offset_ when autofit is on & # of cols has been reduced
    int group_by_;              // How many columns to group together

    int text_height_;           // How high are characters?
    int text_width_;            // How wide is a hex (0-9,A-F) character?
    BCHAR first_char_, last_char_; // Range of chars in current font
    int addr_width_;            // How much room in display does address take?
    const char *addr_format_;   // What format string to use for address

    long mark_;                 // Address of marked position
    BOOL mark_char_;            // Is mark in char area?

    // Display toggles
    BOOL autofit_;              // Change rowsize_ depending on window width?
    BOOL dec_addr_;             // Decimal address (or hex)?
    BOOL display_char_;         // Display chars (ASCII or EBCDIC) as well as hex
    BOOL ebcdic_;               // EBCDIC display (or ACSII)?
    BOOL oem_;                  // Display using OEM rather than ANSO char set
    int display_control_;       // How to display control chars (0, 1, or 2)
    BOOL graphic_;              // Display graphic chars (or a dot)?

    // Display colours
    std::vector<partn> partitions_;  // Describes what colour diff ranges are drawn in
    COLORREF kala[256];         // Actual colours for each byte value

    // Current bg search position displayed
    std::vector<FILE_ADDRESS> search_found_;
    int search_length_;

    // Current edit position and state
    BOOL edit_char_;            // Is edit (caret) in hex or char area?
    short num_entered_;         // How many consecutive characters entered?
    short num_del_;             // How many consec. chars deleted?
    short num_bs_;              // How many consec. back spaces?
    BOOL overtype_;             // Are we in insert or overtype mode?
    BOOL readonly_;             // Are changes disallowed?

    // The following are saved when a new selection is started and are used when
    // it is finished to see what undo info. we need to save
    long prev_start_, prev_end_; // Previous selection
    BOOL area_swap_;            // New sel. swaps between hex/char areas?
    bool mouse_down_;           // Left button down seen but not yet up
    // Note some declarations are bool because this is the new C++ standard,
    // others are BOOL to avoid warnings.  One day I'll change them all to
    // bool and add typecasts where necessary.

    // Info required for macro (km_mouse and km_shift_mouse)
    CPoint dev_down_;           // Mouse down (device coordinates)
    CPoint doc_down_;           // Mouse down (doc coords)
    BOOL shift_down_;           // Shift key was down when left button down seen

    // Display refresh stuff for when running macros
    bool needs_refresh_;        // Does window need redrawing when refresh turned on again
    bool needs_hscroll_;        // Does the horizontal scroll bar need updating
    int h_total_, h_page_, h_pos_; // Last parameters for DoHScroll
    bool needs_vscroll_;        // Does the vertical scroll bar need updating
    int v_total_, v_page_, v_pos_; // Last parameters for DoVScroll

    // Undo array (things done in this view that don't change doc)
    std::vector <view_undo, allocator<view_undo> > undo_;
    long saved_start_, saved_end_;

    // Printing info
    CFont *print_font_;         // Font used when printing (= screen font scaled for printer)
    int curpage_;               // Current page being printed
    bool print_sel_;            // Are we printing the current selection?
    int lines_per_page_;        // Number of text lines on a printed page
    CSize page_size_;           // Size of printer page in logical units
    int print_map_mode_;        // Mapping mode for printer (not MM_TEXT so that text is scaled)
    long print_lfHeight_;       // Logical font height in logical (print_map_mode_) units
    int print_text_width_;      // Width of chars in printing DC
    int print_text_height_ ;    // Height of chars when printing
    // Note: print_text_height_ is analogous to text_height_ but for printing.
    // In OnDraw these values may be different since screen text is drawn using MM_TEXT
    // but printer text uses MM_LOENGLISH so that the text is not tiny on a laser.

    static const char *bin_format_name; // Name of our custom clipboard format
};

#ifndef _DEBUG  // debug version in HexEditView.cpp
inline CHexEditDoc* CHexEditView::GetDocument()
   { return (CHexEditDoc*)m_pDocument; }
#endif

/////////////////////////////////////////////////////////////////////////////
