// HexEditMacro.cpp : Stuff for keystroke macros - part of CHexEditApp
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"

#include "HexEdit.h"

#include "MainFrm.h"
#include "ChildFrm.h"
#include "HexEditDoc.h"
#include "HexEditView.h"
#include "timer.h"
#include "misc.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////

void CHexEditApp::macro_play(long play_times /*=1*/, const std::vector<key_macro> *pmac /*=NULL*/, int halt_lev/*=-1*/)
{
    TRACE0("============== macro_play ======================\n");
    // Fix default parameter values
    if (pmac == NULL) pmac = &mac_;
    if (halt_lev == -1) halt_lev = halt_level_;

    ASSERT(pmac->size() > 0);
    if (pmac->size() == 0) return;
    std::vector <key_macro, allocator<key_macro> >::const_iterator pk;

    ++playing_;

    CHexEditView *pv;                   // Ptr to the view (if any)
    CChildFrame *pc;                    // Ptr to the child (view's) frame
    timer time;                         // Timer tracking time between refreshes
    long plays_done;                    // Number of times macro replayed so far
    long keys_done = 0;                 // Number of "keystrokes" replayed so far

    // Clear previous errors
    mac_error_ = 0;

    // There should be a better way to clear previous macro status bar messages
    bool bb = refresh_off_;             // Save current refresh status
    refresh_off_ = true;                // Turn off refresh so that we clear previous saved messages
    ((CMainFrame *)AfxGetMainWnd())->StatusBarText("");
    refresh_off_ = bb;                  // Restore refresh status

    // Turn off refresh unless refreshing after every keystroke
    if (!(refresh_ == 2 && num_keys_ == 1))
    {
        refresh_off_ = true;
        disable_carets();
    }
    view_changed_ = false;

    CWaitCursor wait;
    pv = GetView();

    for (plays_done = 0; plays_done < play_times; ++plays_done)
    {
        for (pk = pmac->begin(); pk != pmac->end(); ++pk, ++keys_done)
        {
	        MSG msg;

            // Do any redrawing, but nothing else
            while (::PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_NOREMOVE))
            {
                if (::GetMessage(&msg, NULL, 0, 0))
			    {
				    // Process any events for the window so that we can tell if
				    // the timer has finished or if the user clicks the Stop button
                    ::TranslateMessage(&msg);
                    ::DispatchMessage(&msg);
			    }
            }

            // Check if Escape has been pressed
            if (::PeekMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN, PM_NOREMOVE))
            {
                VERIFY(GetMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN) > 0);
                if (msg.wParam == 27 &&
                    ::HMessageBox("Abort macro?", MB_YESNO) == IDYES)
                {
                    ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Macro aborted");
                    mac_error_ = 10;
                    goto exit_play;
                }
            }

            if ((*pk).ktype > km_view)
            {
                // Keep track of any view changes.  (It may be more efficient to
                // do this only after a keystroke that may have changed the view.)
                if (pv != GetView())
                    view_changed_ = true;
                // Get the active view
                if ((pv = GetView()) == NULL)
                {
                    ::HMessageBox("There is no active window - macro terminated");
                    mac_error_ = 10;
                    goto exit_play;
                }
            }

            ASSERT(sizeof(*pk) == 9);
            switch ((*pk).ktype)
            {
            // Handled by the app
            case km_new:
                OnFileNew();
                break;
            case km_open:
                (CHexEditDoc*)AfxGetApp()->OpenDocumentFile(*(*pk).pss);
                break;
            case km_exit:
                OnAppExit();
                break;
            case km_print_setup:
                OnFilePrintSetup();
                break;
            case km_prop:
                OnProperties();
                break;
            case km_prop_file:
                {
                    CPropSheet *pp = ((CMainFrame *)AfxGetMainWnd())->pprop_;
                    if (pp != NULL)
                    {
                        pp->SetActivePage(&pp->prop_file);
                        pp->prop_file.UpdateWindow();
                    }
                }
                break;
            case km_prop_char:
                {
                    CPropSheet *pp = ((CMainFrame *)AfxGetMainWnd())->pprop_;
                    if (pp != NULL)
                    {
                        pp->SetActivePage(&pp->prop_char);
                        pp->prop_char.UpdateWindow();
                    }
                }
                break;
            case km_prop_dec:
                {
                    CPropSheet *pp = ((CMainFrame *)AfxGetMainWnd())->pprop_;
                    if (pp != NULL)
                    {
                        pp->SetActivePage(&pp->prop_dec);
                        pp->prop_dec.UpdateWindow();
                    }
                }
                break;
            case km_prop_float:
                {
                    CPropSheet *pp = ((CMainFrame *)AfxGetMainWnd())->pprop_;
                    if (pp != NULL)
                    {
                        pp->SetActivePage(&pp->prop_float);
                        pp->prop_float.UpdateWindow();
                    }
                }
                break;
            case km_prop_ibmfp:
                {
                    CPropSheet *pp = ((CMainFrame *)AfxGetMainWnd())->pprop_;
                    if (pp != NULL)
                    {
                        pp->SetActivePage(&pp->prop_ibmfp);
                        pp->prop_ibmfp.UpdateWindow();
                    }
                }
                break;
            case km_prop_close:
                if (((CMainFrame *)AfxGetMainWnd())->pprop_ != NULL)
                    ((CMainFrame *)AfxGetMainWnd())->pprop_->DestroyWindow();
                break;
            case km_tips:
                ShowTipOfTheDay();
                break;

            // Handled by the main frame window
            case km_find_text:
                ((CMainFrame *)AfxGetMainWnd())->sec_search_.SetWindowText(*(*pk).pss);
                break;
            case km_find_close:
                if (((CMainFrame *)AfxGetMainWnd())->pfind_ != NULL)
                {
                    delete ((CMainFrame *)AfxGetMainWnd())->pfind_;
                    ((CMainFrame *)AfxGetMainWnd())->pfind_ = NULL;
                }
                break;
            case km_toolbar:
                ((CMainFrame *)AfxGetMainWnd())->OnViewToolbar();
                break;
            case km_editbar:
                ((CMainFrame *)AfxGetMainWnd())->OnViewEditbar();
                break;
            case km_bar:
                ((CMainFrame *)AfxGetMainWnd())->OnBarCheck((*pk).vv);
                break;
            case km_topics:
                ((CMainFrame *)AfxGetMainWnd())->OnHelpFinder();
                break;
            case km_help:
                ((CMainFrame *)AfxGetMainWnd())->OnHelp();
                break;
            case km_focus:
                {
                    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
                    CMDIChildWnd *nextc;        // Loops through all MDI child frames
#if 0 // Use the frame window name for changing focus rather than pointer to frame window
                    // Make sure that the window to activate hasn't been closed
                    CMDIChildWnd *newc = (CChildFrame *)(*pk).vv;

                    for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
                         nextc != newc && nextc != NULL;
                         nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
                             ; /* null statement */
                    if (nextc == NULL)
                    {
                        ::HMessageBox("Window to be activated no longer exists");
                        mac_error_ = 10;
                    }
                    else
                        ((CMainFrame *)AfxGetMainWnd())->MDIActivate(newc);
#else
                    for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
                         nextc != NULL;
                         nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
                    {
                        CString ss;
                        nextc->GetWindowText(ss);
                        if (ss == *(*pk).pss)
                            break;
                    }
                    if (nextc == NULL)
                    {
                        CString ss;
                        ss.Format("Window \"%s\" no longer exists", (const char *)*(*pk).pss);
                        ::HMessageBox(ss);
                        mac_error_ = 10;
                    }
                    else
                        ((CMainFrame *)AfxGetMainWnd())->MDIActivate(nextc);
#endif
                }
                break;
            case km_mainsys:
                // Send message to main frame (see also km_childsys)
                ((CMainFrame *)AfxGetMainWnd())->SendMessage(WM_SYSCOMMAND, (*pk).vv, 0L);
                break;

            // Handled by the view's frame (child frame window)
            case km_childsys:
                // Get active child frame
                pc = (CChildFrame *)((CMainFrame *)AfxGetMainWnd())->MDIGetActive();
                if (pc == NULL)
                {
                    ::HMessageBox("There is no active window - macro terminated");
                    mac_error_ = 10;
                    goto exit_play;
                }
                else
                    pc->SendMessage(WM_SYSCOMMAND, (*pk).vv, 0L);
                break;

            case km_close:
                pv->GetDocument()->OnFileClose();
                break;
            case km_save:
                pv->GetDocument()->OnFileSave();
                break;
            case km_saveas:
                pv->GetDocument()->OnFileSaveAs();
                break;
            case km_print:
                pv->OnFilePrint();
                break;
            case km_preview:
                pv->OnFilePrintPreview();
                break;

            case km_undo:
                pv->OnEditUndo();
                break;
            case km_cut:
                pv->OnEditCut();
                break;
            case km_copy:
                pv->OnEditCopy();
                break;
            case km_paste:
                pv->OnEditPaste();
                break;
            case km_del:
                pv->OnDel();
                break;
            case km_copy_hex:
                pv->OnCopyHex();
                break;
            case km_copy_cchar:
                pv->OnCopyCchar();
                break;
            case km_paste_ascii:
                pv->OnPasteAscii();
                break;
            case km_paste_unicode:
                pv->OnPasteUnicode();
                break;
            case km_paste_ebcdic:
                pv->OnPasteEbcdic();
                break;
            case km_sel_all:
                pv->OnSelectAll();
                break;
            case km_write_file:
                pv->OnEditWriteFile();
                break;
            case km_read_file:
    //      pv->OnReadFile();
                pv->do_read(*(*pk).pss);
                break;
            case km_goto:
                ((CMainFrame *)AfxGetMainWnd())->OnEditGoto();
                break;
            case km_calc_dlg:
                ((CMainFrame *)AfxGetMainWnd())->OnCalculator();
                break;
            case km_calc_close:
                if (((CMainFrame *)AfxGetMainWnd())->pcalc_->m_hWnd != 0)
                {
                    ((CMainFrame *)AfxGetMainWnd())->pcalc_->ShowWindow(SW_HIDE);
                    ((CMainFrame *)AfxGetMainWnd())->pcalc_->visible_ = FALSE;
                }
                break;
            case km_find_dlg:
                ((CMainFrame *)AfxGetMainWnd())->OnEditFind();
                break;
            case km_find_forw:
                pv->OnSearchForw();
                break;
            case km_find_back:
                pv->OnSearchBack();
                break;
            case km_find_sel:
                pv->OnSearchSel();
                break;
            case km_ro_rw:
                pv->OnAllowMods();
                break;
            case km_ovr_ins:
                pv->OnInsert();
                break;
            case km_mark_pos:
                pv->OnMark();
                break;
            case km_goto_mark:
                pv->OnGotoMark();
                break;
            case km_extendto_mark:
                pv->OnExtendToMark();
                break;
            case km_swap_mark:
                pv->OnSwapMark();
                break;
            case km_autofit:
                pv->do_autofit((*pk).vv);
                break;
            case km_font:
                pv->do_font((*pk).plf);
                break;
            case km_inc_font:
                pv->OnFontInc();
                break;
            case km_dec_font:
                pv->OnFontDec();
                break;
            case km_addr:
                pv->OnAddrToggle();
                break;
            case km_char_area:
                pv->do_chartoggle((*pk).vv);
                break;
            case km_ebcdic:
                pv->OnAscEbc();
                break;
            case km_control:
                pv->OnControl();
                break;
            case km_control2:
                pv->OnControlToggle();
                break;
            case km_graphic:
                pv->OnGraphicToggle();
                break;
            case km_oem:
                pv->OnOemToggle();
                break;
            case km_rowsize:
                pv->change_rowsize((*pk).vv);
                break;
            case km_group_by:
                pv->change_group_by((*pk).vv);
                break;
            case km_offset:
                pv->change_offset((*pk).vv);
                break;

            case km_win_next:
                pv->OnWindowNext();
                break;
            case km_win_new:
                ((CMainFrame *)AfxGetMainWnd())->OnWindowNew();
                break;
            case km_win_cmd:
                ((CMainFrame *)AfxGetMainWnd())->OnMDIWindowCmd((*pk).vv);
                break;

            case km_redraw:
                pv->OnRedraw();
                break;
            case km_scroll_up:
                pv->OnScrollUp();
                break;
            case km_scroll_down:
                pv->OnScrollDown();
                break;
            case km_swap_areas:
                pv->OnSwap();
                break;
            case km_start_line:
                pv->OnStartLine();
                break;

            case km_key:
                pv->MovePos((*pk).vv & 0xFFFF, 1, (*pk).vv & 0x10000, (*pk).vv & 0x20000, TRUE);
                break;
            case km_char:
                pv->do_char((*pk).vv);
                break;
            case km_mouse:
                pv->do_mouse((*pk).pms->dev_down, (*pk).pms->doc_dist);
                break;
            case km_shift_mouse:
                pv->do_shift_mouse((*pk).pms->dev_down, (*pk).pms->doc_dist);
                break;
            case km_hscroll:
                pv->OnScroll(MAKEWORD((*pk).vv>>16, -1), (*pk).vv & 0xFFFF);
                break;
            case km_vscroll:
                pv->OnScroll(MAKEWORD(-1, (*pk).vv>>16), (*pk).vv & 0xFFFF);
                break;
            case km_address_tool:
                pv->MoveToAddress((*pk).vv);
                break;

            case km_compare:
                pv->OnEditCompare();
                break;
            case km_inc8:
                pv->OnIncByte();
                break;
            case km_inc16:
                pv->OnInc16bit();
                break;
            case km_inc32:
                pv->OnInc32bit();
                break;
            case km_inc64:
                pv->OnInc64bit();
                break;
            case km_dec8:
                pv->OnDecByte();
                break;
            case km_dec16:
                pv->OnDec16bit();
                break;
            case km_dec32:
                pv->OnDec32bit();
                break;
            case km_dec64:
                pv->OnDec64bit();
                break;
            case km_flip16:
                pv->OnFlip16bit();
                break;
            case km_flip32:
                pv->OnFlip32bit();
                break;
            case km_flip64:
                pv->OnFlip64bit();
                break;
#ifdef ENCOM
            case km_rs_forw:
                pv->OnRsForw();
                break;
            case km_rs_back:
                pv->OnRsBack();
                break;
#endif
            case km_clear:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnClear();
                break;
            case km_clear_entry:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnClearEntry();
                break;
            case km_equals:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnEquals();
                break;
            case km_endian:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->toggle_endian();
                break;
            case km_change_bits:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->change_bits((*pk).vv);
                break;
            case km_change_base:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->change_base((*pk).vv);
                break;
            case km_binop:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->do_binop(CCalcDlg::binop_type((*pk).vv));
                break;
            case km_unaryop:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->do_unary(CCalcDlg::unary_type((*pk).vv));
                break;
            case km_user:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->current_ = (*pk).v64;
                break;
            case km_memget:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMemGet();
                break;
            case km_markget:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkGet();
                break;
            case km_markat:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkAt();
                break;
            case km_selget:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelGet();
                break;
            case km_selat:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelAt();
                break;
            case km_sellen:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelLen();
                break;
            case km_eofget:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnEofGet();
                break;
            case km_memstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMemStore();
                break;
            case km_memclear:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMemClear();
                break;
            case km_memadd:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMemAdd();
                break;
            case km_memsubtract:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMemSubtract();
                break;
            case km_markstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkStore();
                break;
            case km_markclear:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkClear();
                break;
            case km_markadd:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkAdd();
                break;
            case km_marksubtract:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkSubtract();
                break;
            case km_markatstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnMarkAtStore();
                break;
            case km_selstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelStore();
                break;
            case km_go:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnGo();
                break;
            case km_selatstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelAtStore();
                break;
            case km_sellenstore:
                ((CMainFrame *)AfxGetMainWnd())->pcalc_->OnSelLenStore();
                break;

            default:
                ASSERT(0);
            } // switch

            // Check for termination here
            ASSERT(halt_lev == 0 || halt_lev == 1 || halt_lev == 2);
            if (mac_error_ > halt_lev)
            {
                // Display an appropriate error
                if (mac_error_ > 10)
                    ::HMessageBox("System error encountered during macro execution");
                else if (mac_error_  > 2)
                {
                    // Just display mess in status bar as dlg box has already been seen
                    ((CMainFrame *)AfxGetMainWnd())->
                        StatusBarText("Error encountered during macro execution");
                }
                else if (mac_error_ > 1)
                    ::HMessageBox("Minor error encountered during macro execution");
                else if (mac_error_ > 0)
                    ::HMessageBox("Warning encountered during macro execution");
                goto exit_play;
            }

            // Check if its time to refresh the display
            if (refresh_off_ &&
                ((refresh_ == 1 && time.elapsed() > num_secs_) ||
                 (refresh_ == 2 && ((keys_done+1) % num_keys_) == 0) ||
                 (refresh_ == 3 && ((plays_done+1) % num_plays_) == 0 && pk == pmac->end()-1) ) )
            {
                // Turn off refresh temporarily (so refresh is done) and refresh display
                refresh_display(view_changed_);
                view_changed_ = false;

                // If refreshing based on time: reset the timer for next period
                if (refresh_ == 1) time.reset();
            }
            else if (refresh_ == 2 && num_keys_ == 1 && refresh_bars_)
            {
                // If refresh every keystroke refresh is not turned off but
                // some things don't get updated till OnIdle called so force
                // it here (since we're not idle while playing).
                ASSERT(!refresh_off_);
                CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

                // status bar
                mm->m_wndStatusBar.OnUpdateCmdUI(mm, TRUE);     // Update status bar
                mm->m_wndStatusBar.UpdateWindow();              // Make sure window redrawn

                // The find tool
                ASSERT((mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)) != NULL);
                (mm->m_wndEditBar.GetDlgItem(IDC_SEARCH))->UpdateWindow();

                // Find dialog
                if (mm->pfind_ != NULL)
                    mm->pfind_->UpdateWindow();
            }

            // Restore wait cursor if there is any chance a dialog was displayed etc
            if (mac_error_ > 0)
            {
                wait.Restore();         // Restore busy cursor in case a dialog has messed it up
                mac_error_ = 0;         // Reset error so we don't keep restoring after first warning
            }
        } // for keys
    } // for plays

    if (mac_error_ > 1)
        ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Minor error encountered during macro execution");
    else if (mac_error_ > 0)
        ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Warning encountered during macro execution");

exit_play:
    // Keep track of play nesting
    playing_--;

    // If finished playing and display refresh was off refresh the display
    if (playing_ == 0 && refresh_off_)
    {
        refresh_display(true);
        refresh_off_ = false;
        enable_carets();
        CheckBGSearchFinished();
    }
}

void CHexEditApp::disable_carets()
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CMDIChildWnd *nextc;        // Loops through all MDI child frames

    // Invalidate all view windows
    for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
         nextc != NULL;
         nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
    {
        CHexEditView *pv = (CHexEditView *)nextc->GetActiveView();
        ASSERT(pv != NULL);
        pv->DisableCaret();
    }
}

void CHexEditApp::enable_carets()
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CMDIChildWnd *nextc;        // Loops through all MDI child frames

    // Invalidate all view windows
    for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
         nextc != NULL;
         nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
    {
        CHexEditView *pv = (CHexEditView *)nextc->GetActiveView();
        ASSERT(pv != NULL);
        pv->EnableCaret();
    }
}

void CHexEditApp::refresh_display(bool do_all /*=false*/)
{
    CHexEditView *pactive;              // Ptr to the active view (if any)
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // Refresh status bar even if there are no windows open
    if (do_all || refresh_bars_)
    {
        mm->StatusBarText(NULL);                        // Show latest status bar message
        mm->m_wndStatusBar.OnUpdateCmdUI(mm, TRUE);     // Update other panes
        mm->m_wndStatusBar.UpdateWindow();              // Make sure window redrawn
    }

    // Update the calculator
    ASSERT(mm->pcalc_ != NULL);
    if ((do_all || refresh_bars_) && mm->pcalc_->m_hWnd != 0 && mm->pcalc_->visible_)
    {
        mm->pcalc_->UpdateData(FALSE);  // Update base/bits radio buttons etc
        mm->pcalc_->edit_.Put();        // Make sure current calc. value is displayed
        mm->pcalc_->FixFileButtons();   // Make sure file buttons reflect current view (or no view)
        mm->pcalc_->ShowBinop();
        mm->pcalc_->UpdateWindow();     // Make sure all these nice updates are seen
    }

    // Refresh of other things only makes sense if there is a window open
    if ((pactive = GetView()) != NULL)
    {
        // Update properties dialog?
        if (do_all || refresh_props_)
        {
            pactive->show_prop(-2);
        }

        if (do_all || refresh_bars_)
        {
            // Update the hex/decimal address tools
            pactive->show_pos(-2);

            // Update the find tool
            ASSERT((mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)) != NULL);
            (mm->m_wndEditBar.GetDlgItem(IDC_SEARCH))->UpdateWindow();

            // Update the find dialog box
            if (mm->pfind_ != NULL)
                mm->pfind_->UpdateWindow();
        }
        if (do_all)
        {
            CMDIChildWnd *nextc;        // Loops through all MDI child frames

            // Invalidate all view windows
            for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
                 nextc != NULL;
                 nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
            {
                CHexEditView *pv = (CHexEditView *)nextc->GetActiveView();
                ASSERT(pv != NULL);
                if (pv != NULL && pv != pactive)
                {
                    pv->EnableCaret();
                    pv->DisplayCaret();
                    pv->DoUpdate();
                    pv->DisableCaret();
                }
            }
        }
        pactive->EnableCaret();
        pactive->DisplayCaret();
        pactive->DoUpdate();
        pactive->DisableCaret();
    }
}

struct file_header
{
    long plays;                         // Default number of plays
    char dummy[20];                     // Unused (should be zero)
    short halt_lev;                     // Default halt level to use with the macro
    char comment[2];                    // First byte of description of this macro
};

BOOL CHexEditApp::macro_save(const char *filename, const std::vector<key_macro> *pmac /*=NULL*/,
                             const char *comment /*= NULL*/, int halt_lev /*=-1*/, long plays /*=1*/)
{
    unsigned short magic, header_len;   // Magic number to check for valid files and header length

    // Fix default parameter values
    if (comment == NULL) comment = "";
    if (pmac == NULL) pmac = &mac_;
    if (halt_lev == -1) halt_lev = halt_level_;

    // Get memory for the header
    magic = 0xCDAB;
    header_len = sizeof(file_header) + strlen(comment);  // Includes 2 extra bytes for '\0' and '\xCD'
    char *pp = new char[header_len];
    file_header *ph = (file_header *)pp;

    // Build the header
    memset(pp, '\0', header_len);
    pp[header_len-1] = '\xCD';
    ph->halt_lev = halt_lev;
    ph->plays = plays;
    strcpy(ph->comment, comment);
    ASSERT(pp[header_len-2] == '\0' && pp[header_len-1] == '\xCD');

    // Write the file
    try
    {
        short str_len;                  // Length of string for km_find_text etc
        ASSERT(pmac->size() > 0);
        std::vector <key_macro, allocator<key_macro> >::const_iterator pk;

        CFile ff(filename, CFile::modeCreate|CFile::modeWrite|CFile::shareExclusive|CFile::typeBinary);

        ff.Write(&magic, sizeof(magic));
        ff.Write(&header_len, sizeof(header_len));
        ff.Write(pp, header_len);              // Write header

        for (pk = pmac->begin(); pk != pmac->end(); ++pk)
        {
            ASSERT(sizeof((*pk)) == 9);
            ff.Write(&(*pk).ktype, sizeof((*pk).ktype));
            switch ((*pk).ktype)
            {
            case km_find_text:
            case km_open:
            case km_read_file:
            case km_focus:
                // Write the string length
                str_len = (*(*pk).pss).GetLength();
                ff.Write(&str_len, sizeof(str_len));
                ff.Write((const char *)(*(*pk).pss), str_len + 1);
                break;
            case km_mouse:
            case km_shift_mouse:
                ff.Write((*pk).pms, sizeof(*(*pk).pms));
                break;
            case km_font:
                ff.Write(&(*pk).plf->lfHeight, sizeof((*pk).plf->lfHeight));
                ff.Write(&(*pk).plf->lfCharSet, sizeof((*pk).plf->lfCharSet));
                str_len = strlen((*pk).plf->lfFaceName);
                ff.Write(&str_len, sizeof(str_len));
                ff.Write((*pk).plf->lfFaceName, str_len + 1);
                break;
            default:
                ff.Write(&(*pk).v64, 8);
                break;
            }
        }

    }
    catch (CFileException *pfe)
    {
        ::HMessageBox(FileErrorMessage(pfe, CFile::modeWrite));
        pfe->Delete();
        mac_error_ = 10;
        delete[] pp;
        return FALSE;
    }

    delete[] pp;
    return TRUE;
}

BOOL CHexEditApp::macro_load(const char *filename, std::vector<key_macro> *pmac,
                             CString &comment, int &halt_lev, long &plays)
{
    pmac->clear();                      // Empty macro ready to build new one

    try
    {
        unsigned short magic;           // Special number to check file is what we expect
        unsigned short header_len;      // Length of header
        CFileException *pfe;            // This is thrown if we reach EOF before expected

        // Create the thing to be thrown if we reach EOF unexpectedly.
        // Note: the object must be created on the heap as CException::m_bAutoDelete
        // is always TRUE (default CException constructor is called which sets it TRUE).
        pfe = new CFileException(CFileException::endOfFile, -1, filename);

        CFile ff(filename, CFile::modeRead|CFile::shareDenyWrite|CFile::typeBinary);

        if (ff.Read(&magic, sizeof(magic)) < sizeof(magic)) throw pfe;
        if (magic != 0xCDAB)
        {
            ::HMessageBox("This is not a valid HexEdit macro file");
            pfe->Delete();
            return FALSE;
        }

        // Read the header length
        if (ff.Read(&header_len, sizeof(header_len)) < sizeof(header_len)) throw pfe;


        // Now read the whole header
        char *pp = new char[header_len];       // Memory to hold header
        if (ff.Read(pp, header_len) < header_len) throw pfe;
        file_header *ph = (file_header *)pp;
        ASSERT(pp[header_len-2] == '\0' && pp[header_len-1] == '\xCD');

        comment = ph->comment;
        halt_lev = ph->halt_lev;
        plays = ph->plays;

        // Now read the "keystrokes" until EOF
        unsigned char kk;               // "key" type
        unsigned __int64 v64;           // Value associated with normal keys
        short str_len;                  // Length of string for km_find_text etc
        char *buf = NULL;               // Buffer for reading strings
        mouse_sel ms;                   // Info for km_mouse/km_shift_mouse
        LOGFONT lf;                     // Info use to create a new font for km_font

        memset(&lf, '\0', sizeof(lf));  // Make sure all fields are zero since we only set name and height

        while (ff.Read(&kk, sizeof(kk)) == sizeof(kk))
        {
            switch (km_type(kk))
            {
            case km_find_text:
            case km_open:
            case km_read_file:
            case km_focus:
                // Get the string length and allocate memory for the string
                if (ff.Read(&str_len, sizeof(str_len)) < sizeof(str_len)) throw pfe;
                buf = new char[str_len+1];

                // Read the string and add to the macro
                if (ff.Read(buf, str_len+1) < str_len+1) throw pfe;
                ASSERT(buf[str_len] == '\0');
                pmac->push_back(key_macro(km_type(kk), CString(buf)));

                // Free the memory
                delete buf;
                buf = NULL;
                break;
            case km_mouse:
            case km_shift_mouse:
                // Get the mouse data and add to the macro
                if (ff.Read(&ms, sizeof(ms)) < sizeof(ms)) throw pfe;
                pmac->push_back(key_macro(km_type(kk), &ms));
                break;
            case km_font:
                // Read the font height and char set
                if (ff.Read(&lf.lfHeight, sizeof(lf.lfHeight)) < sizeof(lf.lfHeight)) throw pfe;
                if (ff.Read(&lf.lfCharSet, sizeof(lf.lfCharSet)) < sizeof(lf.lfCharSet)) throw pfe;

                // Read font name length, allocate memory for the string and read it in 
                if (ff.Read(&str_len, sizeof(str_len)) < sizeof(str_len)) throw pfe;
                buf = new char[str_len+1];
                if (ff.Read(buf, str_len+1) < str_len+1) throw pfe;
                ASSERT(buf[str_len] == '\0');

                // Create font and add to macro
                if (lf.lfHeight < 2 || lf.lfHeight > 100)
			        lf.lfHeight = 16;
                strncpy(lf.lfFaceName, buf, LF_FACESIZE-1);
                lf.lfFaceName[LF_FACESIZE-1] = '\0';
                pmac->push_back(key_macro(km_type(kk), &lf));

                // Free the memory
                delete[] buf;
                buf = NULL;
                break;
            default:
                ff.Read(&v64, 8);
                pmac->push_back(key_macro(km_type(kk), v64));
                break;
            }
        }

        pfe->Delete();
    }
    catch (CFileException *pfe)
    {
        ::HMessageBox(FileErrorMessage(pfe, CFile::modeRead));
        pfe->Delete();
        mac_error_ = 10;
        return FALSE;
    }

    return TRUE;
}
