// HexEditDoc.h : interface of the CHexEditDoc class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

/////////////////////////////////////////////////////////////////////////////

#include <vector>
#include <list>
#include <set>
#include <afxmt.h>              // For MFC IPC (CEvent etc)

using namespace std;

// This enum is for the different modification types that can be made
// to the document.  It is used for keeping track of changes made in the
// undo array and for passing info about changes made to views.
enum mod_type
{
    mod_unknown = '0',          // Helps bug detection
    mod_insert  = 'I',          // Bytes inserted
    mod_replace = 'R',          // Bytes replaced (overtyped)
    mod_delforw = 'D',          // Bytes deleted (using DEL)
    mod_delback = 'B',          // Bytes deleted (using back space)
    mod_repback = '<',          // Replace back (BS in overtype mode)
};

// This enum represents the different sources of document data
// - it is used in the locations list.
enum { loc_unknown = 'u', loc_file = 'f', loc_mem = 'm' };

// This object is passed to view OnUpdate() functions as the (3rd) hint
// parameter.  It is used by the view to tell what parts of its display
// (if any) need to be updated.
class CHexHint : public CObject
{
public:
    enum mod_type utype;        // Have bytes been inserted, deleted or replaced?
    size_t len;                 // How many were      "         "     "    "
    long address;               // Address in the file that change started

    CView *pview;               // The view that caused the change
    int index;                  // Index into the document undo array
    BOOL is_undo;               // True if undoing

    // Default and proper constructor
    CHexHint() { utype = mod_unknown; }
    CHexHint(enum mod_type u, size_t l, long a, CView *v, int i, BOOL f = FALSE)
    {
        ASSERT(u == mod_insert  || u == mod_replace ||
               u == mod_delforw || u == mod_delback || u == mod_repback);
        utype = u; len = l; address = a; pview = v; index = i; is_undo = f;
    }

protected:
    DECLARE_DYNAMIC(CHexHint)   // Required for MFC run-time type info.
};

// This object is passed to view OnUpdate() functions as the (3rd) hint
// parameter.  It is used to inform the view to undo all view changes
// (moves etc) back to the last doc undo.
class CUndoHint : public CObject
{
public:
    CView *pview;               // The view that caused the change
    int index;                  // Index into the document undo array

    // Default and proper constructor
    CUndoHint() { index = -1; }
    CUndoHint(CView *v, int ii)
    {
        pview = v; index = ii;
    }

protected:
    DECLARE_DYNAMIC(CUndoHint)  // Required for MFC run-time type info.
};

// This object is passed to view OnUpdate() functions as the (3rd) hint
// parameter.  It is used to inform the view to remove all undo info up to
// the last doc undo, probably because the file has been saved.
class CRemoveHint : public CObject
{
public:
    int index;                          // Index of last doc undo
    CRemoveHint(int ii = -1) { index = ii; }

protected:
    DECLARE_DYNAMIC(CRemoveHint)        // Required for MFC run-time type info.
};

// This object is passed to view OnUpdate() functions as the (3rd) hint
// parameter.  It is used to tell the view to update its found string
// display.  It is called when a background search is started (to turn
// off display of found strings) and finished (to display the new strings).
class CBGSearchHint : public CObject
{
public:
    BOOL finished_;                     // FALSE = clear all search string occurrences
                                        // TRUE = get new occurrences from doc and fix current display
                                        // -1 = remove range of address in start_:end_ but keep rest
    FILE_ADDRESS start_, end_;          // range of string occurrence addresses that need to be removed
    CBGSearchHint(BOOL bb = TRUE) { finished_ = bb; start_ = end_ = -1; }
    CBGSearchHint(FILE_ADDRESS ss, FILE_ADDRESS ee) { finished_ = -1; start_ = ss; end_ = ee; }

protected:
    DECLARE_DYNAMIC(CBGSearchHint)        // Required for MFC run-time type info.
};

// These structures used to be declared within class CHexEditDoc but with
// VC++ 5 you get errors in <vector> and <list>

// These structures are used to keep track of all changes made to the doc
struct doc_undo
{
    static const int limit;
    enum mod_type utype;                // Type of modification made to file
    size_t len;                 // Length of mod
    long address;                       // Address in file of start of mod
    unsigned char *ptr;         // NULL if utype is del else new data

    // Normal constructor
    doc_undo(mod_type u, long a, size_t l, unsigned char *p = NULL)
    {
        ASSERT(u == mod_insert  || u == mod_replace ||
               u == mod_delforw || u == mod_delback || u == mod_repback);

        utype = u; len = l; address = a;
        if (p != NULL)
        {
            ptr = new unsigned char[max(limit,len)];
            memcpy(ptr, p, len);
        }
        else
            ptr = NULL;
    }
    // Copy constructor
    doc_undo(const doc_undo &from)
    {
        ASSERT(from.utype != mod_unknown);
        utype = from.utype;
        len = from.len;
        address = from.address;
        if (from.ptr != NULL)
        {
            ptr = new unsigned char[max(limit,from.len)];
            memcpy(ptr, from.ptr, from.len);
        }
        else
            ptr = NULL;
    }
    // Copy assignment operator
    doc_undo &operator=(const doc_undo &from)
    {
        if (&from != this)
        {
            ASSERT(from.utype != mod_unknown);
            if (ptr != NULL)
                delete[] ptr;

            utype = from.utype;
            len = from.len;
            address = from.address;

            if (from.ptr != NULL)
            {
                ptr = new unsigned char[max(limit,from.len)];
                memcpy(ptr, from.ptr, from.len);
            }
            else
                ptr = NULL;
        }
        return *this;
    }
    ~doc_undo()
    {
        ASSERT(utype != mod_unknown);
        if (ptr != NULL)
            delete[] ptr;
    }

    // vector requires a default constructor (even if not used)
    doc_undo() { utype = mod_unknown; }
    operator==(const doc_undo &) const { return false; }
    operator<(const doc_undo &) const { return false; }
};

// These structures are used to easily access the current doc data
struct doc_loc
{
    char location;                      // File or memory?
    size_t len;
    union
    {
        FILE_ADDRESS fileaddr;  // File location (if loc_file)
        unsigned char *memaddr; // Ptr to data (if loc_mem)
    };
    doc_loc(unsigned char *m, size_t l)
    {
        location = loc_mem;
        len = l; memaddr = m;
    }
    doc_loc(long f, size_t l)
    {
        location = loc_file;
        len = l; fileaddr = f;
    }

    // The compiler requires operator== and operator< for list functions
    // move(), unique(), and  merge() [even though they're never called]
    doc_loc() { location = loc_unknown; }       // Default constructor also needed
    operator==(const doc_loc &) const { return false; }
    operator<(const doc_loc &) const { return false; }

    // I can't get rid of warning C4800 except by disabling the warning
#pragma warning(disable : 4800)
    operator!=(const doc_loc &) const { return true; }
    operator>(const doc_loc &) const { return true; }
};

UINT bg_func(LPVOID pParam);    // Entry function for background search

class boyer;

struct adjustment
{
    adjustment(FILE_ADDRESS ss, FILE_ADDRESS ee, FILE_ADDRESS address, long aa)
    { start_ = ss; end_ = ee; address_ = address; adjust_ = aa; }
    FILE_ADDRESS start_, end_;
    FILE_ADDRESS address_;
    long adjust_;
};

class CHexEditDoc : public CDocument
{
protected: // create from serialization only
        CHexEditDoc();
        DECLARE_DYNCREATE(CHexEditDoc)

public:
// Attributes
    CFile file_;
    long length() const { return length_; }
    BOOL read_only() { return readonly_; }
    BOOL readonly_;
    const char *why0() const
    {
        // Return reason (as text) that length is zero
        if (length_ == 0)
            return "Empty file";
        else
        {
            ASSERT(0);
            return "Internal error";
        }
    }

// Operations
    size_t GetData(unsigned char *buf, size_t len, long loc, CFile *pfile = NULL);
    BOOL WriteData(const CString, long start, long end);
    void WriteInPlace();
    void Change(enum mod_type, long address, size_t len,
                         unsigned char *buf, int, CView *pview);
    BOOL Undo(CView *pview, int index, BOOL same_view);

public:
// Overrides
        // ClassWizard generated virtual function overrides
        //{{AFX_VIRTUAL(CHexEditDoc)
        public:
        virtual BOOL OnNewDocument();
        virtual BOOL OnOpenDocument(LPCTSTR lpszPathName);
        virtual void OnCloseDocument();
        virtual void DeleteContents();
        virtual BOOL OnSaveDocument(LPCTSTR lpszPathName);
        protected:
        virtual BOOL SaveModified();
        //}}AFX_VIRTUAL

        void SetModifiedFlag(BOOL bMod = TRUE);

public:
        afx_msg void OnFileClose();
        afx_msg void OnFileSave();
        afx_msg void OnFileSaveAs();
        virtual BOOL DoSave(LPCTSTR lpszPathName, BOOL bReplace = TRUE);

// Implementation
public:
        virtual ~CHexEditDoc();
#ifdef _DEBUG
        virtual void AssertValid() const;
        virtual void Dump(CDumpContext& dc) const;
#endif

protected:

// Generated message map functions
protected:
        //{{AFX_MSG(CHexEditDoc)
	afx_msg void OnDocTest();
	//}}AFX_MSG
        DECLARE_MESSAGE_MAP()

private:
// Private member functions
    void regenerate();      // Rebuild loc_ list from undo_ array
    BOOL only_over();       // Check if file can be saved in place
    BOOL open_file(LPCTSTR lpszPathName); // Open or reopen file_

    // The following are used to modify the locations list (loc_)
    typedef std::vector <doc_undo>::const_iterator pundo_t;
    typedef std::list <doc_loc>::iterator ploc_t;
    void loc_add(pundo_t pu, long &pos, ploc_t &pl);
    void loc_del(long address, size_t len, long &pos, ploc_t &pl);
    void loc_split(long address, long pos, ploc_t pl);

private:
    long length_;

    CView *last_view_;          // Last view that caused change to document

    // Array of changes made to file (last change at end)
    std::vector <doc_undo> undo_;

    // List of locations of where to find doc data (disk file/memory)
    std::list <doc_loc> loc_;

public:
    void CreateThread();        // Createn background thread
    void KillThread();          // Kill background thread ASAP
    UINT RunBGThread();         // Main func in bg thread
    void BGThreadPriority(int ii); // Set bg thread priority
    void StartSearch(FILE_ADDRESS start = -1, FILE_ADDRESS end = -1);
    void StopSearch();          // Stops any current search (via stop_event_), thread waits for start_event_
    void BGSearchFinished();    // Called when BG thread signals that the search is complete
                                // - allows views to be updated with new found strings
    int SearchOccurrences();    // No of search occurrences (-ve if disabled/not finished)
    FILE_ADDRESS GetNextFound(const unsigned char *pat, size_t len,
                              BOOL icase, int tt, FILE_ADDRESS from);
    FILE_ADDRESS GetPrevFound(const unsigned char *pat, size_t len,
                              BOOL icase, int tt, FILE_ADDRESS from);
    std::vector<FILE_ADDRESS> SearchAddresses(FILE_ADDRESS start, FILE_ADDRESS end);

private:
    void FixFound(FILE_ADDRESS start, FILE_ADDRESS end, FILE_ADDRESS address, long adjust);

    // Data for background searches
    CWinThread *pthread_;       // Ptr to background search thread or NULL if bg searches are off

    CEvent start_event_;        // Signal to bg thread to start a new search, or check for termination
    CEvent stopped_event_;      // Signal from bg thread that finished/aborted the search

    CCriticalSection docdata_;  // Protects access from main/bg threads to document data (loc_) the file
                                // and find data below.
                                // Note that this is used for read access of the document from the bg
                                // thread or write (but not read) access from the primary thread. This
                                // stops the bg thread accessing the data while it is being updated.
    CFile file2_;               // We need a copy of file_ so we can access the same file for searching
    int thread_flag_;           // Signals thread to stop search (1) or even kill itself (-1)

    BOOL view_update_;          // Flags that the bg search is finished and the view need updating

    // List of ranges to search in background (first = start, second = byte past end)
    std::list<pair<FILE_ADDRESS, FILE_ADDRESS> > to_search_;
    std::set<FILE_ADDRESS> found_;      // Addresses where current search text was found
    // List of adjustments pending due to insertions/deletions (first = address, second = adjustment amount)
    std::list<adjustment> to_adjust_;

#if 0
    boyer *pboyer_;             // Ptr to current search pattern (NULL if none)
                                // (Also stores search bytes and their length.)
    BOOL icase_;                // Indicates a case-insensitive search
    int text_type_;             // Type of text search (only matters if icase == TRUE)
#endif

    DWORD main_thread_id_;      // Thread ID of main thread so bg thread can send it a message
};

#pragma warning(default : 4800)

/////////////////////////////////////////////////////////////////////////////
