// DocData.cpp : part of implementation of the CHexEditDoc class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

// This module handles the retrieving and modification to the data, including
// updating an undo list.  Two data structures are used in this process:
//  - An undo array which tracks all changes made to the data.  This is
//    separate to undo facilities of the view(s) which track changes in the
//    display as well as keeping track of changes made to the document (via
//    an index into this array).
//  - A locations linked list which represents where every byte of the current
//    displayed "file" comes from.  This list is rebuilt from the original data
//    file plus the undo array everytime a change to the document is made.
//  Note that the original data file is never modified by the user making
//       changes until the file is saved.  When the file is saved the original
//       file plus the locations linked list are used to create the new file.
//       If no changes move any part of the file (i.e. all changes just
//       overwrite existing parts of the file, but do not insert/delete)
//       then the file is modified in place (unless a backup is required).
//       For a very large file this is a lot faster than copying the whole
//       file, and means you can get by without temporary disk space.

#include "stdafx.h"
#include "HexEdit.h"
#include "boyer.h"

#include "HexEditDoc.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

const int doc_undo::limit = 5;

IMPLEMENT_DYNAMIC(CHexHint, CObject)
IMPLEMENT_DYNAMIC(CUndoHint, CObject)
IMPLEMENT_DYNAMIC(CRemoveHint, CObject)
IMPLEMENT_DYNAMIC(CBGSearchHint, CObject)

size_t CHexEditDoc::GetData(unsigned char *buf, size_t len, FILE_ADDRESS address, CFile *pfile /*=NULL*/)
{
    ASSERT(address >= 0);
    FILE_ADDRESS pos;           // Tracks file position of current location record
    ploc_t pl;                  // Current location record

    if (pfile == NULL) pfile = &file_;  // Default to normal file

//    CSingleLock sl(&docdata_, TRUE);

    // Find the 1st loc record that has (some of) the data
    for (pos = 0, pl = loc_.begin(); pl != loc_.end(); pos += (*pl).len, ++pl)
        if (address < pos + (*pl).len)
            break;

    // Get the data from each loc record until buf is full
    size_t left = len;                  // How much is left to copy
    size_t start = size_t(address - pos); // Where to start copy in this block
    size_t tocopy;                      // How much to copy in this block
    for (left = len; left > 0 && pl != loc_.end(); left -= tocopy, buf += tocopy)
    {
        tocopy = min(left, (*pl).len - start);
        if ((*pl).location == loc_mem)
            memcpy(buf, (*pl).memaddr + start, tocopy);
        else
        {
            // Read data from the original file
            ASSERT((*pl).location == loc_file);
            UINT actual;                // Number of bytes actually read from file

            pfile->Seek((*pl).fileaddr + start, CFile::begin);
            if ((actual = pfile->Read((void *)buf, (UINT)tocopy)) != tocopy)
            {
                // If we run out of file there is something wrong with our data
                ASSERT(0);
                left -= actual;
                break;
            }
        }

        // Move to start of next loc record
        start = 0;
        pos += (*pl).len;
        ++pl;
    }

    // Return the actual number of bytes written to buf
    return len - left;
}

// Change allows the document to be modified.  After adding it to the undo
// array it regenerates the loc list and sends update notices to all views.
//   utype indicates the type of change
//   address is the byte within the document for the start of the change
//   len is the number of bytes inserted/deleted or replaced
//   buf points to the characters being inserted or replaced (NULL for deletes)
//       - the data pointed to by buf is copied so buf does not have to be kept
//   num_done indicates if this change is to be merged with a previous change,
//            0 means that this is the start of a new change
//   pview is a pointer to the view that is making the change (or NULL)
void CHexEditDoc::Change(enum mod_type utype, long address, size_t len,
                         unsigned char *buf, 
                         int num_done, CView *pview)
{
    // Lock the doc data (automatically releases the lock when it goes out of scope)
    CSingleLock sl(&docdata_, TRUE);

    int index;          // index into undo array
    ASSERT(utype == mod_insert || utype == mod_replace ||
           utype == mod_delforw || utype == mod_delback || utype == mod_repback);
    ASSERT(address <= length_);
    ASSERT(len > 0);

    // Can this change be merged with the previous change?
    // Note: if num_done is odd then we must merge this change
    // since it is the 2nd nybble of a hex edit change
    if (((num_done % 2) == 1 && len == 1) ||
        (num_done > 0 && len + undo_.back().len < doc_undo::limit))
    {
        // Join this undo to the previous one (end of undo array).
        // This allows (up to 8) multiple consecutive changes in the same
        // view of the same type to be merged together.
        ASSERT(pview == last_view_);
        ASSERT(utype == undo_.back().utype);

        if (utype == mod_delforw)
        {
            // More deletes forward
            ASSERT(buf == NULL);
            ASSERT(undo_.back().address == address);
            ASSERT(address + len <= length_);
            undo_.back().len += len;
        }
        else if (utype == mod_delback)
        {
            // More deletes backward
            ASSERT(buf == NULL);
            ASSERT(undo_.back().address == address + len);
            undo_.back().address = address;
            undo_.back().len += len;
        }
        else if (utype == mod_repback)
        {
            ASSERT(buf != NULL);
            ASSERT(undo_.back().address == address + len);
            memmove(undo_.back().ptr + len,
                    undo_.back().ptr, undo_.back().len);
            memcpy(undo_.back().ptr, buf, len);
            undo_.back().address = address;
            undo_.back().len += len;
        }
        else if (undo_.back().address + undo_.back().len == address)
        {
            // mod_insert/mod_replace - add to end of current insert/replace
            ASSERT(buf != NULL);
            memcpy(undo_.back().ptr+undo_.back().len, buf, len);
            undo_.back().len += len;
        }
        else
        {
            // mod_insert/mod_replace but last byte of previous undo changed.
            // This happens in hex edit mode when 2nd nybble entered
            ASSERT(undo_.back().address + undo_.back().len - 1 == address);
            memcpy(undo_.back().ptr+undo_.back().len - 1, buf, len);
            len -= 1;                   // Fix for later calcs of length_
            undo_.back().len += len;
        }
        index = -1;             // Signal that this is not a new undo
    }
    else
    {
        // Add a new elt to undo array
        if (utype == mod_insert || utype == mod_replace || utype == mod_repback)
            undo_.push_back(doc_undo(utype, address, len, buf));
        else
            undo_.push_back(doc_undo(utype, address, len));
        index = undo_.size() - 1;
    }

    last_view_ = pview;

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // If there is a current search string and background searches are on
    if (aa->pboyer_ != NULL && pthread_ != NULL)
    {
        ASSERT(aa->bg_search_);
        long adjust;

        if (utype == mod_delforw || utype == mod_delback)
            adjust = -len;
        else if (utype == mod_insert)
            adjust = len;
        else if (utype == mod_replace || utype == mod_repback)
            adjust = 0;
        else
            ASSERT(0);

        // Adjust pending search addresses
        std::list<pair<FILE_ADDRESS, FILE_ADDRESS> >::iterator pcurr, pend;
        for (pcurr = to_search_.begin(), pend = to_search_.end(); pcurr != pend; ++pcurr)
        {
            if (pcurr->first >= address)
            {
                pcurr->first += adjust;
                if (pcurr->first < address)  // Make sure it doesn't go too far backwards
                    pcurr->first = address;
            }
            if (pcurr->second >= address)
            {
                pcurr->second += adjust;
                if (pcurr->second < address)  // Make sure it doesn't go too far backwards
                    pcurr->second = address;
            }
        }

        // Remove invalidated found_ occurrences and adjust those for insertions/deletions
        if (!to_search_.empty())
        {
            // Ask bg thread to do it so that changes are kept consistent
            ASSERT(!view_update_);
            to_adjust_.push_back(adjustment(address - (aa->pboyer_->length() - 1),
                                            utype == mod_insert ? address : address + len,
                                            address,
                                            adjust));
            // Tell views to remove currently displayed occurrences
            CBGSearchHint bgsh(FALSE);
            UpdateAllViews(NULL, 0, &bgsh);
        }
        else
        {
            FixFound(address - (aa->pboyer_->length() - 1),
                     utype == mod_insert ? address : address + len,
                     address,
                     adjust);

#if 0 // xxx also fix Undo
            // Signal views to update display for changed search string occurrences
            if (utype == mod_replace || utype == mod_repback)
            {
                // Just invalidate replaced area (plus overlaps at ends for search string size)
                CBGSearchHint bgsh(address - aa->pboyer_->length() + 1, address + len);
                UpdateAllViews(NULL, 0, &bgsh);
            }
            else if (utype == mod_insert)
            {
                // Invalidate to new EOF
                CBGSearchHint bgsh(address - aa->pboyer_->length() + 1, length_ + adjust);
                UpdateAllViews(NULL, 0, &bgsh);
            }
            else
            {
                // Invalidate to old EOF
                CBGSearchHint bgsh(address - aa->pboyer_->length() + 1, length_);
                UpdateAllViews(NULL, 0, &bgsh);
            }
#else
            // Invalidate area of change (for deletions/insertions displayed bit towards EOF is invalidated below)
            CBGSearchHint bgsh(address - aa->pboyer_->length() + 1, address + len);
            UpdateAllViews(NULL, 0, &bgsh);
#endif
        }

        // Add new area to be searched
        if (utype == mod_delforw || utype == mod_delback)
            to_search_.push_back(pair<FILE_ADDRESS, FILE_ADDRESS>(address - (aa->pboyer_->length() - 1), address));
        else
            to_search_.push_back(pair<FILE_ADDRESS, FILE_ADDRESS>(address - (aa->pboyer_->length() - 1), address + len));

        // Restart bg search thread in case it is waiting
        view_update_ = FALSE;
        TRACE1("Restarting bg search (change) for %p\n", this);
        start_event_.SetEvent();
    }

    // Adjust file length according to bytes added or removed
    if (utype == mod_delforw || utype == mod_delback)
        length_ -= len;
    else if (utype == mod_insert)
        length_ += len;
    else if (utype == mod_replace && address + len > length_)
        length_ = address + len;
    else
        ASSERT(utype == mod_replace);

    // Rebuild the location list
    regenerate();

    // Update views to show changed doc
    CHexHint hh(utype, len, address, pview, index);
    if (utype == mod_insert && len == 0)
    {
        // Insert hex changed low nybble without inserting
        hh.utype = mod_replace;
        hh.len = 1;
    }
    SetModifiedFlag(TRUE);
    UpdateAllViews(NULL, 0, &hh);
}

// Undo removes the last change made by calling Change() [above]
//   pview is a pointer to the view requesting the undo
//   index is the undo index [at the moment should always be undo_.size()-1]
//   same_view is true if the view undo change originally made the change
BOOL CHexEditDoc::Undo(CView *pview, int index, BOOL same_view)
{
    ASSERT(index == undo_.size() - 1);
    ASSERT(undo_.back().utype == mod_insert  ||
           undo_.back().utype == mod_replace ||
           undo_.back().utype == mod_delforw ||
           undo_.back().utype == mod_delback ||
           undo_.back().utype == mod_repback );

    // If undoing change made in different view ask the user to confirm
    if (!same_view)
    {
        CString mess;
        mess.Format("Undo %s made in different window?",
            undo_.back().utype == mod_insert  ? "insertion" :
            undo_.back().utype == mod_replace ? "change" :
            undo_.back().utype == mod_repback ? "backspace" : "deletion");
        if (::HMessageBox(mess, MB_OKCANCEL) == IDCANCEL)
            return FALSE;
    }

    // Inform views to undo everything up to last doc change
    // before changing the document.  This ensures that any view changes
    // that rely on the document as it was are undone before the document
    // is change is undone.
    CUndoHint uh(pview, index);
    UpdateAllViews(NULL, 0, &uh);

    // Work out the hint which says where and how the document
    // has changed.  Note that undoing an insert is equivalent
    // to deleting bytes, undoing a delete == insert etc.
    CHexHint hh(undo_.back().utype == mod_insert ? mod_delforw :
                undo_.back().utype == mod_replace ? mod_replace :
                undo_.back().utype == mod_repback ? mod_replace : mod_insert,
                undo_.back().len, undo_.back().address, pview, index, TRUE);

    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // Lock the doc data (automatically releases the lock when it goes out of scope)
        CSingleLock sl(&docdata_, TRUE);

        // If there is a current search string and background searches are on
        if (aa->pboyer_ != NULL && pthread_ != NULL)
        {
            ASSERT(aa->bg_search_);

            long adjust;

            if (undo_.back().utype == mod_delforw || undo_.back().utype == mod_delback)
                adjust = undo_.back().len;
            else if (undo_.back().utype == mod_insert)
                adjust = -undo_.back().len;
            else if (undo_.back().utype == mod_replace || undo_.back().utype == mod_repback)
                adjust = 0;
            else
                ASSERT(0);

            // Adjust pending search addresses
            std::list<pair<FILE_ADDRESS, FILE_ADDRESS> >::iterator pcurr, pend;
            for (pcurr = to_search_.begin(), pend = to_search_.end(); pcurr != pend; ++pcurr)
            {
                if (pcurr->first >= undo_.back().address)
                {
                    pcurr->first += adjust;
                    if (pcurr->first < undo_.back().address)  // Make sure it doesn't go too far backwards
                        pcurr->first = undo_.back().address;
                }
                if (pcurr->second >= undo_.back().address)
                {
                    pcurr->second += adjust;
                    if (pcurr->second < undo_.back().address)  // Make sure it doesn't go too far backwards
                        pcurr->second = undo_.back().address;
                }
            }

            // Remove invalidated found_ occurrences and adjust those for insertiosn deletions
            if (!to_search_.empty())
            {
                // Ask bg thread to do it so that changes are kept consistent
                ASSERT(!view_update_);
                to_adjust_.push_back(adjustment(undo_.back().address - (aa->pboyer_->length() - 1),
                            undo_.back().utype == mod_delforw || undo_.back().utype == mod_delback ?
                                undo_.back().address : undo_.back().address + undo_.back().len,
                            undo_.back().address,
                            adjust));

                // Tell views to remove currently displayed occurrences
                CBGSearchHint bgsh(FALSE);
                UpdateAllViews(NULL, 0, &bgsh);
            }
            else
            {
                FixFound(undo_.back().address - (aa->pboyer_->length() - 1),
                         undo_.back().utype == mod_delforw || undo_.back().utype == mod_delback ? 
                             undo_.back().address : undo_.back().address + undo_.back().len,
                         undo_.back().address,
                         adjust);

#if 0
                // Signal view to update display for changed search string occurrences
                if (undo_.back().utype == mod_replace || undo_.back().utype == mod_repback)
                {
                    // Just invalidate replaced area (plus overlaps at ends for search string size)
                    CBGSearchHint bgsh(undo_.back().address - aa->pboyer_->length() + 1, undo_.back().address + undo_.back().len);
                    UpdateAllViews(NULL, 0, &bgsh);
                }
                else if (undo_.back().utype == mod_insert)
                {
                    // Invalidate to old EOF
                    CBGSearchHint bgsh(undo_.back().address - aa->pboyer_->length() + 1, length_);
                    UpdateAllViews(NULL, 0, &bgsh);
                }
                else
                {
                    // Invalidate to new EOF
                    CBGSearchHint bgsh(undo_.back().address - aa->pboyer_->length() + 1, length_ + adjust);
                    UpdateAllViews(NULL, 0, &bgsh);
                }
#else
            // Invalidate area of change (for deletions/insertions displayed bit towards EOF is invalidated below)
            CBGSearchHint bgsh(undo_.back().address - aa->pboyer_->length() + 1, undo_.back().address + undo_.back().len);
            UpdateAllViews(NULL, 0, &bgsh);
#endif
            }

            // Add new area to be searched
            if (undo_.back().utype == mod_insert)
                to_search_.push_back(pair<FILE_ADDRESS, FILE_ADDRESS>
                                         (undo_.back().address - aa->pboyer_->length() + 1,
                                          undo_.back().address));
            else
                to_search_.push_back(pair<FILE_ADDRESS, FILE_ADDRESS>
                                         (undo_.back().address - aa->pboyer_->length() + 1,
                                          undo_.back().address + undo_.back().len));

            // Restart bg search thread in case it is waiting
            view_update_ = FALSE;
            TRACE1("Restarting bg search (undo) for %p\n", this);
            start_event_.SetEvent();
        }

        // Recalc doc size if nec.
        if (undo_.back().utype == mod_delforw || undo_.back().utype == mod_delback)
            length_ += undo_.back().len;
        else if (undo_.back().utype == mod_insert)
            length_ -= undo_.back().len;
        else if (undo_.back().utype == mod_replace &&
                 undo_.back().address + undo_.back().len > length_)
            length_ = undo_.back().address + undo_.back().len;

        // Remove the change from the undo array since it has now been undone
        undo_.pop_back();
        if (undo_.size() == 0)
            SetModifiedFlag(FALSE);     // Undid everything so clear changed flag

        // Rebuild locations list as undo array has changed
        regenerate();
    }

    // Update views because doc contents have changed
    UpdateAllViews(NULL, 0, &hh);
    return TRUE;
}

#if 0
// Adjust addresses in file and remove invalidated already found addresses
// start, end = the range of found addresses to remove
// address = where address adjustments start, adjust = amount to adjust
void CHexEditDoc::fix_address(FILE_ADDRESS start, FILE_ADDRESS end,
                              FILE_ADDRESS address, long adjust)
{
    // Remove invalidated found_ occurrences and adjust those for insertiosn deletions
    if (!to_search_.empty())
    {
        // Ask bg thread to do it so that changes are kept consistent
        ASSERT(!view_update_);
        to_adjust_.push_back(adjustment(start, end, address, adjust));
    }
    else
        FixFound(start, end, address, adjust);

    // Adjust pending search addresses
    std::list<pair<FILE_ADDRESS, FILE_ADDRESS> >::iterator pcurr, pend;
    for (pcurr = to_search_.begin(), pend = to_search_.end(); pcurr != pend; ++pcurr)
    {
        if (pcurr->first >= address)
        {
            pcurr->first += adjust;
            if (pcurr->first < address)  // Make sure it doesn't go too far backwards
                pcurr->first = address;
        }
        if (pcurr->second >= address)
        {
            pcurr->second += adjust;
            if (pcurr->second < address)  // Make sure it doesn't go too far backwards
                pcurr->second = address;
        }
    }
}
#endif

// Returns TRUE if all changes to the document are only overtyping of
// parts of original file.  That is, all parts of the original file that
// would be written by a save are not moved from their orig. location.
// Ie. changes are replaces or inserts with matching deletes.
BOOL CHexEditDoc::only_over()
{
    long pos;
    ploc_t pl;          // Current location record

    // Check that each file record is at right place in file
    for (pos = 0L, pl = loc_.begin(); pl != loc_.end(); pos += (*pl).len, ++pl)
        if ((*pl).location == loc_file && (*pl).fileaddr != pos)
            return FALSE;
    // Make sure file length not changed
    if (file_.m_hFile == CFile::hFileNull || pos != file_.GetLength())
        return FALSE;

    return TRUE;
}

// Write changes from memory to file (all mods must be 'in place')
void CHexEditDoc::WriteInPlace()
{
    ASSERT(file_.m_hFile != CFile::hFileNull);

    long pos;
    ploc_t pl;          // Current location record

    // Lock the doc data (automatically releases the lock when it goes out of scope)
    CSingleLock sl(&docdata_, TRUE);

    // Write in memory bits at appropriate places in file
    try
    {
        for (pos = 0L, pl = loc_.begin(); pl != loc_.end(); pos += (*pl).len, ++pl)
            if ((*pl).location == loc_mem)
            {
                file_.Seek(pos, CFile::begin);
                file_.Write((*pl).memaddr, (*pl).len);
            }
            else
                ASSERT((*pl).fileaddr == pos);
    }
    catch (CFileException *fe)          // xxx test for one of these errors somehow?
    {
        // Display info about why the open failed
        CString mess;
        mess.Format("Error writing file");

        switch (fe->m_cause)
        {
        case CFileException::sharingViolation:
        case CFileException::lockViolation:
            mess = "File locked error";
            break;
        case CFileException::hardIO:
            mess += "\r- hardware error";
            break;
        case CFileException::diskFull:
            mess += "\r- disk full";
            break;
        }
        fe->Delete();
        ::HMessageBox(mess);

        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 10;
        return;
    }

    ASSERT(pos == file_.GetLength());
}

// Write the document (or part thereof) to file with name 'filename'.
// The range to write is given by 'start' and 'end'.
BOOL CHexEditDoc::WriteData(const CString filename, long start, long end)
{
    const size_t copy_buf_len = 16384;
    CFile ff;
    CFileException fe;

    // Open the file to write to
    if (!ff.Open(filename,
        CFile::modeCreate|CFile::modeWrite|CFile::shareExclusive|CFile::typeBinary,
        &fe) )
    {
        // Display info about why the open failed
        CString mess;
        mess.Format("File \"%s\"",filename);
        CFileStatus fs;

        switch (fe.m_cause)
        {
        case CFileException::badPath:
            mess += "\ris an invalid file name";
            break;
        case CFileException::tooManyOpenFiles:
            mess += "\r- too many files already open";
            break;
        case CFileException::directoryFull:
            mess += "\r- directory is full";
            break;
        case CFileException::accessDenied:
            if (!CFile::GetStatus(filename, fs))
                mess += "\rcannot be created";
            else
            {
                if (fs.m_attribute & CFile::directory)
                    mess += "\ris a directory";
                else if (fs.m_attribute & (CFile::volume|CFile::hidden|CFile::system))
                    mess += "\ris a special file";
                else if (fs.m_attribute & CFile::readOnly)
                    mess += "\ris a read only file";
                else
                    mess += "\rcannot be used (reason unknown)";
            }
            break;
        case CFileException::sharingViolation:
        case CFileException::lockViolation:
            mess += "\ris in use";
            break;
        case CFileException::hardIO:
            mess += "\r- hardware error";
            break;
        default:
            mess += "\rcould not be opened (reason unknown)";
            break;
        }
        ::HMessageBox(mess);

        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 10;
        return FALSE;
    }

    // Get memory for a buffer
    unsigned char *buf = new unsigned char[copy_buf_len];   // Where we store data
    size_t got;                                 // How much we got from GetData

    // Copy the range to file catching exceptions (probably disk full)
    try
    {
        for (long address = start; address < end; address += (long)got)
        {
            got = GetData(buf, min(end-address, copy_buf_len), address);
            ASSERT(got > 0);

            ff.Write(buf, got);
        }
        ASSERT(address == end);
    }
    catch (CFileException *fe)
    {
        // Display info about why the copy failed
        CString mess;
        mess.Format("Error writing file");

        switch (fe->m_cause)
        {
        case CFileException::sharingViolation:
        case CFileException::lockViolation:
            mess = "File locked error";
            break;
        case CFileException::hardIO:
            mess += "\r- hardware error";
            break;
        case CFileException::diskFull:
            mess += "\r- disk full";
            break;
        }

        fe->Delete();           // Pretty odd, but required by MFC

        // Close and delete the (probably incomplete) file
        ff.Close();
        remove(filename);
        delete[] buf;

        ::HMessageBox(mess);

        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 10;
        return FALSE;
    }

    ff.Close();
    delete[] buf;
    return TRUE;
}

void CHexEditDoc::regenerate()
{
    pundo_t pu;         // Current modification (undo record) being checked
    long pos;           // Tracks file position of current location record
    ploc_t pl;          // Current location record

    // Rebuild locations list starting with original file as only loc record
    loc_.clear();
    if (file_.m_hFile != CFile::hFileNull && file_.GetLength() > 0)
        loc_.push_back(doc_loc(0L, file_.GetLength()));

    // Now check each modification in order to build up location list
    for (pu = undo_.begin(); pu != undo_.end(); ++pu)
    {
        // Find loc record where this modification starts
        for (pos = 0, pl = loc_.begin(); pl != loc_.end(); pos += (*pl).len, ++pl)
            if (pu->address < pos + (*pl).len)
                break;

        // Modify locations list here (according to type of mod)
        // Note: loc_add & loc_del may modify pos and pl (passed by reference)
        switch (pu->utype)
        {
        case mod_insert:
            loc_add(pu, pos, pl);       // Insert record into list
            break;
        case mod_replace:
        case mod_repback:
            loc_del(pu->address, pu->len, pos, pl); // Delete what's replaced
            loc_add(pu, pos, pl);       // Add replacement before next record
            break;
        case mod_delforw:
        case mod_delback:
            loc_del(pu->address, pu->len, pos, pl); // Just delete them
            break;
        default:
            ASSERT(0);
        }
    } /* for */

#ifndef NDEBUG
    // Check that the length of all the records gels with stored doc length
    for (pos = 0, pl = loc_.begin(); pl != loc_.end(); pos += (*pl).len, ++pl)
        ; /* null statement*/

    ASSERT(pos == length_);
#endif
}

// loc_add inserts a record into the location list
// pu describes the record to insert
// pos is the location in the doc of "pl"
// - on return is the new location of pl (the rec after the one inserted)
// pl is the record before or in which the insertion is to take place
// - on return is the rec after one inserted (!= entry value if rec split)
// (if pl is the end of the list then we just append)
void CHexEditDoc::loc_add(pundo_t pu, long &pos, ploc_t &pl)
{
    if (pu->address != pos)
    {
        // We need to split a block into 2 to insert between the two pieces
        loc_split(pu->address, pos, pl);
        pos += (*pl).len;
        ++pl;
    }
    loc_.insert(pl, doc_loc(pu->ptr, pu->len));
    pos += pu->len;
}

// loc_del deletes record(s) or part(s) thereof from the location list
// It returns the next undeleted record [which may be loc_.end()]
// address is where the deletions are to commence
// len is the number of bytes to be deleted
// pos is the location in the doc of "pl"
// - on return it's the loc of "pl" which may be different if a record was split
// pl is the record from or in which the deletion is to start
// - on return points to the record after the deletion [may be loc_.end()]
// Note: loc_del can be called for a mod_replace modification.  Replacements
// can go past EOF so deleting past EOF is also required to be handled here.
void CHexEditDoc::loc_del(long address, size_t len, long &pos, ploc_t &pl)
{
    ploc_t byebye;              // Saved current record to be deleted

    if (address != pos)
    {
        ASSERT(pl != loc_.end());

        // We need to split this block so we can erase just the 2nd bit of it
        loc_split(address, pos, pl);
        pos += (*pl).len;
        ++pl;
    }

    // Erase all the blocks until we get a block or part thereof to keep
    // or we hit end of document location list (EOF)
    size_t deleted = 0;
    while (pl != loc_.end() && len >= deleted + (*pl).len)
    {
        byebye = pl;            // Save current record to delete later
        deleted += (*pl).len;   // Keep track of how much we've seen
        ++pl;                   // Advance to next rec before deleting current
        loc_.erase(byebye);
    }

    if (pl != loc_.end() && len > deleted)
    {
        // We need to split this block and erase the 1st bit
        loc_split(address + len, address + deleted, pl);
        byebye = pl;
        deleted += (*pl).len;
        ++pl;
        loc_.erase(byebye);
    }
//    ASSERT(deleted == len);
}

void CHexEditDoc::loc_split(long address, long pos, ploc_t pl)
{
    ASSERT(pl != loc_.end());
    ASSERT(address > pos && address < pos + (*pl).len);

    // Work out exactly where to split the record
    size_t split = pos + (*pl).len - address;

    // Insert a new record before the next one and store location and length
    ploc_t plnext = pl; ++plnext;
    ASSERT((*pl).location == loc_file || (*pl).location == loc_mem);
    if ((*pl).location == loc_file)
        loc_.insert(plnext, doc_loc((*pl).fileaddr + (*pl).len - split, split));
    else
        loc_.insert(plnext, doc_loc((*pl).memaddr + (*pl).len - split, split));

    // Orig record now smaller by amount split off
    (*pl).len -= split;
}
