/*
  tad.c - Time and Date.

  Display the current time and date; set (optionally without display) the time
  and date from the real-time clock or from the command line; correct for
  real-time clock inaccuracies.

  Jason Hood, 14 & 15 January, 1997.

  30 November & 1 December, 1998:
    reformatted;
    able to set time and date from command line.

  3 April, 1999:
    display of "approximate" time.

  10 May, 2000:
    clock display.

  30 & 31 May, 3 June, 2000:
    rewrote option processing (and modified parse() accordingly);
    added correction to counter RTC drift;
    added option to set system time, but not RTC;
    return the current 10-minute interval (midnight is 0, noon is 72);
    used Time and Date structures.

  15 to 21 June, 2000, v1.00:
    corrected bug when displaying calibration end (was using same Date pointer);
    used a function to display calibration time (and removed seconds) and
     seconds-a-day calculation;
    improved all the display output;
    added get_ and set_{system,rtc}_tad() and {read,write}_calib();
    display of "brief" time;
    added copyright and version info for distribution;
    recognize '/' as a switch character;
    use brief option to shorten the duration display.

  15 July, 2000, v1.01:
    more reliable monitoring;
    quiet clock - remove display when finished.


  Feel free to use any portion of this code for your own use.
*/


#define VERSION "1.01"
#define DATE	"15 July, 2000"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <conio.h>
#include <io.h>


typedef struct
{
  int hour,
      minute,
      second;
} Time;

typedef struct
{
  int day,
      month,
      year;
} Date;


char* monthname[] = { "January",   "February", "March",    "April",
		      "May",       "June",     "July",     "August",
		      "September", "October",  "November", "December" };

char* dayname[] = { "Sunday",   "Monday", "Tuesday", "Wednesday",
		    "Thursday", "Friday", "Saturday" };

char* hourname[] = { "twelve", "one",  "two", "three",
		     "four",   "five", "six", "seven",
		     "eight",  "nine", "ten", "eleven" };

char* fives[] = { "o'clock", "Five", "Ten", "Quarter",
		  "Twenty",  "Twenty-five", "Half" };

char* rel[] = { "past", "to" };


int approx  = 0;
int brief   = 0;
int cont    = 0;
int rtc2sys = 0;
int leave   = 0;
int calib   = 0;
int ref     = 0;
int monitor = 0;
int drift   = 0;
int view    = 0;
int quiet   = 0;

char* tadfile;

int dayofweek;


void get_system_time( Time* );
void get_system_date( Date* );
void get_system_tad( Time*, Date* );

void set_system_time( Time* );
void set_system_date( Date* );
void set_system_tad( Time*, Date* );

void get_rtc_time( Time* );
void get_rtc_date( Date* );
void get_rtc_tad( Time*, Date* );

void set_rtc_time( Time* );
void set_rtc_date( Date* );
void set_rtc_tad( Time*, Date* );

void parse( char*, Time*, Date*	);

void read_calib( FILE*, long*, long*, long* );
void write_calib( FILE*, long, long, long );
void calibrate( Time*, Date*, Time*, Date* );
void update_ref( void );
void monitor_time( void );
void adjust( void );
void view_calibration( void );

char* str_time( Time*, Date* );
char* str_dur( long );
void  secs_per_day( long, long );

long time_to_secs( Date*, Time* );
void secs_to_time( long, Date*, Time* );


#define BCD2BIN( bcd ) ((((bcd) & 0xf0) >> 4) * 10 + ((bcd) & 0x0f))
#define BIN2BCD( bin ) ((((bin) / 10) << 4) | ((bin) % 10))


void help( void )
{
fputs(
"\n"
"Time and Date by Jason Hood <jadoxa@hotmail.com>.\n"
"Version "VERSION" ("DATE"). Freeware.\n"
"http://adoxa.homepage.com/tad/\n"
"\n"
"tad [-abcsknrmdvq] [hh:[mm[:ss]]] [dd[/mm[/yy[yy]]]]\n"
"\n"
"<none>   display the time, day and date\n"
"-a       display the approximate time\n"
"-b       display the date and time briefly (concisely)\n"
"-c       display as a clock\n"
"-s       set the system time/date from the real-time clock\n"
"-k       set system time/date, keep the RTC as is\n"
"-n       start a new calibration\n"
"-r       take the given time/use the current system time as reference\n"
"-m       monitor the time for external update\n"
"-d       correct for drift\n"
"-v       view calibration details\n"
"-q       quiet, no display\n"
"\n"
"Notes: Hour is 24-hour format.\n"
"       Two-digit year is assumed 1980 to 2079.\n"
"       Missing minute or second defaults to zero.\n"
"       Missing month or year defaults to current.\n"
"       Return value is in 10-minute intervals (midnight is 0, noon is 72).\n"
"       Calibration/reference/monitor should not be done near midnight.\n"
, stdout);
}


int main( int argc, char *argv[] )
{
  Time t = { -1, -1, -1 }, t1;
  Date d = { -1, -1, -1 }, d1;
  int  special = 0;
  int  j, k;

  tadfile = malloc( (j = strlen( argv[0] )) + 8 );
  strcpy( tadfile, argv[0] );
  while (argv[0][--j] != '\\') ;
  strcpy( tadfile + j + 1, "tad.dat" );

  for (j = 1; j < argc; ++j)
  {
    if (*argv[j] == '-' || *argv[j] == '/')
    {
      for (k = 1; argv[j][k] != 0; ++k)
      {
	switch (tolower(argv[j][k]))
	{
	  case 'a': ++approx;             break;
	  case 'b': ++brief;		  break;
	  case 'c': ++cont;               break;
	  case 's': ++rtc2sys; ++special; break;
	  case 'k': ++leave;   ++special; break;
	  case 'n': ++calib;   ++special; break;
	  case 'r': ++ref;     ++special; break;
	  case 'm': ++monitor; ++special; break;
	  case 'd': ++drift;   ++special; break;
	  case 'v': ++view;    ++special; break;
	  case 'q': ++quiet;              break;
	  case '/': 			  break;
	  default:
	    help();
	    return 255;
	}
      }
    }
    else
      parse( argv[j], &t, &d );
  }

  if (!quiet && isatty( fileno( stdout ) )) putchar( '\n' );

  if (!special) set_system_tad( &t, &d );

  if (leave)
  {
    if (t.hour == -1 && d.day == -1)
    {
      fputs( "It would help to specify a time and/or date.\n", stderr );
      return 255;
    }
    get_rtc_tad( &t1, &d1 );
    set_system_tad( &t, &d );
    set_rtc_tad( &t1, &d1 );
  }

  if (rtc2sys)
  {
    get_rtc_tad( &t, &d );
    set_system_tad( &t, &d );
  }

  if (ref)
  {
    if (access( tadfile, 0 ) != 0) ++calib;
    else if (t.hour == -1) update_ref();
    else
    {
      get_system_tad( &t1, &d );
      set_system_time( &t );
      calibrate( &t1, &d, &t, &d );
    }
  }

  if (calib && !monitor)
  {
    set_system_tad( &t, &d );
    calibrate( NULL, NULL, &t, &d );
  }

  if (monitor) monitor_time();

  if (drift) adjust();

  if (view) view_calibration();

  if (cont)
  {
    if (quiet) putch( '\n' );
    while (!kbhit())
    {
      get_system_time( &t );
      cprintf( "\r%2d:%02d:%02d", t.hour, t.minute, t.second );
    }
    while (kbhit()) getch();
    if (quiet)
    {
      cprintf( "\r        " );
      gotoxy( 1, wherey() - 1 );	// Counteract COMMAND.COM's newline
    }
    else
      putchar( '\n' );
  }
  else if (!quiet && !calib && !ref && !monitor && !drift && !view)
  {
    get_system_time( &t );
    if (approx)
    {
      int r = 0;
      if (t.hour > 11) t.hour -= 12;
      t.minute = ((t.minute * 60) + t.second + 150) / 300;
      if (t.minute > 6) t.minute = 12 - t.minute, ++t.hour, ++r;
      if (t.minute == 0)
	printf( "%c%s %s.\n",
		toupper( *hourname[t.hour] ), hourname[t.hour]+1, fives[0] );
      else
	printf( "%s %s %s.\n", fives[t.minute], rel[r], hourname[t.hour] );
    }
    else
    {
      get_system_date( &d );
      if (brief)
	printf( "%d%02d%02d%02d%02d%02d\n",
		d.year, d.month, d.day, t.hour, t.minute, t.second );
      else
	printf( "%2d:%02d:%02d  %s, %d %s, %d\n",
		t.hour, t.minute, t.second, dayname[dayofweek],
		d.day, monthname[d.month-1], d.year );
    }
  }

  get_system_time( &t );
  return (t.hour * 60 + t.minute) / 10;
}


void read_calib( FILE* dat, long* ref, long* diff, long* dur )
{
  fread( ref,  sizeof(long), 1, dat );
  fread( diff, sizeof(long), 1, dat );
  fread( dur,  sizeof(long), 1, dat );
}

void write_calib( FILE* dat, long ref, long diff, long dur )
{
  fwrite( &ref,  sizeof(long), 1, dat );
  fwrite( &diff, sizeof(long), 1, dat );
  fwrite( &dur,  sizeof(long), 1, dat );
}


void calibrate( Time* told, Date* dold, Time* tnew, Date* dnew )
{
  FILE* dat;
  long	sold, snew, ref;
  long	diff, dur;

  if (told == NULL)
  {
    get_system_tad( tnew, dnew );
    snew = time_to_secs( dnew, tnew );
    if (!quiet) printf( "Calibration started %s.\n", str_time( tnew, dnew ) );
    diff = dur = 0;
    goto new_calibration;
  }

  sold = time_to_secs( dold, told );
  snew = time_to_secs( dnew, tnew );

  if ((dat = fopen( tadfile, "rb" )) != NULL)
  {
    read_calib( dat, &ref, &diff, &dur );
    fclose( dat );
    if (!quiet)
    {
      Time tref;
      Date dref;
      secs_to_time( ref, &dref, &tref );
      if (dur == 0)
      {
	printf( "Calibration started %s.\n"
		"Calibration ended   %s.\n"
		"System time was     %s.\n"
		"\n"
		"Calibration period was %s,\n"
		"during which time the clock %s %s.\n",
		str_time( &tref, &dref ),
		str_time( tnew, dnew ),
		str_time( told, dold ),
		str_dur( snew - ref ),
		(snew <= sold) ? "gained" : "lost", str_dur( snew - sold ) );
      }
      else
      {
	printf( "Correct time: %s\n"
		"System  time: %s\n"
		"\n"
		"It was %s since the last adjustment\n"
		"or reference (which was at %s),\n"
		"resulting in a %s of %s.\n",
		str_time( tnew, dnew ),
		str_time( told, dold ),
		str_dur( snew - ref ),
		str_time( &tref, &dref ),
		(snew <= sold) ? "gain" : "loss", str_dur( snew - sold ) );
      }
    }
  }
  else
  {
    fprintf( stderr, "Unable to read calibration file %s!\n", tadfile );
    exit( 255 );
  }

  diff += snew - sold;
  dur  += sold - ref;

  if (!quiet) secs_per_day( diff, dur );

new_calibration:
  if ((dat = fopen( tadfile, "wb" )) != NULL)
  {
    write_calib( dat, snew, diff, dur );
    fclose( dat );
  }
  else
  {
    fprintf( stderr, "Unable to create calibration file %s!\n", tadfile );
    exit( 255 );
  }
}


void update_ref( void )
{
  FILE* dat;
  Time	t, tr;
  Date	d, dr;
  long	ref, old, diff, dur;

  if ((dat = fopen( tadfile, "rb+" )) != NULL)
  {
    get_system_tad( &t, &d );
    ref = time_to_secs( &d, &t );
    read_calib( dat, &old, &diff, &dur );
    rewind( dat );
    write_calib( dat, ref, diff, dur );
    fclose( dat );
    if (!quiet)
    {
      secs_to_time( old, &dr, &tr );
      printf( "%s %s.\n"
	      "\n"
	      "Previous reference or adjustment was at %s,\n"
	      "which was %s ago.\n",
	      (dur == 0) ? "Calibration restarted" :
			   "Reference set to", str_time( &t, &d ),
	      str_time( &tr, &dr ),
	      str_dur( ref - old ) );
    }
  }
  else
  {
    fprintf( stderr, "Unable to update calibration file %s!\n", tadfile );
    exit( 255 );
  }
}


void monitor_time( void )
{
  Time t, t1;
  Date d;
  long last, next, now;
  volatile long far* const ticks = (long far*)0x46c;
  static char* ind = "|/-\\";
  static char* str = "\b|";
  int p = 0;

  fputs( "Waiting for update...  ", stdout );
  while (!kbhit())
  {
    get_system_time( &t );
    last = *ticks;
    next = last + 4;
    while (*ticks < next && *ticks >= last)
      cputs( str );			// Need to be busy for Windows
    now = *ticks;
    if (now < last || now > next + 4)   // next+3 was the slowest I saw
    {
      get_system_tad( &t1, &d );
      puts( "\bFound." ); if (!quiet) putchar( '\n' );
      if (calib || access( tadfile, 0 ) != 0) calibrate( NULL, NULL, &t, &d );
      else calibrate( &t, &d, &t1, &d );
      break;
    }
    if (++p == 4) p = 0;
    str[1] = ind[p];
  }
  if (kbhit())
  {
    puts( "\bAborted." );
    while (kbhit()) getch();
  }
}


void adjust( void )
{
  FILE* dat;
  Time	t, tn, tr;
  Date	d, dn, dr;
  long	s, n, r, diff, dur;

  if (access( tadfile, 0 ) != 0)
    puts( "Calibration has not been started." );
  else if ((dat = fopen( tadfile, "rb+" )) == NULL)
  {
    fprintf( stderr, "Unable to open calibration file %s!\n.", tadfile );
    exit( 255 );
  }
  else
  {
    read_calib( dat, &r, &diff, &dur );
    get_system_tad( &t, &d );
    s = time_to_secs( &d, &t );
    if (dur != 0)
    {
      n = s + (s - r) * diff / dur;
      secs_to_time( n, &dn, &tn );
      set_system_tad( &tn, &dn );
      if (!quiet)
      {
	secs_to_time( r, &dr, &tr );
	printf( "Expected time: %s\n"
		"Current  time: %s (a %s of %s)\n"
		"\n"
		"It was %s since the last adjustment\n"
		"(which was at %s).\n",
		str_time( &tn, &dn ),
		str_time( &t, &d ),
		(n <= s) ? "gain" : "loss", str_dur( n - s ),
		str_dur( n - r ),
		str_time( &tr, &dr ) );
      }
      diff += n - s;
      dur  += s - r;
      rewind( dat );
      write_calib( dat, n, diff, dur );
    }
    else if (!quiet)
    {
      secs_to_time( r, &dr, &tr );
      printf( "Calibration started at %s\n"
	      "and has been in progress for %s.\n",
	      str_time( &tr, &dr ), str_dur( s - r ) );
    }
    fclose( dat );
  }
}


void view_calibration( void )
{
  FILE* dat;
  Time	t, ts;
  Date	d, ds;
  long	ref, diff, dur, s;

  if (access( tadfile, 0 ) != 0)
    puts( "Calibration has not been started." );
  else if ((dat = fopen( tadfile, "rb" )) == NULL)
  {
    fprintf( stderr, "Unable to open calibration file %s!\n.", tadfile );
    exit( 255 );
  }
  else
  {
    read_calib( dat, &ref, &diff, &dur );
    fclose( dat );
    secs_to_time( ref, &d, &t );
    get_system_tad( &ts, &ds );
    s = time_to_secs( &ds, &ts ) - ref;
    if (dur == 0)
    {
      printf( "Calibration started at %s\n"
	      "and has been in progress for %s.\n",
	      str_time( &t, &d ),
	      str_dur( s ) );
    }
    else
    {
      printf( "Last reference or adjustment was at %s,\n"
	      "which was %s ago.\n",
	      str_time( &t, &d ),
	      str_dur( s ) );
      secs_per_day( diff, dur );
      printf( "(An overall %s of %s%s over %s.)\n",
	      (diff < 0) ? "gain" : "loss", str_dur( diff ),
	      (brief) ? "" : "\n", str_dur( diff + dur ) );
    }
  }
}


// str_time() and str_dur() allocate memory from the heap, which is not free'd.
// This is bad programming but saves having to declare variables. (I originally
// had static arrays, but that wouldn't work because I call it more than once
// in the same printf().)

char* str_time( Time* t, Date* d )
{
  // 12345678901234567890
  // 23:59:59 31/12/99
  char* time_str = malloc( 18 );

  sprintf( time_str, "%d:%02d:%02d %d/%d/%02d",
	   t->hour, t->minute, t->second, d->day, d->month, d->year % 100 );

  return time_str;
}


#define NUMBER(num) (num), "s"+((num) == 1)     // A great trick by Bob Stout.

char* str_dur( long secs )
{
  // 12345678901234567890123456789012345678901234567890
  // 365 days, 23 hours, 59 minutes and 59 seconds
  // 365d 23h 59m 59s (using brief option)
  char* dur_str = malloc( 46 );
  int	days, hours, minutes, seconds;
  int	pos;


  if (secs == 0) strcpy( dur_str, "less than a second" );
  else
  {
    if (secs < 0) secs = -secs;

    days    = (int)(secs / 86400L); secs %= 86400L;
    hours   = (int)(secs / 3600);   secs %= 3600;
    minutes = (int)secs / 60;
    seconds = (int)secs % 60;

    if (brief)
    {
      pos = (days) ? sprintf( dur_str, "%dd", days ) : 0;
      if (hours)
	pos += sprintf( dur_str+pos, "%s%dh",
			(days) ? " " : "", hours );
      if (minutes)
	pos += sprintf( dur_str+pos, "%s%dm",
			(days || hours) ? " " : "", minutes );
      if (seconds)
	pos += sprintf( dur_str+pos, "%s%ds",
			(days || hours || minutes) ? " " : "", seconds );
    }
    else
    {
      pos = (days) ? sprintf( dur_str, "%d day%s", NUMBER(days) ) : 0;
      if (hours)
	pos += sprintf( dur_str+pos, "%s%d hour%s",
			(!days) ? "" : (minutes || seconds) ? ", " : " and ",
			NUMBER(hours) );
      if (minutes)
	pos += sprintf( dur_str+pos, "%s%d minute%s",
			(!days && !hours) ? "" : (seconds) ? ", " : " and ",
			NUMBER(minutes) );
      if (seconds)
	       sprintf( dur_str+pos, "%s%d second%s",
			(!days && !hours && !minutes) ? "" : " and ",
			NUMBER(seconds) );
    }
  }

  return dur_str;
}


void secs_per_day( long diff, long dur )
{
  double f;
  long	 d;
  char*  s;

  if (diff < 0) d = -diff, s = "gains";
  else		d =  diff, s = "loses";
  f = 86400L * d / (double)(dur + diff);
  printf( "\nThe clock %s %.2f seconds a day.\n", s, f );
}


void parse( char* str, Time* t, Date* d )
{
  int	hd, mm = -1, sy = -1;
  int	seen_time;
  char* bp;

  hd = (int)strtol( str, &bp, 10 );
  seen_time = (*bp == ':');
  if (*bp != 0)
  {
    mm = (int)strtol( bp+1, &bp, 10 );
    if (*bp != 0)
      sy = (int)strtol( bp+1, &bp, 10 );
  }
  if (seen_time)
  {
    t->hour   = hd;
    t->minute = mm;
    t->second = (sy == -1) ? 0 : sy;
  }
  else
  {
    if (sy == -1) get_system_date( d );
    d->day = hd;
    if (mm != -1)
    {
      d->month = mm;
      if (sy != -1)
      {
	if (sy < 100) sy += (sy >= 80) ? 1900 : 2000;
	d->year = sy;
      }
    }
  }
}


void get_system_time( Time* t )
{
  union REGS regs;

  regs.h.ah = 0x2c;
  intdos( &regs, &regs );
  t->hour   = regs.h.ch;
  t->minute = regs.h.cl;
  t->second = regs.h.dh;
}

void get_system_date( Date* d )
{
  union REGS regs;

  regs.h.ah = 0x2a;
  intdos( &regs, &regs );
  d->year   = regs.x.cx;
  d->month  = regs.h.dh;
  d->day    = regs.h.dl;
  dayofweek = regs.h.al;
}

void get_system_tad( Time* t, Date* d )
{
  get_system_time( t );
  get_system_date( d );
}


void set_system_time( Time* t )
{
  union REGS regs;

  regs.h.ch = t->hour;
  regs.h.cl = t->minute;
  regs.h.dh = t->second;
  regs.h.dl = 10;			// It goes back 1 to 6 hundredths,
  regs.h.ah = 0x2d;			//  depending on the time being set.
  intdos( &regs, &regs );
}

void set_system_date( Date* d )
{
  union REGS regs;

  regs.x.cx = d->year;			// 1980 - 2099
  regs.h.dh = d->month;
  regs.h.dl = d->day;
  regs.h.ah = 0x2b;
  intdos( &regs, &regs );
}

void set_system_tad( Time* t, Date* d )
{
  set_system_time( t );
  set_system_date( d );
}


void get_rtc_time( Time* t )
{
  union REGS regs;

  regs.h.ah = 0x02;
  int86( 0x1a, &regs, &regs );
  t->hour   = BCD2BIN( regs.h.ch );
  t->minute = BCD2BIN( regs.h.cl );
  t->second = BCD2BIN( regs.h.dh );
}

void get_rtc_date( Date* d )
{
  union REGS regs;

  regs.h.ah = 0x04;
  int86( 0x1a, &regs, &regs );
  d->year  = BCD2BIN( regs.h.ch ) * 100 + BCD2BIN( regs.h.cl );
  d->month = BCD2BIN( regs.h.dh );
  d->day   = BCD2BIN( regs.h.dl );
}

void get_rtc_tad( Time* t, Date* d )
{
  get_rtc_time( t );
  get_rtc_date( d );
}


void set_rtc_time( Time* t )
{
  union REGS regs;

  regs.h.ch = BIN2BCD( t->hour );
  regs.h.cl = BIN2BCD( t->minute );
  regs.h.dh = BIN2BCD( t->second );
  regs.h.ah = 0x03;
  int86( 0x1a, &regs, &regs );
}

void set_rtc_date( Date* d )
{
  union REGS regs;

  regs.h.ch = BIN2BCD( d->year / 100 );
  regs.h.cl = BIN2BCD( d->year % 100 );
  regs.h.dh = BIN2BCD( d->month );
  regs.h.dl = BIN2BCD( d->day );
  regs.h.ah = 0x05;
  int86( 0x1a, &regs, &regs );
}

void set_rtc_tad( Time* t, Date* d )
{
  set_rtc_time( t );
  set_rtc_date( d );
}


/*
** scalar date routines    --    public domain by Ray Gardner
** Numerically, these will work over the range 1/01/01 thru 14699/12/31.
** Practically, these only work from the beginning of the Gregorian 
** calendar thru 14699/12/31.  The Gregorian calendar took effect in
** much of Europe in about 1582, some parts of Germany in about 1700, in
** England and the colonies in about 1752ff, and in Russia in 1918.
*/
// From the C Snippets 9707, by Bob Stout. Slightly modified.

#define D19800101 722815L

int isleap( unsigned year )
{
  return ((year & 3) == 0 && (year % 100 != 0 || year % 400 == 0));
}

unsigned months_to_days( unsigned month )
{
  return (month * 3057 - 3007) / 100;
}

long years_to_days( unsigned year )
{
  return (year * 365L + (year >> 2) - year / 100 + year / 400);
}

long time_to_secs( Date* d, Time* t )
{
  long scalar = d->day + months_to_days( d->month );
  if (d->month > 2) 			/* adjust if past February */
    scalar -= isleap( d->year ) ? 1 : 2;
  scalar += years_to_days( d->year - 1 );

  scalar -= D19800101;			// normalize to 1 January, 1980
  scalar *= 86400L;			// convert to seconds
  scalar += t->hour * 3600L + t->minute * 60 + t->second;
  return scalar;
}

void secs_to_time( long scalar, Date* d, Time* t )
{
   unsigned n;		      /* compute inverse of years_to_days() */
   long sec;

   sec = scalar % 86400L;
   t->hour   = (int)(sec / 3600); sec %= 3600;
   t->minute = (int)sec / 60;
   t->second = (int)sec % 60;

   scalar /= 86400L;
   scalar += D19800101;

   for (n = (unsigned)((scalar * 400L) / 146097L); years_to_days(n) < scalar;)
     n++;				/* 146097 == years_to_days(400) */
   d->year = n;
   n = (unsigned)(scalar - years_to_days( n-1 ));
   if (n > 59)				/* adjust if past February */
   {
     n += 2;
     if (isleap( d->year ))
       n -= n > 62 ? 1 : 2;
   }
   d->month = (n * 100 + 3007) / 3057;	/* inverse of months_to_days() */
   d->day   = n - months_to_days( d->month );
}
