/*  This program is a modification of graph
 * 
 *  The main body contains lines form the program graph(1) of the
 *  Unix distribution. Nevertheless it has been extensively
 *  modified and not much remains of graph(1). I do not know the
 *  complete hacking history, but by comparing it with existing
 *  code, it looks that graph(1) was modified in
 *
 *  june 83  - SLAC UGS added by John Richardson and Bob Pearson
 *  june 84  - options in the input and Sun viewer by Doug Toussaint
 *  may  87  - multiplots and error handling by Jim Gubernatis
 *             SunView support
 *  jan  89  - adapted xyplot to produce ascii output RM
 */

#include <stdio.h>
#include <math.h>
#include "patchlevel.h"
#define YES (1)
#define NO (0)

struct xy {
    int             xlbf, xubf; /* flag:explicit lower and upper bound */
    int             xqf;        /* flag:explicit quantum */
    double          (*xf) ();   /* transform function, e.g. log */
    float           xa, xb;     /* scaling coefficients */
    float           xlb, xub;   /* lower and upper bound */
    float           xquant;     /* quantum */
    float           xoff;       /* screen offset fraction */
    float           xsize;      /* screen fraction */
    int             xbot, xtop; /* screen coords of border */
    float           xmult;      /* scaling constant */
}               xd, yd;

struct datum {
    float           xv;		/* x coordinate */
    float           yv;		/* y coordinate */
    float           yerr;	/* optional error on y */
    char            line_mode;  /* current line mode */
    char            err_flag;   /* does data have an error bar */
    char            cflag;      /* conect to previous point ? */
    char           *lblptr;     /* optional plot symbol or label */
    float	    lblsiz;     /* the size of the symbol or label */
}              *xx;

#define LSIZ 1024      		/* size of each label buffer */
char           *labs;   	/* pointer to current label  */
int             labsleft = 0;   /* space left in allocated buffer */

# define TICKSZ 50     		/* standard tick size */
int             tick = TICKSZ;  /* current tick size */
int 		tickspertick = 5;  /* sub-divisions between big ticks */
int             topx = 4000;
int             botx = 750;
int             leftx = 300;
int             topy = 3750;
int             boty = 500;
float           absbot;

#define 	THINLIN  0.50		/* how thinner error bars should be */
float           symsize = 1.5;  	/* default symbol size */
float           lblsize = 1.5;  	/* default label character size */
float           titlesize = 2.0;        /* default title character size */
float           xtitlesize = 1.5;       /* default x title character size */
float           ytitlesize = 1.5;       /* default y title character size */
float		stdwidth = 1.0;		/* standard width factor */
float 		currentwidth = 1.0;	/* the current width at a time */
float           titlex = 0.0;
float           titley = 0.0;
float           xtitlex = 0.0;
float           xtitley = 0.0;
float           ytitlex = 0.0;
float           ytitley = 0.0;
int             titlexf = 0;
int             titleyf = 0;
int             xtitlexf = 0;
int             xtitleyf = 0;
int             ytitlexf = 0;
int             ytitleyf = 0;

#define BSIZ 256
char            linebuf[BSIZ];
char            titlebuf[BSIZ];
char            xtitlebuf[BSIZ];
char            ytitlebuf[BSIZ];
char 		xformat[50]="%g";  /* the default format for the x numbers */
char 		yformat[50]="%g";  /* the default format for the y numbers */
char           *plotsymb;

int             n = 0;          /* no. of data points */
int             erasf = 1;      /* erase screen ? */
int             gridf = 1;      /* grid style */
int             symbf = 0;      /* default plot symbol in effect ? */
int             absf = 0;       /* supply abscissas ? */
int             errf = 0;       /* data has error bars ? */
int             transf = 0;     /* transpose ? */
int             lbflag = 1;     /* break after labels ? */
int             cflag = 1;      /* connect to previous ? */
int		nextf = 1;      /* continue plotting? */
float           dx = 1.0;

char           *modes[] = {
                           "solid", /* point plot needs to draw solid lines! */
                           "solid",
                           "dotted",
                           "dotdashed",
                           "shortdashed",
                           "longdashed",
                           "dashdotdotted"
};
int             mode = 1;

char           *realloc();
char           *malloc();
char           *copystring();

int UseScannerQ;
char ProgName[50] = "xyplot"; /* the name of the program  */
FILE *inputfile,*outputfile;

double
ident(x)
    double          x;
{
    return (x);
}

float
modceil(f, t)
    float           f, t;
{
    t = fabs((double)t);
    t = t * ceil( (double)(f/t) );
    return( (t==0.0) ? 0 : t);
}

float
modfloor(f, t)
    float           f, t;
{
    t = fabs(t);
    t = t * floor( (double)(f/t) );
    return( (t==0.0) ? 0 : t);
}


main(argc, argv)
int argc;
char *argv[];
{
    extern FILE *inputfile,*outputfile;
    extern char ProgName[];

    strcpy(ProgName,argv[0]);

    readopts(argc,argv);

    while (nextf) {
        n=0;
        init(&xd);
        init(&yd);
        xx = (struct datum *) malloc((unsigned) sizeof(struct datum));
        readin(inputfile);
        space(0, 0, 4096, 4096);/* this size is hard-coded throughout */
        transpose();
        getxlim(&xd, xx);
        getylim(&yd, xx);
        xscale(&xd);
        yscale(&yd);
	if(!UseScannerQ) bbox(&xd,&yd);
        resetw();
        plot();
        axes();
        title();
        if (erasf)
            erase();
    }
    if (UseScannerQ) pclose(outputfile);
}

init(p)
    struct xy      *p;
{
    p->xlb = (float) (HUGE);
    p->xub = -(float) (HUGE);
    p->xsize = 1;
    p->xoff = 0;
    p->xf = ident;
    p->xmult = 1;
}


readin(inputfile)
    FILE           *inputfile;
{
    register char  *t, *tt;
    char           *index(), *argv[32];
    struct datum   *temp;
    int             i, inquote, argc, inarg;
    int             setopt();

    if (absf == 1) {
        if (xd.xlbf)
            absbot = xd.xlb;
        else if (xd.xf == log10)
            absbot = 1;
    }
    for (;;) {
        if (getline(inputfile, linebuf) == -1) {
            nextf = 0;
            erasf = 1;
            return;         /* End of file or garbage */
            }

        t = linebuf;
        /* find first nonwhite character */
        while (*t == ' ' || *t == '\t') t++;
        if (*t == '#') {        /* Control line */
            /* make argc and argv and then call setopt() */
            inarg = NO;
            inquote = NO;
            argc = 1;
            for (t++; *t != '\0'; t++) {        /* scan line */
                switch (*t) {
                case ' ':    /* replace white space by nulls except */
                case '\t':   /* in quotes. inarg=0 means next nonwhite is new */
                    if (!inquote) {
                        *t = '\0';
                        inarg = NO;
                    } else {
			if (!inarg)  argv[argc++]=t;
			if (argc > 32 ) break;
			inarg = YES;
		    }
                    break;
                case '"':       /* start or finish a quotation */
                    if (!inquote)
                        inquote = YES;
                    else {
                        *t = '\0';
                        inquote = NO;
                    }
                    break;
                default:
                    if (!inarg)
                        argv[argc++] = t;
                    if (argc > 32)
                        break;
                    inarg = YES;      /* inarg=1 means in middle of arg. */
                    break;
                }
            }
            if (setopt(argc, argv))
                return;
	} else {
            /* make a new structure for next value */
            temp = (struct datum *) realloc((char *) xx,
                                 (unsigned) (n + 1) * sizeof(struct datum));
            if (temp == NULL) {
                fprintf(stderr,
                     "%s:  Unable to allocate more memory for data structures",
		       ProgName);
                exit (-1);
            }
            xx = temp;

            /* read in the value */
            if (absf) {
                xx[n].xv = n * dx + absbot;
                if (errf) {
                    if (sscanf(linebuf, "%f%f", &xx[n].yv, &xx[n].yerr)
                        != 2)
                        continue;
                } else {
                    if (sscanf(linebuf, "%f", &xx[n].yv) != 1)
                        continue;
                }
            } else {
                if (errf) {
                    if (sscanf(linebuf, "%f%f%f", &xx[n].xv, &xx[n].yv,
                               &xx[n].yerr) != 3)
                        continue;
                } else {
                    if (sscanf(linebuf, "%f%f", &xx[n].xv, &xx[n].yv) != 2)
                        continue;
                }
            }
            xx[n].lblptr = NULL;
	    xx[n].lblsiz = symsize;
            if ((t = index(linebuf, '"')) != NULL) {
                if ((tt = index(++t, '"')) != NULL)
                    *tt = '\0';
                i = utou(t);
                xx[n].lblptr = copystring(t, i);
            } else if (symbf)
                xx[n].lblptr = plotsymb;

            /*
             * decide whether point should be connected to previous 
             */
            xx[n].cflag = cflag;
            cflag = 1;
            /*
             * if a label break is set and there is a symbol or an error bar,
             * set break for next point 
             */
            if (lbflag && (xx[n].lblptr != NULL || errf))
                cflag = 0;

            xx[n].line_mode = mode;
            xx[n].err_flag = errf;
            n++;
        }
    }
}

int
setopt(argc, argv)
    int            argc;
    char           *argv[];
{
    char           *p1, *p2;
    register int    i, j;

    /* an options line sets a break! */
    cflag = 0;

    /* scan the options */
    for (i = 1; i < argc; i++) {

        if (strcmp(argv[i], "e") == 0)
            errf = 1;
        else if (strcmp(argv[i], "e0") == 0)
            errf = 0;
        /* "e" and "e0" options toggle error flag */

        else if (strcmp(argv[i], "lt") == 0) {  /* title */
            p1 = titlebuf;
            if (++i < argc) {
                p2 = argv[i];
                while (*p1++ = *p2++);
            }
        } else if (strcmp(argv[i], "lx") == 0) {        /* x label */
            p1 = xtitlebuf;
            if (++i < argc) {
                p2 = argv[i];
                while (*p1++ = *p2++);
            }
        } else if (strcmp(argv[i], "ly") == 0) {        /* y label */
            p1 = ytitlebuf;
            if (++i < argc) {
                p2 = argv[i];
                while (*p1++ = *p2++);
            }
        } else if (strcmp(argv[i], "ltx") == 0) {       /* title x location */
            if (!argv[++i] || sscanf(argv[i], "%e", &titlex) != 1)
                badarg(argc, argv);
            titlexf = 1;
        } else if (strcmp(argv[i], "lty") == 0) {       /* title y location */
            if (!argv[++i] || sscanf(argv[i], "%e", &titley) != 1)
                badarg(argc, argv);
            titleyf = 1;
        } else if (strcmp(argv[i], "lxx") == 0) {       /* x label  x loc. */
            if (!argv[++i] || sscanf(argv[i], "%e", &xtitlex) != 1)
                badarg(argc, argv);
            xtitlexf = 1;
        } else if (strcmp(argv[i], "lxy") == 0) {       /* x label  y loc. */
            if (!argv[++i] || sscanf(argv[i], "%e", &xtitley) != 1)
                badarg(argc, argv);
            xtitleyf = 1;
        } else if (strcmp(argv[i], "lyx") == 0) {       /* y label  x loc. */
            if (!argv[++i] || sscanf(argv[i], "%e", &ytitlex) != 1)
                badarg(argc, argv);
            ytitlexf = 1;
        } else if (strcmp(argv[i], "lyy") == 0) {       /* y label  y loc. */
            if (!argv[++i] || sscanf(argv[i], "%e", &ytitley) != 1)
                badarg(argc, argv);
            ytitleyf = 1;
        } else if (strcmp(argv[i], "lts") == 0) {       /* title size */
            if (!argv[++i] || sscanf(argv[i], "%e", &titlesize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "lxs") == 0) {       /* x label size */
            if (!argv[++i] || sscanf(argv[i], "%e", &xtitlesize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "lys") == 0) {       /* y label size */
            if (!argv[++i] || sscanf(argv[i], "%e", &ytitlesize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "lns") == 0) {       /* label number size */
            if (!argv[++i] || sscanf(argv[i], "%e", &lblsize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "m") == 0) { /* line style */
            if (!argv[++i] || sscanf(argv[i], "%d", &mode) != 1)
                badarg(argc, argv);
            if (mode >= sizeof(modes) / sizeof(*modes))
                mode = 1;
        } else if (strcmp(argv[i], "a") == 0) { /* automatic abscissas */
            absf = 1;
            dx = 1.0;
            if (i + 1 < argc)
                if (!argv[i + 1] || sscanf(argv[i + 1], "%e", &dx) != 1)
                    continue;
            i++;
            if (i + 1 < argc)
                if (!argv[i + 1] || sscanf(argv[i + 1], "%e", &absbot) != 1)
                    continue;
            i++;
            absf = 2;
        }

        /* grid style 0 none, 1 ticks, 2 full */
        else if (strcmp(argv[i], "g") == 0) {
            if (!argv[++i] || sscanf(argv[i], "%d", &gridf) != 1)
                badarg(argc, argv);
        }

        /* x format string */
	else if (strcmp(argv[i],"xf") == 0) { 
            if (argc!=3 || strlen(strcpy(xformat,argv[2]))==0 ) 
                badarg(argc, argv);
        }

        /* x format string */
	else if (strcmp(argv[i],"yf") == 0) { 
            if (argc!=3 || strlen(strcpy(yformat,argv[2]))==0 ) 
                badarg(argc, argv);
        }

	/* set linewidth */
	else if (strcmp(argv[i],"lw") == 0) {
            if (!argv[++i] || sscanf(argv[i], "%f", &stdwidth) != 1)
                badarg(argc, argv);
        }

	/* number of sub ticks */
	else if (strcmp(argv[i],"nd") == 0) {
            if (!argv[++i] || sscanf(argv[i], "%d", &tickspertick) != 1)
                badarg(argc, argv);
        }

        else if (strcmp(argv[i], "c0") == 0)
            symbf = 0;
        /* turn off plot symbols */

        else if (strcmp(argv[i], "cs") == 0) {  /* symbol size */
            if (!argv[i++] || sscanf(argv[i], "%e", &symsize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "c") == 0) {
            /* character(s) for plotting */
            if (++i >= argc)
                badarg(argc, argv);
            symbf = 1;
            j = utou(argv[i]);
            plotsymb = copystring(argv[i], j);
        } else if (strcmp(argv[i], "t") == 0)
            transf = 1;
        /* transpose */

        else if (strcmp(argv[i], "lb") == 0)
            lbflag = 1;
        else if (strcmp(argv[i], "lb0") == 0)
            lbflag = 0;
        /* set or unset breaks after labels */

        else if (strcmp(argv[i], "x") == 0) {   /* x limits */
            if (strcmp(argv[i + 1], "l") == 0) {
                i++;
                xd.xf = log10;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &xd.xlb) == 1) {
                i++;
                xd.xlbf = 1;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &xd.xub) == 1) {
                i++;
                xd.xubf = 1;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &xd.xquant) == 1) {
                i++;
                xd.xqf = 1;
            }
        } else if (strcmp(argv[i], "y") == 0) { /* y limits */
            if (strcmp(argv[i + 1], "l") == 0) {
                i++;
                yd.xf = log10;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &yd.xlb) == 1) {
                i++;
                yd.xlbf = 1;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &yd.xub) == 1) {
                i++;
                yd.xubf = 1;
            }
            if (i + 1 < argc && sscanf(argv[i + 1], "%e", &yd.xquant) == 1) {
                i++;
                yd.xqf = 1;
            }
        } else if (strcmp(argv[i], "h") == 0) { /* fraction of height */
            if (sscanf(argv[++i], "%e", &yd.xsize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "w") == 0) { /* fraction of width */
            if (sscanf(argv[++i], "%e", &xd.xsize) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "r") == 0) { /* move to right */
            if (sscanf(argv[++i], "%e", &xd.xoff) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "u") == 0) { /* move up */
            if (sscanf(argv[++i], "%e", &yd.xoff) != 1)
                badarg(argc, argv);
        } else if (strcmp(argv[i], "s") == 0) { /* save screen, overlay plot */
            erasf = 0;
            if (++i < argc ) {   /* flush comment string */
                ;
            }
            return(1);
        } else if (strcmp(argv[i], "n") == 0) { /* next plot flag */
            erasf = 1;
            return(1);             /* ignore comment string */
        } else if (strcmp(argv[i], "k") == 0) { /* comment statement */
            if (++i < argc) {
                ;               /* flush comment string */
            }
        } else
            badarg(argc, argv);
        return(0);
    }
}

getline(inputfile, p)   /* get a line from inputfile, put in string p */
    FILE           *inputfile;
    char           *p;
{
    register int    i, c;       /* return length of string, -1 if EOF */
    i = -1;
    while ((c = getc(inputfile)) != EOF) {
        if (c == '\n') {
            p[++i] = '\0';
            return (i);
        } else
            p[++i] = c;
    }
    if (i >= 0)
        p[++i] = '\0';
    return (i);
}

transpose()
{
    register        i;
    float           f;
    struct xy       t;
    if (!transf)
        return;
    t = xd;
    xd = yd;
    yd = t;
    for (i = 0; i < n; i++) {
        if (xx[i].err_flag) {
            fprintf(stderr, "%s: Cannot transpose data with error bars!\n",ProgName);
            exit(-1);
        }
        f = xx[i].xv;
        xx[i].xv = xx[i].yv;
        xx[i].yv = f;
    }
}

char           *
copystring(p, length)
    int        length;
    char           *p;  /* copy a string into the label list */
{
    register char  *temp;
    register        i;

    length++;                   /* length of string plus null char */
    if (length > labsleft) {    /* need more space */
        labs = (char *) malloc(LSIZ);
        labsleft = LSIZ;
        if (labs == NULL) {
            fprintf(stderr,
                    "%s:  Unable to allocate memory for label.\n",ProgName);
            exit(-1);
        }
    }
    temp = labs;
    for (i = 0; i < length; i++)
        labs[i] = p[i];
    labs += length;
    labsleft -= length;
    return (temp);
}

getxlim(p, v)
    register struct xy *p;
    register struct datum *v;
{
    register        i;

    if (n == 0) {
        if (p->xf == log10) {
            if (!p->xlbf)
                p->xlb = 1.0;
            if (!p->xubf)
                p->xub = 10.0;
        } else {
            if (!p->xlbf)
                p->xlb = 0.0;
            if (!p->xubf)
                p->xub = 1.0;
        }
    } else
        for (i = 0; i < n; i++) {
            if (!p->xlbf && p->xlb > v[i].xv)
                p->xlb = v[i].xv;
            if (!p->xubf && p->xub < v[i].xv)
                p->xub = v[i].xv;
        }
}

getylim(p, v)
    register struct xy *p;
    register struct datum *v;
{
    register        i;
    float           t;

    if (n == 0) {
        if (p->xf == log10) {
            if (!p->xlbf)
                p->xlb = 1.0;
            if (!p->xubf)
                p->xub = 10.0;
        } else {
            if (!p->xlbf)
                p->xlb = 0.0;
            if (!p->xubf)
                p->xub = 1.0;
        }
    } else
        for (i = 0; i < n; i++) {
            if (v[i].err_flag && p->xf == log10) {
                if (!p->xlbf &&
                    p->xlb > (v[i].yv * (t = exp(-v[i].yerr / v[i].yv))))
                    p->xlb = v[i].yv * t;
                if (!p->xubf && p->xub < v[i].yv / t)
                    p->xub = v[i].yv / t;
            } else if (v[i].err_flag) {
                if (!p->xlbf && p->xlb > (v[i].yv - (t = v[i].yerr)))
                    p->xlb = v[i].yv - t;
                if (!p->xubf && p->xub < (v[i].yv + t))
                    p->xub = v[i].yv + t;
            } else {
                if (!p->xlbf && p->xlb > v[i].yv)
                    p->xlb = v[i].yv;
                if (!p->xubf && p->xub < v[i].yv)
                    p->xub = v[i].yv;
            }
        }
}

xscale(p)
    register struct xy *p;
{
    float           edge;

    setlim(p);
    edge = topx - botx;
    p->xa = p->xsize * edge / ((*p->xf) (p->xub) - (*p->xf) (p->xlb));
    p->xbot = botx + edge * p->xoff;
    p->xtop = p->xbot + edge * p->xsize;
    p->xb = p->xbot - (*p->xf) (p->xlb) * p->xa +.5;
}

yscale(p)
    register struct xy *p;
{
    float           edge;

    setlim(p);
    edge = topy - boty;
    p->xa = p->xsize * edge / ((*p->xf) (p->xub) - (*p->xf) (p->xlb));
    p->xbot = boty + edge * p->xoff;
    p->xtop = p->xbot + edge * p->xsize;
    p->xb = p->xbot - (*p->xf) (p->xlb) * p->xa +.5;
}

struct z {
    float           lb, ub, mult, quant;
}               setloglim(), setlinlim();

setlim(p)
    register struct xy *p;
{
    float           t, delta, sign;
    struct z        z;
    int             lmark[512], mark[512];
    float           lb, ub;
    int             lbf, ubf;

    lb = p->xlb;
    ub = p->xub;
    delta = ub - lb;
    if (p->xqf) {               /* user supplied quant */
        if (delta * p->xquant <= 0) {
            fprintf(stderr,
                    "%s:  User-supplied axis limits error.",ProgName);
            fprintf(stderr, "%e%e%e", p->xlb, p->xub, p->xqf);
            exit(-1);
        }
        return;
    }
    sign = 1;
    lbf = p->xlbf;              /* is user supplied lower bound */
    ubf = p->xubf;              /* is user supplied upper bound */
    if (delta < 0) {
        sign = -1;
        t = lb;
        lb = ub;
        ub = t;
        t = lbf;
        lbf = ubf;
        ubf = t;
    } else if (delta == 0) {
        if (ub > 0) {
            ub = 2 * ub;
            lb = 0;
        } else if (lb < 0) {
            lb = 2 * lb;
            ub = 0;
        } else {
            ub = 1;
            lb = -1;
        }
    }
    if (p->xf == log10 && lb > 0 && ub > lb) {
        z = setloglim(lbf, ubf, lb, ub);
        p->xlb = z.lb;
        p->xub = z.ub;
        p->xmult *= z.mult;
        p->xquant = z.quant;
        if (setmark(mark, lmark, p) < 2) {
            p->xqf = lbf = ubf = 1;
            lb = z.lb;
            ub = z.ub;
        } else
            return;
    }
    z = setlinlim(lbf, ubf, lb, ub);
    if (sign > 0) {
        p->xlb = z.lb;
        p->xub = z.ub;
    } else {
        p->xlb = z.ub;
        p->xub = z.lb;
    }
    p->xmult *= z.mult;
    p->xquant = sign * z.quant;
}

struct z
setloglim(lbf, ubf, lb, ub)
    int           lbf, ubf; 
    float           lb, ub;
{
    float           r, s, t;
    struct z        z;

    for (s = 1; lb * s < 1; s *= 10);
    lb *= s;
    ub *= s;
    for (r = 1; 10 * r <= lb; r *= 10);
    for (t = 1; t < ub; t *= 10);
    z.lb = !lbf ? r : lb;
    z.ub = !ubf ? t : ub;
    if (ub / lb < 100) {
        if (!lbf) {
            if (lb >= 5 * z.lb)
                z.lb *= 5;
            else if (lb >= 2 * z.lb)
                z.lb *= 2;
        }
        if (!ubf) {
            if (ub * 5 <= z.ub)
                z.ub /= 5;
            else if (ub * 2 <= z.ub)
                z.ub /= 2;
        }
    }
    z.mult = s;
    z.quant = r;
    return (z);
}


struct z
setlinlim(lbf, ubf, xlb, xub)
    int             lbf, ubf;
    float           xlb, xub;
{
    struct z        z;
    float           r, s, delta;
    float           ub, lb;

loop:
    ub = xub;
    lb = xlb;
    delta = ub - lb;
    /* scale up by s, a power of 10, so range (delta) exceeds 1 */
    /* find power of 10 quantum, r, such that delta/10<=r<delta */
    r = s = 1;

    while (delta * s < 10)
        s *= 10;
    delta *= s;

    while (10 * r < delta)
        r *= 10;

    lb *= s;
    ub *= s;

    /* set r=(1,2,5)*10**n so that 3-5 quanta cover range */

    if (r >= delta / 2)
        r /= 2;
    else if (r < delta / 5)
        r *= 2;

    z.ub = ubf ? ub : modceil(ub, r);
    z.lb = lbf ? lb : modfloor(lb, r);
    if (!lbf && z.lb <= r && z.lb > 0) {
        xlb = 0;
        goto loop;
    } else if (!ubf && z.ub >= -r && z.ub < 0) {
        xub = 0;
        goto loop;
    }
    z.quant = r;
    z.mult = s;
    return (z);
}

plot()
{
    int             ix, iy, iy1, iy2;
    int             jx, jy, jy1, jy2;
    int             ix1, ix2;
    int             bar, dbar, delta;
    int             i;
    int             conn;
    float           t;

    bar = 3 * tick * (n + 10) / (3 * n + 10);
    conn = 0;
    linemod(modes[1]);

    for (i = 0; i < n; i++) {
        conn &= xx[i].cflag;    /* check for break */
        if (xx[i].line_mode != mode) {
            mode = xx[i].line_mode;
            linemod(modes[mode]);
        }
        if (xx[i].err_flag) {
            jx = conv(xx[i].xv, &xd, &ix);
            jy = conv(xx[i].yv, &yd, &iy);
            if (yd.xf == log10) {
                jy1 = conv(xx[i].yv *
                           (t = exp(xx[i].yerr / xx[i].yv)),
                           &yd, &iy1);
                jy2 = conv(xx[i].yv / t, &yd, &iy2);
            } else {
                jy1 = conv(xx[i].yv + xx[i].yerr, &yd, &iy1);
                jy2 = conv(xx[i].yv - xx[i].yerr, &yd, &iy2);
            }
            if (jx == 0)
                continue;
            if (jy == 0)
                continue;
            if (conn) {
		resetw();
                cont(ix, iy);   /* connect center of bar */
	    }
            if (jy1 == 0)
                iy1 = yd.xtop;
            if (jy2 == 0)
                iy2 = yd.xbot;
            delta = iy1 - iy2;
            delta = delta / 5;
            dbar = (bar < delta) ? bar : delta;
            ix1 = (xd.xbot < ix - dbar) ? ix - dbar : xd.xbot;
            ix2 = (xd.xtop > ix + dbar) ? ix + dbar : xd.xtop;
            if (mode != 1)
                linemod(modes[1]);
	    neww(THINLIN); /* change linewidth */
            if (iy1) {
                move(ix1, iy1);
                cont(ix2, iy1);
            }
            if (iy2) {
                move(ix1, iy2);
                cont(ix2, iy2);
            }
            move(ix, iy1);
            cont(ix, iy2);
	    move(ix,iy);
            symbol(ix, iy, xx[i].lblptr,xx[i].lblsiz);
            if (mode != 1)
                linemod(modes[mode]);
            conn = 1;
            continue;
        } else {
            if (!conv(xx[i].xv, &xd, &ix) ||
                !conv(xx[i].yv, &yd, &iy)) {
                /* point out of bounds */
                conn = 0;
                continue;
            }
            if (mode != 0) {
                if (conn) {
		    resetw();
                    cont(ix, iy);
		}
                else
                    move(ix, iy);
                conn = 1;
            }
            symbol(ix, iy, xx[i].lblptr,xx[i].lblsiz);
        }
    }
    resetw();
}

conv(xv, p, ip)
    float           xv;
    register struct xy *p;
    int            *ip;
{
    long            ix;
    float           tmp;
    tmp = xv * p->xmult;
    if (p->xf == log10 && tmp <= 0) {
        ix = p->xb;
    } else {
        ix = p->xa * (*p->xf) (tmp) + p->xb;
    }
    if (ix < p->xbot || ix > p->xtop)
        return (0);
    *ip = ix;
    return (1);
}

symbol(ix, iy, ptr,size)
    int          ix, iy;
    char           *ptr;
    float	   size;
{
    if (ptr == NULL) {
        if (mode == 0)
            point(ix, iy);
    } else {
        if (mode != 1)
            linemod(modes[1]);
        ascii(ptr, ix, iy, size);
        if (mode != 1)
            linemod(modes[mode]);
        move(ix, iy);
    }
}

axes()
{
    register        i;
    int             lmark[512], mark[512];
    int             xn, yn;
    double 	    log();

    linemod(modes[1]);
    if (gridf == 0)
        return;

    /* decide tick size */
    tick = (int)(
        TICKSZ * 0.417 *
	log(1.0+10.0*((xd.xsize > yd.xsize)? xd.xsize : yd.xsize ))
    );  /* crazy formula to scale tick non-linearly */

    /* set width to default */
    neww(1.00);

    /* draw a box */
    move(xd.xbot, yd.xbot);
    cont(xd.xtop, yd.xbot);
    cont(xd.xtop, yd.xtop);
    cont(xd.xbot, yd.xtop);
    cont(xd.xbot, yd.xbot);

    xn = setmark(mark, lmark, &xd);
    for (i = 0; i < xn; i++) {
        if (lmark[i] == 2 * tick && gridf == 2) {
            move(mark[i], yd.xbot);
            cont(mark[i], yd.xtop);
        } else {
            move(mark[i], yd.xbot);
            cont(mark[i], yd.xbot + lmark[i]);
            move(mark[i], yd.xtop - lmark[i]);
            cont(mark[i], yd.xtop);
        }
    }
    yn = setmark(mark, lmark, &yd);
    for (i = 0; i < yn; i++) {
        if (lmark[i] == 2 * tick && gridf == 2) {
            move(xd.xbot, mark[i]);
            cont(xd.xtop, mark[i]);
        } else {
            move(xd.xbot, mark[i]);
            cont(xd.xbot + lmark[i], mark[i]);
            move(xd.xtop - lmark[i], mark[i]);
            cont(xd.xtop, mark[i]);
        }
    }
}

setmark(xmark, lmark, p)
    int            *lmark, *xmark;
    register struct xy *p;
{
    int             i, xn = 0;
    float           x, xl, xu;
    float           q;
    if (p->xf == log10 && !p->xqf) {
        for (x = p->xquant; x < p->xub; x *= 10) {
            lmark[xn] = 2 * tick;
            submark(xmark, &xn, x, p);
            for (i = 2; i < 10; i += 1) { 
                lmark[xn] = tick;
                submark(xmark, &xn, i * x, p);
            }
        }
    } else {
        xn = 0;
        q = p->xquant;
        if (q > 0) {
            xl = modfloor(p->xlb, q);
            xu = modfloor(p->xub - q / 10, q / 5) + q / 10;
        } else {
            xl = modceil(p->xub, q);
            xu = modfloor(p->xlb + q / 10, q / 5) - q / 10;
        }
        for(i=0,x=xl;x<=xu;i++,x += fabs(p->xquant)/(float)tickspertick) { 
            if (x <= p->xlb || x >= p->xub)
                continue;
            xmark[xn] = (*p->xf) (x) * p->xa + p->xb;
            if (i % tickspertick) {
                lmark[xn++] = tick;
            } else
                lmark[xn++] = 2 * tick;
        }
    }
    return (xn);
}

submark(xmark, pxn, x, p)
    int            *xmark;
    int            *pxn;
    float           x;
    struct xy      *p;
{
    if (1.001 * p->xlb < x &&.999 * p->xub > x) {
        xmark[(*pxn)++] = log10(x) * p->xa + p->xb;
        return;
    }
    return;
}

title()
{
    int             ix, iy;

    if (*titlebuf) {
        utou(titlebuf);
        ix = titlexf ?
            titlex * (xd.xtop - xd.xbot) + xd.xbot :
            (xd.xtop + xd.xbot) / 2.;
        iy = titleyf ?
            titley * (yd.xtop - yd.xbot) + yd.xbot :
            yd.xtop - 4 * tick - (int) (titlesize * 40);
        cascii(titlebuf, ix, iy, titlesize);
    }
    leftx = xd.xbot;
    if (gridf) {
        axlab('x', &xd,xformat);
        axlab('y', &yd,yformat);
    }
    if (*xtitlebuf) {
        utou(xtitlebuf);
        ix = xtitlexf ?
            xtitlex * (xd.xtop - xd.xbot) + xd.xbot :
            (xd.xtop + xd.xbot) / 2.;
        iy = xtitleyf ?
            xtitley * (yd.xtop - yd.xbot) + yd.xbot :
            yd.xbot - (int) (lblsize * 160 + xtitlesize * 80);
        cascii(xtitlebuf, ix, iy, xtitlesize);
    }
    if (*ytitlebuf) {
        utou(ytitlebuf);
        ix = ytitlexf ?
            ytitlex * (xd.xtop - xd.xbot) + xd.xbot :
            leftx - (int) (80 * ytitlesize);
        iy = ytitleyf ?
            ytitley * (yd.xtop - yd.xbot) + yd.xbot :
            (yd.xtop + yd.xbot) / 2;
        cvascii(ytitlebuf, ix, iy, ytitlesize);
    }
}

axlab(c, p,format_s)
char *format_s,c;
register struct xy *p;
{
    int             mark;
    int             width, digits;
    char            buf[128];
    float           x, xl, xu;
    float           q;

    if (p->xf == log10 && !p->xqf) {
        for (x = p->xquant; x <= p->xub; x *= 10) {
            if (p->xlb <= x && p->xub >= x) {
                sprintf(buf,format_s, x / p->xmult);
                mark = log10(x) * p->xa + p->xb;
                num_lbl(c, buf, mark);
            }
            if (p->xub / p->xlb <= 100) {
                if (p->xlb <= 2 * x && p->xub >= 2 * x) {
                    sprintf(buf,format_s, 2 * x / p->xmult);
                    mark = log10(2 * x) * p->xa + p->xb;
                    num_lbl(c, buf, mark);
                }
                if (p->xlb <= 5 * x && p->xub >= 5 * x) {
                    sprintf(buf,format_s, 5 * x / p->xmult);
                    mark = log10(5 * x) * p->xa + p->xb;
                    num_lbl(c, buf, mark);
                }
            }
        }
    } else {                    /* its linear */
        q = p->xquant;
        if (q > 0) {
            xl = modceil(p->xlb - q / 6, q);
            xu = modfloor(p->xub + q / 6, q);
        } else {
            xl = modceil(p->xub + q / 6, q);
            xu = modfloor(p->xlb - q / 6, q);
        }
        digit(xl / p->xmult, xu / p->xmult, (int) ((xu - xl) / q + 1.5),
              &width, &digits);

        for (x = xl; x <= xu; x += fabs(p->xquant)) {
            sprintf(buf,format_s, x / p->xmult);
            mark = (*p->xf) (x) * p->xa + p->xb;
            num_lbl(c, buf, mark);
        }
    }
}


num_lbl(c, buf, mark)
    int                mark;
    char            c, *buf;
{
    char            pri[BSIZ];
    int             i;

    gtou(buf, pri);

    switch (c) {
    case 'x':
        iascii(pri, mark, yd.xbot - (int) (120 * lblsize),
               lblsize);
        break;
    case 'y':
        i = xd.xbot - 50 * lblsize - 60 * strlen(pri) * lblsize;
        leftx = (leftx > i) ? i : leftx;
        rascii(pri, xd.xbot - (int) (50 * lblsize),
               mark, lblsize);
        break;
    }
}

badarg(argc, argv)
    int             argc;
    char           *argv[];
{
    int             i;

    fprintf(stderr, "%s: error in input stream\n",ProgName);
    for (i = 1; i < argc; i++)
        fprintf(stderr, "%s%c ", argv[i], (i < argc - 1) ? ' ' : '\n');
    exit(-1);
}

neww(w)
float w;
/*  change the width to w if it is not that */
{
	if ( w!=currentwidth ) {
		currentwidth = w;
		width(w);
	}
}

resetw()
/* returns width to standard width */
{
	if (currentwidth != stdwidth) 
	 	width(currentwidth=stdwidth);
}

readopts(ac,av)
int ac;
char *av[];
{
	extern int UseScannerQ;
	extern FILE *inputfile;
	extern int optind;
	extern char *optarg;

	int c;
	FILE *fopen();

	/* defaults */
	UseScannerQ = YES;
	inputfile = stdin;
	outputfile = stdout;

	/* scan options (more may be added, so lets do it cleanly) */
	while( (c=getopt(ac,av,"?s")) != EOF ) {
	  switch (c) {
		case 's' :  UseScannerQ = NO;
			    break;
		case '?' :  fprintf(stderr,"xyplot 2.0 patchlevel %-d\n",
						PATCHED);
			    exit(1);
			    break;
		default  :  fprintf(stderr,"%s: Unknown option %c\n",
				ProgName,c);
			    exit(-1);
			    break;
	  }
	}
	
	/* only one xy file per program call */
	if (optind<ac) {
	  inputfile = fopen(av[optind],"r");
	  if (inputfile == NULL ) {
		fprintf(stderr,"%s: Cannot open input file\n",ProgName);
		exit(-1);
	  }
	}

	if (UseScannerQ) {
	   outputfile = popen(SCANNER,"w");
	   if (outputfile == NULL ) {
		fprintf(stderr,"%s: Cannot open output\n",ProgName);
		exit(-1);
	   }
	}

}

bbox(xd,yd)
struct xy *xd,*yd;
{
	extern FILE *outputfile;
	extern int titleyf;
	extern float titley,titlesize;

	fprintf(outputfile,"b %6d %6d %6d %6d\n",
	   xd->xbot,
	   yd->xbot,
	   xd->xtop,
           (int)
	     ((titleyf) ?
               titley * (yd->xtop - yd->xbot) + yd->xbot + 110*titlesize :
	       yd->xtop
	     )
	);
}
@@@ ronnie Mon Sep 14 16:15:35 MDT 1992 Version before use of put
