/* algebra.c - vector algebra functions on a sphere
 *
 * $Id: algebra.c,v 1.3 96/02/11 21:27:51 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	algebra.c,v $
 * Revision 1.3  96/02/11  21:27:51  leech
 * Add a distance-computing function to ForceLaw.
 * 
 * Revision 1.2  96/02/04  22:46:31  leech
 * Encapsulated force laws in a class with common interface.
 *
 * Revision 1.1  92/04/27  04:25:11  leech
 * Initial revision
 *
 */

#include "algebra.h"
#include <math.h>

extern "C" {
    void srand48(long);
    double drand48();
};

inline double square(double x) { return x * x; }

// Generate a random vector on the unit sphere
dvector unit_random() {
    dvector r;

    r[0] = drand48() - 0.5;
    r[1] = drand48() - 0.5;
    r[2] = drand48() - 0.5;

    return r.normalize();
}

// Return the component of v normal to vref
// vref is assumed to be normalized
dvector normal_component(const dvector &v, const dvector &vref) {
    return v - (v * vref) * vref;
}

// Return the component of v parallel to vref
// vref is assumed to be normalized
dvector tangent_component(const dvector &v, const dvector &vref) {
    return (v * vref) * vref;
}

// Return arclength between two vectors on the unit sphere
// r1 and r2 are assumed to be normalized
double arclength(const dvector &r1, const dvector &r2) {
    double dotprod = r1 * r2;

    // Range check; should not happen
    if (dotprod > 1)
	return 0;
    else if (dotprod < -1)
	return M_PI;
    else
	return acos(dotprod);
}

// Implementation of different types of force laws.
ForceLaw::ForceLaw(double _k, double _epsilon, double _r0)
    : k(_k), epsilon(_epsilon), r0(_r0),
      distanceFunction(kArcDistance), falloffFunction(kSquareLaw),
      morseDepth(1)
{
    SetForceFunction();
    UpdateTerms();
}

// Accessors
double ForceLaw::GetSoftening() const {
    return epsilon;
}

double ForceLaw::GetKForce() const {
    return k;
}

double ForceLaw::GetR0() const {
    return r0;
}

ForceLaw::DistanceFunction ForceLaw::GetDistanceFunction() const {
    return distanceFunction;
}

ForceLaw::FalloffFunction ForceLaw::GetFalloffFunction() const {
    return falloffFunction;
}

// Modifiers
void ForceLaw::SetSoftening(double _e) {
    epsilon = _e;
    UpdateTerms();
}

void ForceLaw::SetKForce(double _k) {
    k = _k;
    UpdateTerms();
}

void ForceLaw::SetR0(double _r0) {
    r0 = _r0;
    UpdateTerms();
}

void ForceLaw::SetDistanceFunction(DistanceFunction d) {
    distanceFunction = d;
    SetForceFunction();
}

void ForceLaw::SetFalloffFunction(FalloffFunction f) {
    falloffFunction = f;
    SetForceFunction();
}

// Find the force exerted between points 'to' and 'from' using the
//  current force-calculation function.
dvector ForceLaw::FindForce(const dvector &to, const dvector &from) {
    return (this->*forceFunction)(to, from);
}

// Find the distance between points 'to' and 'from'.
double ForceLaw::Distance(const dvector &to, const dvector &from) {
    if (distanceFunction == kArcDistance)
	return arclength(to, from);
    else
	return (to-from).magnitude();
}

// PRIVATE METHODS

// Update parameters used in computing various force terms when a
//  parameter changes
void ForceLaw::UpdateTerms() {
    morseBeta = sqrt(k / (2 * morseDepth));
}

// Set the internal pointer to the current force-calculation function
//  based on the desired distance and force functions.
void ForceLaw::SetForceFunction() {
    if (distanceFunction == kArcDistance) {
	switch (falloffFunction) {
	    case kSquareLaw:  forceFunction = &ForceLaw::ArcSquare; break;
	    case kLinearLaw:  forceFunction = &ForceLaw::ArcLinear; break;
	    case kLennartLaw: forceFunction = &ForceLaw::ArcLennart; break;
	    case kMorseLaw:   forceFunction = &ForceLaw::ArcMorse; break;
	}
    } else {	// distanceFunction == kEuclidDistance
	switch (falloffFunction) {
	    case kSquareLaw:  forceFunction = &ForceLaw::EuclidSquare; break;
	    case kLinearLaw:  forceFunction = &ForceLaw::EuclidLinear; break;
	    case kLennartLaw: forceFunction = &ForceLaw::EuclidLennart; break;
	    case kMorseLaw:   forceFunction = &ForceLaw::EuclidMorse; break;
	}
    }
}

// The following methods implement the various force laws.
// The first part of the method name specifies how distance between
//  points is computed. 'Arc' means arc distance on the unit sphere,
//  'Euclid' means Euclidean 3-space distance.
// The second part of the method name specifies the force falloff.
//  'Square' means 1/r^2, 'Linear' means 1/r, 'Lennart' means the
//  Lennart-Jones potential, and 'Morse' means the Morse potential.
dvector ForceLaw::ArcSquare(const dvector &to, const dvector &from) {
    dvector r = (to - from).normalize();    // Direction of force vector
    double d = arclength(to, from);	    // Distance between to & from
    return (k / square(d + epsilon)) * r;
}

dvector ForceLaw::ArcLinear(const dvector &to, const dvector &from) {
    dvector r = (to - from).normalize();    // Direction of force vector
    double d = arclength(to, from);	    // Distance between to & from
    return (k / (d + epsilon)) * r;
}

dvector ForceLaw::ArcLennart(const dvector &to, const dvector &from) {
    dvector r = (to - from).normalize();    // Direction of force vector
    double d = arclength(to, from);	    // Distance between to & from

    return (k * LennartJones(d)) * r;
}

dvector ForceLaw::ArcMorse(const dvector &to, const dvector &from) {
    dvector r = (to - from).normalize();    // Direction of force vector
    double d = arclength(to, from);	    // Distance between to & from

    return MorseForce(d) * r;
}


dvector ForceLaw::EuclidSquare(const dvector &to, const dvector &from) {
    dvector r = to - from;		    // Direction of force vector
    double d = r.magnitude();		    // Distance between to & from
    double dinv = (d != 0.0) ? 1 / d : 0.0; // Normalization factor for r

    return (k * dinv / square(d + epsilon)) * r;
}

dvector ForceLaw::EuclidLinear(const dvector &to, const dvector &from) {
    dvector r = to - from;		    // Direction of force vector
    double d = r.magnitude();		    // Distance between to & from
    double dinv = (d != 0.0) ? 1 / d : 0.0; // Normalization factor for r

    return (k * dinv / (d + epsilon)) * r;
}

dvector ForceLaw::EuclidLennart(const dvector &to, const dvector &from) {
    dvector r = to - from;		    // Direction of force vector
    double d = r.magnitude();		    // Distance between to & from
    double dinv = (d != 0.0) ? 1 / d : 0.0; // Normalization factor for r

    return (k * dinv * LennartJones(d)) * r;
}

dvector ForceLaw::EuclidMorse(const dvector &to, const dvector &from) {
    dvector r = to - from;		    // Direction of force vector
    double d = r.magnitude();		    // Distance between to & from
    double dinv = (d != 0.0) ? 1 / d : 0.0; // Normalization factor for r

    return (dinv * MorseForce(d)) * r;
}


// Lennart-Jones (r0/r)^6 - (r0/r)^12 potential function
double ForceLaw::LennartJones(double r) {
    double rinv = (r <= epsilon) ? r0 / epsilon : r0 / r;

    return pow(rinv, 6) - pow(rinv, 12);
}

//				-beta(r-r0)^2
// Morse potential = depth ((1-e	     ) -1)
//
double ForceLaw::MorsePotential(double r) {
    return morseDepth * ((1 - exp(-morseBeta * square(r-r0)))-1);
}

//					-2 beta (r-r0)	    -beta (r-r0)
// Morse force = 2 * beta * depth * (exp	       - exp		)
//
double ForceLaw::MorseForce(double r) {
    double term = morseBeta * (r - r0);

    return 2 * morseBeta * morseDepth * (exp(-2 * term) - exp(-term));
}
