/* support.c  --  returns support area of a structuring element */

/* contains: main()                                */

/* dependencies:                                   */

/* support version 1.0  10 June 1993               */    
/* image morphology program                        */
/*                                                 */
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */
/* Nashville, TN 37235                             */
/* rap2@vuse.vanderbilt.edu                        */
/*                                                 */
/* This software is freely redistributable if      */
/* the author's name and affiliation are included. */


#include <ctype.h>
#include <malloc.h>            
#include <math.h>
#include <memory.h>
#include <stdio.h>    
#include <stdlib.h>
#include <strings.h>
#include "support.h"

main( argc, argv ) 
   unsigned int argc;  /* count of arguments parsed by operating system    */
   char *argv[];       /* pointers to arguments parsed by operating system */
   {
   char *SEname;     /* structuring element (SE) path name */
   int sX,sY,sZ;     /* SE horizontal, vertical, time_or_depth dimensions */
   int sorgx,sorgy,sorgz;
   int glev;         /* largest gray level in SE */
   int autoSE;       /* T => create a disk-shaped SE locally */
   int SEType;       /* binary or gray-level SE */
   int SorF;
   int n;
   int *SE;


   /* begin */

   if (argc < 2) /* display usage */
      {
      fprintf(stderr,
         "usage: support SEFile [2d|3d] | 3x3 | 5x5 | plus | 3x3x3 | 5x5x5 | 3dplus |\n\
       auto xxx yyy [zzz] | [cone | cylinder | sphere xxx yyy zzz [vvv]]\n");
      exit(0);
      }

   SEType = GRASE;
   SorF = SET;

   /* get args */
   if ( !strcasecmp( argv[1], "3x3" ) )
      {
      autoSE = S3X3;
      SEname = NULL;
      SE = MakeSE( SEname, autoSE, &sX, &sY, glev, &sorgx, &sorgy);
      }
   else if ( !strcasecmp( argv[1], "plus" ) )
      {
      autoSE = PLUS;
      SEname = NULL;
      SE = MakeSE( SEname, autoSE, &sX, &sY, glev, &sorgx, &sorgy);
      }
   else if ( !strcasecmp( argv[1], "5x5" ) )
      {
      autoSE = S5X5;
      SEname = NULL;
      SE = MakeSE( SEname, autoSE, &sX, &sY, glev, &sorgx, &sorgy);
      }
   else if ( !strcasecmp( argv[1], "3x3x3" ) )
      {
      autoSE = S3X3X3;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else if ( !strcasecmp( argv[1], "3dplus" ) )
      {
      autoSE = PLUS3D;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else if ( !strcasecmp( argv[1], "5x5x5" ) )
      {
      autoSE = S5X5X5;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else if ( !strcasecmp( argv[1], "auto" ) )
      {
      autoSE = AUTO;
      sX = atoi( argv[2] );
      sY = atoi( argv[3] );
      if ( (4 < argc) && isdigit( argv[4][0] ) )
         glev = atoi( argv[4] );
      SEname = NULL;
      SE = MakeSE( SEname, autoSE, &sX, &sY, glev, &sorgx, &sorgy);
      }
   else if ( !strcasecmp( argv[1], "sphere" ) )
      {
      autoSE = SPHERE;
      sX = atoi( argv[2] );
      sY = atoi( argv[3] );
      sZ = atoi( argv[4] );
      if ( (5 < argc) && isdigit( argv[5][0] ) )
         glev = atoi( argv[5] );
      else
         glev = BLACK;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else if ( !strcasecmp( argv[1], "cylinder" ) )
      {
      autoSE = CYLINDER;
      sX = atoi( argv[2] );
      sY = atoi( argv[3] );
      sZ = atoi( argv[4] );
      if ( (5 < argc) && isdigit( argv[5][0] ) )
         glev = atoi( argv[5] );
      else
         glev = BLACK;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else if ( !strcasecmp( argv[1], "cone" ) )
      {
      autoSE = CONE;
      sX = atoi( argv[2] );
      sY = atoi( argv[3] );
      sZ = atoi( argv[4] );
      if ( ( 5 < argc) && isdigit( argv[5][0] ) )
         glev = atoi( argv[5] );
      else
         glev = BLACK;
      SEname = NULL;
      SE = GetSE3D( 
           SEname,autoSE,&SEType,&SorF,&sX,&sY,&sZ,&glev,&sorgx,&sorgy,&sorgz);
      }
   else
      {
      SEname = argv[1];
      if ( (argc == 3) && !strcasecmp( argv[2], "2d" ) )
         {
         printf("Reading 2D SE file %s\n",SEname);
         SE = ReadSE2D( SEname, &sX, &sY, &sorgx, &sorgy );
         }
      else if ( (argc == 3) && !strcasecmp( argv[2], "3d" ) )
         {
         printf("Reading 3D SE file %s\n",SEname);
         SE = ReadSE3D( SEname, &sX, &sY, &sZ, &sorgx, &sorgy, &sorgz);
         }
      else
         {
         printf("Assuming %s is a 2D SE file.\n",SEname);
         SE = ReadSE2D( SEname, &sX, &sY, &sorgx, &sorgy );
         }
      }

   if ( !sZ )
      n = GetSupport( SE, sX, sY, SEType );
   else
      n = GetSupport3D( SE, sX, sY, sZ, SEType );

   if (SEname)
      fprintf(stderr,"SE %s has support of %d pixels.\n",SEname,n);
   else
      fprintf(stderr,"The SE has support of %d pixels.\n",n);

   fprintf(stderr,"The median is %d.\n",(n/2)+1);

   }




int *ReadSE2D( SEname, sX, sY, sorgx, sorgy )
   char *SEname;       /* structuring element file name */
   int *sX,*sY;    /* x,y dimensions of bounding box */
   int *sorgx,*sorgy; /* x,y position of SE origin */
   {    
   int i,n;            /* indicies */
   int *SE;            /* SE array */
   char *prefix;       /* leading component of pathnmae */
   char pathname[256]; /* pathname string */
   char firstline[256];
   FILE *sefp;         /* structuring element file pointer */


   *sX=*sY=0;
   *sorgx=*sorgy=0;

   /* if environment variable SEPATH is set, get it and prepend to path */
   pathname[0] = '\0';
   prefix = getenv( SEPATH );
   if ( prefix && (n = strlen(prefix))  ) 
      {
      strcpy(pathname,prefix);
      if ( pathname[n-1] != '/' )
         {
         pathname[n] = '/';
         pathname[n+1] = '\0';
         }
      }
   strcat( pathname, SEname );

   /* try to open the SE file */
   if ( !(sefp = fopen( pathname, "r")) )
      {
      fprintf(stderr,"Unable to open %s\n", pathname);
      exit(0);
      }

   fscanf(sefp,"%d %d %d %d",sX,sY,sorgx,sorgy);

   /* number of pixels covered by bounding box */
   n = (*sX) * (*sY);
   if ( !n )
      {
      fprintf(stderr,"SE %s has size zero.\n", pathname);
      exit(0);
      }

   /* allocate the space for it */
   if ( !(SE = (int *)calloc( n, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for SE %s\n", SEname);
      exit(0);
      }

   /* read it in from the file */
   for ( i=0; i<n; ++i ) fscanf(sefp, "%d", SE+i);

   /* close the file */
   fclose( sefp );

   /* pass the pointer to the SE array */
   return( SE );
   }



int *ReadSE3D( SEname, sX, sY, sZ, sorgx, sorgy, sorgz )
   char *SEname;       /* structuring element file name */
   int *sX,*sY,*sZ;    /* x,y,z dimensions of bounding box */
   int *sorgx,*sorgy,*sorgz; /* x,y position of SE origin */
   {    
   int i,n;            /* indicies */
   int *SE;            /* SE array */
   char *prefix;       /* leading component of pathnmae */
   char pathname[256]; /* pathname string */
   char firstline[256];
   FILE *sefp;         /* structuring element file pointer */


   *sX=*sY=*sZ=0;
   *sorgx=*sorgy=*sorgz=0;

   /* if environment variable SEPATH is set, get it and prepend to path */
   pathname[0] = '\0';
   prefix = getenv( SEPATH );
   if ( prefix && (n = strlen(prefix))  ) 
      {
      strcpy(pathname,prefix);
      if ( pathname[n-1] != '/' )
         {
         pathname[n] = '/';
         pathname[n+1] = '\0';
         }
      }
   strcat( pathname, SEname );

   /* try to open the SE file */
   if ( !(sefp = fopen( pathname, "r")) )
      {
      fprintf(stderr,"Unable to open %s\n", pathname);
      exit(0);
      }

   fscanf(sefp,"%d %d %d %d %d %d",sX,sY,sZ,sorgx,sorgy,sorgz);

   /* number of pixels covered by bounding box */
   n = (*sX) * (*sY) * (*sZ) ;

   /* allocate the space for it */
   if ( !(SE = (int *)calloc( n, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for SE %s\n", SEname);
      exit(0);
      }

   /* read it in from the file */
   for ( i=0; i<n; ++i ) fscanf(sefp, "%d", SE+i);

   /* close the file */
   fclose( sefp );

   /* pass the pointer to the SE array */
   return( SE );
   }




/* create a stucturing element per command SEtype */

int *MakeSE( SEname, SEtype, sX, sY, smax, sorgx, sorgy )
   char *SEname;         /* SE path name */
   int SEtype;          /* type of SE to get or make */
   int *sX,*sY;         /* x,y, dims of SE bpounding box */
   int smax;            /* peak value of SE of type auto */
   int *sorgx, *sorgy;  /* x,y location of SE origin */
   {
   int *SE;


   /* act according to type */
   switch ( SEtype )
      {
      case S3X3 :  /* 3x3 square */
         {
         SE = SE3x3;
         *sX = *sY = 3; 
         *sorgx = *sorgy = 1;
         break;
         }
      case S5X5 : /* 5x5 "fat plus": 5x5 square with corners deleted */
         {
         SE = SE5x5;
         *sX = *sY = 5; 
         *sorgx = *sorgy = 2;
         break;
         }
      case PLUS : /* 3x3 plus: 3x3 square with corners deleted */
         {
         SE = SEplus;
         *sX = *sY = 3; 
         *sorgx = *sorgy = 1;
         break;
         }
      case AUTO : /* create disk-shaped SE of arbitrary size */
         {
         SE = MakeDisk( SEname, *sX, *sY, smax, sorgx, sorgy );
         break;
         }
      default : /* this should never happen */
         {
         fprintf(stderr, "Catastrophic failure!\n");
         fprintf(stderr,
         "   The kaelstrom directory has overwritten the lamoid pointer.\n");
         exit( 0 );
         }
      }
   return( SE );
   }



/* create a structuring element of type AUTO. */ 
/* That is, a digital approximation to a disk */

int *MakeDisk( SEname, sX, sY, smax, sorgx, sorgy )
   char *SEname;
   int sX,sY;
   int smax;
   int *sorgx, *sorgy;
   {    
   int i,j,n;
   int *s,*SE;
   double a,b,r,t,z;

   /* check for zero dims */
   if ( !( sX && sY )  )
      {
      fprintf(stderr,
         "-k auto x y z :  neither x nor y may be zero.\n");
      exit(0);
      }

   /* make sure peak value is legal */ 
   if ( smax < BLACK  ||  smax > WHITE )
      {
      fprintf(stderr,
         "-k auto x y z :  z must be in [0,255].\n");
      exit(0);
      }

   /* compute the origin. differs depending on parity of dimension */
   *sorgx = (sX % 2) ? sX/2 : (sX/2)-1;
   *sorgy = (sY % 2) ? sY/2 : (sY/2)-1;

   /* number of pixels covered by bounding box */
   n = sX * sY;

   /* allocate space for it */
   if ( !(SE = (int *)calloc( n, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for auto SE.\n");
      exit(0);
      }

   if ( n > 1 ) /* then SE is not a single point */
      {
      a = (sX>1) ? 2/(double)(sX-1) : 0.0 ;
      b = (sY>1) ? 2/(double)(sY-1) : 0.0 ;
      z = (double)smax;

      /* t is the distance from the center of the disk out to the edge */
      /* t is one within a tolerance of max(a,b) */
      t = MAX(a,b);
      t = sqrt( 1.0 + t*t + 0.0001*t );

      /* compute disk */
      s = SE;
      if ( sX>1 && sY>1 )
         for ( j=0; j<sY; ++j )
            for ( i=0; i<sX; ++i )
               {
               r = sqrt( pow(-1.0+(double)i*a,2.0) + pow(1.0-(double)j*b,2.0) );
               if ( r*r > t ) *(s++) = -1;
               else *(s++) = (int)( z*sqrt( t - r*r ) + 0.5 );
               }
      else  /* either sX==1 or sY==1 */
         for ( j=0; j<n; ++j )
            {
            r = sqrt( pow(-1.0+(double)j*(a+b),2.0) );
            r = ( r < 1 ) ? r : 1.0 ;
            *(s++) = (int)( z*sqrt( 1.0 - r*r ) + 0.5 );
            }
      }
   else
      *SE = smax; /* SE is a single point */

   return( SE );
   }


/* Return the structuring element to the calling program. */
/* The function returns the address of a newly allocated array */
 
int *GetSE3D( SEname, AutoSE, SEType, SorF, sX, sY, sZ, glev, sorgx, sorgy, sorgz )
   char *SEname;        /* structuring element (SE) file name */
   int AutoSE;          /* T ==> have program make one to specs. */
   int *SEType;         /* type of SE to get or make */
   int *SorF;           /* flag indicates set or function morphology */
   int *sX,*sY,*sZ,*glev; /* SE x,y,z dimensions of bounding box */
   int *sorgx, *sorgy, *sorgz; /* SE x,y,z origina cooridinates */
   {    
   int *S;  /* SE array */

   /* load the SE */
   if ( AutoSE )
      {
      *SEType = GRASE;          /* automatic SE's are necessarily gray-scale */
      *SorF = *glev == 0 ? SET : FUNCT; /* if gray-level is const, do a set op */
      S = MakeSE3D( AutoSE, sX, sY, sZ, glev, sorgx, sorgy, sorgz );
      }
   else
      {
      S = LoadSE3D( SEname, sX, sY, sZ, sorgx, sorgy, sorgz );
      *glev = 0;
      }
   return( S );
   }


/* Get the strcututring element from a file. Allocate the space */
/* for it and return a pointer to it.                           */

int *LoadSE3D( SEname, sX, sY, sZ, sorgx, sorgy, sorgz )
   char *SEname;                /* structuring element file name */
   int *sX,*sY,*sZ;            /* x,y,z dimensions of bounding box */
   int *sorgx, *sorgy, *sorgz; /* x,y,z position of SE origin */
   {    
   int i,n;            /* indicies */
   int *S;             /* SE array */
   char *prefix;       /* leading component of pathnmae */
   char pathname[256]; /* pathname string */
   FILE *sefp;         /* structuring element file pointer */

   /* if environment variable SEPATH is set, get it and prepend to path */
   pathname[0] = '\0';
   prefix = getenv( SEPATH );
   if ( prefix && (n = strlen(prefix))  ) 
      {
      strcpy(pathname,prefix);
      if ( pathname[n-1] != '/' )
         {
         pathname[n] = '/';
         pathname[n+1] = '\0';
         }
      }
   strcat( pathname, SEname );

   /* try to open the SE file */
   if ( !(sefp = fopen( pathname, "r")) )
      {
      fprintf(stderr,"Unable to open %s\n", pathname);
      exit(0);
      }

   /* get the xy dimensions and origin coordinates */
   fscanf(sefp,"%d %d %d %d %d %d",sX,sY,sZ,sorgx,sorgy,sorgz);

   /* make sure origin is within bounding box */
   if ( *sorgx>=*sX || *sorgy>=*sY || *sorgz>=*sZ )
      {
      fprintf(stderr,"The SE origin must be inside the bounding box.\n");
      exit(0);
      }

   /* number of pixels covered by bounding box */
   n = (*sX) * (*sY) * (*sZ);

   /* allocate the space for it */
   if ( !(S = (int *)calloc( n, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for SE %s\n", SEname);
      exit(0);
      }

   /* read it in from the file */
   for ( i=0; i<n; ++i ) fscanf(sefp, "%d", S+i);

   /* close the file */
   fclose( sefp );
 
   /* pass the pointer to the SE array */
   return( S );
   }



/* create a stucturing element per command SEtype */

int *MakeSE3D( SEtype, sX, sY, sZ, glev, sorgx, sorgy, sorgz )
   int SEtype;
   int *sX,*sY,*sZ;
   int *glev;
   int *sorgx, *sorgy, *sorgz;
   {
   int *S;

   /* act according to type */
   switch ( SEtype )
      {
      case S3X3X3 :  /* 3x3x3 cube */
         {
         S = SE3x3x3;
         *sX = *sY = *sZ = 3; 
         *sorgx = *sorgy = *sorgz = 1;
         break;
         }
      case S5X5X5 : /* 5x5x5 "fat plus": 5x5x5 cube with corners deleted */
         {
         S = SE5x5x5;
         *sX = *sY = *sZ = 5; 
         *sorgx = *sorgy = *sorgz = 2;
         break;
         }
      case PLUS3D : /* 3x3x3 plus: 3x3x3 cube with corners deleted */
         {
         S = SEplus3D;
         *sX = *sY = *sZ = 3; 
         *sorgx = *sorgy = *sorgz = 1;
         break;
         }
      case CONE : /* make dual-cone-shaped SE. axis || z. origin at point */
         {
         S = MakeCone( *sX, *sY, *sZ, *glev, sorgx, sorgy, sorgz );
         break;
         }
      case CYLINDER : /* make cylindrical SE. axis || z. origin at center */
         {
         S = MakeCylinder( *sX, *sY, *sZ, *glev, sorgx, sorgy, sorgz );
         break;
         }
      case SPHERE : /* make spherical SE. origin at center */
         {
         S = MakeSphere( *sX, *sY, *sZ, *glev, sorgx, sorgy, sorgz );
         break;
         }
      default : /* this should never happen */
         {
         fprintf(stderr, "Catastrophic failure!\n");
         fprintf(stderr,
         "   The kaelstrom directory has overwritten the cugot map.\n");
         exit( 0 );
         }
      }
   return( S );
   }



/* create a cylindrical structuring element. The axis of the cylinder */
/* is along z; it is perpendicular to the image frames or slices.     */
/* generally this is used with a time-sequence of images to cause a   */
/* disk to persist over several frames */

int *MakeCylinder( sX, sY, sZ, smax, sorgx, sorgy, sorgz )
   int sX,sY,sZ;
   int smax;
   int *sorgx, *sorgy, *sorgz;
   {    
   int i,j,n;
   int *s,*SE;
   double a,b,r,t,v;

   if ( !( sX && sY && sZ)  )
      {
      fprintf(stderr,
         "-k cylinder x y z v:  x, y, nor z may be zero.\n");
      exit(0);
      }

   if ( smax < BLACK  ||  smax > WHITE )
      {
      fprintf(stderr,
         "-k cylinder x y z v:  v must be in the range [0,255].\n");
      exit(0);
      }

   *sorgx = (sX % 2) ? sX/2 : (sX/2)-1;
   *sorgy = (sY % 2) ? sY/2 : (sY/2)-1;
   *sorgz = (sZ % 2) ? sZ/2 : (sZ/2)-1;

   n = sX * sY;

   if ( !(SE = (int *)calloc( n*sZ, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for auto SE.\n");
      exit(0);
      }

   if ( n > 1 )
      {
      a = (sX>1) ? 2/(double)(sX-1) : 0.0 ;
      b = (sY>1) ? 2/(double)(sY-1) : 0.0 ;
      v = (double)smax;
  
      /* t is one within a tolerance of max(a,b) */
      t = MAX(a,b);
      t = sqrt( 1.0 + t*t + 0.0001*t );

      s = SE;
      if ( sX>1 && sY>1 )
         for ( j=0; j<sY; ++j )
            for ( i=0; i<sX; ++i )
               {
               r = sqrt( pow(-1.0+(double)i*a,2.0) + pow(1.0-(double)j*b,2.0) );
               if ( r*r > t ) *(s++) = -1;
               else *(s++) = (int)( v*sqrt( t - r*r ) + 0.5 );
               }
      else  /* either sX==1 or sY==1 */
         for ( j=0; j<n; ++j )
            {
            r = sqrt( pow(-1.0+(double)j*(a+b),2.0) );
            r = ( r < 1 ) ? r : 1.0 ;
            *(s++) = (int)( v*sqrt( 1.0 - r*r ) + 0.5 );
            }
      for ( j=1; j<sZ; ++j)
         for ( i=0; i<n; ++i )
             *(s++) = *(SE+i);

      }
   else /* n==1 (sZ by 1 cylinder */
      {
      for ( j=0; j<sZ; ++j)
         *(SE+j) = smax;
      }

   return( SE );
   }


/* create a conical structuring element. This is a dual cone; that is */
/* two cones along tha same axis that share the same endpoint. The    */
/* axis of the cone is along z; it is perpendicular to the image      */
/* frames or slices. */

int *MakeCone( sX, sY, sZ, smax, sorgx, sorgy, sorgz )
   int sX,sY,sZ;
   int smax;
   int *sorgx, *sorgy, *sorgz;
   {    
   int i,j,k,n;
   int *s,*SE;
   double a,b,r,t,v;

   if ( !( sX && sY && sZ)  )
      {
      fprintf(stderr,
         "-k cone x y z v:  x, y, nor z may be zero.\n");
      exit(0);
      }

   if ( !( sX%2 && sY%2 && sZ%2 )  ||  smax < BLACK  ||  smax > WHITE )
      {
      fprintf(stderr,
         "-k cone x y z v:  x, y, & z must be odd;  v in [0,255].\n");
      exit(0);
      }

   *sorgx = sX/2;
   *sorgy = sY/2;
   *sorgz = sZ/2;

   n = sX * sY;

   if ( !(SE = (int *)calloc( n*sZ, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for auto SE.\n");
      exit(0);
      }

   if ( n > 1 )
      {
      /* t is one within a tolerance of 1/255 */
      t = 1.0 + (1.0 / (2.0 * (double)WHITE));

      s = SE;

      for (k=0; k<*sorgz; ++k)   /* fill in top half of cone */
         {
         /* first calculate radius for slice k */
         a = *sorgz 
           ? ((double)*sorgx/(double)*sorgz)*(double)(*sorgz-k) 
           : *sorgx ;
         b = *sorgz 
           ? ((double)*sorgy/(double)*sorgz)*(double)(*sorgz-k) 
           : *sorgy ;
         a = *sorgx ? 1.0 / a : 1.0  ;
         b = *sorgy ? 1.0 / b : 1.0  ;
         v = (double)smax;
  
         /* then fill in slice k */
         for ( j=-*sorgy; j<=*sorgy; ++j )
            for ( i=-*sorgx; i<=*sorgx; ++i )
               {
               r = pow(a*(double)i,2.0) + pow(b*(double)j,2.0);
               if ( r > t ) *(s++) = -1;
               else *(s++) = (int)( v*sqrt( t - r ) );
               }
         }

      for ( j=0; j<n; ++j )     /* median slice is all don't cares ... */
         {
         *(s++) = -1;
         }

      *(SE + (n*sZ)/2) = smax;  /* ... except for the origin */

      for (k=0; k<(n*(sZ-1))/2; ++k)  /* fill in bottom half from top half */
         {
         *(s+k) = *(s-n-1 - k);
         }
      }

   else  /* n == 1 (sZ by single pixel SE) */
      {
      for ( j=0; j<sZ; ++j)
         *(SE+j) = smax;
      }

   return( SE );
   }



/* create a spherical or ellipsoidal structuring element.     */ 
/* That is, a digital approximation to a sphere or ellipsoid. */

int *MakeSphere( sX, sY, sZ, smax, sorgx, sorgy, sorgz )
   int sX,sY,sZ;
   int smax;
   int *sorgx, *sorgy, *sorgz;
   {    
   int i,j,k,n;
   int *s,*S;
   double a,b,c,r,t,v;

   if ( !( sX && sY && sZ)  )
      {
      fprintf(stderr,
         "-k sphere x y z v:  x, y, nor z may be zero.\n");
      exit(0);
      }

   if ( smax < BLACK  ||  smax > WHITE )
      {
      fprintf(stderr,
         "-k sphere x y z v:  v must be in the range [0,255].\n");
      exit(0);
      }

   *sorgx = (sX % 2) ? sX/2 : (sX/2)-1;
   *sorgy = (sY % 2) ? sY/2 : (sY/2)-1;
   *sorgz = (sZ % 2) ? sZ/2 : (sZ/2)-1;

   n = sX * sY * sZ;

   if ( !(S = (int *)calloc( n, sizeof(int) )) )
      {
      fprintf(stderr,"Unable to allocate memory for auto SE.\n");
      exit(0);
      }

   if ( n > 1 )
      {
      a = (sX>1) ? 2/(double)(sX-1) : 0.0 ;
      b = (sY>1) ? 2/(double)(sY-1) : 0.0 ;
      c = (sZ>1) ? 2/(double)(sZ-1) : 0.0 ;
      v = (double)smax;
  
      /* t is one within a tolerance of max(a,b,c) */
      t = MAX(a,b);
      t = MAX(t,c);
      t = sqrt( 1.0 + t*t + 0.0001*t );

      s = S;
      if ( sX>1 && sY>1 && sZ>1)
         for ( k=0; k<sY; ++k )
            for ( j=0; j<sY; ++j )
               for ( i=0; i<sX; ++i )
                  {
                  r = sqrt( pow(-1.0+(double)i*a,2.0) 
                          + pow( 1.0-(double)j*b,2.0) 
                          + pow(-1.0+(double)k*c,2.0) );
                  if ( r*r > t ) *(s++) = -1;
                  else *(s++) = (int)( v*sqrt( t - r*r ) + 0.5 );
                  }
      else if ( sX>1 && sY>1 )
         for ( j=0; j<sY; ++j )
            for ( i=0; i<sX; ++i )
               {
               r = sqrt( pow(-1.0+(double)i*a,2.0) + pow(1.0-(double)j*b,2.0) );
               if ( r*r > t ) *(s++) = -1;
               else *(s++) = (int)( v*sqrt( t - r*r ) + 0.5 );
               }
      else if ( sY>1 && sZ>1 )
         for ( k=0; k<sZ; ++k )
            for ( j=0; j<sY; ++j )
               {
               r = sqrt( pow(-1.0+(double)j*b,2.0) + pow(1.0-(double)k*c,2.0) );
               if ( r*r > t ) *(s++) = -1;
               else *(s++) = (int)( v*sqrt( t - r*r ) + 0.5 );
               }
      else if ( sZ>1 && sX>1 )
         for ( k=0; k<sZ; ++k )
            for ( i=0; i<sX; ++i )
               {
               r = sqrt( pow(-1.0+(double)k*c,2.0) + pow(1.0-(double)i*a,2.0) );
               if ( r*r > t ) *(s++) = -1;
               else *(s++) = (int)( v*sqrt( t - r*r ) + 0.5 );
               }
      else  /* 2 of the 3 are equal to 1 */
         for ( j=0; j<n; ++j )
            {
            r = sqrt( pow(-1.0+(double)j*(a+b+c),2.0) );
            r = ( r < 1 ) ? r : 1.0 ;
            *(s++) = (int)( v*sqrt( 1.0 - r*r ) + 0.5 );
            }
      }
   else
      *S = smax;

   return( S );
   }


int GetSupport( s, sX, sY, SEisBIN )
   int *s;
   int sX, sY;
   int SEisBIN;
   {
   int i;
   int sup = 0;

   if ( SEisBIN )  for ( i=0; i<sX*sY; ++i ) s[i] = s[i] ? s[i] : -1 ;

   for ( i=0; i<sX*sY; ++i ) if ( s[i] >= 0 ) ++sup;

   return( sup );
   }

int GetSupport3D( s, sX, sY, sZ, SEisBIN )
   int *s;
   int sX, sY, sZ;
   int SEisBIN;
   {
   int i;
   int sup = 0;

   if ( SEisBIN )  for ( i=0; i<sX*sY*sZ; ++i ) s[i] = s[i] ? s[i] : -1 ;

   for ( i=0; i<sX*sY*sZ; ++i ) if ( s[i] >= 0 ) ++sup;

   return( sup );
   }

