/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/vcrEdit/RCS/vcrEditEdit.c,v 0.29 92/09/08 14:29:05 drapeau Exp $ */
/* $Log:	vcrEditEdit.c,v $
 * Revision 0.29  92/09/08  14:29:05  drapeau
 * * Fixed errors in the way edit lists are read in, especially
 *   in reading the label part of an edit.
 * * Removed a test for minimum edit length of 3 seconds during
 *   edit preview; this minimum length is based on a specific
 *   limitation of the NEC PC-VCR.  The limitation is now encoded
 *   in the NEC Driver, as it should be.
 * 
 * Revision 0.28  92/07/22  19:17:25  drapeau
 * Fixed small error in OpenHandler(), introduced in the last
 * revision.  The error was in a call to fscanf(); a classic error,
 * passing in a variable as argument instead of a pointer to that
 * variable.  The error is fixed now.
 * 
 * Revision 0.27  92/06/15  14:45:19  drapeau
 * Minor changes, in function OpenHandler():
 * In each line the third argument to the fscanf function is already the address
 * of a string, so the additional & isn't necessary (it does work with the & because the
 * compiler ignores any & before an array or function).  This fixed the corresponding
 * warning messages.
 * 
 * Revision 0.26  92/05/14  00:00:35  drapeau
 * Made several major additions and changes:
 * * Added functions to implement menu items in the edit panel's
 *   Document menu (Save, SaveAs, CloseFile, etc.).  Also, changed
 *   the semantics of some of these functions to simplify the
 *   interface.
 * * Added a function UpdateHeader() to print the name of the current
 *   document on the title bar of the edit list panel, and to print
 *   the word "modified" along with the title if the document was
 *   modified in any way.
 * * Modified several functions that manipulate the edit list (EditAdd(),
 *   EditModify(), etc.) to update modified status, and to call the
 *   UpdateHeader() function to reflect that status in the interface.
 * * Formatting changes to conform to coding standards.
 * 
 * Revision 0.25  92/01/03  18:00:04  drapeau
 * Changed all calls to Browse() to use "0" instead of "NULL", to
 * take into account ANSI-C definition of NULL as (void*)0.
 * 
 * Revision 0.24  91/09/26  12:32:27  lim
 * 1. Did casting for xv_get's.
 * 2. Corrected duration for SetDuration in the case when
 * no device is installed.
 * 
 * Revision 0.23  91/08/28  17:02:41  lim
 * Clears current selection panel in EditDeleteAll.
 * 
 * Revision 0.22  91/08/28  12:52:22  lim
 * Corrected duration. 
 * 
 * Revision 0.21  91/08/24  18:00:05  lim
 * Implemented PrintDiagnostics. With command line flag '-d', the 
 * app will launch such that it prints all diag messages.
 * 
 * Revision 0.20  91/08/23  17:23:44  lim
 * 1. Document format has changed.
 * 2. Duration can be calculated regardless of whether
 * myVideo has been initialized or not.
 * 
 * Revision 0.19  91/08/20  15:21:16  lim
 * currTape is no longer reset to "NULL" if tape doesn't match edit file.
 * 
 * Revision 0.18  91/08/20  15:18:09  lim
 * Fixed deleting old edits.
 * 
 * Revision 0.17  91/08/17  20:49:37  lim
 * OpenPanel now called Browse.
 * 
 * Revision 0.16  91/08/16  13:02:50  lim
 * Always display 2-digit numbers for hr, sec, min, frame.
 * 
 * Revision 0.15  91/08/16  12:33:19  lim
 * Interface changes, such as merging editCurrSelMsg into editNewEditMsg.
 * 
 * Revision 0.14  91/08/11  18:44:14  lim
 * Added warning message before loading new file and there are unsaved
 * changes.
 * 
 * Revision 0.13  91/08/09  16:42:52  lim
 * Included openPanel.
 * 
 * Revision 0.12  91/08/08  16:28:49  lim
 * 1. Added instance pointer to each public videoObject function call.
 * 2. DisplayError() now has 2 arguments.
 * 3. currentAudio is used to denote current audio setting in edit list.
 *    audioStg is set only when EditAdd() or EditModify() is called.
 * 4. Checks if speed is zero when adding/modifying. This is disallowed.
 * 
 * Revision 0.11  91/07/20  11:43:53  lim
 * Added code to support audio and speed settings for each edit.
 * 
 * Revision 0.10  91/07/16  16:16:24  lim
 * Initial revision.
 *  */

#include "vcrEdit.h"

static char editrcsid[] = "$Header: /Source/Media/collab/vcrEdit/RCS/vcrEditEdit.c,v 0.29 92/09/08 14:29:05 drapeau Exp $";

int currentAudio;
char currentFilename[256];					    /* Name of file currently open in edit list */

/* 	VIDEO EDIT FUNCTIONS (POPUP WINDOW 2) 	*/

/* format an edit entry for the panel list */
void
PrintOnList(current, buf)
     int current;
     char buf[70];
{
  char audio[7];

  switch (currentAudio)
  {
   case Stereo:
    strcpy(audio, "Stereo");
    break;
   case Right:
    strcpy(audio, "Right");
    break;
   case Left:
    strcpy(audio, "Left");
    break;
   case Mute:
    strcpy(audio, "Mute");
    break;
   default:
    DisplayError("Invalid audio setting", " ");
    return;
  }
  
  sprintf (buf, " %5d.  %-26.22s%.11s %.11s   %-8s%d", 
	   current+1, label[current], startframe[current], 
	   endframe[current], audio, speedStg[current]);
}

/* print the number of edit entries on the list */
void
PrintNumEdits()
{
  char edit[20];
  
  sprintf(edit, "No. of Edits : %d", lines);
  xv_set(vcrEdit_editPopup->editNumEditMsg, PANEL_LABEL_STRING, 
	 edit, NULL);
}

/* file error handling procedure */
void 
FileError(optype)
     int optype;
{
  char buf[40];
  
  if (optype == SAVE) 
    strcpy(buf, "Error in writing file.");
  else 
    switch(errno) 
    {
     case ENOTDIR:
     case EFAULT: 
     case ENOENT:	
      strcpy(buf, "File or directory does not exist.");
      break;
     case EACCES: 
      strcpy(buf, "File path search permission denied.");
      break;
     default:	
      strcpy(buf, "Error in opening file.");
       break;
     }

  DisplayError(buf, " ");
}

/*
 * Menu handler for `DocMenu (Open)'.
 */
Menu_item
FileLoad(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  int result;

  if (op != MENU_NOTIFY)
    return item;
  
  if ((change) && (lines > 0)) 
  {
    result = notice_prompt(vcrEdit_editPopup->editPopup, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Loading a file will erase all changes.",
			   "Go ahead and load the file?",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
      return;
  }
  
  Browse(NULL, BrowseOpen, 0, "#VCR Edit Document#", "vcrEdit");
  return item;
}



void
DisplayEditList(oldlines, newlines)
     int oldlines;
     int newlines;
{
  char buf[70];
  int i; 

  if (editnum != -1)						    /* Deselect any current selection */
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, FALSE,
	   NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* To prevent blinking of list, set XV_SHOW to FALSE first */
	 XV_SHOW, FALSE,
	 NULL);
  for (i=0; i<oldlines; i++)					    /* Replace the old string by the new one */
  {    
    currentAudio = audioStg[i];
    PrintOnList(i, buf);
    xv_set (vcrEdit_editPopup->editScrollList,
	    PANEL_LIST_STRING, i, buf,
	    PANEL_LIST_FONT, i, font,
	    NULL);
  }
  if (oldlines < newlines)					    /* Insert the additional new strings */
    for (i=oldlines; i<newlines; i++) 
    {
      currentAudio = audioStg[i];
      PrintOnList(i, buf);
      xv_set (vcrEdit_editPopup->editScrollList,
	      PANEL_LIST_INSERT, i,
	      PANEL_LIST_STRING, i, buf,
	      PANEL_LIST_FONT, i, font,
	      NULL);
    }
  else for (i=oldlines-1; i>newlines-1; i--)			    /* Delete all additional old entries in the edit list */
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_DELETE, i,
	   NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);
  xv_set(vcrEdit_editPopup->editModButton,			    /* Set modify, delete buttons to inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);
  PrintNumEdits();
  editnum = -1;							    /* Housekeeping stuff */
  change = 0;
  sprintf(currentFilename, "Untitled");
}								    /* end function DisplayEditList */


int 
  OpenHandler(char* proposedPath, int id)
{
  int i;
  int oldlines;
  char header[MAX_LABELSIZE];
  char tempLabel[32];
  char tapeCode[10];
  char tmp[25];
  char diagMsg[30];
  FILE* fp;
  
  fp = fopen(proposedPath, "r");				    /* Open file and load onto edit list */
  fgets(header, 20, fp);	
  
  if (strcmp(header, "#VCR Edit Document#") != 0)		    /* This isn't of correct file type */
  {
    fclose(fp);
    return 1;
  }
  else
  {
    fscanf(fp, "%*s%*s%s\n", tapeCode);				    
    oldlines = lines;
    fscanf(fp, "%*s%*s%*s%d", &lines);
    for (i=0; i<lines; i++) 
    {
      fscanf(fp, "%*s%*s%*d");				    
      fscanf(fp, "%*s%*s%s\n", startframe[i]);
      fscanf(fp, "%*s%*s%s\n", endframe[i]);
      fscanf(fp, "%*s%*s%d\n", &(audioStg[i]));
      fscanf(fp, "%*s%*s%*s%d\n", &(speedStg[i]));
      fgets(tempLabel, 32, fp);
      strncpy(label[i], &tempLabel[8], 22);
      if (strlen(label[i]) < MAX_LABELSIZE)			    /* Remove newline character if necessary */
	label[i][strlen(label[i])-1] = '\0';
      if ((strlen(label[i]) == MAX_LABELSIZE) &&
	  (label[i][MAX_LABELSIZE - 1] == '\n'))
	label[i][MAX_LABELSIZE -1] = '\0';
    }
    fclose (fp);
    change = 0;
    DisplayEditList(oldlines, lines);				    /* Print the new file on edit list */
    UpdateHeader(proposedPath, 0);				    /* Print new filename on the title bar */
    xv_set(vcrEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, 
	   "Current Selection : New Edit", NULL);
    strcpy(currentFilename, proposedPath);			    /* Copy filename into currentFilename */
    return 0;							    /* Returns 0 to openPanel */
  }
}								    /* end function OpenHandler */


/*
 * Menu handler for `DocMenu (Save)'.
 */
Menu_item
FileSave(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if ((op == MENU_NOTIFY) && lines)
    if (strcmp(currentFilename, "Untitled") != 0)
      Browse(currentFilename, BrowseCheckSave, 1, 
	     "#VCR Edit Document#", "vcrEdit");
    else
      Browse(NULL, BrowseSave, 0, "#VCR Edit Document#", "vcrEdit");
  
  return item;
}

/*
 * Menu handler for `DocMenu (Save As ...)'.
 */
Menu_item
FileSaveAs(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if ((op == MENU_NOTIFY) && lines)
    Browse(NULL, BrowseSave, 0, "#VCR Edit Document#", "vcrEdit");
  return item;
}



int
SaveHandler(proposedPath, id)
     char* proposedPath;
     int id;
{
  int i, result;
  char header[20];
  char buf[40];
  struct stat stbuf;
  FILE* fp;
  
  if (stat(proposedPath, &stbuf) == 0)				    /* check if file existed previously ... */
  {								    /* ...and if it is a vcr edit file */
    fp = fopen(proposedPath, "r");
    fgets(header, 20, fp);
    fclose(fp);
    if (strcmp(header, "#VCR Edit Document#") != 0) 
      sprintf(buf, "This file is not a vcr edit file.");
    else
      sprintf(buf, "This vcr edit file exists.");
    if (!id)
    {
      result = DisplayChoice(buf, "Do you wish to overwrite it?",
			     "No", "Yes");
      if (result == NOTICE_YES) 
	return 1;						    /* Returns 1 to openPanel */
    }
  }
  fp = fopen(proposedPath, "w");
  if (fp == NULL) 
  {
    FileError(SAVE);
    return 1;
  }
  lines = xv_get(vcrEdit_editPopup->editScrollList, PANEL_LIST_NROWS);
  fprintf(fp, "#VCR Edit Document#\n");				    /* Open and write the file */
  fprintf(fp, "#Tape Code:\tNULL\n");				    /* No longer check for medium identification */
  fprintf(fp, "#Number of Edits:\t%d\n\n", lines);
  for (i=0; i<lines; i++) 
  {
    fprintf(fp, "#Edit Number:\t%d\n", i+1);
    fprintf(fp, "#Start Address:\t%.11s\n", startframe[i]);
    fprintf(fp, "#End Address:\t%.11s\n", endframe[i]);
    fprintf(fp, "#Audio Setting:\t%d\n", audioStg[i]);
    fprintf(fp, "#Speed in frames/second:\t%d\n", speedStg[i]);
    fprintf(fp, "#Label:\t%s\n\n", label[i]);
  }
  fclose (fp);
 
  xv_set(vcrEdit_editPopup->editPopup, XV_LABEL,		    /* Changes title bar of window */
	 proposedPath, NULL);	
  if (!id)							    /* Was a SaveAs */
  {
    bzero(currentFilename, 256);
    strcpy(currentFilename, proposedPath);
  }
  change = 0;
  UpdateHeader(currentFilename, 0);
  return 0;
}								    /* end function SaveHandler */



/*
 * Menu handler for `DocMenu (Close File)'.
 */
Menu_item
  EditFileClose(item, op)
Menu_item	item;
Menu_generate	op;
{
  int result;
  
  if (op != MENU_NOTIFY)
    return item; 
  
  if ((change) && (lines > 0)) 
  {
    result = notice_prompt(vcrEdit_editPopup->editPopup, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Go ahead and close file without saving changes? ",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
      return item;
  }			
  DisplayEditList(lines, 0);					    /* Display new edit list which originally had 'lines'... */
  lines = 0;							    /* ...and now has 0 lines. */
  change = 0;
  UpdateHeader("Untitled", 0);
  xv_set(vcrEdit_editPopup->editNewEditMsg, 
	 PANEL_LABEL_STRING, "Current Selection : None", 
	 NULL);
  xv_set(vcrEdit_editPopup->editStartTxt1, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editStartTxt2, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editStartTxt3, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editStartTxt4, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editEndTxt1, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editEndTxt2, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editEndTxt3, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editEndTxt4, 
	 PANEL_VALUE, "00", NULL);
  xv_set(vcrEdit_editPopup->editDurTxt, 
	 PANEL_VALUE, "0", NULL);
  xv_set(vcrEdit_editPopup->editLabelTxt, 
	 PANEL_VALUE, " ", NULL);
  xv_set(vcrEdit_editPopup->editSpeedTxt, 
	 PANEL_VALUE, FrameRate, NULL);
  return item;
}								    /* end function EditFileClose */



/*
 * Menu handler for `DocMenu (Close Window)'.
 */
Menu_item
EditCloseWindow(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {  
    xv_set(vcrEdit_previewPopup->previewPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(vcrEdit_previewPopup->previewPopup, XV_SHOW, FALSE, NULL);
    xv_set(vcrEdit_editPopup->editPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
    xv_set(vcrEdit_editPopup->editPopup, XV_SHOW, FALSE, NULL);
  }

  return item;
}

void
EditDone(item, event)
     Panel_item	item;
     Event		*event;
{
  xv_set(vcrEdit_previewPopup->previewPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(vcrEdit_previewPopup->previewPopup, XV_SHOW, FALSE, NULL);
  xv_set(vcrEdit_editPopup->editPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(vcrEdit_editPopup->editPopup, XV_SHOW, FALSE, NULL);
}



/*
 * Menu handler for `editAudioMenu (Stereo)'.
 */
Menu_item
EditAudioStereo(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(vcrEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Stereo", NULL);
    currentAudio = Stereo;
  }
 
  return item;
}


/*
 * Menu handler for `editAudioMenu (Right)'.
 */
Menu_item
EditAudioRight(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(vcrEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Right", NULL);
    currentAudio = Right;
  }
  
  return item;
}


/*
 * Menu handler for `editAudioMenu (Left)'.
 */
Menu_item
EditAudioLeft(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(vcrEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Left", NULL);
    currentAudio = Left;
  }
  return item;
}


/*
 * Menu handler for `editAudioMenu (Mute)'.
 */
Menu_item
EditAudioMute(item, op)
	Menu_item	item;
	Menu_generate	op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(vcrEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, "Mute", NULL);
    currentAudio = Mute;
  }
 
  return item;
}

/*
 * Notify callback function for `editSpeedTxt'.
 */
Panel_setting
EditSpeedSet(item, event)
	Panel_item	item;
	Event		*event;
{
  if ((event_action(event) != '\r') && (event_action(event) != '\n'))
    return panel_text_notify(item, event);

  if (editnum != -1)
    speedStg[editnum]  =  xv_get(item, PANEL_VALUE);
  SetDuration();
  
  return panel_text_notify(item, event);
}


/* Procedures to display addresses */

void 
DisplayStartFrame(address)
     int address;
{
  int hr;
  int min;
  int sec;
  int fr;
  char hour[2];
  char minute[2];
  char second[2];
  char frame[2];
  extern void ConvertToTime();
  
  ConvertToTime(&hr, &min, &sec, &fr,  address);
  sprintf(hour, "%.2d", hr);
  sprintf(minute, "%.2d", min);
  sprintf(second, "%.2d", sec);
  sprintf(frame, "%.2d", fr);
  
  xv_set(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE, hour, NULL);
  xv_set(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE, minute, NULL);
  xv_set(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE, second, NULL);
  xv_set(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE, frame, NULL);
}

/*
  Button notify proc to get the start frame number (editGetStartButton)
  */
void
GetStart(item, event)
     Panel_item	item;
     Event		*event;
{
  DisplayStartFrame(DevQueryFrame(myVideo));
}


void 
DisplayEndFrame(address)
     int address;
{
  int hr; 
  int min;
  int sec;
  int fr;
  char hour[2];
  char minute[2];
  char second[2];
  char frame[2];
  
  ConvertToTime(&hr, &min, &sec, &fr, address);
  sprintf(hour, "%.2d", hr);
  sprintf(minute, "%.2d", min);
  sprintf(second, "%.2d", sec);
  sprintf(frame, "%.2d", fr);
  
  xv_set(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE, hour, NULL);
  xv_set(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE, minute, NULL);
  xv_set(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE, second, NULL);
  xv_set(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE, frame, NULL);
  SetDuration();
}

/*
  Button notify proc to get the end frame number (editGetEndButton)
  */
void
GetEnd(item, event)
     Panel_item	item;
     Event		*event;
{
  DisplayEndFrame(DevQueryFrame(myVideo));
}

/*
 * Menu handler for `previewMenu (Play part of edit...)'.
 */
Menu_item
EditPreview(item, op)
     Menu_item       item;
     Menu_generate   op;
{
  if (op == MENU_NOTIFY)
  {
    xv_set(vcrEdit_previewPopup->previewPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(vcrEdit_previewPopup->previewPopup, XV_SHOW, TRUE, NULL);
  }
  return item;
}

/* Procedure to display the approximate duration of the current selection.
 * Durations must be at least 3 seconds long.
 * Returns 0 if end address less than start address.
 */
int
SetDuration()
{
  int start, starthr, startmin, startsec, startfr;
  int end, endhr, endmin, endsec, endfr;
  int duration;
  int speedInFrames;
  int speedInDevice;
  char hr[2];
  char min[2];
  char sec[2];
  char frame[2];
  char buf[80];
  
  starthr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE));
  startmin = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE));
  startsec = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE));
  startfr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE));
  start = ConvertToAddress(starthr, startmin, startsec, startfr);
  
  endhr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE));
  endmin = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE));
  endsec = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE));
  endfr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE));
  end = ConvertToAddress(endhr, endmin, endsec, endfr);
  
  if (end - start < 0) 
  {
    DisplayError("End address is less than start address.", " ");
    return 0;
  }	
  speedInFrames =  xv_get(vcrEdit_editPopup->editSpeedTxt,
				    PANEL_VALUE);
  if (myVideo)
  {
    speedInDevice =  DevCalcSpeed(myVideo, speedInFrames, 1);
    
    if ((speedInDevice) && (end - start))
      duration = ((((end - start)%speedInDevice)/speedInDevice > 0.5) ? /* Round to nearest second */
		  (((end - start)/speedInDevice) + 1) 
		  : ((end - start)/speedInDevice));	   
    else 
      duration = 0;
  }
  else
  {
    if ((speedInFrames) && (end - start))
      duration = ((end - start)/speedInFrames +1);	
    else
      duration = 0;
  }
  
  sprintf (buf, "%d", duration);
  xv_set(vcrEdit_editPopup->editDurTxt, PANEL_VALUE, buf, NULL);
  return duration;
}								    /* end function SetDuration */



/*
  Button notify proc to add the current selection to the edit list
  (editAddButton)
  */
void
EditAdd(item, event)
     Panel_item	item;
     Event		*event;
{
  char buf[70];
  char s[30];
  int starthr; 
  int startmin;
  int startsec;
  int startfr;
  int endhr;
  int endmin;
  int endsec;
  int endfr;
  char start[11];
  char end[11];
  
  if ((int) xv_get(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE) <= 0)
  {
    DisplayError("Zero speed is not acceptable", "Please enter another value");
    return;
  }
  starthr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt1,   /* Get start frame and label */
				PANEL_VALUE));
  startmin = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt2,
				 PANEL_VALUE));
  startsec = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt3,
				 PANEL_VALUE));
  startfr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt4,
				PANEL_VALUE));

  endhr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt1,	    /* Get end frame & label */
			      PANEL_VALUE));
  endmin = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt2,
			       PANEL_VALUE));
  endsec = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt3,
			       PANEL_VALUE));
  endfr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt4,
			      PANEL_VALUE));
  
  if (!SetDuration())
    return;
  
  sprintf(start, "%.2d:%.2d:%.2d:%.2d", starthr, startmin, 
	  startsec, startfr);	
  sprintf(end, "%.2d:%.2d:%.2d:%.2d", endhr, endmin,
	  endsec, endfr);
  strncpy(startframe[lines], start, 11);
  strncpy(endframe[lines], end, 11);
  strcpy(label[lines], 
	 (char *) xv_get (vcrEdit_editPopup->editLabelTxt, PANEL_VALUE));
  speedStg[lines] =  xv_get (vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE);
  audioStg[lines] = currentAudio;
 
  xv_set(vcrEdit_editPopup->editScrollList,			    /* To prevent blinking of list, set XV_SHOW to FALSE first */
	 XV_SHOW, FALSE,
	 NULL);

  PrintOnList(lines, buf);					    /* Add this entry to the edit list */
  xv_set(vcrEdit_editPopup->editScrollList,
	 PANEL_LIST_INSERT, lines,
	 PANEL_LIST_STRING, lines, buf,
	 PANEL_LIST_FONT, lines, font,
	 NULL);
  if (editnum != -1)
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum, FALSE,
	   NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Select the newly added entry */
	 PANEL_LIST_SELECT, lines, TRUE,
	 NULL);
  editnum = lines;
  lines++;
  xv_set(vcrEdit_editPopup->editModButton,			    /* Make the modify and delete buttons active */
	 PANEL_INACTIVE, FALSE, NULL);
  xv_set(vcrEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, FALSE, NULL);
  sprintf(s, "Current Selection : Edit #%d", lines);		    /* Update edit status info */
  xv_set(vcrEdit_editPopup->editNewEditMsg,
	 PANEL_LABEL_STRING, s, NULL);
  PrintNumEdits();
  change = 1;
  UpdateHeader(currentFilename, 1);
}								    /* end function EditAdd */



/*
  Button notify proc to delete the current selection from the edit list
  (editDelButton)
  */
void
  EditDelete(item, event)
Panel_item      item;
Event           *event;
{
  int i;
  char buf[70];
  
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Delete the current selection */
	 PANEL_LIST_SELECT, editnum, FALSE,
	 NULL);
  if (editnum != 0) 
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum - 1, TRUE,
	   NULL);
  xv_set(vcrEdit_editPopup->editScrollList,
	 PANEL_LIST_DELETE, editnum,
	 NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* To prevent blinking of list, set XV_SHOW to FALSE first */
	 XV_SHOW, FALSE,
	 NULL);
  lines --;							    /* Shift all the following edit entries up by one */
  for (i=editnum; i < lines;i++) 
  {
    strncpy(startframe[i], startframe[i+1], 11);
    strncpy(endframe[i], endframe[i+1], 11);
    strcpy(label[i], label[i+1]);
    audioStg[i] = audioStg[i+1];
    speedStg[i] = speedStg[i+1];
    currentAudio = audioStg[i];
    
    PrintOnList(i, buf);
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_STRING, i, buf,
	   PANEL_LIST_FONT, i, font,
	   NULL);
  }
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);
  
  if (editnum != 0) 
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_SELECT, editnum - 1, FALSE,
	   NULL);
  xv_set(vcrEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, 
	 "Current Selection : New Edit", NULL);
  xv_set(vcrEdit_editPopup->editModButton,			    /* Make the delete, modify buttons inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);
  PrintNumEdits();
  change = 1;
  editnum = -1;
  UpdateHeader(currentFilename, 1);
}								    /* end function EditDelete */



/*
  Button notify proc to modify the current selection (editModButton)
  */
void
  EditModify(item, event)
Panel_item	item;
Event		*event;
{
  char buf[70];
  char start[11];
  char end[11];
  int starthr;
  int startmin; 
  int startsec; 
  int startfr;
  int endhr;
  int endmin; 
  int endsec;
  int endfr;
  
  if ((int) xv_get(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE) <= 0)
  {
    DisplayError("Zero speed is not acceptable", "Please enter another value");
    return;
  }
  starthr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE));
  startmin = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE));
  startsec = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE));
  startfr = atoi((char*) xv_get(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE));
  
  endhr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE));
  endmin = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE));
  endsec = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE));
  endfr = atoi((char*) xv_get(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE));
  
  if (!SetDuration())
    return;
  sprintf(start, "%.2d:%.2d:%.2d:%.2d", starthr, startmin, 
	  startsec, startfr);
  sprintf(end, "%.2d:%.2d:%.2d:%.2d", endhr, endmin,
	  endsec, endfr);
  strncpy(startframe[editnum], start, 11);
  strncpy(endframe[editnum], end, 11);
  strcpy (label[editnum], (char *) xv_get(vcrEdit_editPopup->editLabelTxt,
					  PANEL_VALUE));
  speedStg[editnum] = xv_get(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE);
  audioStg[editnum] = currentAudio;

  xv_set(vcrEdit_editPopup->editScrollList,			    /* To prevent blinking of list, set XV_SHOW to FALSE first */
	 XV_SHOW, FALSE,
	 NULL);
  PrintOnList(editnum, buf);
  xv_set(vcrEdit_editPopup->editScrollList,
	 PANEL_LIST_STRING, editnum, buf,
	 PANEL_LIST_FONT, editnum, font,
	 NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);
  change = 1;
  UpdateHeader(currentFilename, 1);
}								    /* end function EditModify */



/*
 * Notify callback function for `editScrollList'.
 */
int
List(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  int selection;
  char s[30];
  char starthr[2];
  char startmin[2];
  char startsec[2];
  char startfr[2];
  char endhr[2];
  char endmin[2];
  char endsec[2];
  char endfr[2];
  char audio[7];

  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    xv_set(vcrEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, 
	   "Current Selection : New Edit", NULL);
    xv_set(vcrEdit_editPopup->editModButton, PANEL_INACTIVE,	    /* Make the modify, delete buttons inactive */
	   TRUE, NULL);
    xv_set(vcrEdit_editPopup->editDelButton, PANEL_INACTIVE,
	   TRUE, NULL);
    break;
    
   case PANEL_LIST_OP_SELECT:
    sscanf(string, "%d", &selection);
    editnum = selection - 1;
    sprintf(s, "Current Selection : Edit #%d", selection);
    
    sscanf(startframe[editnum], "%2s:%2s:%2s:%2s", 
	   starthr, startmin, startsec, startfr);
    sscanf(endframe[editnum], "%2s:%2s:%2s:%2s", 
	   endhr, endmin, endsec, endfr);
    xv_set(vcrEdit_editPopup->editNewEditMsg,			    /* Load the appropriate values into the textfields */
	   PANEL_LABEL_STRING, s, NULL);
    xv_set(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE, starthr, 
	   NULL);
    xv_set(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE, startmin, 
	   NULL);
    xv_set(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE, startsec, 
	   NULL);
    xv_set(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE, startfr, 
	   NULL);
    xv_set(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE, endhr, 
	   NULL);
    xv_set(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE, endmin, 
	   NULL);
    xv_set(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE, endsec, 
	   NULL);
    xv_set(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE, endfr, 
	   NULL);
    xv_set(vcrEdit_editPopup->editLabelTxt, PANEL_VALUE, label[editnum], NULL);
    xv_set(vcrEdit_editPopup->editSpeedTxt, PANEL_VALUE, (int) speedStg[editnum], NULL);

    switch (audioStg[editnum])
    {
     case Stereo:
      strcpy(audio, "Stereo");
      break;
     case Right:
      strcpy(audio, "Right");
      break;
     case Left:
      strcpy(audio, "Left");
      break;
     case Mute:
      strcpy(audio, "Mute");
      break;
     default:
      DisplayError("Invalid audio setting", " ");
      return XV_OK;
    }
    xv_set(vcrEdit_editPopup->editAudioMenuButton, PANEL_LABEL_STRING, audio, NULL);
    SetDuration();
    xv_set(vcrEdit_editPopup->editModButton,			    /* Make the modify, delete buttons active */
	   PANEL_INACTIVE, FALSE, NULL);
    xv_set(vcrEdit_editPopup->editDelButton,
	   PANEL_INACTIVE, FALSE, NULL);
    break;
  }
  return XV_OK;
}								    /* end function List */


/*
 * Notify callback function for `editDelAllButton'.
 * Delete all entries in the edit list
 */
void
  EditDeleteAll(item, event)
Panel_item      item;
Event           *event;
{
  int i;
  int result;
  char edit[20];
  
  if ((change) && (lines > 0))					    /* Check if unsaved changes exist in edit list */
  {
    result = DisplayChoice("Unsaved changes exist in the edit list.",
			   "Go ahead and delete all entries in the edit list? ",
			   "No", "Yes");
    if (result == NOTICE_YES)
      return;
  }
  xv_set(vcrEdit_editPopup->editScrollList,			    /* To prevent blinking of list, set XV_SHOW to FALSE first */
	 XV_SHOW, FALSE,
	 NULL);
  for (i=0; i < lines; i++) 
    xv_set(vcrEdit_editPopup->editScrollList,
	   PANEL_LIST_DELETE, 0,
	   NULL);
  xv_set(vcrEdit_editPopup->editScrollList,			    /* Reshow the scrolling list */
	 XV_SHOW, TRUE,
	 NULL);
  lines  = 0;
  editnum = -1;
  change = 0;
  sprintf(currentFilename, "Untitled");
  UpdateHeader(currentFilename, 0);
  PrintNumEdits();
  sprintf(edit, "No. of Edits : 0");
  xv_set(vcrEdit_editPopup->editNumEditMsg, PANEL_LABEL_STRING, 
	 edit, NULL);
  xv_set(vcrEdit_editPopup->editNewEditMsg, PANEL_LABEL_STRING, 
	 "Current Selection : New Edit", NULL);
  xv_set(vcrEdit_editPopup->editEndTxt1, PANEL_VALUE, "  ");	    /* Clear out current selection */
  xv_set(vcrEdit_editPopup->editEndTxt2, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editEndTxt3, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editEndTxt4, PANEL_VALUE, "  ");
  
  xv_set(vcrEdit_editPopup->editStartTxt1, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editStartTxt2, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editStartTxt3, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editStartTxt4, PANEL_VALUE, "  ");
  
  xv_set(vcrEdit_editPopup->editLabelTxt, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editDurTxt, PANEL_VALUE, "  ");
  xv_set(vcrEdit_editPopup->editModButton,			    /* Make modify, delete buttons inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(vcrEdit_editPopup->editDelButton,
	 PANEL_INACTIVE, TRUE, NULL);
}								    /* end function EditDeleteAll */



void UpdateHeader (char* documentName, int modified)
{
  char	label[MAXPATHLEN+32];
  
  if (modified == 1)
  {
    sprintf(label, "%s (modified)",
	    documentName);
  }
  else
  {
    sprintf(label, "%s", documentName);
  }
  xv_set(vcrEdit_editPopup->editPopup, XV_LABEL,		    /* Print the new label string on the title bar */
	 label, NULL);	
}								    /* end function UpdateHeader */
