/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/cdEdit/RCS/cdEdit.c,v 2.11 92/05/14 16:40:54 drapeau Exp $ */
/* $Log:	cdEdit.c,v $
 * Revision 2.11  92/05/14  16:40:54  drapeau
 * * Removed code that modifies size of the cdEdit remote control panel.
 *   This code was a workaround for an error in OpenWindows 2.0, which has
 *   since been fixed.
 * * Made slight formatting changes to better conform to coding standards.
 * 
 * Revision 2.10  92/01/03  17:47:27  drapeau
 * Modified cdInit() so that call to Browse() uses "0" instead of "NULL",
 * to accommodate new ANSI-C definition of NULL as (void*)0.
 * 
 * Revision 2.0  91/10/06  21:01:02  chua
 * Update to version 2.0
 * 
 * Revision 1.37  91/09/18  17:26:18  chua
 * Removed the hard path for the include files.
 * 
 * Revision 1.36  91/09/17  11:41:12  chua
 * In InitializeCD, do not stop the cd player when initializing.  Reset the
 * volume only if the device is being opened for the first time.
 * 
 * Revision 1.35  91/09/16  14:19:52  chua
 * In CheckChanges, initialize result to NOTICE_NO, so that if there were no changes,
 * NOTICE_NO will be returned as the return value.
 * 
 * Revision 1.34  91/09/10  13:33:52  chua
 * Changed the AlertMessage a little in InitializeCD.
 * 
 * Revision 1.33  91/09/04  11:34:26  chua
 * In main(), use strcpy to assign senderPort.hostName instead of just using "=".
 * 
 * In QuitNotify, check that the CD device is open before attempting to stop the CD or eject
 * it.
 * 
 * 
 * Revision 1.32  91/09/03  15:33:48  chua
 * Include the getopt.h file from the collab directory.
 * 
 * Revision 1.31  91/09/03  13:45:45  chua
 * In main(), call ConnectPortManager to make a connection with the port manager.
 * In QuitNotify, check that there is a valid sender before attempting to perform
 * a DestroyReceiver.
 * A new function, InitializeCD, which will try to open the CD device (if necessary),
 * and check if there is a disc inserted.  If so, it will initialize the variables
 * connected with the CD (such as total tracks etc).
 * 
 * Removed the options pop-up window, which is replaced by a menu button.
 * 
 * A new function, UpdateHeader, which will update the header of the edit popup 
 * window to indicate which file is currently being edited and its status (whether
 * modified or not).
 * 
 * A new function, AlertMessage, which puts up a notice prompt displaying the
 * messages passed in as the parameters.
 * 
 * Revision 1.3  91/08/13  17:43:47  chua
 * *** empty log message ***
 * 
 * Revision 1.2  91/07/10  11:32:06  chua
 * Added a QuitNotify interpose function which will be called when the
 * user hits the Quit button or when the frame's quit menu is chosen.
 * This interpose function will stop the player and eject the disc, as well
 * as unregister the application from the Port Manager.
 * 
 * Revision 1.1  91/07/09  16:45:12  chua
 * In the main program, after the xv_main_loop line, check that the quit button has been
 * pressed.  If not, this means that the user has quit using the frame quit button.
 * In that case, call the quit procedure to eject the disc and deregister from the
 * Port Manager.
 * 
 * Revision 1.0  91/07/08  13:45:28  chua
 * Initial revision
 *  */

static char cdEditrcsid[] = "$Header: /Source/Media/collab/cdEdit/RCS/cdEdit.c,v 2.11 92/05/14 16:40:54 drapeau Exp $";

#include "main.h"
#include <getopt.h>
#include <xview/rect.h>
#include <Browse.h>

char *startFilename;
int editnum;							    /* current edit */
int lines;							    /* number of lines in the edit list */
int change;							    /* indicates if unsaved changes exist in the edit list */
char *cdfilename;						    /* Stores the full path name of the file currently being edited */
int offset;							    /* Keeps track of the offset into the current edit to be played */
Msf offsetStart;						    /* Starting time of the offset */
int playerState;						    /* current state of the cdEdit player */
int discInPlayer;						    /* Indicates whether there is a disc in the player */
Sender* sender;							    /* Sender and receiver to send and receive messages to other applications */
Receiver* receiver;
Port senderPort;
int ReceiverPortNumber;						    /* Port number for the cdEdit to listen on */

static struct itimerval timer;					    /* Timer to update the playing time display */
static char deviceOpen = 0;					    /* Flag to indicate if the cd device is open */

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;

void main(argc, argv)
     int		argc;
     char		**argv;
{
  static DispatchTable  DT = 
  {
    OpenDoc,
    GetDoc,
    GetSelection,
    SetSelection,
    PerformSelection,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    HaltSelection,
    PauseSelection,
    ResumeSelection,
    HideApplication,
    ShowApplication,
    GetAppIcon,
    };
  
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /* Initialize XView */
  INSTANCE = xv_unique_key();

  sender = NULL;
  receiver = NULL;
  senderPort.hostName = (char *) malloc(MAXPATHLEN);
  strcpy(senderPort.hostName, "localhost");			    /* Set the default values for hostname and port number and then */
  senderPort.portNumber = PortMgrPortNumber;
  ReceiverPortNumber = AnyPort;					    /* Check command line to see if user has specified any of these */
  cdfilename = (char *) malloc(MAXPATHLEN);			    /* Allocate space for the full path name of the file */
  strcpy(cdfilename, "untitled");
  CheckOptions(argc, argv);
  BuildDispatchTable (&DT);
  cdInit();							    /* Initialize the variables and cd player */
  xv_set(cdEdit_PortManagerPopup->NewPortManagerText
	 , PANEL_VALUE, senderPort.hostName, NULL);
  strcpy(senderPort.hostName, "(None)");			    /* Reset to (None) */
  ConnectPortManager(cdEdit_PortManagerPopup->ConnectPortManagerButton,
		     NULL);
  (void) notify_enable_rpc_svc (TRUE);
  
  timer.it_value.tv_sec = 1;					    /* Start the timer for managing the playing time display */
  timer.it_value.tv_usec = 0;
  timer.it_interval.tv_sec = 1;
  timer.it_interval.tv_usec = 0;
  notify_set_itimer_func(cdEdit_window1->window1, TimerNotify, 
			 ITIMER_REAL, &timer, NULL);
  xv_main_loop(cdEdit_window1->window1);			    /* Turn control over to Xview */
  exit(0);
}

/* 
 * This is the interpose function for the frame destroy procedure.  It is called when the quit button is pressed, or when the frame's "Quit"
 * menu selection is chosen.
 * The frame destroying process happens in two phases (meaning this function will be called twice).  The first time it is called, status =
 * DESTROY_CHECKING (this is done automatically by XView).  At this point, the interpose function can choose to veto the destruction, or 
 * let it proceed to phase two, where the actual destruction occurs.
 * If destruction is to proceed, the CD player will be stopped and any CD in it will be ejected.  The CD device is then closed, and the
 * application unregisters itself from the port manager.
 */
Notify_value QuitNotify(client, status)
     Notify_client client;
     Destroy_status status;
{
  if (status == DESTROY_CHECKING)				    /* Check to see if we really want to destroy the frame */
  {
    if (CheckChanges(QuitcdEdit) == NOTICE_YES)			    /* check if unsaved changes exist */
    {
      return notify_veto_destroy(client);
    }
  }
  else								    /* Second phase. Proceed to destroy */
  {
    notify_set_itimer_func(cdEdit_window1->window1, NOTIFY_FUNC_NULL, /* Turn off the timer */
			   ITIMER_REAL, NULL, NULL);
    if (deviceOpen == 1) 
    {
      if (discInPlayer == 1)
      {
	cdrom_stop(fd);		
	cdrom_eject (fd);
      }
      close(fd);
    }
    if (sender && receiver)					    /* Destroy sender and receiver, and disconnect from port manager */
    {
      DestroyReceiver(sender, receiver);
    }
    return notify_next_destroy_func(client, status);
  }
  return NOTIFY_DONE;
} 

/*
 * This function will initialize the CD-ROM player by trying to open the device (if it is not already open), and then checking if there is a CD
 * inserted.
 */
int InitializeCD()
{
  if (deviceOpen == 0) 
  {
    if ((fd = open("/dev/rsr0", O_RDONLY | O_EXCL)) == -1)	   
    {								    /* Initialization failed */
      xv_set(cdEdit_window1->window1, 
	     FRAME_RIGHT_FOOTER, "Failed to initialize player",
	     NULL);
      return Error;
    }	
    cdrom_volume(fd, 185, 185);					    /* Set the CD volume level */
  }
  deviceOpen = 1;
  if ((toc = (Toc) ReadDiscToc()) == (Toc)NULL) 		    /* Read the table of contents */
  {
    return Error;
  }
  if (editTotalTracks != 0 && 
      (editTotalTracks != toc->size || editmin != toc->total_msf->min || editsec != toc->total_msf->sec))
  {								    /* Check if the disc matches the currently loaded edit document (if any) */
    AlertMessage("This disc is not compatible with the current document.",
		 "Please try another disc, or close the current document first.");
    Eject(NULL, NULL);
    return Error;
  }
  numTracks = toc->size;
  startTrack = 1;
  endTrack = toc->size;
  playerState = StopMode;
  UpdateTime(Whole);						    /* update the time field to the total disc time */
  discInPlayer = 1;
  DimButtons(FALSE);
  xv_set(cdEdit_window1->window1, 
	 FRAME_RIGHT_FOOTER, "",
	 NULL);
  return OK;
}

/* 
 * This function initializes all the global variables used, the position of the popup windows and also the CD-ROM player itself.
 */
void cdInit() 
{
  Rect rect;
  
  cdEdit_window1 = cdEdit_window1_objects_initialize(NULL, NULL);   /* Initialize all the windows */
  cdEdit_EditPopup = cdEdit_EditPopup_objects_initialize(NULL, cdEdit_window1->window1);
  cdEdit_PreviewPopup = cdEdit_PreviewPopup_objects_initialize(NULL, cdEdit_window1->window1);
  cdEdit_InfoPopup = cdEdit_InfoPopup_objects_initialize(NULL, cdEdit_window1->window1);
  cdEdit_PortManagerPopup = ConnectPortManager_PortManagerPopup_objects_initialize(NULL, cdEdit_window1->window1);
  notify_interpose_destroy_func(cdEdit_window1->window1, QuitNotify); /* Interpose the destroy function */
  CreateBrowse(OpenHandler, SaveHandler, cdEdit_window1->window1);
  frame_get_rect(cdEdit_window1->window1, &rect);		
  rect.r_left = 100;						    /* Set the position of the main frame (remote controller) */
  rect.r_top = 250;
  frame_set_rect(cdEdit_window1->window1, &rect);		
  xv_set(cdEdit_EditPopup->EditPopup,				    /* Set the position of the edit popup window */
	 XV_X, 350,
	 XV_Y, 250,
	 FRAME_DONE_PROC, EditDone,
	 NULL);
  xv_set(cdEdit_PreviewPopup->PreviewPopup,			    /* Set the position of the preview popup window */
	 XV_X, 850,
	 XV_Y, 525,
	 NULL);
  xv_set(cdEdit_InfoPopup->InfoPopup,				    /* Set the position of the info popup window */
	 XV_X, 250,
	 XV_Y, 250,
	 NULL);
  xv_set(cdEdit_PortManagerPopup->PortManagerPopup,		    /* Set the position of the info popup window */
	 XV_X, 250,
	 XV_Y, 250,
	 NULL);
  xv_set(cdEdit_window1->VolumeSlider,				    /* Set the value of the volume slider so that it is in the middle */
	 PANEL_NOTIFY_LEVEL, PANEL_ALL, 
	 PANEL_VALUE, 180, 
	 NULL);
  xv_set(cdEdit_window1->BalanceSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(cdEdit_window1->DurationSlider, PANEL_NOTIFY_LEVEL, PANEL_ALL, NULL);
  xv_set(cdEdit_PreviewPopup->PreviewTimeText, PANEL_VALUE, 4, NULL);

  xv_set(cdEdit_EditPopup->ModifyButton,			    /* Set the modify, delete buttons to inactive */
	 PANEL_INACTIVE, TRUE, NULL);
  xv_set(cdEdit_EditPopup->DeleteButton,
	 PANEL_INACTIVE, TRUE, NULL);
  font = (Xv_font *) xv_find(cdEdit_EditPopup->EditPopup, FONT,	    /* Set the font for the panel list */
			     FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			     FONT_STYLE,  FONT_STYLE_NORMAL,
			     FONT_SIZE, 12, NULL);
  xv_set(cdEdit_window1->window1, 
	 FRAME_RIGHT_FOOTER, "No disc in player",
	 NULL);
  discInPlayer = 0;
  DimButtons(TRUE);
  playerState = Nothing;
  InitializeCD();
  editmin = 0;
  editsec = 0;
  editTotalTracks = 0;
  lines = 0;
  editnum = -1;
  displayType = Relative;
  offset = 0;
  UpdateHeader(0);
  if (strcmp(cdfilename, "untitled") != 0)			    /* Check if a file is to be loaded */
  {
    Browse(cdfilename, BrowseCheckOpen, 0, "#CD Edit Document#", "cdEdit");
  }
}

/* 
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  In this case, it will be the filename to be
 * loaded.  At present, only one filename will be handled.  So if there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"hostname", 1, 0, 'h'},		
    {"portnumber", 1, 0, 'p'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "h:p:",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
    {
      break;
    }
    switch (optionChar)
    {
     case 'h':
      if (optarg) 
      {
	strcpy (senderPort.hostName, optarg);
      }
      break;
     case 'p':
      if (optarg) 
      {
	ReceiverPortNumber = atoi(optarg);
      }
      break;
     default:
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
  {
    strcpy (cdfilename, argv[optind]);
  }
}

/*
 * Checks if there are any unsaved changes to the current cdEdit document.  If so, it warns the user by using a notice prompt, and
 * gives the user the choice of either going ahead with the operation or cancelling the operation.
 */
int CheckChanges (optype)
     int	 optype;
{
  int 	result;
  char 	buf[100], buf1[100];

  result = NOTICE_NO;
  if (change)							    /* Perform the following only if there are unsaved changes */
  {  
    switch (optype) 
    {
     case LoadFile :						    /* Loading a new document */
      sprintf (buf, "Loading a file will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and load the file?");
      break;
     case DeleteAll :						    /* Clearing the edit list */
      sprintf (buf, "This clear all operation will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and clear the cdEdit edit list");
      break;
     case QuitcdEdit :						    /* Quit the cdEdit application */
      sprintf (buf, "Quiting the cdEdit application will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and quit?");
      break;
     case CloseFile :						    /* Closing a file */
      sprintf (buf, "Closing the file will cause unsaved changes to be lost.");
      sprintf (buf1, "Go ahead and quit?");
      break;
     default : 
      break;
    }
    result = notice_prompt(cdEdit_window1->window1, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the current cdEdit edit list.",
			   buf, buf1, NULL,
			   NOTICE_BUTTON_NO,       "Yes",
			   NOTICE_BUTTON_YES,      "No",
			   NULL);
  }
  return (result);
}

/*
 * This function will update the header of the cdEdit Edit Popup window to either the current document name, or untitled.
 * It will also specify the status of the file, if it has been modified.
 */
void UpdateHeader(modified)
     int modified;
{
  char label[100];
  
  if (modified) 
  {
    sprintf(label, "CD Edit Document :  \"%s\" (modified)", cdfilename);
  }
  else 
  {
    sprintf(label, "CD Edit Document :  \"%s\" ", cdfilename);
  }
  xv_set(cdEdit_EditPopup->EditPopup,				    /* Set the edit popup header */
	 XV_LABEL, label,
	 NULL);
}

/*
 * This function will displace a notice prompt, which is used to alert the user that something is wrong.
 */
void AlertMessage(msg1, msg2)
     char *msg1;
     char *msg2;
{
  notice_prompt(cdEdit_window1->window1, NULL,
		NOTICE_MESSAGE_STRINGS,
		msg1, msg2,
		NULL,
		NOTICE_BUTTON, "OK", 100,
		NULL);
  
}
