/******************************************************************************\
*                                                                              *
* The software for `MaMA' was developed by:    H. K. Blokhuis and              *
*                                              S. L. Voogd                     *
*                                                                              *
* as part of their graduation for:     the Hoger Informatica Onderwijs (HIO)   *
* in Enschede.                             ---------------------------------   *
* The project was carried out at:          Sun Microsystems Nederland B.V.     *
* in Amersfoort.                           -------------------------------     *
*                                                                              *
*                                                                              *
* Copyright (c) 1992 Sun Microsystems Nederland B.V.                           *
*                                                                              *
* Permission to use, copy, modify, and distribute this software and            *
* its documentation for any purpose is hereby granted without fee, provided    *
* that (i) the above copyright notices and this permission notice appear in    *
* all copies of the software and related documentation, and (ii) the name      *
* Sun may not be used in any advertising or publicity relating to              *
* the software without the specific, prior written permission of               *
* Sun.                                                                         *
*                                                                              *
* THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,           *
* EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY             *
* WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.             *
*                                                                              *
* IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INCIDENTAL,                 *
* INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES                *
* WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT       *
* ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,        *
* ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS          *
* SOFTWARE.                                                                    *
*                                                                              *
\******************************************************************************/

/*----------------------------------------------------------------------------*\
|                                                                              |
|        File : MaMA_perform.c                                                 |
|                                                                              |
|    Contents : Functions/Procedures for performing/playing an Edit/Action     |
|                                                                              |
\*----------------------------------------------------------------------------*/

#include <MaMA_include.h>
#include <MaMA_global.h>

void CancelTimers();
void CancelTimer(Xv_opaque);
void CancelButtonGlowTimer();
void CancelButtonGlow();
void UpdateButtonGlow();

extern double ActionTime;
extern EDITLIST *CurEdit;
extern ACTIONLIST *ActionListPerformPtr;
extern int EditOffset;

double   timerInterval;
double   endTime;
double   currentTime;
double   actionTime;

static struct itimerval timer;
static struct itimerval updater;
static struct itimerval actiontimer;

extern int BlackColorIndex;
extern int GreenColorIndex;
extern int OrangeColorIndex;
extern int EditPlaying;
extern int EditPlayPaused;
extern int ActionPlaying;
extern int ActionPlayPaused;

char TargetDocName[100];
int RepeatValue;
extern unsigned int DocHandler;

/******************************************************************************\
*                                                                              *
*        Name : InitTimer                                                      *
*                                                                              *
* Description : inits                                                          *
*                                                                              *
\******************************************************************************/

void
InitTimer()
{
    currentTime = 0.0;
    actionTime = 0.0;
    timerInterval = 0.05;
    endTime = 0.0;
    EditPlaying = False;
    EditPlayPaused = False;
    ActionPlaying = False;
    ActionPlayPaused = False;
    RepeatValue = 0;

    BlackColorIndex = gcm_color_index("black");
    GreenColorIndex = gcm_color_index("green");
    OrangeColorIndex = gcm_color_index("orange");

    xv_set(MaMA_Main_Window->Action_Time_Gauge, PANEL_VALUE, 10, NULL);
    CancelButtonGlow();
}

/******************************************************************************\
*                                                                              *
*        Name : TimerAndActionHandler                                          *
*                                                                              *
* Description : Explicit Time and Actions handler                              *
*                                                                              *
\******************************************************************************/

Notify_value
TimerAndActionHandler(Notify_client client, int which)
{
    currentTime += timerInterval;
    actionTime += timerInterval;

    /* 
     * Is it time for the next action ?
     */
    if (currentTime >= ActionTime)
    {
        if (((EditPlaying) && (ActionListPerformPtr->next!=(NIL(ACTIONLIST *))))
            || (RepeatValue > 0))
        {
            /*
             * Not busy with a `repeated' action ?
             */
            
	    if (RepeatValue == 0)
            {
                ActionListPerformPtr = ActionListPerformPtr->next; 

                if((ActionListPerformPtr->Action->ActionType == GotoNP)
                    || (ActionListPerformPtr->Action->ActionType == GotoPP))
                {
                    RepeatValue = ActionListPerformPtr->Action->Param->Times;
                }
            }
            PerformAction(ActionListPerformPtr);
            ActionTime += (double)ActionListPerformPtr->Duration;
    	    actionTime = 0;
    	    xv_set(MaMA_Main_Window->Action_Time_Gauge, PANEL_VALUE,
                10, NULL);
        }
    }

    return(NOTIFY_DONE);
}

/******************************************************************************\
*                                                                              *
*        Name : UpdateGaugeHandler                                             *
*                                                                              *
* Description : Entered whenever it is time to do gauge updating               *
*                                                                              *
\******************************************************************************/

Notify_value
UpdateGaugeHandler(Notify_client client, int which)
{
    static int time;
    static int atime;

    time = ((currentTime/((double)endTime)) * 100.0);
    atime = ((actionTime/(double)ActionListPerformPtr->Duration) * 10.0);

    if (EditPlaying)
    {
        xv_set(MaMA_Main_Window->Time_Gauge1, PANEL_VALUE, time, NULL);
        xv_set(MaMA_Main_Window->Action_Time_Gauge, PANEL_VALUE,
	    10 - atime, NULL);

    } else {
        xv_set(MaMA_Edit_Panel->Time_Gauge2, PANEL_VALUE, time, NULL);
    }
  
    if (currentTime >= endTime) /* TIME_COEFF */
    {
        CancelTimers();
        CancelButtonGlowTimer();
        CancelButtonGlow();
    
        EditPlaying = False;
        ActionPlaying = False;

	xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    "", NULL);

#ifdef SaM
	if(!(xv_get(MaMA_Main_Window->Repeat_Setting, PANEL_VALUE)))	
	{
	    play_button_proc(MaMA_Main_Window->Main_Window, NIL(Event *));
	}
#endif

    }

    return(NOTIFY_DONE);
}

/******************************************************************************\
*                                                                              *
*        Name : SetTimer                                                       *
*                                                                              *
* Description : Set a periodic timer to poll for device open availability      *
*                                                                              *
\******************************************************************************/

void
SetTimer(double time)
{
    int min;
    double sec;
    char msg[6];

    endTime = (double)time;

    timer.it_value.tv_usec = 50000;
    timer.it_value.tv_sec = 0;
    timer.it_interval.tv_usec = 50000;
    timer.it_interval.tv_sec = 0;

    notify_set_itimer_func(MaMA_Main_Window->Main_Window,
        TimerAndActionHandler,
        ITIMER_REAL, &timer, NULL);
  
    updater.it_value.tv_usec = 50000;
    updater.it_value.tv_sec = 0;
    updater.it_interval.tv_usec = 50000;
    updater.it_interval.tv_sec = 0;

    notify_set_itimer_func(MaMA_About_Panel->About_Panel,
        UpdateGaugeHandler,
        ITIMER_REAL, &updater, NULL);
}

/******************************************************************************\
*                                                                              *
*        Name : CancelTimers                                                   *
*               CancelTimer                                                   *
*                                                                              *
* Description : Cancel any outstanding periodic timer                          *
*                                                                              *
\******************************************************************************/

void
CancelTimers()
{
    CancelTimer(MaMA_Main_Window->Main_Window);
    CancelTimer(MaMA_About_Panel->About_Panel);
    CancelTimer(MaMA_Edit_Panel->Edit_Panel);
    CancelTimer(MaMA_Edit_Panel->Action_List);
}

void
CancelTimer(Xv_opaque item)
{
    (void) notify_set_itimer_func(item, NOTIFY_FUNC_NULL,ITIMER_REAL,NULL,NULL);
}
  
/******************************************************************************\
*                                                                              *
*        Name : ButtonGlowTimerHandler                                         *
*                                                                              *
* Description : Called when it is time to do a "flash"                         *
*                                                                              *
\******************************************************************************/

Notify_value
ButtonGlowTimerHandler(Notify_client client, int which)
{
    UpdateButtonGlow();
    return(NOTIFY_DONE);
}

/******************************************************************************\
*                                                                              *
*        Name : SetButtonGlowTimer                                             *
*                                                                              *
* Description : Set a periodic timer to poll for device open availability.     *
*               This timer is mainly for updating the Button glow.             *
*                                                                              *
\******************************************************************************/

void
SetButtonGlowTimer(double time)
{
    static  struct itimerval        timer;
    int        secs;
    int        usecs;
  
    secs = (int)time;
    usecs = (int)((time - (int)secs) * 1000000.);
  
    timer.it_value.tv_usec = usecs;
    timer.it_value.tv_sec = secs;
    timer.it_interval.tv_usec = usecs;
    timer.it_interval.tv_sec = secs;
    (void)notify_set_itimer_func((Notify_client)MaMA_Edit_Panel,
         (Notify_func)ButtonGlowTimerHandler,
         ITIMER_REAL, &timer, 
         ((struct itimerval *)0));
}

/******************************************************************************\
*                                                                              *
*        Name : CancelButtonGlowtimer                                          *
*                                                                              *
* Description : Cancel any outstanding periodic timer.  This is used for       *
*               cancelling the timer which updates the button color.           *
*                                                                              *
\******************************************************************************/

void
CancelButtonGlowTimer()
{
    (void)notify_set_itimer_func((Notify_client)MaMA_Edit_Panel,
         (Notify_func)ButtonGlowTimerHandler,
         ITIMER_REAL, ((struct itimerval *)0), ((struct itimerval *)0));
}

/******************************************************************************\
*                                                                              *
*        Name : UpdateButtonGlow                                               *
*                                                                              *
* Description : Let play-buttons flash from green to black to green to black...*
*                                                                              *
\******************************************************************************/

void
UpdateButtonGlow()
{
    static short count = 0;
  
    if (count == 0)
    {
        count = 1;
        if (EditPlaying)
            xv_set(MaMA_Main_Window->Play_Button, PANEL_ITEM_COLOR, 
                GreenColorIndex, NULL);

        if (ActionPlaying)
            xv_set(MaMA_Edit_Panel->Play_Button2, PANEL_ITEM_COLOR, 
                GreenColorIndex, NULL);

    } else {
        count = 0;
        if (EditPlaying)
            xv_set(MaMA_Main_Window->Play_Button, PANEL_ITEM_COLOR, 
                BlackColorIndex, NULL);

        if (ActionPlaying)
            xv_set(MaMA_Edit_Panel->Play_Button2, PANEL_ITEM_COLOR, 
                BlackColorIndex, NULL);
    }  
}

/******************************************************************************\
*                                                                              *
*        Name : CancelButtonGlow                                               *
*                                                                              *
* Description : Make all buttons black again                                   *
*                                                                              *
\******************************************************************************/

void
CancelButtonGlow()
{
    xv_set(MaMA_Main_Window->Play_Button, PANEL_ITEM_COLOR, BlackColorIndex, 
        NULL);
    xv_set(MaMA_Main_Window->Pause_Button, PANEL_ITEM_COLOR, BlackColorIndex, 
        NULL);
    xv_set(MaMA_Edit_Panel->Play_Button2, PANEL_ITEM_COLOR, BlackColorIndex, 
        NULL);
    xv_set(MaMA_Edit_Panel->Pause_Button2, PANEL_ITEM_COLOR, BlackColorIndex, 
        NULL);
}

/******************************************************************************\
*                                                                              *
*        Name : PerformAction                                                  *
*                                                                              *
* Description : Performs an Action on a FrameMaker document                    *
*                                                                              *
\******************************************************************************/

PerformAction(ACTIONLIST *Action)
{
    char NewTargetDocName[FILELEN + PATHLEN];
    int code[2];
    char RepeatString[5];
    char TimesString[5];
    char ActionString[50];
    char *mama_home;

    strcpy(NewTargetDocName, Action->Path->Dir);

    if (strcmp("$MaMAHOME/Demo", NewTargetDocName) == 0) 
    {
	if ((mama_home = getenv( "MaMAHOME" )) != NULL)
	{
        	strcpy(NewTargetDocName, getenv( "MaMAHOME" ));
        	strcat(NewTargetDocName, "/Demo/");
	} else { 
		    fprintf(stderr,"MaMA ERROR: $MaMAHOME undefined!\n");
		    exit(1);
	}
    } else {
 
        strcat(NewTargetDocName, "/");
    } 

    strcat(NewTargetDocName, Action->Path->File);
 
    if (!OpenFM4EachAction)
    {
        if(Action->Action->ActionType != OpenDoc)
        {
	    /*
	     * Open FrameMaker document
	     */
            if (OpenDocProc(NewTargetDocName) == -1) return;
        }
    }
 
    switch(Action->Action->ActionType)
    {
        case Jump:
	    strcpy(ActionString, dgettext("MaMA_labels", "Jump To "));
            strcat(ActionString, Action->Action->Param->Hyperlink);
	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	        ActionString, NULL);
            GoToLink(DocHandler, "",
                Action->Action->Param->Hyperlink, False);
            break;

        case GotoFP:
	    strcpy(ActionString, dgettext("MaMA_labels", "Goto First Page"));
	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	        ActionString, NULL);
            code[0] = KBD_FIRSTPAGE;
            doFmCommands(DocHandler, code, 1);
            break;

        case GotoNP:
	    strcpy(ActionString, dgettext("MaMA_labels", "Goto Next Page ("));
            itoa((Action->Action->Param->Times - RepeatValue) + 1,RepeatString);
            strcat(ActionString, RepeatString);
            strcat(ActionString, dgettext("MaMA_labels", " of "));
            itoa(Action->Action->Param->Times, TimesString);
            strcat(ActionString, TimesString);
            strcat(ActionString, ")");
	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	        ActionString, NULL);
            code[0] = KBD_NEXTPAGE;
            doFmCommands(DocHandler, code, 1);
            RepeatValue--;
            break;

        case GotoPP:
	    strcpy(ActionString, dgettext("MaMA_labels", "Goto Prev Page ("));
            itoa((Action->Action->Param->Times - RepeatValue) + 1,RepeatString);
            strcat(ActionString, RepeatString);
            strcat(ActionString, dgettext("MaMA_labels", " of "));
            itoa(Action->Action->Param->Times, TimesString);
            strcat(ActionString, TimesString);
            strcat(ActionString, ")");
	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	        ActionString, NULL);
            code[0] = KBD_PREVPAGE;
            doFmCommands(DocHandler, code, 1);
            RepeatValue--;
            break;

        case GotoLP:
	    strcpy(ActionString, dgettext("MaMA_labels", "Goto Last Page"));
	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	        ActionString, NULL);
            code[0] = KBD_LASTPAGE;
            doFmCommands(DocHandler, code, 1);
            break;

        case ZoomIn:
            code[0] = KBD_ZOOMIN;
	    strcpy(ActionString, dgettext("MaMA_labels", "Zoom In, Fit: "));
            switch (Action->Action->Param->Fit)
            {
                case NONE:
                    strcat(ActionString, dgettext("MaMA_labels", "None"));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    doFmCommands(DocHandler, code, 1);
                    break;
 
                case WTP:
                    strcat(ActionString, dgettext("MaMA_labels", "WTP "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_WINDOW;
                    doFmCommands(DocHandler, code, 2);
                    break;
 
                case PTW:
                    strcat(ActionString, dgettext("MaMA_labels", "PTW "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_PAGE;
                    doFmCommands(DocHandler, code, 2);
                    break;
            }

            break;

        case ZoomOut:
            code[0] = KBD_ZOOMOUT;
	    strcpy(ActionString, dgettext("MaMA_labels", "Zoom Out, Fit: "));

            switch (Action->Action->Param->Fit)
            {
                case NONE:
                    strcat(ActionString, dgettext("MaMA_labels", "None"));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    doFmCommands(DocHandler, code, 1);
                    break;
 
                case WTP:
                    strcat(ActionString, dgettext("MaMA_labels", "WTP "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_WINDOW;
                    doFmCommands(DocHandler, code, 2);
                    break;
 
                case PTW:
                    strcat(ActionString, dgettext("MaMA_labels", "PTW "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_PAGE;
                    doFmCommands(DocHandler, code, 2);
                    break;
            }
            break;

        case Zoom100:
            code[0] = KBD_ZOOM100;
	    strcpy(ActionString,dgettext("MaMA_labels", "Zoom To 100%, Fit: "));

            switch (Action->Action->Param->Fit)
            {
                case NONE:
                    strcat(ActionString, dgettext("MaMA_labels", "None"));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    doFmCommands(DocHandler, code, 1);
                    break;
 
                case WTP:
                    strcat(ActionString, dgettext("MaMA_labels", "WTP "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_WINDOW;
                    doFmCommands(DocHandler, code, 2);
                    break;
 
                case PTW:
                    strcat(ActionString, dgettext("MaMA_labels", "PTW "));
	    	    xv_set(MaMA_Main_Window->Main_Window, FRAME_LEFT_FOOTER,
	    	        ActionString, NULL);
                    code[1] = KBD_ZOOM_FIT_PAGE;
                    doFmCommands(DocHandler, code, 2);
                    break;
            }
            break;

        case OpenDoc:
            strcpy(ActionString, dgettext("MaMA_labels", "Open Document"));
	    /*
	     * Open FrameMaker document
	     */
            if (OpenDocProc(NewTargetDocName) == -1) return;
            break;

        case CloseDoc:
            strcpy(ActionString, dgettext("MaMA_labels", "Close Document"));
            code[0] = KBD_CLOSEWIN; /*Quit's the doc in FM-3.0X, Frame BUG!*/
            doFmCommands(DocHandler, code, 1);
            break;

        case QuitDoc:
            strcpy(ActionString, dgettext("MaMA_labels", "Quit Document"));
            code[0] = KBD_QUITWIN;
            doFmCommands(DocHandler, code, 1);
            break;
        }
}
