/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/* $Header: /Source/Media/bryant/DisplayTool/RCS/slides.c,v 1.1 92/10/29 13:58:09 drapeau Exp $ */
/* $Log:	slides.c,v $
 * Revision 1.1  92/10/29  13:58:09  drapeau
 * Initial revision
 *  */
static char dtSlides[] = "$Header: /Source/Media/bryant/DisplayTool/RCS/slides.c,v 1.1 92/10/29 13:58:09 drapeau Exp $";

#include "DisplayTool.h"
#include "externs.h"



/*
 * Event callback function for `slides'.
 */
Notify_value
  DisplayTool_baseWindow_slides_event_callback(Xv_window win, Event *event, Notify_arg arg, Notify_event_type type)
{
  if (event_id(event) == LOC_DRAG) 
  {
    SlidesEventProc(win, event, arg, type);
  }
  /* gxv_start_connections DO NOT EDIT THIS SECTION */
  
  if (event_action(event) == ACTION_SELECT)
  {
    SlidesEventProc(win, event, arg, type);
  }
  
  if (event_action(event) == ACTION_ADJUST)
  {
    SlidesEventProc(win, event, arg, type);
  }
  
  if (event_action(event) == ACTION_MENU)
  {
    SlidesEventProc(win, event, arg, type);
  }
  
  /* gxv_end_connections */
  
  return notify_next_event_func(win, (Notify_event) event, arg, type);
}




/*
 * Called when an event is received in the slides canvas.
 */
Notify_value
  SlidesEventProc(Xv_window win, Event *event, Notify_arg arg, Notify_event_type type)
{
  static int	first, dragging, i, dragFirst = TRUE;
  static int	imageNum, selectionMade, resizing;
  static int	oldRectWidth, oldRectHeight;
  static XPoint	oldRectCorner;
  static XPoint pointerOffset;
  static int	slideOffsetX;
  static int	slideOffsetY;
  IMAGE		tempImage;
  XPoint	loc;
  int		width, height, effectiveWidth = None, effectiveHeight = None;
  XEvent	compressedEvent;
  
  PrintDTDiagnostics("Entered SlidesEventProc.\n");
  if (event_action(event) != event_id(event))			    /* Same reasoning as with the GalleryEventProc */
    {
      switch (event_action(event)) 
	{
	case ACTION_SELECT :					    /* the action */
	case MS_LEFT :						    /* the actual (literal) event */
	case ACTION_ADJUST :
	case MS_MIDDLE :
	case ACTION_MENU :
	case MS_RIGHT :
	  if (slideSize == Large)
	    notice_prompt(baseWindow->baseWindow, NULL,
			  NOTICE_MESSAGE_STRINGS,
			  "Please change 'Show Full Size' to NO before manipulating the slides.",
			  NULL,
			  NOTICE_BUTTON_YES, "OK",
			  NULL);
	  if (event_is_down(event))
	    SlideEventDown(event, &loc, &selectionMade, &imageNum, 
			   &resizing, &pointerOffset);
	      
	  else 
	    if (event_is_up(event))
              {
                SlideEventUp(&resizing, &first, dragging, selectionMade,
			     oldRectCorner, oldRectWidth, oldRectHeight);
		imageNum      = None;
		selectionMade = FALSE;
		dragging      = FALSE;
		dragFirst     = TRUE;
		RedrawRectangles();
	      }
	  break;
	  default :
	    break;
	}							    /* end switch (event_action(event)) */
    }
  else								    /* Same reasoning as with the GalleryEventProc */
    switch (event_id(event)) 
      {
      case LOC_DRAG:
	if (imageNum == None)					    /* Was an image selected? */
	  break;						    /* No, get out of this loop now */
	tempImage = dtImage[imageNum];
	if (selectionMade)
	  {
	    dragging = TRUE;
	    if (resizing)
	      SlideEventResize(event, loc, &first, &oldRectCorner, &oldRectWidth, &oldRectHeight);
		
	    else						    /* not resizing */
	      {
		if (dragFirst)
		  {
		    GetEffectiveDimensions(dtImage[numSlidesImages - 1],
					   &effectiveWidth, &effectiveHeight);
		    width  = (effectiveWidth > None)? effectiveWidth :
		      tempImage->origWidth / largeFactor;
		    height = (effectiveHeight > None)? effectiveHeight :
		      tempImage->origWidth / largeFactor;
		    dragFirst = FALSE;
		  }
		else
		  {
		    width = tempImage->origWidth / largeFactor;
		    height = tempImage->origWidth / largeFactor;
		  }
		GetOffsets(tempImage->slide,
			   &slideOffsetX, &slideOffsetY);
		if (tempImage->imageData == (byte*)NULL)
		{
		  BusyApp();					    /* Tell XView that the app is occupied */
		  dtImage[imageNum] = ReCreateImage(tempImage);
		  tempImage = dtImage[imageNum];
		  NormalApp();					    /* Tell XView that the app is ready-to-go again */
		}
		PaintSlideImage(tempImage, 0, 0,		    /* Draw this image on the slidesWin */
				slideOffsetX + tempImage->corner.x,
				slideOffsetY + tempImage->corner.y);
		tempImage->corner.x = (int)(event_x(event)-pointerOffset.x);
		tempImage->corner.y = (int)(event_y(event)-pointerOffset.y);
		PaintSlideImage(tempImage, 0, 0,		    /* Draw this image on the slidesWin */
				slideOffsetX + tempImage->corner.x,
				slideOffsetY + tempImage->corner.y);
		changes = TRUE;
	      }
	  }
	break;
      default:
	;
      }
  if (display)
    {
      for (i=XEventsQueued(display, QueuedAlready); i>1; i--)
	XNextEvent(display, &compressedEvent); 
    }  
  
  /* gxv_start_connections DO NOT EDIT THIS SECTION */
  
  /* gxv_end_connections */
  
  return notify_next_event_func(win, (Notify_event) event, arg, type);
}								    /* end function SlidesEventProc */



int 
  WhichImage(int x, int y)
{
  int i, xoffset, yoffset;
  
  GetOffsets(selectedSlide, &xoffset, &yoffset);
  for (i=0; i<numSlidesImages; i++)
    {
      if (dtImage[i]->slide == selectedSlide)
	{
	  if (x>=xoffset+dtImage[i]->corner.x && 
	      x<=xoffset+dtImage[i]->corner.x+(dtImage[i]->largeWidth / largeFactor) && 
	      y>=yoffset+dtImage[i]->corner.y && 
	      y<=yoffset+dtImage[i]->corner.y+(dtImage[i]->largeHeight / largeFactor))
	  {
	    sprintf(diagString, "In WhichImage, Image selected is %d.\n", i);
	    PrintDTDiagnostics(diagString);
	    return i;
	  }
	}
    }
  return None;
}								    /* end function WhichImage */



/*
 *  called from SlidesEventProc ...
 */
void
  SlideEventDown(Event*		event,
		 XPoint*	loc,
		 int*		selectionMade,
		 int*		imageNum,
		 int*		resizing,
		 XPoint*	pointerOffset)
{
  XPoint	adjustedLoc;
  int		xoffset, yoffset;
  IMAGE		tempImage;
  
  loc->x = event_x(event);
  loc->y = event_y(event);
  previousSlide = selectedSlide;
  
  if ((selectedSlide = WhichSlide(loc->x, loc->y, slideSize, &adjustedLoc)) < MaxNumSlides)
    {
      RedrawRectangles();
      xv_set(slidePopup->duration,
	     PANEL_VALUE,
	     slide[selectedSlide]->duration,
	     NULL);
      if (slide[selectedSlide]->label)
	if (strncmp(slide[selectedSlide]->label, "No Label", 8) != 0)
	  xv_set(slidePopup->label, PANEL_VALUE, slide[selectedSlide]->label, NULL);
	else
	  xv_set(slidePopup->label, PANEL_VALUE, "", NULL);
      SetCurrentSlide(selectedSlide+1);
    }
  if (*selectionMade == FALSE)
    {
      if ((*imageNum = WhichImage(loc->x, loc->y)) != None)
	{
	  tempImage = dtImage[*imageNum];
	  
	  if (event_action(event)==ACTION_MENU || event_action(event)==MS_RIGHT)
	    *resizing = TRUE;
	  else
	    *resizing = FALSE;
	  *selectionMade = TRUE;
	  
	  GetOffsets(numSlidesImages-1, &xoffset, &yoffset);
	  if (tempImage->imageData == (byte*)NULL)
	  {
	    BusyApp();							    /* Tell XView that the app is occupied */
	    dtImage[*imageNum] = ReCreateImage(tempImage);
	    tempImage = dtImage[*imageNum];
	    NormalApp();					    /* Tell XView that the app is ready-to-go again */
	  }
	  PaintSlideImage(tempImage, 0, 0,			    /* Draw this image on the slidesWin */
			  tempImage->corner.x,
			  tempImage->corner.y);
	  pointerOffset->x = loc->x - tempImage->corner.x;
	  pointerOffset->y = loc->y - tempImage->corner.y;
	  PaintSlideImage(tempImage, 0, 0,			    /* Draw this image on the slidesWin */
			  loc->x-pointerOffset->x,
			  loc->y-pointerOffset->y);
	  changes = TRUE;
	}
    } 
}								    /* end function SlideEventDown */


/*
 *  called from SlidesEventProc ....
 */
void
  SlideEventUp(int*	resizing,
	       int*	first,
	       int	dragging,
	       int	selectionMade,
	       XPoint	oldRectCorner,
	       int	oldRectWidth,
	       int	oldRectHeight)
{
  int	whichImageNum = None;
  char	errorMessage[MaxLength];
  IMAGE	tempImage;
  
  whichImageNum = WhichImage(oldRectCorner.x, oldRectCorner.y);	    /* Try to determine which image was being pressed */
  if (whichImageNum == None)					    /* No image?  Return immediately without doing anything */
    return;
  tempImage = dtImage[whichImageNum];
  if (*resizing && dragging)
    {
      XDrawRectangle(display, slidesWin, theGC,
		     oldRectCorner.x, oldRectCorner.y, 
		     oldRectWidth, oldRectHeight);
      if (tempImage->imageData == (byte*)NULL)
      {
	BusyApp();						    /* Tell XView that the app is occupied */
	dtImage[whichImageNum] = ReCreateImage(tempImage);
	tempImage = dtImage[whichImageNum];
	NormalApp();						    /* Tell XView that the app is ready-to-go again */
      }
      PaintSlideImage(tempImage, 0, 0,				    /* Draw this image on the slidesWin */
		      tempImage->corner.x,
		      tempImage->corner.y);
      tempImage->corner.x = oldRectCorner.x; 
      tempImage->corner.y = oldRectCorner.y; 
      PaintSlideImage(tempImage, 0, 0,				    /* Draw this image on the slidesWin */
		      tempImage->corner.x,
		      tempImage->corner.y);
      if (numSlidesImages > 0 && selectionMade)
	{
	  tempImage->slide = 
	    WhichSlide(tempImage->corner.x, 
		       tempImage->corner.y,
		       slideSize, &tempImage->corner);
	  if (tempImage->slide > MaxNumSlides-1)
	    {
	      sprintf(errorMessage, "Can't exceed %d slides.", MaxNumSlides);
	      notice_prompt(baseWindow->baseWindow, NULL,
			    NOTICE_MESSAGE_STRINGS,
			    errorMessage,
			    " ",
			    "Images beyond this limit shall",
			    "be placed in the last slide.",
			    NULL,
			    NOTICE_BUTTON_YES, "OK",
			    NULL);
	      tempImage->slide = MaxNumSlides - 1;
	      SlidesRepaint(NULL, NULL, NULL, NULL);
	    }
	  SetTotalNumberOfSlides();
	}
      XSetFunction(display, theGC, GXxor); 
      changes  = TRUE;
      *first    = FALSE;
      *resizing = FALSE;
    }
  else
    {
      if (numSlidesImages > 0 && selectionMade)
	{
	  tempImage->slide =
	    WhichSlide(tempImage->corner.x, 
		       tempImage->corner.y,
		       slideSize, &tempImage->corner);
	  
	  if (tempImage->slide > MaxNumSlides-1)
	    {
	      sprintf(errorMessage, "Can't exceed %d slides.", MaxNumSlides);
	      notice_prompt(baseWindow->baseWindow, NULL,
			    NOTICE_MESSAGE_STRINGS,
			    errorMessage,
			    " ",
			    "Images beyond this limit shall",
			    "be placed in the last slide.",
			    NULL,
			    NOTICE_BUTTON_YES, "OK",
			    NULL);
	      tempImage->slide = MaxNumSlides - 1;
	      SlidesRepaint(NULL, NULL, NULL, NULL);
	    }
	  SetTotalNumberOfSlides();
	  selectedSlide = tempImage->slide;
	  SetCurrentSlide(selectedSlide+1);
	}
    }
}								    /* end function SlideEventUp */


/*
 *
 */
void
  SlideEventResize(Event *event, XPoint loc, int *first, XPoint *oldRectCorner, 
		   int *oldRectWidth, int *oldRectHeight)
{
  int		rectWidth, rectHeight;
  int		whichImageNum = None;
  XPoint	rectCorner;
  IMAGE		tempImage;
  
  whichImageNum = WhichImage(oldRectCorner->x, oldRectCorner->y);   /* Try to determine which image was being pressed */
  if (whichImageNum == None)					    /* No image?  Return immediately without doing anything */
    return;
  tempImage = dtImage[whichImageNum];
  if (loc.x > (tempImage->corner.x + 
	       (int)((tempImage->largeWidth / largeFactor)/2.0)))
    {
      rectCorner.x = tempImage->corner.x;
      rectWidth = event_x(event)- tempImage->corner.x;
    }
  else 
    {
      rectCorner.x = event_x(event);
      rectWidth    = (tempImage->largeWidth / largeFactor) -
	(event_x(event)- tempImage->corner.x);
    }
  if (loc.y > tempImage->corner.y + 
      (int)((tempImage->largeHeight / largeFactor) / 2.0))
    {
      rectCorner.y = tempImage->corner.y;
      rectHeight = event_y(event)- tempImage->corner.y;
    }
  else 
    {
      rectCorner.y = event_y(event);
      rectHeight   = (tempImage->largeHeight / largeFactor) -
	(event_y(event)- tempImage->corner.y);
    }
  if (*first)
    XDrawRectangle(display, slidesWin, theGC,
		   oldRectCorner->x, oldRectCorner->y, 
		   *oldRectWidth, *oldRectHeight);
  else 
    *first = TRUE;
  
  if (rectCorner.x > (tempImage->corner.x + 
		      (tempImage->largeWidth / largeFactor) - 4))
    rectCorner.x = (tempImage->corner.x + 
		    (tempImage->largeWidth / largeFactor) - 4);
  
  if (rectCorner.y > (tempImage->corner.y + 
		      (tempImage->largeHeight / largeFactor) - 4))
    rectCorner.y = (tempImage->corner.y + 
		    (tempImage->largeHeight / largeFactor) - 4);
  
  if (rectWidth  < 4)
    rectWidth  = 4;
  if (rectHeight < 4)
    rectHeight = 4;
  if (rectWidth  > SmallWidth  - 5)
    rectWidth  = SmallWidth  - 5;
  if (rectHeight > SmallHeight - 5)
    rectHeight = SmallHeight - 5;
  
  XDrawRectangle(display, slidesWin, theGC,
		 rectCorner.x, rectCorner.y,
		 rectWidth, rectHeight);
  
  oldRectCorner->x = rectCorner.x; 
  oldRectCorner->y = rectCorner.y; 
  *oldRectWidth    = rectWidth; 
  *oldRectHeight   = rectHeight; 
  changes         = TRUE;
}								    /* end function SlideEventResize */


int 
  WhichSlide(int x, int y, int size, XPoint *corner)
{
  int	theSlide = 0;
  
  if (size == Small)
    {
      if (corner)
	{
	  corner->x = x % SmallWidth;
	  corner->y = y % SmallHeight;
	}
      theSlide = (x / SmallWidth) + (SlidesPerRow * (y / SmallHeight));
    }
  else if (size == Large)
    {
      if (corner)
	{
	  corner->x = (int)(x / largeFactor);
	  corner->y = (int)((y % FullHeight) / largeFactor);
	}
      theSlide = (int)(y / FullHeight);
    }
  sprintf(diagString, "In WhichSlide, theSlide is %d.\n", theSlide);
  return(theSlide);
}								    /* end function WhichSlide */



/*
 * Repaint callback function for `slides'.
 */
void
  SlidesRepaint(Canvas canvas, Xv_window paint_window, 
		Window xid, Xv_xrectlist *rects)
{
  int	i;
  
  PrintDTDiagnostics("Entering SlidesRepaint.\n");
  BusyApp();							    /* Tell XView that the app is occupied */
  XClearArea(display, slidesWin, 0, 0, (int)xv_get(baseWindow->slides, CANVAS_WIDTH),
	     (int)xv_get(baseWindow->slides, CANVAS_HEIGHT), True); 
  if (slideSize == Small)
  {
    for (i = 0; i < MaxNumSlides; i++)
    {
      PaintSlide(i);
    }								    /* end for... */
    RedrawRectangles();
  }
  XFlush(display);
  NormalApp();							    /* Tell XView that the app is ready-to-go again */
}								    /* end function SlidesRepaint */



void
  RedrawRectangles(void)
{
  int i;
  if (!display)
    return;
  XSetLineAttributes(display, theGC, 2, LineSolid, CapProjecting, JoinMiter);
  XSetFunction(display, theGC, GXcopy); 
  XSetForeground(display, theGC, BlackPixel(display, DefaultScreen(display)));
  for (i=0; i< MaxNumSlides; i++)
    {
      if (i == selectedSlide)
	XSetForeground(display, theGC, pixelValues[Red]);
      XDrawRectangle(display, slidesWin, theGC, 
		     (i%SlidesPerRow)*(SmallWidth-4)+4*(i%SlidesPerRow)+1, 
		     (i/SlidesPerRow)*(SmallHeight-4)+4*(i/SlidesPerRow)+1, 
		     SmallWidth-3, SmallHeight-3);
      if (i == selectedSlide)
	XSetForeground(display, theGC, BlackPixel(display, DefaultScreen(display)));
    }
  XSetFunction(display, theGC, GXxor); 
}


void InitSlide(SCR theSlide)
{
  int	i;
  
  theSlide->duration = NULL;
  theSlide->label = NULL;
  theSlide->numberOfImages = 0;
  for (i = 0; i < MaxNumImages / 2; i++)
    theSlide->imageIds[i] = 0;
  return;
}								    /* end function InitSlide */


void
  GetOffsets(int i, int *xoffset, int *yoffset)
{
  if (slideSize == Small)
    {
      *xoffset = SmallWidth*(i % SlidesPerRow);
      *yoffset = SmallHeight*(i / SlidesPerRow);
    }
  else								    /*  slideSize == Large */
    {
      *xoffset = 0;
      *yoffset = 0;    
    }
}								    /* end function GetOffsets */

void PaintSlideImage(IMAGE		theImage,		    /* This function paints the slide specified by... */
		     int		srcX,			    /* ..."theImage->slideImage" onto the appropriate slide... */
		     int		srcY,			    /* ...in the Slides window. */
		     int		destX,
		     int		destY)
{
  int	effectiveWidth;
  int	effectiveHeight;
  
  effectiveWidth = theImage->largeWidth / largeFactor;
  effectiveHeight = theImage->largeHeight / largeFactor;
  GetEffectiveDimensions(theImage,				    /* Determine the width to use for drawing this image... */
			 &effectiveWidth, &effectiveHeight);	    /* ...on its slide */
  if (theImage->slideImage == (XImage*)NULL)			    /* If the slideImage doesn't exist, create it first. */
    theImage->slideImage = Resize(theImage->imageData,
				  theImage->origWidth,
				  theImage->origHeight,
				  theImage->largeWidth / largeFactor,
				  theImage->largeHeight / largeFactor);
  XPutImage(display, slidesWin, theGC, 
	    theImage->slideImage,
	    srcX, srcY, destX, destY,
	    effectiveWidth, effectiveHeight);
}								    /* end function PaintSlideImage */



void PaintSlide(int slideNum)
{
  int		i;
  int		xOffset;
  int		yOffset;
  IMAGE		tempImage = (IMAGE)NULL;
  int		tempSlide, width, height;
  int		hold;
  XPoint	corner;
  
  sprintf(diagString, "Entering PaintSlide, slide %d.\n", slideNum);
  PrintDTDiagnostics(diagString);
  if (performing != 0)						    /* Don't redraw slides during performance of MAEstro... */
    return;							    /* ...messages */
  GetOffsets(slideNum, &xOffset, &yOffset);			    /* Determine coordinates in which to draw slides */
  XClearArea(display, slidesWin, xOffset, yOffset,
	     SmallWidth, SmallHeight, True);
  for (i = 0; i < numSlidesImages; i++)
  {
    if (dtImage[i]->slide == slideNum)				    /* Does this dtImage belong in this slide? */
    {
      if (dtImage[i]->imageData == (byte*)NULL)			    /* Is this image empty? */
      {								    /* Yes, create it on demand before trying to draw it. */
	sprintf(diagString, "In PaintSlide, creating image %d (file: %s) on demand.\n",
		i, dtImage[i]->filename);
	PrintDTDiagnostics(diagString);
	dtImage[i] = ReCreateImage(dtImage[i]);
      }								    /* end if image doesn't yet exist */
      PaintSlideImage(dtImage[i], 0, 0,				    /* Yes, paint this image in this slide */
		      xOffset + dtImage[i]->corner.x,
		      yOffset + dtImage[i]->corner.y);
    }								    /* end if this image belongs in this slide */
  }								    /* end for */
}								    /* end function PaintSlide */
