/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/* $Header: /Source/Media/bryant/DisplayTool/RCS/DisplayTool.c,v 1.41 92/10/30 19:09:38 drapeau Exp $ */
/* $Log:	DisplayTool.c,v $
 * Revision 1.41  92/10/30  19:09:38  drapeau
 * Minor modifications to remove extraneous diagnostic messages.
 * 
 * Revision 1.4  92/10/30  18:48:43  drapeau
 * Slight modifications to improve reporting of diagnostics, and in the
 * NullFields() function to initialize a new ImageStruct.
 * 
 * Revision 1.3  92/10/29  17:53:37  drapeau
 * Minor adjustments to functions responsible for creating and painting the
 * popup canvases used when displaying an image full-size.  The problem occurred
 * when a slide contained multiple images to be drawn: the result was that the
 * same image would be drawn in each of the full-sized canvases, instead of each
 * separate image being drawn in the separate canvases.
 * 
 * Revision 1.2  92/10/29  13:51:47  drapeau
 * Re-designed major parts of the application.  Cleaned up the interface to
 * the underlying image creation code.  Improved memory usage.  Improved
 * performance.  Improved robustness and error recovery.  Too many other
 * changes to mention here.  This is the first public release of the code.
 * 
 * Revision 1.1  92/09/29  01:04:00  drapeau
 * Initial revision
 * 
 * Revision 1.2  92/04/17  00:55:32  bryant
 * Integrated the xv-211 code
 * 
 * Revision 1.1  92/04/01  21:26:30  bryant
 * Initial revision
 *  */
static char rcsid[] = "$Header: /Source/Media/bryant/DisplayTool/RCS/DisplayTool.c,v 1.41 92/10/30 19:09:38 drapeau Exp $";

#include "DisplayTool.h"
#include "externs.h"

int main(int argc, char *argv[])
{
  int  i;
  FILE *fp;
  char *firstLine = "#DisplayTool Document#";
  char header[30];
  
  static DispatchTable  DT = 
    {
      OpenDoc,
      GetDoc,
      GetSelection,
      SetSelection,
      PerformSelection,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      HaltSelection,
      PauseSelection,
      ResumeSelection,
      HideApplication,
      ShowApplication,
      GetAppIcon
      }; 
  
  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /* Initialize XView. */
  INSTANCE = xv_unique_key(); 
  canvasKeyData = xv_unique_key();				    /* Create data key to store image numbers in popup canvases */
  theDisp = NULL;
  for (i=0; i<NumColors; i++)
    cmap[i] = 0;
  for (i = 0; i < MaxNumSlides; i++)				    /* Allocate space for all the slides */
  {
    slide[i] = (SCR)malloc(sizeof(struct SlideStruct));
    InitSlide(slide[i]);
  }
  
  /*
   * Initialize user interface components.
   */
  baseWindow = DisplayTool_baseWindow_objects_initialize(NULL, NULL);
  infoPopup = DisplayTool_infoPopup_objects_initialize(NULL, baseWindow->baseWindow);
  resizePopup = DisplayTool_resizePopup_objects_initialize(NULL, baseWindow->baseWindow);
  helpPopup = DisplayTool_helpPopup_objects_initialize(NULL, baseWindow->baseWindow);
  slidePopup = DisplayTool_slidePopup_objects_initialize(NULL, baseWindow->baseWindow);
  slideNumberPopup = DisplayTool_slideNumberPopup_objects_initialize(NULL, baseWindow->baseWindow);
  strcpy(currentFilename, "untitled");
  imagesPerRow = xv_get(baseWindow->gallery, XV_WIDTH)/102;
  CreateBrowse(OpenHandler, SaveHandler, baseWindow->baseWindow);      
  notify_interpose_destroy_func(baseWindow->baseWindow, QuitNotify);
  cms = (Cms)xv_create(NULL, CMS, CMS_SIZE, cmsSize, 
		       CMS_COLORS, colors, NULL);
  listFont = (Xv_font) xv_find(baseWindow->baseWindow, FONT,
			       FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			       FONT_STYLE, FONT_STYLE_NORMAL,
			       FONT_SIZE, 12,
			       NULL);
  
  /*
   *	Set input mask
   */
  xv_set(canvas_paint_window(baseWindow->gallery),
	 WIN_CONSUME_EVENTS,
	 WIN_NO_EVENTS, LOC_DRAG, WIN_MOUSE_BUTTONS, NULL,
         NULL);
  
  xv_set(baseWindow->gallery,
	 CANVAS_HEIGHT, 4*MaxNumImages,
	 CANVAS_AUTO_EXPAND, FALSE,
	 CANVAS_AUTO_SHRINK, FALSE, 
	 OPENWIN_AUTO_CLEAR, FALSE,  
	 NULL);
  
  galleryScrollbar = xv_create(baseWindow->gallery, SCROLLBAR,
			       SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
			       SCROLLBAR_PIXELS_PER_UNIT, 104,
			       SCROLLBAR_OBJECT_LENGTH, MaxNumSlides/2,
			       SCROLLBAR_PAGE_LENGTH, 1,
			       SCROLLBAR_VIEW_LENGTH,3,
			       NULL);
  
  xv_set(canvas_paint_window(baseWindow->slides),
	 WIN_CONSUME_EVENTS,
	 WIN_NO_EVENTS, LOC_DRAG, WIN_MOUSE_BUTTONS, NULL,
         NULL);

  xv_set(baseWindow->slides,
	 CANVAS_HEIGHT, SmallHeight*(MaxNumSlides/SlidesPerRow),
	 CANVAS_AUTO_EXPAND, FALSE,
	 CANVAS_AUTO_SHRINK, FALSE,       
	 OPENWIN_AUTO_CLEAR, FALSE,  
	 NULL);
  
  slidesScrollbar = xv_create(baseWindow->slides, SCROLLBAR,
			      SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
			      SCROLLBAR_PIXELS_PER_UNIT, SmallHeight,
			      SCROLLBAR_OBJECT_LENGTH, MaxNumSlides/SlidesPerRow,
			      SCROLLBAR_PAGE_LENGTH, 1,
			      SCROLLBAR_VIEW_LENGTH, 2,
			      NULL);
  
  pixelValues = (unsigned long *)xv_get(cms, CMS_INDEX_TABLE);
  
  display = (Display *)xv_get(baseWindow->baseWindow, XV_DISPLAY);
  galleryWin = (Window)xv_get(canvas_paint_window(baseWindow->gallery), XV_XID);
  slidesWin = (Window)xv_get(canvas_paint_window(baseWindow->slides), XV_XID);
  slidesCmap = DefaultColormap(display, DefaultScreen(display));
  galleryCmap = DefaultColormap(display, DefaultScreen(display));
  
  theCmap   = galleryCmap;
  theDisp   = display;
  theScreen = DefaultScreen(theDisp);
  rootW     = RootWindow(theDisp,theScreen);
  theGC     = DefaultGC(theDisp,theScreen);
  theVisual = DefaultVisual(theDisp,theScreen);
  dispWIDE  = DisplayWidth(theDisp,theScreen);
  dispHIGH  = DisplayHeight(theDisp,theScreen);
  dispDEEP  = DisplayPlanes(theDisp,theScreen);
  XSetLineAttributes(display, theGC, 2, LineSolid, CapProjecting, JoinMiter);
  borderOffset = BorderOffset + xv_get(baseWindow->borderControls, XV_HEIGHT);
  
  CreateCursor();						    /* Create the busy and normal cursors */

  for (i = 0; i < MaxNumImages; i++)				    /* Initialize pointers to Frames and Canvases that... */
  {								    /* ...display full-sized images */
    globalPopup[i] = (Frame)NULL;
    globalCanvas[i] = (Canvas)NULL;
    oldPopups[i] = (Frame)NULL;
    oldCanvases[i] = (Canvas)NULL;
  }
  senderPort.hostName = "localhost";				    /* Networking stuff */
  ReceiverPortNumber = AnyPort;
  verbose = 0;
  CheckOptions(argc, argv);
  
  senderPort.portNumber = PortMgrPortNumber;
  sender = NewSender(&senderPort);
  if (!sender)
    {
      notice_prompt(baseWindow->baseWindow, NULL,
		    NOTICE_MESSAGE_STRINGS,
		    "The PortManager is not running.",
		    "Please check that it is running and try again.",
		    NULL,
		    NOTICE_BUTTON_YES, "OK",
		    NULL);
      exit(0);
    }
  receiver = NewReceiver(sender, "DisplayTool", ReceiverPortNumber);
  if (verbose)
    printf("Listening on port %d.\n", receiver->transport->xp_port);
  
  BuildDispatchTable (&DT);
  canonFilename = (char *) malloc(MaxLength);
  
  (void) notify_enable_rpc_svc (TRUE);
  
  xv_set(slidePopup->duration, PANEL_VALUE, "Indefinite", NULL);
  xv_set(baseWindow->baseWindow, XV_LABEL, 
	 "DisplayTool Document :  \"untitled\"", NULL);
  xv_set(baseWindow->showFullSize, PANEL_VALUE, 2, NULL);
  if (argc > 1)
    for (i=1; i<argc; i++)
      if (!strncmp(argv[i], "-h", 2) || !strncmp(argv[i], "-p", 2) )
	i++;
      else if (strncmp(argv[i], "-v", 2))
	if ((fp=fopen(realpath(argv[i], dummy), "r")) != NULL)
	  {
	    fgets(header, strlen(firstLine)+1, fp);
	    header[strlen(firstLine)+1] = '\0';
	    if (!strncmp(header, firstLine, strlen(firstLine)))
	      OpenHandler(realpath(argv[i], dummy), 1);
	    else
	      LoadGalleryImage(realpath(argv[i], dummy));
	    fclose(fp);
	  }
  timer.it_value.tv_usec = 250000;
  timer.it_interval.tv_usec = 250000;
  xv_main_loop(baseWindow->baseWindow);				    /* Turn control over to XView. */
  exit(0);
}								    /* end function main */



/*
 * Menu handler for `DocumentMenu (Info)'.
 */
Menu_item 
  Info(Menu_item item, Menu_generate op)
{
  static int	first = TRUE;
  static	Rect	rect;
  
  if (first == TRUE)
    frame_get_rect(infoPopup->infoPopup, &rect);
  if (op == MENU_NOTIFY)
    if ((int)xv_get(infoPopup->infoPopup, XV_SHOW) == FALSE && first)			   
      {
	ShowPopup(&infoPopup->infoPopup, 300, 170, rect.r_width, rect.r_height);
	first = FALSE;
      }
    else
      ShowPopup(&infoPopup->infoPopup, None, None, rect.r_width, rect.r_height);
  return item;
}

/*
 * Menu handler for `DocumentMenu (Quit)'.
 */
Menu_item
  QuitDisplayTool(Menu_item item, Menu_generate op)
{
  int i;
  if (op == MENU_NOTIFY)
    {
      KillPopups();
      xv_destroy_safe(baseWindow->baseWindow);
    }
  return item;
}

/*
 * Menu handler for `imagesMenu (Load Images)'.
 */
Menu_item
  LoadImages(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    Browse(getwd(dummy), BrowseMultiple, NULL, NULL, NULL);
  return item;
}

/*
 * Menu handler for `imagesMenu (Clear Selected Images)'.
 */
Menu_item
  ClearSelectedImages(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    printf("Not implemented yet\n");
  return item;
}


/*
 * Menu handler for `imagesMenu (Clear Gallery)'.
 */
Menu_item
  ClearGallery(Menu_item item, Menu_generate op)
{
  int i;
  if (op == MENU_NOTIFY)
    {
      theCmap = galleryCmap;
      theDisp = display;
      for (i=0; i<numGalleryImages; i++)
      {
	if (dtImage[i]->galleryImage != (XImage*)NULL)
	{
	  XDestroyImage(dtImage[i]->galleryImage);
	  dtImage[i]->galleryImage = NULL;
	}
      }
      numGalleryImages = 0;
      GalleryRepaint(NULL, NULL, NULL, NULL);
    }
  return item;
}


/*
 * Menu handler for `documentMenu (Open)'.
 */
Menu_item
  OpenSequence(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    Browse(NULL, BrowseOpen, 1, "#DisplayTool Document#", "DisplayTool");
  return item;
}

/*
 * Menu handler for `documentMenu (Help)'.
 */
Menu_item
  Help(Menu_item item, Menu_generate op)
{
  static int	first = TRUE;
  static	Rect	rect;
  
  if (first == TRUE)
    frame_get_rect(helpPopup->helpPopup, &rect);
  if (op == MENU_NOTIFY)
    if ((int)xv_get(helpPopup->helpPopup, XV_SHOW) == FALSE && first)
      {
	ShowPopup(&helpPopup->helpPopup, 300, 170, rect.r_width, rect.r_height);
	first = FALSE;
      }
    else
      ShowPopup(&helpPopup->helpPopup, None, None, rect.r_width, rect.r_height);
  return item;
}


/*
 * Menu handler for `documentMenu (Save Sequence)'.
 */
Menu_item
  SaveSequence(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    if (strlen(currentFilename) > 0 &&
	strstr(currentFilename, "untitled") == NULL)
      Browse(currentFilename, BrowseCheckSave, NULL, "#DisplayTool Document#", "DisplayTool"); 
    else
      Browse(getwd(dummy), BrowseSave, NULL, "#DisplayTool Document#", "DisplayTool");
  return item;
}


/*
 * Menu handler for `documentMenu (Save Sequence As)'.
 */
Menu_item
  SaveSequenceAs(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    Browse(getwd(dummy), BrowseSave, NULL, "#DisplayTool Document#", "DisplayTool");
  return item;
}

/*
 * Menu handler for `documentMenu (Close)'.
 */
Menu_item
  Close(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    printf("Not implemented yet\n");
  return item;
}

/*
 * Menu handler for `slidesMenu (Insert Slide)'.
 */
Menu_item
  InsertNewSlide(Menu_item item, Menu_generate op)
{
  int	i;
  int	notifyFlag = 0;
  char	errorMessage[MaxLength];
  
  if (op == MENU_NOTIFY)
    {
      for (i=0; i < numSlidesImages; i++)
	{
	  if (dtImage[i]->slide >= selectedSlide)
	    if (dtImage[i]->slide < MaxNumSlides-1)
	      dtImage[i]->slide = dtImage[i]->slide+1;
	    else
	      {
		dtImage[i]->slide = MaxNumSlides-1;
		notifyFlag = 1;
	      }
	}
      if ((int)item != (int)baseWindow->baseWindow) 
	SlidesRepaint(NULL, NULL, NULL, NULL);
      if (notifyFlag)
	{
	  sprintf(errorMessage, "Can't exceed %d slides.", MaxNumSlides);
	  notice_prompt(baseWindow->baseWindow, NULL,
			NOTICE_MESSAGE_STRINGS,
			errorMessage,
			" ",
			"Images beyond this limit shall",
			"be placed in the last slide.",
			NULL,
			NOTICE_BUTTON_YES, "OK",
			NULL);
	  return(item);
	}
      changes = True;
      SetTotalNumberOfSlides();
    } 
  return(item);
}								    /* end function InsertNewSlide */

/*
 * Menu handler for `slidesMenu (Slide Info ...)'.
 */
Menu_item
  SlideInfo(Menu_item item, Menu_generate op)
{
  static int	first = TRUE;
  static	Rect	rect;
  
  if (first == TRUE)
    frame_get_rect(slidePopup->slidePopup, &rect);
  if (op == MENU_NOTIFY)
    {
      if ((int)xv_get(slidePopup->slidePopup, XV_SHOW) == FALSE && first)		  
	{
	  ShowPopup(&slidePopup->slidePopup, 300, 170, rect.r_width, rect.r_height);
	  first = FALSE;
	}
      else
	ShowPopup(&slidePopup->slidePopup, None, None, rect.r_width, rect.r_height);
      SetCurrentSlide(selectedSlide+1);
    }
  return item;
}



/*
 * Menu handler for `slidesMenu (Refresh Slides)'.
 */
Menu_item
  RedrawSlides(Menu_item item, Menu_generate op)
{
  if (op == MENU_NOTIFY)
    SlidesRepaint(NULL, NULL, NULL, NULL);
  return item;
}



/*
 * Menu handler for `slidesMenu (Clear All)'.
 */
Menu_item
  ClearAll(Menu_item item, Menu_generate op)
{
  int	i;
  IMAGE	tempImage;
  
  if (op == MENU_NOTIFY)
    {
      if (NotifyChoice(baseWindow->baseWindow,
		       "Are you sure you want to clear all the images in all the slides?") == Cancel)
	return item;
      theCmap = slidesCmap;
      theDisp = display;
      for (i=0; i<numClipboardImages; i++)
	FreeImage(clipboardImage[i]); 
      numClipboardImages = 0;
      for (i=0; i<MaxNumSlides; i++)
	{
	  if (slide[i]->duration != NULL)
	    free((char *)slide[i]->duration);
	  if (slide[i]->label != NULL)
	    free((char *)slide[i]->label);
	  slide[i]->duration = NULL;
	  slide[i]->label = NULL;
	  slide[i]->numberOfImages = 0;
	}
      for (i=0; i < numSlidesImages; i++)
      {
	tempImage = dtImage[i];
	if (tempImage->slideImage != (XImage*)NULL)
	{
	  XDestroyImage(tempImage->slideImage);
	  tempImage->slideImage = NULL;
	}
	if (tempImage->largeImage != (XImage*)NULL)
	{
	  XDestroyImage(tempImage->largeImage);
	  tempImage->largeImage = NULL;
	}
      }
      numSlidesImages = 0;
      SlidesRepaint(NULL, NULL, NULL, NULL);
      changes = True;
      SetTotalNumberOfSlides();
      RedrawRectangles();
    }
  return item;
}								    /* end function ClearAll */


/*
 * Event callback function for `duration'.
 */
void
  DurationEventProc(Panel_item item, Event *event)
{
  if (event_is_up(event) && event_is_ascii(event))
    slide[selectedSlide]->duration = 
      (char *)strdup((char *)xv_get(slidePopup->duration, PANEL_VALUE));
  changes = True;
  panel_default_handle_event(item, event);
}


/*
 * Event callback function for `label'.
 */
void
  LabelEventProc(Panel_item item, Event *event)
{
  if (event_is_up(event) && event_is_ascii(event))
    {
      if (strlen((char *)xv_get(slidePopup->label, PANEL_VALUE)) > 0)
	slide[selectedSlide]->label = 
	  (char *)strdup((char *)xv_get(slidePopup->label, PANEL_VALUE));
      else
	slide[selectedSlide]->label = (char *)strdup("No Label");
    }
  changes = True;
  panel_default_handle_event(item, event);
}


/*
 * Notify callback function for `indefiniteButton'.
 */
/* ARGSUSED */
void
  Indefinite(Panel_item item, Event *event)
{
  xv_set(slidePopup->duration, PANEL_VALUE, "Indefinite", NULL);
  slide[selectedSlide]->duration = (char *)strdup("Indefinite");
  changes = True;
}


/*
 * Notify callback function for `showFullSize'.
 */
/* ARGSUSED */
void
  ShowFullSize(Panel_item item, int value, Event *event)
{
  char temp[MaxLength];
  if (slideSize == Small)
    {
      xv_set(baseWindow->showFullSize, PANEL_VALUE, 1, NULL);
      slideSize = Large;
    }
  else
    {
      xv_set(baseWindow->showFullSize, PANEL_VALUE, 2, NULL);      
      KillPopups();
      slideSize = Small;
    }
  if (slideSize == Small)
    {
      if (slideSize == Small)
	{
	  previousSlide = selectedSlide;
	  selectedSlide = 0;
	}
      if ((int)xv_get(slidesScrollbar, SCROLLBAR_VIEW_START) != selectedSlide)
	xv_set(slidesScrollbar, SCROLLBAR_VIEW_START, selectedSlide, NULL); 
      SetCurrentSlide(selectedSlide+1);
      xv_set(slidePopup->duration, PANEL_VALUE, slide[selectedSlide]->duration, NULL);
      if (slide[selectedSlide]->label)
	if (strncmp(slide[selectedSlide]->label, "No Label", 8) != 0)
	  xv_set(slidePopup->label, PANEL_VALUE, slide[selectedSlide]->label, NULL);
	else
	  xv_set(slidePopup->label, PANEL_VALUE, "", NULL);
      SlidesRepaint(NULL, NULL, NULL, NULL);
    }
  else
    {
      beginDoubleBuffer = 1;
      sprintf(temp, "Current Slide    # %d", selectedSlide+1);
      PrepareMultipleWindowsForDisplay(selectedSlide);
      DisplayMultipleWindows(selectedSlide);
    }
  CheckGeometry();
}								    /* end function ShowFullSize */



IMAGE
  CreateImage(char *file)
{
  IMAGE		image = (IMAGE)NULL;
  int		i;
  
  sprintf(diagString, "Entering CreateImage with filename %s.\n",
	  file);
  PrintDTDiagnostics(diagString);
  image = xvimage_main(file);					    /* Try to create a new image */
  if (image == (IMAGE)NULL)					    /* Did the attempt fail? */
    return((IMAGE)NULL);					    /* Yes, return a null IMAGE, don't do anything else */
  if (verbose)
    printf("CREATEIMAGE: width=%d height=%d\n", image->origWidth, image->origHeight);
  image->largeWidth = image->origWidth;
  image->largeHeight = image->origHeight;
  for (i=0; i<numcols && i<=NumColors; i++)
    image->cols[i] = cols[i];
  image->numcols = numcols;
  image->nfcols = nfcols;
  for (i=0; i<nfcols && i<=NumColors; i++)
  {
    image->freecols[i] = freecols[i];
    cmap[(int)image->freecols[i]]++;
  }
  sprintf(diagString, "In CreateImage, format = %s\n", image->format);
  PrintDTDiagnostics(diagString);
  return(image);   
}								    /* end function CreateImage */



IMAGE ReCreateImage(IMAGE image)
{
  IMAGE		tempImage = (IMAGE)NULL;
  int		tempSlide, width, height;
  XPoint	corner;
  
  sprintf(diagString, "Entered ReCreateImage, creating image (file: %s) on demand.\n",
	  image->filename);
  PrintDTDiagnostics(diagString);
  width = image->largeWidth;
  height = image->largeHeight;
  corner.x = image->corner.x;
  corner.y = image->corner.y;
  tempSlide = image->slide;
  
  tempImage = CreateImage(image->filename);			    /* Create a new image for this file */
  if (tempImage == (IMAGE)NULL)					    /* Did the image creation fail? */
  {								    /* Yes, print a diagnostic message and return */
    sprintf(diagString, "In PaintSlide, could not create new image for file %s.\n",
	    image->filename);
    PrintDTDiagnostics(diagString);
    return((IMAGE)NULL);
  }
  tempImage->largeWidth = width;				    /* Restore crucial values to newly-created IMAGE structure */
  tempImage->largeHeight = height;
  tempImage->corner.x = corner.x;
  tempImage->corner.y = corner.y;
  tempImage->slide = tempSlide;
  return(tempImage);
}								    /* end function ReCreateImage */



void 
  ShowPopup(Frame *popup, int cornerx, int cornery, int width, int height)
{
  Rect rect;
  if (cornerx >= 0 && cornery >= 0)
    {
      sprintf(diagString,"ShowPopup: x=%d y=%d w=%d h=%d popup=%d\n",
	      cornerx, cornery, width, height, *popup);
      PrintDTDiagnostics(diagString);
      rect.r_left = cornerx;
      rect.r_top = cornery;
      rect.r_width = width;
      rect.r_height = height;
      frame_set_rect(*popup, &rect);
    }
  if ((int)xv_get(*popup, FRAME_CMD_PUSHPIN_IN) == FALSE)		   
    xv_set(*popup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);		   
  if ((int)xv_get(*popup, XV_SHOW) == FALSE)			   
    xv_set(*popup, XV_SHOW, TRUE, NULL);			   
  XRaiseWindow(display, (Window)xv_get(*popup, XV_XID));
  XFlush(display);
  PrintDTDiagnostics("Leaving ShowPopup.\n");
}	

void 
  HidePopup(Frame *popup)
{
  sprintf(diagString,"HidePopup: popup=%d\n", *popup);
  PrintDTDiagnostics(diagString);
  xv_set (*popup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);		   
  xv_set (*popup,XV_SHOW, FALSE, NULL);			   
/*
  if ((int)xv_get(*popup,FRAME_CMD_PUSHPIN_IN) == TRUE)		   
    xv_set (*popup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);		   
  if ((int)xv_get(*popup, XV_SHOW) == TRUE)			   
    xv_set (*popup,XV_SHOW, FALSE, NULL);			   
*/
}      



void
  GetEffectiveDimensions(IMAGE tempImage, int *width, int *height)  /* Given an image, determine the width and height... */
{								    /* ...necessary to make it fit into a slide for display */
  if (slideSize == Small)
  {
    if (tempImage->corner.x%SmallWidth +			    /* Would this image run off the right side of the... */
	(tempImage->largeWidth / largeFactor) > SmallWidth-2)	    /* ...slide? */
      *width = SmallWidth - 2 - tempImage->corner.x%SmallWidth;	    /* Yes, adjust the width to make the slide fit */
    if (tempImage->corner.y%SmallHeight +			    /* Same test for height: would this image run off the... */
	(tempImage->largeHeight / largeFactor) > SmallHeight-2)	    /* ...bottom of the slide? */
      *height = SmallHeight - 2  - tempImage->corner.y%SmallHeight; /* Yes, truncate the height to make the slide fit */
  }
  else								    /*  slideSize == Large */
  {
    *width = tempImage->largeWidth / largeFactor;
    *height = tempImage->largeHeight / largeFactor;
    return;
  }
}								    /* end function GetEffectiveDimensions */


void
  CopyImage(IMAGE srcImage, IMAGE destImage)			    /* Assumes "destImage" is always a dtImage */
{
  int	i;
  
  if (srcImage == destImage)					    /* Nothing to copy; images are identical */
    return;
  if (destImage == (IMAGE)NULL)
  {
    PrintDTDiagnostics("In CopyImage, destImage was NULL.\n");
    return;
  }
  if (destImage->filename != (char*)NULL)			    /* Begin copying data from "srcImage" to "destImage" */
    free(destImage->filename);
  destImage->filename = (char *)strdup(srcImage->filename);
  if (destImage->format != (char*)NULL)
    free(destImage->format);
  destImage->format = (char *)strdup(srcImage->format);
  destImage->corner.x = srcImage->corner.x;
  destImage->corner.y = srcImage->corner.y;
  if (destImage->imageData != (byte*)NULL)			    /* Is there already space allocated for the destination's... */
  {								    /* image data? */
    if ((destImage->origWidth * destImage->origHeight) <	    /* Is the destination image data too small? */
	(srcImage->origWidth * srcImage->origHeight))
    {								    /* Yes, free up old space and allocate enough for the... */
      free(destImage->imageData);				    /* ...new image data */
      destImage->imageData = (byte*)MyMalloc(srcImage->origWidth *
					     srcImage->origHeight);
    }
  }
  bcopy(srcImage->imageData, destImage->imageData,		    /* Copy the original image's data into the new image */
	srcImage->origWidth * srcImage->origHeight);
  destImage->origWidth = srcImage->origWidth;
  destImage->origHeight = srcImage->origHeight;
  
  if (destImage->slideImage != (XImage*)NULL)			    /* Re-create the slide image (this is inefficient: a... */
    XDestroyImage(destImage->slideImage);			    /* ...bcopy of the appropriate information from the... */
  destImage->slideImage = Resize(destImage->imageData,		    /* ...srcImage would be much better if it exists) */
				 destImage->origWidth,
				 destImage->origHeight,
				 srcImage->largeWidth / largeFactor,
				 srcImage->largeHeight / largeFactor);
  destImage->largeWidth = srcImage->largeWidth;
  destImage->largeHeight = srcImage->largeHeight;
  
  for (i=0; i<srcImage->numcols && i<=NumColors; i++)
    destImage->cols[i] = srcImage->cols[i];
  destImage->numcols = srcImage->numcols;
  destImage->nfcols = srcImage->nfcols;
  for (i=0; i<srcImage->nfcols && i<=NumColors; i++)
    {
      destImage->freecols[i] = srcImage->freecols[i];  
      cmap[(int)destImage->freecols[i]]++;
    }
}								    /* end function CopyImage */



void
  FreeImage(IMAGE image)
{
  if (image == (IMAGE)NULL)
    return;
  FreeColors(image);
  if (image->filename != NULL)
    {
      free(image->filename);
      image->filename = NULL;
    }
  if (image->format != NULL)
    {
      free(image->format);
      image->format = NULL;
    }
  if (image->galleryImage != (XImage*)NULL)
    {
      XDestroyImage(image->galleryImage);
      image->galleryImage = NULL;
    }
  if (image->slideImage != (XImage*)NULL)
    {
      XDestroyImage(image->slideImage);
      image->slideImage = NULL;
    }
  if (image->largeImage != (XImage*)NULL)
    {
      XDestroyImage(image->largeImage);
      image->largeImage = NULL;
    }
  if (image->imageData != (byte*)NULL)
    {
      free((char *)image->imageData);
      image->imageData = (byte*)NULL;
    }
  free((char *)image);
  image = (IMAGE)NULL;
}								    /* end function FreeImage */


void
  SetTotalNumberOfSlides(void)
{
  int maxSlide;
  char temp[MaxLength];
  maxSlide = GetMaxSlide();
  if (numSlidesImages > 0)
    {
      sprintf(temp, "Total Number of Slides :  %d", maxSlide + 1);
      xv_set(baseWindow->totalNumberOfSlidesMessage, PANEL_LABEL_STRING, temp, NULL);
    }
  else 
    {
      xv_set(baseWindow->totalNumberOfSlidesMessage, PANEL_LABEL_STRING, 
	     "Total Number of Slides :  0", NULL);
    }
}


int 
  GetMaxSlide(void)
{
  int i, maxSlide = 0;
  for (i=0; i<numSlidesImages; i++)
    {
      if (dtImage[i]->slide > maxSlide)
	maxSlide = dtImage[i]->slide;
    }
  return maxSlide;
}


void
  FreeColors(IMAGE image)
{
  int i;
  int count = 0;
  
  for (i=0; i<image->nfcols; i++) 
    {
      /*      if (i==image->nfcols-1)
	      printf("cmap[image->freecols[nfcols-1]]=%d\n", cmap[(int)image->freecols[i]]);  */
      if (cmap[(int)image->freecols[i]]--==1)   
	{
	  XFreeColors(theDisp, theCmap, image->freecols + i, 1, 0L);
	  sprintf(diagString, "f%d ", (int)image->freecols[i]);
	  PrintDTDiagnostics(diagString);
	  count++;
	}
      else if (cmap[(int)image->freecols[i]] < 0)
	cmap[(int)image->freecols[i]] = 0;
    }
  if (verbose)
    {
      printf("\nFreed %d colors for %s   nfcols = %d\n", count, image->filename, image->nfcols);
      PrintCmap();
    }
}								    /* end function FreeColors */




/*
  This is the interpose function for the frame destroy procedure.  It
  is called when the quit button is pressed, or when the frame's "Quit"
  menu selection is chosen.
  The frame destroying process happens in two phases (meaning this
  function will be called twice).  The first time it is called, status =
  DESTROY_CHECKING (this is done automatically by XView).  At this
  point, the interpose function can choose to veto the destruction, or
  let it proceed to phase two, where the actual destruction occurs.
  */
Notify_value 
  QuitNotify(Notify_client client, Destroy_status status)
{
  /* First phase:  Check to see if we really want to destroy the frame */
  if (status == DESTROY_CHECKING)
    {
      if (CheckChanges(client) == CancelQuit)
	return notify_veto_destroy(client);
    }
  else /* Second phase:  Proceed to destroy */
    {
      SenderDisconnectFromPortMgr(sender, &(receiver->receivePort));
      return notify_next_destroy_func(client, status);
    } 
  return NOTIFY_DONE;
}



int
  CheckChanges(Notify_client client)
{
  int	response;
  char	message[MaxLength];
  
  if (changes == 0)						    /* No changes, it's okay to quit right now */
    return(~CancelQuit);
  sprintf(message, "Save changes to '%s' ?", currentFilename);
  response = notice_prompt(baseWindow->baseWindow, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   message,
			   NULL,
			   NOTICE_BUTTON_YES, "Yes",
			   NOTICE_BUTTON_NO,  "No",
			   NOTICE_BUTTON, "Cancel", 100,
			   NULL);
  if (response == 100)
    return(CancelQuit);
  if (response == NOTICE_YES)
  {
    WriteSequenceToFile();
    notify_veto_destroy(client);
  }
  return(~CancelQuit);
}								    /* end function CheckChanges */



/* 
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  In this case, it will be the filename to be
 * loaded.  At present, only one filename will be handled.  So if there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(int argc, char **argv)
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
    {
      {"hostname", 1, 0, 'h'},		
      {"portnumber", 1, 0, 'p'},
      {"verbose", 0, 0, 'v'},
      {0, 0, 0, 0}
    };
  while (1)							    /* Start parsing all known options */
    {
      optionChar = getopt_long_only (argc, argv, "h:p:v",
				     long_options, &option_index);
      if (optionChar == EOF)					    /* Done with all known options */
	break;
      switch (optionChar)
	{
	case 'h':
	  if (optarg) 
	    senderPort.hostName = strdup(optarg);
	  break;
	case 'p':
	  if (optarg) 
	    ReceiverPortNumber = atoi(optarg);
	  break;
	case 'v':
	  verbose = 1;
	  printDiags = 1;
	  break;
	default:
	  break;
	}
    }
}

void
  DisplayImages(int slideNum)
{
  int		i, j, tempSlide, width, height;
  int		hold;
  XPoint	corner;
  IMAGE		tempImage;
  
  PrintDTDiagnostics("Entering DisplayImages.\n");
  for (i=0; i<slide[slideNum]->numberOfImages; i++)		    /* Load all the images for this slide only */
  {
    j = slide[slideNum]->imageIds[i];				    /* Set j to one of this slide's images (the 'ith image) */
    width = dtImage[j]->largeWidth;
    height = dtImage[j]->largeHeight;
    corner.x = dtImage[j]->corner.x;
    corner.y = dtImage[j]->corner.y;
    tempSlide = dtImage[j]->slide;
    if (verbose)
      printf("filename = %s\n", dtImage[j]->filename);
    hold = numSlidesImages;
    numSlidesImages = 0; 
    if (dtImage[j]->imageData == (byte*)NULL)
    {
      BusyApp();						    /* Tell XView that the app is occupied */
      tempImage = ReCreateImage(dtImage[j]);
      dtImage[j] = tempImage;
      NormalApp();						    /* Tell XView that the app is ready-to-go again */
    }
    if (AlreadyInSlidesBuffer(tempImage) == NotInBuffer)	    /* Does this image have its slide image built yet? */
    {								    /* No, build it */
      sprintf(diagString, "Slide %d (filename : %s) is not in the Slides buffer.\n",
	      j, tempImage->filename);
      PrintDTDiagnostics(diagString);
      tempImage->slideImage = Resize(tempImage->imageData,
				     tempImage->origWidth,
				     tempImage->origHeight,
				     tempImage->origWidth / largeFactor,
				     tempImage->origHeight / largeFactor);
    }
    numSlidesImages = hold;
  }   
}								    /* end function DisplayImages */



int
  AlreadyInSlidesBuffer(IMAGE tempImage)
{
  if (tempImage->slideImage == (XImage*)NULL)
    return(NotInBuffer);
  else
    return(0);
} 



/*
 * Notify callback function for `DoneHandler'.
 */
/* ARGSUSED */
void
  DoneHandler(Panel_item item, Event *event)
{
  Panel panel = (Panel)xv_get(item, PANEL_PARENT_PANEL);
  Frame frame = (Frame)xv_get(panel, XV_OWNER);
  HidePopup(&frame);  
}


void
  PrintCmap(void)
{
  int i;
  
  printf("In PrintCmap, colormap is:\n");
  for (i=0; i< NumColors; i++)
    printf("%d ", cmap[i]);
  printf("\n\n");
}

void
  NullFields(IMAGE image)
{
  int	i;
  
  image->filename = NULL;
  image->format = NULL;
  image->origWidth = 0;
  image->origHeight = 0;
  image->largeWidth = 0;
  image->largeHeight = 0;
  image->galleryWidth = 0;
  image->galleryHeight = 0;
  image->imageData = (byte*)NULL;
  image->galleryImage = (XImage*)NULL;
  image->slideImage = (XImage*)NULL;
  image->largeImage = (XImage*)NULL;
  image->slide = 0;
  image->corner.x = 0;
  image->corner.y = 0;
  image->numcols = 0;
  image->nfcols = 0;
  for (i = 0; i < NumColors; i++)
     {
       image->cols[i] = 0;
       image->freecols[i] = 0;
     }
}								    /* end function NullFields */


/* ARGSUSED */
Notify_value
  IncrementCounter(Notify_client client, int which)
{
  if (!pauseMode)
    {
      timeCount++;
      if (atoi(slide[selectedSlide]->duration))
	if ((timeCount-1)/4 >= atoi(slide[selectedSlide]->duration)) /* Subtract 1 from timeCount since it start counting from 0 */
	  {
	    KillPopups();
	    notify_set_itimer_func((Notify_client)baseWindow->baseWindow,
				   NOTIFY_FUNC_NULL, ITIMER_REAL, NULL, NULL);
	  }
    }
  return(NOTIFY_DONE);
}								    /* end function IncrementCounter */



void
  CheckGeometry(void)
{
  Rect rect;
  if (dontCheckGeometry == 0)
    {
      /* Bry --      frame_get_rect(fullslidePopup->fullslidePopup, &rect);  */
      /*      printf("x=%d y=%d w=%d h=%d\n", rect.r_left, rect.r_top, rect.r_width, rect.r_height); 
	      if (rect.r_left != slide[selectedSlide]->corner.x || rect.r_top != slide[selectedSlide]->corner.y ||
	      rect.r_width != slide[selectedSlide]->width || rect.r_height != slide[selectedSlide]->height)
	      {
	      slide[selectedSlide]->corner.x = rect.r_left;
	      slide[selectedSlide]->corner.y = rect.r_top;
	      slide[selectedSlide]->width = rect.r_width;
	      slide[selectedSlide]->height = rect.r_height;
	      changes = True;
	      printf("x=%d y=%d w=%d h=%d\n", rect.r_left, rect.r_top, rect.r_width, rect.r_height);
	      }
	      */
    }
}



void DisplayMultipleWindows(int slide)
{
  int		i, tot = 0;
  
  PrintDTDiagnostics("Entering DisplayMultipleWindows.\n");
  for (i=0; i<numSlidesImages; i++)
    if (dtImage[i]->slide == slide)
      {
	ShowPopup(&(globalPopup[tot]),
		  (int)(largeFactor*(dtImage[i]->corner.x-2)),
		  (int)(largeFactor*(dtImage[i]->corner.y-2)),
		  dtImage[i]->largeWidth,
		  dtImage[i]->largeHeight);
	tot++;
      }
}								    /* end function DisplayMultipleWindows */



void PrepareMultipleWindowsForDisplay(int slide)
{
  int		i, tot = 0;
  IMAGE		tempImage;
  
  for (i=0; i<numSlidesImages; i++)
    if (dtImage[i]->slide == slide)
      tot++;
  sprintf(diagString, "In PrepareMult, number of popups is %d.\n",
	  tot);
  PrintDTDiagnostics(diagString);
  tot = 0;
  for (i=0; i<numSlidesImages; i++)
    if (dtImage[i]->slide == slide)
      {
	tempImage = dtImage[i];
	globalPopup[tot] = PopupCreate(baseWindow->baseWindow,
				       tempImage->largeWidth,
				       tempImage->largeHeight);
        globalCanvas[tot] = CanvasCreate(globalPopup[tot],
					 tempImage->largeWidth,
					 tempImage->largeHeight);
	xv_set(globalCanvas[tot], XV_KEY_DATA, canvasKeyData, i, NULL);	/* Tell this canvas which image it's supposed to draw */
	XSetFunction(display, theGC, GXcopy); 
	
	if (tempImage->largeImage == (XImage*)NULL)		    /* If the largeImage doesn't exist, create it first. */
	tempImage->largeImage = Resize(tempImage->imageData,
				      tempImage->origWidth,
				      tempImage->origHeight,
				      tempImage->largeWidth,
				      tempImage->largeHeight);
	tot++;
      }
  popupTot = tot;
}								    /* end function PrepareMultipleWindowsForDisplay */


void PlaceImage(Window win, int imageNum)
{
  IMAGE	tempImage;
  
  PrintDTDiagnostics("Entering PlaceImage.\n");
  tempImage = dtImage[imageNum];
  XPutImage(display, win, theGC, tempImage->largeImage, 0, 0, 0, 0,
	    tempImage->largeWidth, tempImage->largeHeight);
}								    /* end function PlaceImage */



/*
 * Repaint callback function for `popups'.
 */
void
  PopupRepaint(Canvas		canvas,
	       Xv_window	paint_window,
	       Display*		display,
	       Window		xid,
	       Xv_xrectlist*	rects)
{
  int		i, tot;
  int		imageNum = 0;
  Canvas	whichCanvas = (Canvas)NULL;
  int		whichImage = -1;
  
  PrintDTDiagnostics("Entering PopupRepaint.\n");
  for (i=0; i < popupTot; i++)					    /* Look for the canvas to be redrawn in the list of all... */
  {								    /* ...canvases currently being shown */
    if (globalCanvas[i] == canvas)				    /* Was the correct canvas found? */
    {
      whichCanvas = canvas;					    /* Yes, set a variable saying so */
      imageNum = i;
    }
  }
  sprintf(diagString, "whichCanvas is %d.\n", whichCanvas);
  PrintDTDiagnostics(diagString);
  if (whichCanvas == (Canvas)NULL)				    /* No, the correct canvas was not found, exit immediately */
    return;
  XSetFunction(display, theGC, GXcopy);				    /* Continue preparing to re-draw this canvas */
  tot = 0;
  whichImage = xv_get(whichCanvas, XV_KEY_DATA, canvasKeyData);
  sprintf(diagString, "whichImage is %d.\n", whichImage);
  PrintDTDiagnostics(diagString);
  if (whichImage == -1)						    /* No, the correct image was not found, exit immediately */
    return;
  PlaceImage(xid, whichImage);					    /* Redraw the correct image on the correct canvas */
}								    /* end function PopupRepaint */



/*
 * Create object `Popup' in the specified instance.
 */
Frame
  PopupCreate(Frame owner, int width, int height)
{
  Frame frame = (Frame)NULL;
  
  PrintDTDiagnostics("Entering PopupCreate...\t");
  frame = xv_create(owner, FRAME_CMD,
		    XV_KEY_DATA, INSTANCE, xv_get(owner, XV_KEY_DATA, INSTANCE),
		    XV_WIDTH, width,
		    XV_HEIGHT, height,
		    XV_LABEL, "",
		    XV_SHOW, FALSE,
		    FRAME_SHOW_FOOTER, FALSE,
		    FRAME_SHOW_HEADER, FALSE,
		    FRAME_SHOW_RESIZE_CORNER, TRUE,
		    FRAME_CMD_PUSHPIN_IN, FALSE,
		    NULL);
  sprintf(diagString, "Popup id is %d.\n", frame);
  PrintDTDiagnostics(diagString);
/*  xv_set(xv_get(frame, FRAME_CMD_PANEL), WIN_SHOW, FALSE, NULL);*/
  return frame;
}


/*
 * Create object `canvas' in the specified instance.
 */
Canvas
  CanvasCreate(Frame owner, int width, int height)
{
  Canvas canvas;
  
  PrintDTDiagnostics("Entering CanvasCreate...\t");
  canvas = xv_create(owner, CANVAS,
		     XV_X, 0,
		     XV_Y, 0,
		     XV_WIDTH, width,
		     XV_HEIGHT, height,
		     CANVAS_REPAINT_PROC, PopupRepaint,
		     CANVAS_X_PAINT_WINDOW, TRUE,
		     NULL);
  sprintf(diagString, "Canvas id is %d.\n", canvas);
  PrintDTDiagnostics(diagString);
  xv_set(canvas_paint_window(canvas),     /* canvas_paint_window(canvas),  */
	 WIN_CONSUME_EVENTS,
	 WIN_ASCII_EVENTS, WIN_MOUSE_BUTTONS,
	 NULL,
	 WIN_EVENT_PROC, CanvasEventProc,
         NULL);
  return(canvas);
}								    /* end function CanvasCreate */



/*
 * Called when an event is received in the gallery canvas.
 */
void
  CanvasEventProc(Xv_Window pw, Event *event, Notify_arg arg)
{
  if (event_is_ascii(event))
    {
      switch (event_id(event))
	{
	case 'q' : case 'Q' :
	  window_done(pw);
	  break;
	  default : ;
	}
      return;
    }
  switch (event_action(event)) 
    {
    case ACTION_ADJUST :
    case MS_MIDDLE :
      window_done(pw);
      break;
      default : ;
    }
}


void KillPopups()
{
  int i;
  for (i=0; i<popupTot; i++)
  {
    xv_destroy_safe(globalPopup[i]);
    globalPopup[i] = (Frame)NULL;
/*    HidePopup(&(globalPopup[i]));*/
  }
  popupTot = 0;
  timeCount = 0;
  KillOldPopupsOnly();
}

void KillOldPopupsOnly()
{
  int i;
  
  sprintf(diagString, "In KillOldPopupsOnly, trying to erase %d popups.\n",
	  oldPopupTot);
  PrintDTDiagnostics(diagString);
  for (i=0; i < oldPopupTot; i++)
  {
/*
    sprintf(diagString,
	    "Commented the actual destroying of popup %d.\n",
	    oldPopups[i]);
    PrintDTDiagnostics(diagString);
*/
    HidePopup(&(oldPopups[i]));
    xv_destroy_safe(oldPopups[i]);
    oldPopups[i] = (Frame)NULL;
  }
  oldPopupTot = 0;
}


void EraseOldImage(int selectedSlideImage)
{
  PaintSlide(selectedSlideImage);
}								    /* end function EraseOldImage */


void DrawNewImage(int selectedSlideImage)
{
  PaintSlide(selectedSlideImage);
  changes = True;
}


void NoImageSelected()
{
  notice_prompt(baseWindow->baseWindow, NULL,
		NOTICE_MESSAGE_STRINGS,
		"No image is selected.",
		"You must first select an image from the scrollong list.",
		NULL,
		NOTICE_BUTTON_YES, "OK",
		NULL);
}
  

void ClearList(Panel_item list)
{
  xv_set(list, PANEL_LIST_DELETE_ROWS, 0,
	 (int)xv_get(list, PANEL_LIST_NROWS), NULL);
}


int NotifyChoice(Xv_opaque owner, char *string)
{
  int response = notice_prompt(owner, NULL,
			       NOTICE_MESSAGE_STRINGS,
			       string,
			       NULL,
			       NOTICE_BUTTON_YES, "Cancel",
			       NOTICE_BUTTON_NO, "Confirm",
			       NULL);
  if (response == NOTICE_YES)
    return Cancel;
  else
    return Confirm;
}


void FreeMemoryForSlide(int slideNum)
{
  int	imageID;
  SCR	theSlide;
  IMAGE	thisImage;
  
  theSlide = slide[slideNum];					    /* Point to the slide referenced by argument passed in */
  for (imageID = 0; imageID < theSlide->numberOfImages; imageID++)  /* Look at each image for this slide */
  {
    thisImage = dtImage[theSlide->imageIds[imageID]];		    /* Point to one of this slide's image structures */
    if (thisImage == NULL)					    /* Don't bother trying to release memory that isn't there */
      return;
    if (thisImage->largeImage != (XImage*)NULL)
    XDestroyImage(thisImage->largeImage);
    thisImage->largeImage = NULL;
  }
  return;
}								    /* end function FreeMemoryForSlide */


char*	MyMalloc(unsigned int size)
{
  sprintf(diagString, "+++++++++++++++In MyMalloc, about to allocate %d bytes.\n", size);
  PrintDTDiagnostics(diagString);
  return((char*)malloc(size));
}								    /* end function MyMalloc */


void	PrintDTDiagnostics(char* theString)
{
  if (printDiags != 0)
    printf("%s", theString);
  return;
}								    /* end function PrintDTDiagnostics */
