/*****************************************************************************/
/* module maplib.c							     */
/*									     */
/* Author: Rene W. Lutz							     */
/*	   Labo Image							     */
/*	   Computing Science Center					     */
/*	   University of Geneva, Switzerland				     */
/* Date:   June 1989							     */
/* Copyright (c) A. Jacot-Descombes, T. Pun, C. Pellegrini, Uni. of Geneva   */
/* (This copyright notice should appear).				     */
/*									     */
/*****************************************************************************/

/********************************************************************/
/*								    */
/*			    MAPEDIT, version 2.1		    */
/*								    */
/*	    Gestionnaire interactif de tables de couleurs	    */
/*		  version utilisant 4 descripteurs		    */
/*								    */
/* Derniere modif : aide interactif en deux langues,		    */
/*		    boutton "Help" ajoute dans init_main_map_frame  */
/*								    */
/********************************************************************/


#include <suntool/sunview.h>
#include <suntool/panel.h>
#include <suntool/canvas.h>
#include <suntool/textsw.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <stdio.h>
#include <math.h>


extern char *paneltabs[];
extern char *mastertabs[];
extern char *labeltabs[];
extern char *filetabs[];
extern char *canvastabs[];
extern Frame frame;

static short ME_icon[] = {
#include "icon/mapedit.icon"
};
DEFINE_ICON_FROM_IMAGE(mapedit_icon, ME_icon);


/*
    definition de constantes 
*/

/*  pour l'attribution d'un Pixwin   */

#define MAPPANELNAME	    "mainpanel"
#define MAPFRAMENAME	    "mainframe"
#define MAPCANVASNAME	    "mapcanvas"
#define ARROWCANVASNAME	    "arrowcanvas"

/*  constantes generales   */

#define XSUP 64
#define YSUP 0
#define YINF 95

#define BLACK	255
#define WHITE	0
#define E	2.7182818
#define CMAP_SIZE 256

#define RM 1 
#define HM 2 
#define RO 3 
#define HO 4 

#define RGB 1
#define HLS 0

#define FRENCH 1 
#define ENGLISH 0
#define MULT 1
#define ONE 0

extern int flag_help;
extern int flag_bother;

/*
    definition de nouveau curseurs
*/

static short crosscurs[] = {
#include "icon/mappanel.cursor"    
};
mpr_static(panelcursor, 16, 16, 1, crosscurs);

static short mapcurs[] = {
#include "icon/cmap.cursor"    
};
mpr_static(mapcursor, 16, 16, 1, mapcurs);

void extern hproc_MapEdit();

/*******************************************************************

		toutes les procedures de ce module

********************************************************************/


/*
    procedures d'initialisation 
*/

caddr_t init_main_map_frame() ;

static void create_one_items() ;
static void set_one_items() ;
static void destroy_one_items() ;

static void create_mult_items() ;
static void set_mult_items() ;
static void destroy_mult_items() ;

/*
    procedures associees aux objets du panneau de MAPEDIT
*/

static void load_proc() ;
static void store_proc() ;
static void freeze_proc();
static void quit_proc() ;
static void help_proc() ;
static void set_map_edit() ;
static void write_message() ;

/*
    procedures pour le canevas graphique
*/

void draw_map_on_canvas() ;

static void draw_arrow() ;
static void erase_arrows() ;
static void get_field() ;
static void arrows() ;
static void one_arrow() ;

/*
    procedure pour la table de couleurs
*/

static void init_map() ;

/*
    procedures de test sur des objets
*/

static int lin_on();
static int freeze_on();
static int is_in_dir() ;
static char *get_filename() ;

/*
    procedures liees aux panneaux et aux evenements 
*/

static void change_one_field() ;
static void change_multiple_field() ;
static void one_field_proc() ;
static void fields_proc() ;
static void of_proc() ;
static void fs_proc() ;

/*
    procedures de gestion de couleur RGB 
*/

static void RGB_fix_mult_map() ;
static void fix_colourRGB() ;

/*
    procedures de gestion de couleur HLS 
*/

static void HLS_fix_mult_map() ;
static void fix_colourHLS() ;
static void RGB_to_HLS() ;
static float RGBconv() ;

/*
    procedures de test de variables d'environnement 
*/

static int RGB_on() ;
static int MULT_on() ;

/*
    procedures du panneau d'information 
*/

static void create_info_frame() ;
static void insert_text_proc() ;
static void quit_info_proc() ;
static void ret_proc() ;
static void text_proc() ;
static void clear_text() ;


/*******************************************************************

		declaration de variables globales

********************************************************************/


	Frame	    map_main_frame,
		    info_frame ;

	Panel	    map_panel,
		    info_panel ;            

	Textsw	    info_text ;

	Canvas	    map_canvas, 
		    arrow_canvas ;

static  Panel_item  dir_item,
		    file_item,
		    iter_item,
		    freeze_item,
                    message_item,
		    edit_option,
                    one_number_item,
                    one_red_H_slider, 
                    one_green_L_slider, 
                    one_blue_S_slider, 
                    mult_number_item[2],
                    mult_red_H_slider[2],
                    mult_green_L_slider[2],
                    mult_blue_S_slider[2],

		    text_item,
		    files_button,
		    quit_button,
		    mouse_button ; 

     Pixwin	    *ppw,   
		    *mappw ,
                    *framepw, 
		    *arrowpw;

/*
    table de couleurs complete
*/

static unsigned char	red[256] ;
static unsigned char	green[256] ;
static unsigned char	blue[256] ;

/*
    borne de l'intervale (freeze = ON)
*/

static unsigned char	cred[2] ;
static unsigned char	cgreen[2] ;
static unsigned char	cblue[2] ;

/*
    champ unique (freeze = ON)
*/

static unsigned char	one_cred ;
static unsigned char	one_cgreen ;
static unsigned char	one_cblue ;

extern int Confirm();

static int o_num, m_num[2] ;
static int mode, info_lang;

Menu main_map_menu;
int flag_mapedit_win_open = FALSE;
int flag_in_mapedit = FALSE;

/****************************************************************/
/*							    	*/
/* nom	    : main						*/
/*								*/								
/* fonction : lance MAPEDIT					*/ 
/*								*/								
/* entree   : ---						*/								
/*							    	*/								
/* globales : map_main_frame.					*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : init_main_loop.					*/								
/*								*/								
/****************************************************************/
/*
main()
{
    init_main_map_frame() ;
    window_main_loop(map_main_frame) ;
    exit(0) ;
}
*/


/*******************************************************************

		    procedures d'initialisation

********************************************************************/


/****************************************************************/
/*							    	*/
/* nom	    : init_main_frame					*/
/*								*/								
/* fonction : initialise les paneaux et canvas		    	*/ 
/*								*/								
/* entree   : ---						*/								
/*							    	*/								
/* globales : map_main_frame, map_panel, dir_item, file_item	*/								
/*	      iter_item,message_item, edit_option		*/								
/*	      arrow_canvas, map_canvas				*/
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : init_map, draw_map_on_canvas, get_field		*/								
/*	      load_proc, store_proc, freeze_proc, set_map_edit,	*/								
/*	      help_proc						*/								
/*								*/								
/* modif    : help_button rattache a help_proc			*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

caddr_t init_main_map_frame(m,mi)
Menu	m ;
Menu_item mi ;
{
    static void proc_quit_mapedit();

    char    current_dir[30] ;

    if (flag_bother){
	 hproc_MapEdit();
	 return;
    }
    if (flag_help) hproc_MapEdit();
    flag_mapedit_win_open = TRUE;
    if (flag_in_mapedit) return ;
    flag_in_mapedit = TRUE;

    map_main_frame = window_create(frame, FRAME,
				    FRAME_LABEL,labeltabs[5],
				    FRAME_EMBOLDEN_LABEL, TRUE, 
				    FRAME_ICON, &mapedit_icon,
				    FRAME_NO_CONFIRM, TRUE,
				    FRAME_SHOW_LABEL, TRUE,
				    WIN_SHOW, TRUE,
				    WIN_X, 0 ,
				    WIN_Y, 435 , 
				    0);
    main_map_menu = (Menu) window_get(map_main_frame, WIN_MENU, 0);
    menu_set(main_map_menu, MENU_REMOVE, 1,
		MENU_ACTION_ITEM, "quit*", proc_quit_mapedit,
		0);
    window_set(map_main_frame, WIN_MENU, main_map_menu,0);
    map_panel = window_create(map_main_frame, PANEL, 
				   WIN_ROWS, 15 ,
				   WIN_CURSOR, cursor_create(CURSOR_IMAGE, &panelcursor,
				   CURSOR_XHOT,8,
				   CURSOR_YHOT,8,
				   0),
				   0);

    
    dir_item = panel_create_item(map_panel, PANEL_TEXT,
                                    PANEL_LABEL_X, ATTR_COL(0),
				    PANEL_LABEL_Y, ATTR_ROW(0),
                                    PANEL_LABEL_STRING,paneltabs[301], 
                                    PANEL_VALUE_DISPLAY_LENGTH, 35,
				    PANEL_VALUE, getwd(current_dir),
				    PANEL_LABEL_BOLD, TRUE,
				    0);
    
    file_item = panel_create_item(map_panel, PANEL_TEXT,
                                    PANEL_LABEL_X, ATTR_COL(0),
				    PANEL_LABEL_Y, ATTR_ROW(1),
                                    PANEL_LABEL_STRING,paneltabs[302], 
                                    PANEL_VALUE_DISPLAY_LENGTH, 35,
				    PANEL_LABEL_BOLD, TRUE,
				    0);

    panel_create_item(map_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, 560,
				    PANEL_LABEL_Y, ATTR_ROW(0) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(map_panel,paneltabs[303],
				    8,0),
				    PANEL_NOTIFY_PROC, load_proc,
				    0);

    panel_create_item(map_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, 560,
				    PANEL_LABEL_Y, ATTR_ROW(1), 
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(map_panel,paneltabs[304],
				    8,0),
				    PANEL_NOTIFY_PROC, store_proc,
				    0);

     panel_create_item(map_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, 560,
				    PANEL_LABEL_Y, ATTR_ROW(2) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE, 
				    panel_button_image(map_panel,paneltabs[312],8,0),
				    PANEL_NOTIFY_PROC, help_proc,
				    0);

    panel_create_item(map_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, 560,
				    PANEL_LABEL_Y, ATTR_ROW(3) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,


				    panel_button_image(map_panel,paneltabs[313],8,0),
				    PANEL_NOTIFY_PROC, proc_quit_mapedit,
				    0);

    iter_item = panel_create_item(map_panel, PANEL_CYCLE, 
				    PANEL_LABEL_X, 390,
				    PANEL_LABEL_Y, ATTR_ROW(0) ,
                                    PANEL_LABEL_BOLD, TRUE,
                                    PANEL_LABEL_STRING, paneltabs[305],
				    PANEL_CHOICE_STRINGS, "LIN","LOG",0,
				    0);

    freeze_item = panel_create_item(map_panel, PANEL_CYCLE, 
				    PANEL_LABEL_X, 390,
				    PANEL_LABEL_Y, ATTR_ROW(1) ,
                                    PANEL_LABEL_BOLD, TRUE,
                                    PANEL_LABEL_STRING, paneltabs[306],
				    PANEL_CHOICE_STRINGS, "OFF","ON",0,
				    PANEL_NOTIFY_PROC, freeze_proc,
				    0);

    message_item = panel_create_item(map_panel, PANEL_TEXT,
				    PANEL_LABEL_X, ATTR_COL(0),
				    PANEL_LABEL_Y, ATTR_ROW(4) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_STRING,"Messages : ",
                                    PANEL_VALUE_DISPLAY_LENGTH, 60,
                                    0);

    edit_option = panel_create_item(map_panel, PANEL_CHOICE,
				    PANEL_ITEM_X, 60,
  				    PANEL_ITEM_Y, ATTR_ROW(2),       
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_STRING, "Map editing :",
				    PANEL_CHOICE_STRINGS,"RGB_multi","HLS_multi","RGB_one","HLS_one", 0,
				    PANEL_CLIENT_DATA, "ABCD",
				    PANEL_NOTIFY_PROC, set_map_edit,
				    0);
    create_mult_items() ;

    window_fit_height(map_panel) ;
    window_fit(map_main_frame) ;

    arrow_canvas = window_create(map_main_frame, CANVAS,
				    WIN_WIDTH, 640,
                                    WIN_HEIGHT, 35,
                                    WIN_CURSOR,
				    cursor_create(CURSOR_IMAGE,
				    &mapcursor,
				    CURSOR_XHOT,7,
				    CURSOR_YHOT,1,
				    0),
                                    WIN_EVENT_PROC, get_field,
				    0);
    window_fit(map_main_frame) ;

    map_canvas = window_create(map_main_frame, CANVAS,
                                    WIN_BELOW, arrow_canvas,
                                    WIN_X, 0,
                                    WIN_WIDTH, 640,
                                    WIN_HEIGHT,100,
				    CANVAS_RETAINED, TRUE,
				    CANVAS_AUTO_EXPAND, TRUE,
				    CANVAS_AUTO_SHRINK, TRUE,
                                    WIN_EVENT_PROC, get_field,
				    0);

    init_map() ;
    draw_map_on_canvas() ;
    set_mult_items(RGB,0,0,0,0,255,0,0,0) ;
    window_fit(map_main_frame) ;
    window_set(map_main_frame, WIN_SHOW, TRUE, 0) ;
    
}

static void
proc_quit_mapedit()
{
  if (Confirm("quit MapEdit?") == 1){
     window_destroy(map_main_frame);
     flag_mapedit_win_open = FALSE;
     flag_in_mapedit = FALSE;
  }
}


/****************************************************************/
/*								*/
/* nom	    : create_one_items					*/   
/*								*/								
/* fonction : creation des items pour l'edition champ unique	*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : map_panel, one_number_item, one_red_H_slider,	*/
/*	      one_green_L_slider, one_blue_S_slider		*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : of_proc, change_one_field				*/
/*								*/
/****************************************************************/

static void
create_one_items()
{
	one_number_item = panel_create_item(map_panel, PANEL_TEXT,
		PANEL_ITEM_Y, ATTR_ROW(7),
                PANEL_LABEL_BOLD, TRUE ,
		PANEL_LABEL_STRING, " Field Number : ",
                PANEL_VALUE_DISPLAY_LENGTH, 3,
		PANEL_NOTIFY_PROC, of_proc,
		0) ;

	one_red_H_slider  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(9),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
                PANEL_NOTIFY_LEVEL, PANEL_ALL,
		PANEL_NOTIFY_PROC, change_one_field,
		0) ;

	one_green_L_slider  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(10),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
                PANEL_NOTIFY_LEVEL, PANEL_ALL,
		PANEL_NOTIFY_PROC, change_one_field,
		0) ;

	one_blue_S_slider  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(11),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
                PANEL_NOTIFY_LEVEL, PANEL_ALL,
		PANEL_NOTIFY_PROC, change_one_field,
		0) ;

}


/****************************************************************/
/*								*/
/* nom	    : set_one_items					*/   
/*								*/								
/* fonction : initialise les items pour l'edition champ	unique	*/ 
/*								*/								
/* entree   : int type, f, rh, gl, bs				*/								
/*								*/								
/* globales : map_panel, one_number_item, one_red_H_slider,	*/
/*	      one_green_L_slider, one_blue_S_slider		*/								
/*	      mode						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
set_one_items(type, f, rh, gl, bs)
int type, f, rh, gl, bs ;
{
    char *buf ;
    Panel_item item ;

    buf = (char *) calloc (3, sizeof *buf) ;
    sprintf(buf,"%3d",f) ;

    o_num = f ;
        
    switch (type)
    {

	case RGB : panel_set(one_number_item, PANEL_VALUE, f,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(one_red_H_slider, 
			    PANEL_LABEL_STRING, "R :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, rh,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(one_green_L_slider, 
			    PANEL_LABEL_STRING, "G :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, gl,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(one_blue_S_slider, 
			    PANEL_LABEL_STRING, "B :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, bs,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

                   mode = RO ;
		   break ;

	case HLS : panel_set(one_number_item, PANEL_VALUE, f,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(one_red_H_slider, 
			    PANEL_LABEL_STRING, "H :",
			    PANEL_MAX_VALUE,  360,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, rh,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(one_green_L_slider, 
			    PANEL_LABEL_STRING, "L :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, gl,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(one_blue_S_slider, 
			    PANEL_LABEL_STRING, "S :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, bs,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

                   mode = HO ;
		   break ;
    }

    panel_each_item(map_panel, item)
	panel_set(item, PANEL_SHOW_ITEM, TRUE, 0) ;
    panel_end_each ;
}


/****************************************************************/
/*								*/
/* nom	    : destroy_one_items					*/   
/*								*/								
/* fonction : destruction des items pour l'edition champ unique	*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : one_number_item, one_red_H_slider,		*/
/*	      one_green_L_slider, one_blue_S_slider		*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
destroy_one_items()
{
    panel_set(one_number_item, PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(one_red_H_slider, PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(one_green_L_slider, PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(one_blue_S_slider, PANEL_SHOW_ITEM, FALSE,0) ;
    panel_destroy(one_number_item) ;
    panel_destroy(one_red_H_slider) ;
    panel_destroy(one_green_L_slider) ;
    panel_destroy(one_blue_S_slider) ;
}


/****************************************************************/
/*								*/
/* nom	    : create_mult_items					*/   
/*								*/								
/* fonction : creation des items pour l'edition champ multiples	*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : map_panel,mult_number_item[],mult_red_H_slider[],	*/
/*	      mult_green_L_slider[], mult_blue_S_slider[]	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : fs_proc, change_multiple_field			*/
/*								*/
/****************************************************************/

static void
create_mult_items()
{
        
	mult_number_item[0] = panel_create_item(map_panel, PANEL_TEXT,
		PANEL_ITEM_Y, ATTR_ROW(6),
                PANEL_LABEL_BOLD, TRUE ,
		PANEL_LABEL_STRING, " Field Number : ",
                PANEL_VALUE_DISPLAY_LENGTH, 3,
		PANEL_NOTIFY_PROC, fs_proc,
		0) ;

	mult_red_H_slider[0]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(7),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	mult_green_L_slider[0]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(8),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	mult_blue_S_slider[0]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(9),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	mult_number_item[1] = panel_create_item(map_panel, PANEL_TEXT,
		PANEL_ITEM_Y, ATTR_ROW(10),
                PANEL_LABEL_BOLD, TRUE ,
		PANEL_LABEL_STRING, " Field Number : ",
                PANEL_VALUE_DISPLAY_LENGTH, 3,
		PANEL_NOTIFY_PROC, fs_proc,
		0) ;

	mult_red_H_slider[1]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(11),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	mult_green_L_slider[1]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(12),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	mult_blue_S_slider[1]  = panel_create_item(map_panel, PANEL_SLIDER,
		PANEL_ITEM_Y, ATTR_ROW(13),
                PANEL_LABEL_BOLD, TRUE ,
                PANEL_MIN_VALUE,  0,
		PANEL_NOTIFY_PROC, change_multiple_field,
		0) ;

	window_fit_height(map_panel);

}


/****************************************************************/
/*								*/
/* nom	    : set_mult_items					*/   
/*								*/								
/* fonction : initialise les items pour l'edition champ	multi.	*/ 
/*								*/								
/* entree   : int type, f0, f1, rh0, rh1, gl0, gl1, bs0, bs1    */								
/*								*/								
/* globales : map_panel,mult_number_item[],mult_red_H_slider[],	*/
/*	      mult_green_L_slider[], mult_blue_S_slider[]	*/								
/*	      mode						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
set_mult_items(type, f0, rh0, gl0, bs0, f1, rh1, gl1, bs1)
int type, f0, f1, rh0, rh1, gl0, gl1, bs0, bs1 ;
{
    char *buf ;
    Panel_item item ;

    m_num[0] = f0 ;
    m_num[1] = f1 ;
   
    buf = (char *) calloc (3, sizeof *buf) ;
        
    switch (type)
    {
	case RGB : sprintf(buf,"%3d",f0) ;

		   panel_set(mult_number_item[0], PANEL_VALUE, buf,
			     PANEL_ITEM_X, ATTR_COL(15),
			     0) ;

		   panel_set(mult_red_H_slider[0], 
			    PANEL_LABEL_STRING, "R :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, rh0,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(mult_green_L_slider[0], 
			    PANEL_LABEL_STRING, "G :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, gl0,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(mult_blue_S_slider[0], 
			    PANEL_LABEL_STRING, "B :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, bs0,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   sprintf(buf,"%3d",f1) ;

		   panel_set(mult_number_item[1], PANEL_VALUE, buf,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(mult_red_H_slider[1], 
			    PANEL_LABEL_STRING, "R :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, rh1,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(mult_green_L_slider[1], 
			    PANEL_LABEL_STRING, "G :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, gl1,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

		   panel_set(mult_blue_S_slider[1], 
			    PANEL_LABEL_STRING, "B :",
			    PANEL_MAX_VALUE,  255,
			    PANEL_SLIDER_WIDTH, 256,
			    PANEL_VALUE, bs1,
			    PANEL_ITEM_X, ATTR_COL(15),
			    0) ;

                   mode = RM ;
		   break ; 

	case HLS : sprintf(buf,"%3d",f0) ;

		   panel_set(mult_number_item[0], PANEL_VALUE, buf,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_red_H_slider[0], 
			    PANEL_LABEL_STRING, "H :",
			    PANEL_MAX_VALUE,  359,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, rh0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_green_L_slider[0], 
			    PANEL_LABEL_STRING, "L :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, gl0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_blue_S_slider[0], 
			    PANEL_LABEL_STRING, "S :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, bs0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   sprintf(buf,"%3d",f1) ;

		   panel_set(mult_number_item[1], PANEL_VALUE, buf,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_red_H_slider[1], 
			    PANEL_LABEL_STRING, "H :",
			    PANEL_MAX_VALUE,  359,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, rh0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_green_L_slider[1], 
			    PANEL_LABEL_STRING, "L :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, gl0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

		   panel_set(mult_blue_S_slider[1], 
			    PANEL_LABEL_STRING, "S :",
			    PANEL_MAX_VALUE,  100,
			    PANEL_SLIDER_WIDTH, 360,
			    PANEL_VALUE, bs0,
			    PANEL_ITEM_X, ATTR_COL(7),
			    0) ;

                   mode = HM ;
		   break ; 
    }

    panel_each_item(map_panel, item)
	panel_set(item, PANEL_SHOW_ITEM, TRUE, 0) ;
    panel_end_each ;
}


/****************************************************************/
/*								*/
/* nom	    : destroy_mult_items				*/   
/*								*/								
/* fonction : destruction des items pour l'edition champ multi.	*/ 
/*								*/								
/* entree   : ---					        */								
/*								*/								
/* globales : map_panel,mult_number_item[],mult_red_H_slider[],	*/
/*	      mult_green_L_slider[], mult_blue_S_slider[]	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
destroy_mult_items()
{
    panel_set(mult_number_item[0], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_number_item[1], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_red_H_slider[0], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_red_H_slider[1], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_green_L_slider[0], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_green_L_slider[1], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_blue_S_slider[0], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_set(mult_blue_S_slider[1], PANEL_SHOW_ITEM, FALSE,0) ;
    panel_destroy(mult_number_item[0]) ;
    panel_destroy(mult_red_H_slider[0]) ;
    panel_destroy(mult_green_L_slider[0]) ;
    panel_destroy(mult_blue_S_slider[0]) ;
    panel_destroy(mult_number_item[1]) ;
    panel_destroy(mult_red_H_slider[1]) ;
    panel_destroy(mult_green_L_slider[1]) ;
    panel_destroy(mult_blue_S_slider[1]) ;
}


/*******************************************************************

    procedures associees aux objets du panneau de MAPEDIT

********************************************************************/


/****************************************************************/
/*								*/
/* nom	    : load_proc						*/   
/*								*/								
/* fonction : charge une table de couleur, lue depuis un fichier*/ 
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : red[], green[], blue[], mappw, ppw, arrowpw	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : get_filename, write_message			*/								
/*								*/								
/****************************************************************/

static void
load_proc()
{
    FILE *fp ;
    char *name,*dir,*file,*buf ;

    buf = (char *) calloc (100,sizeof(*buf)) ;

    dir = (char *) panel_get_value(dir_item) ;
    file = (char *) panel_get_value(file_item) ;    
    name = get_filename() ;

    if (is_in_dir(file,dir))
    {
	if ((fp = fopen(name,"r")) == NULL )
	    write_message(" Unable to open file ") ;
	else
	{
	    fread(red,sizeof (unsigned char),256,fp) ;
	    fread(green,sizeof(unsigned char),256,fp) ;
	    fread(blue,sizeof(unsigned char),256,fp) ;

	    pw_putcolormap(mappw, 0, CMAP_SIZE, red, green, blue);

	    red[BLACK] = 0 ;
	    green[BLACK] = 0 ;
	    blue[BLACK] = 0 ;
    
	    red[1] = 0 ;
	    green[1] = 0 ;
	    blue[1] = 0 ;

	    red[WHITE] = 255 ;
	    green[WHITE] = 255 ;
	    blue[WHITE] = 255 ;

	    red[254] = 255 ;
	    green[254] = 255 ;
	    blue[254] = 255 ;

	    pw_putcolormap(ppw, 0, CMAP_SIZE, red, green, blue);
	    pw_putcolormap(arrowpw, 0, CMAP_SIZE, red, green, blue);
	    pw_putcolormap(framepw, 0, CMAP_SIZE, red, green, blue);

	    sprintf(buf," \"%s\" is now loaded ",name);
	    write_message(buf) ;

	    fclose(fp) ; 
	}
    }
    else
    {	
	sprintf(buf,"Can't find \"%s\" in directory \"%s\" ",file, dir);
	write_message(buf) ;
    }
}


/****************************************************************/
/*								*/
/* nom	    : store_proc					*/   
/*								*/								
/* fonction : ecrit la table de couleur dans un fichier		*/ 
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : red[], green[], blue[], mappw			*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : write_message					*/								
/*								*/								
/****************************************************************/

static void
store_proc()
{
    FILE *fp ;
    char *name,*buf ;

    buf = (char *) calloc (100,sizeof(*buf)) ;

    name = get_filename() ;

    if ((fp = fopen(name,"w")) == NULL )
	write_message(" Unable to write file ") ;
    else
    {
	pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;

	fwrite(red,sizeof (unsigned char),256,fp) ;
	fwrite(green,sizeof(unsigned char),256,fp) ;
	fwrite(blue,sizeof(unsigned char),256,fp) ;

        sprintf(buf," \"%s\" is now stored ",name) ;
        write_message(buf) ;
    }
    fclose(fp);
}


/****************************************************************/
/*								*/
/* nom	    : freeze_proc					*/   
/*								*/
/* fonction : gele les couleurs sur chaque fleche si le		*/ 
/*	      changement de mode est OFF -> ON			*/					
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : freeze_item,cred[],cgreen[],cblue[]		*/								
/*	      mult_number_item[], one_number_item		*/
/*	      red[], green[], blue[],				*/								
/*	      one_cred, one_cgreen, one_cblue			*/
/*							    	*/								
/* return   : TRUE si "ON" FALSE si "OFF"			*/
/*								*/								
/* routines : write_message					*/
/*								*/
/****************************************************************/

static void
freeze_proc()
{
    int	index,
	n0,n1;

    index = (int) panel_get_value(freeze_item) ;

    if (index == 1)
    {
	pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;

	if (MULT_on(mode))
	{
	    n0 = atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE)) ;	
	    n1= atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE)) ;
 
	    if ((n0 >= 0) && (n0 <= 255) && (n1 >= 0) && (n1 <= 255))
	    {
		cred[0] = red[n0] ;
		cgreen[0] = green[n0] ;
		cblue[0] = blue[n0] ;
		cred[1] = red[n1] ;
		cgreen[1] = green[n1] ;
		cblue[1] = blue[n1] ;
	    }
	    else
		write_message("Field Number out of range ...") ;
	}
	else
	{
	    n0 = atoi((char *) panel_get(one_number_item, PANEL_VALUE)) ; 
	    if ((n0 >= 0) && (n0 <= 255))
	    {
		pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;
		one_cred = red[n0] ;
		one_cgreen = green[n0] ;
		one_cblue = blue[n0] ;
	    }   
	    else
		write_message("Field Number out of range ...") ;
	}
    }
}

/****************************************************************/
/*								*/
/* nom	    : quit_proc						*/   
/*								*/
/* fonction : termine la session avec MapEdit			*/ 
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : map_main_frame					*/								
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
quit_proc()
{
    window_set(map_main_frame, FRAME_NO_CONFIRM, TRUE, 0) ;
    window_destroy(map_main_frame) ;
}


/****************************************************************/
/*								*/
/* nom	    : help_proc						*/   
/*								*/
/* fonction : ouvre une fenetre d'aide				*/ 
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : info_frame, info_lang				*/								
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static void
help_proc()
{

    info_lang = ENGLISH ;

    create_info_frame() ;
    window_set(info_frame , WIN_SHOW, TRUE, 0) ;
   
}


/****************************************************************/
/*								*/
/* nom	    : set_map_edit					*/   
/*								*/								
/* fonction : fixe le mode d'edition selon le choix "clique"    */ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : edit_option, mode, mappw,				*/								
/*	      cred[],cgreen[],cblue[]				*/								
/*	      mult_number_item[], one_number_item		*/								
/*	      red[], green[], blue[],				*/								
/*	      one_cred, one_cgreen, one_cblue			*/		
/*								*/								
/* return   : ---						*/								
/*								*/								
/* routines : destroy..., create..., set_one_items,		*/								
/*	      destroy..., create..., set_mult_items,		*/								
/*	      one_arrow, arrows					*/								
/*								*/								
/****************************************************************/

static void
set_map_edit()
{

    char	*client_data ;
    char	flag ;
    int		index,
		h0,h1,l0,l1,s0,s1 ;

    client_data = panel_get(edit_option, PANEL_CLIENT_DATA, 0) ;
    index = (int) panel_get_value(edit_option);
    flag = client_data[index] ;

    switch(flag) 
    {
	case 'A' :  if (mode == RM) return ;
		
		 if (!MULT_on(mode))
		 {
		    destroy_one_items() ;
		    create_mult_items() ;
		 }

                 if (freeze_on())
		     set_mult_items(RGB, m_num[0], cred[0], cgreen[0], cblue[0],		  
				    m_num[1], cred[1], cgreen[1], cblue[1]) ;		  
		 else
		 {
		    pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;
		    set_mult_items(RGB, m_num[0], red[m_num[0]], green[m_num[0]], blue[m_num[0]],		  
				   m_num[1], red[m_num[1]], green[m_num[1]], blue[m_num[1]]) ;
		 }

		 arrows() ;
		 break ;

	case 'B' :  if (mode == HM) return ;
		
		 if (!MULT_on(mode))
		 {
		    destroy_one_items() ;
		    create_mult_items() ;
		 }

                 if (freeze_on())
		 {
		    RGB_to_HLS(cred[0], cgreen[0], cblue[0],&h0,&l0,&s0);
		    RGB_to_HLS(cred[1], cgreen[1], cblue[1],&h1,&l1,&s1);
		 }
		 else
		 {
		    pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;
		    RGB_to_HLS(red[m_num[0]], green[m_num[0]], blue[m_num[0]],&h0,&l0,&s0) ;
		    RGB_to_HLS(red[m_num[1]], green[m_num[1]], blue[m_num[1]],&h1,&l1,&s1) ;		     
		 }

		 set_mult_items(HLS,m_num[0], h0, l0, s0, m_num[1], h1, l1, s1);

		 arrows() ;
		 break ;

	case 'C' :  if (mode == RO) return ;
		
		 if (MULT_on(mode))
		 {
		    destroy_mult_items() ;
		    create_one_items() ;
		 }

		 if (freeze_on())
		    set_one_items(RGB, o_num, one_cred, one_cgreen, one_cblue) ;		     
		 else
		    set_one_items(RGB,o_num , red[o_num], green[o_num], blue[o_num]);
	
		 one_arrow();
		 break ;

	case 'D' :  if (mode == HO) return ;
		
		 if (MULT_on(mode))
		 {
		    destroy_mult_items() ;
		    create_one_items() ;
		 }

		 if (freeze_on())
		    RGB_to_HLS(one_cred, one_cgreen, one_cblue,&h0,&l0,&s0) ;
		 else
		    RGB_to_HLS(red[o_num], green[o_num], blue[o_num], &h0,&l0,&s0);		  

		 set_one_items(HLS,o_num , h0, l0, s0);	
	  
		 one_arrow();
		 break ;
       
    }

}


/****************************************************************/
/*								*/
/* nom	    : write_message					*/   
/*								*/								
/* fonction : ecrit le message "str" dans le paneau principal   */ 
/*								*/								
/* entree   : char *str						*/								
/*								*/								
/* globales : message_item, map_panel				*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static void
write_message(str) 
char *str ;
{
    panel_set(message_item, PANEL_VALUE, str, 0);
    window_bell(map_panel) ;
}


/*******************************************************************

		procedures pour le canevas graphique

********************************************************************/


/****************************************************************/
/*								*/
/* nom	    : draw_map_on_canvas	    			*/   
/*								*/								
/* fonction : dessine la table de couleur dans le canevas	*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : mappw						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

void
draw_map_on_canvas() 
{
    int i,x = XSUP ;

    
    pw_rop(mappw,0,0,640,100,PIX_SRC | PIX_COLOR(WHITE), NULL, 0, 0) ;

    pw_batch_on(mappw) ;
    for (i = 0 ; i < CMAP_SIZE ; i++)
    {
	pw_vector(mappw,x,YSUP,x,YINF, PIX_SRC | PIX_COLOR(i), 1) ;
	pw_vector(mappw,x+1,YSUP,x+1,YINF, PIX_SRC | PIX_COLOR(i), 1) ;
        x=x+2 ; 
    }
    pw_text(mappw, 38, 95, PIX_SRC | PIX_COLOR(0), 0, "[0]");
    pw_text(mappw, 580, 95, PIX_SRC | PIX_COLOR(255), 0, "[255]");
    pw_batch_off(mappw) ;
    arrows() ;
}


/****************************************************************/
/*								*/
/* nom	    : draw_arrow					*/   
/*								*/								
/* fonction : dessine une fleche de pointant vers le bas	*/ 
/*								*/								
/* entree   : int x , colour					*/								
/*								*/								
/* globales : arrowpw						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static void
draw_arrow(x) 
int	x ;
{
    pw_vector(arrowpw,x,5,x,35, PIX_SRC | PIX_COLOR(BLACK), 1) ;
    pw_vector(arrowpw,x+1,5,x+1,35, PIX_SRC | PIX_COLOR(BLACK), 1) ;
    pw_vector(arrowpw,x,35,x-5,30, PIX_SRC | PIX_COLOR(BLACK), 1) ;
    pw_vector(arrowpw,x+1,35,x-4,30, PIX_SRC | PIX_COLOR(BLACK), 1) ;
    pw_vector(arrowpw,x,35,x+5,30, PIX_SRC | PIX_COLOR(BLACK), 1) ;
    pw_vector(arrowpw,x+1,35,x+6,30, PIX_SRC | PIX_COLOR(BLACK), 1) ;
}


/****************************************************************/
/*								*/
/* nom	    : erase_arrows					*/   
/*								*/								
/* fonction : efface les fleches dans le Pixwin arrowpw		*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : arrowpw						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static void
erase_arrows()
{
    pw_rop(arrowpw,0,0, 640, 36,PIX_SRC | PIX_COLOR(WHITE), NULL, 0, 0) ;
}


/****************************************************************/
/*								*/
/* nom	    : get_field						*/   
/*								*/								
/* fonction : gerre la souris dans le canevas contenant les     */ 
/*	      fleches qui pointent les zones de table de cou-	*/								
/*	      leur.						*/								
/*								*/								
/* entree   : Canvas canvas Event *event			*/								
/*								*/								
/* globales : one_number_item, mult_number_item[],		*/								
/*	      cred[], cblue[], cgreen[]	, mode			*/								
/*	      one_cred, one_cblue, one_cgreen			*/								
/*								*/								
/* return   : ---						*/								
/*								*/								
/* routines : arrows, one_arrow, fs_proc, of_proc		*/								
/*	      fields_proc, one_field_proc,			*/								
/*	      freeze_on, erase_arrows, MULT_on, RGB_on		*/								
/*								*/								
/****************************************************************/

static void
get_field(canvas,event) 
Canvas canvas ;
Event	*event ;
{
    int	    xpos,
	    x,
	    xleft, xright;

    char    *buf;

    buf = (char *) calloc (3, sizeof *buf) ;

    if (event_is_up(event)) return ;

    xpos = event_x(event) ;

    if (event_id(event) == MS_MIDDLE)
    {
	switch (mode)
	{	    
	    case RM : fs_proc() ;
		      break ;

	    case HM : fs_proc() ;
		      break ;

	    case RO : of_proc() ;
		      break ;

	    case HO : of_proc() ;
		      break ;
	}
    }

    else
    {
	if (MULT_on(mode))
	{
	    x = (int) rint((double)(xpos - XSUP)/2)  ;

	    xleft = atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE));
	    xright = atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE));

	    if ((xleft >= 0) && (xleft <= 255) && (xright >= 0) && (xright <= 512))
	    {              
		if (event_id(event) == MS_RIGHT)
		{
			 
		    if (x>= xleft)
		    {
			if (x > 255)
			    sprintf(buf,"255") ;
			else
			{
			    sprintf(buf,"%d",x) ;
			}

			panel_set(mult_number_item[1], PANEL_VALUE, buf, 0);
		    }
		    else
		    {
			if (x < 0)
			    sprintf(buf, "0") ;
			else
			    sprintf(buf,"%d",x) ;

			panel_set(mult_number_item[0], PANEL_VALUE, buf, 0) ;
			sprintf(buf,"%d",xleft) ;
			panel_set(mult_number_item[1], PANEL_VALUE, buf, 0) ;
		    }

		    if (freeze_on())
			arrows() ;
		    else
			fields_proc() ;
		}
		else 
		if (event_id(event) == MS_LEFT)
		{
		    if (x<=xright)
		    {
			if (x < 0)
			    sprintf(buf, "0") ;
			else
			    sprintf(buf,"%d",x) ;

			panel_set(mult_number_item[0], PANEL_VALUE, buf, 0);
		    }
		    else
		    {
			if (x > 255)
			    sprintf(buf,"255") ;
			else
			    sprintf(buf,"%d",x) ;

			panel_set(mult_number_item[1], PANEL_VALUE, buf, 0) ;
			sprintf(buf,"%d",xright) ;
			panel_set(mult_number_item[0], PANEL_VALUE, buf, 0) ;
		    }

		    if (freeze_on())
			arrows() ;
		    else
		    {
			if (RGB_on(mode))
			    fields_proc() ;
			else
			    fields_proc() ;
		    }
		}
	    }
	    else
		write_message("Field Number out of range ...") ;
	}
	else  
	{	   
            x = (int) rint((double)(xpos - XSUP)/2) ;

	    if (event_id(event) == MS_RIGHT || event_id(event) == MS_LEFT)
	    {
		if (x < 0)
		    sprintf(buf, "0") ;
		else if (x > 255)
                    sprintf(buf, "255") ;
		else
		    sprintf(buf,"%3d",x) ; 

		panel_set(one_number_item, PANEL_VALUE, buf, 0) ;
		if (freeze_on())
		    one_arrow() ;
		else
		    one_field_proc() ;
	    }
	}
    }    
}


/****************************************************************/
/*								*/
/* nom	    : arrows						*/   
/*								*/								
/* fonction : dessine les fleche delimitant la zone de couleurs	*/ 
/*							    	*/								
/* entree   : ---						*/								
/*								*/								
/* globales : mult_number_item[]				*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : erase_arrows, draw_arrows				*/								
/*								*/								
/****************************************************************/

static void
arrows() 
{
    int	    xleft,xright;
    
    xright = 2 * (atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE))) + XSUP;
    xleft = 2 * (atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE))) + XSUP;

    erase_arrows() ;
    draw_arrow(xright);  
    draw_arrow(xleft) ;
}


/****************************************************************/
/*								*/
/* nom	    : one_arrow						*/   
/*								*/								
/* fonction : dessine la fleche pointant sur un champ		*/ 
/*							    	*/								
/* entree   : ---						*/								
/*								*/								
/* globales : one_number_item					*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : erase_arrows, draw_arrow, write_message		*/								
/*								*/								
/****************************************************************/

static void 
one_arrow() 
{
    int	    xpos,
	    num ;

    num = atoi((char *) panel_get(one_number_item, PANEL_VALUE));
    if ((num >= 0) && (num <= 255))
    {    
	xpos = 2 * num + XSUP ; 
	erase_arrows() ;
	draw_arrow(xpos);  	
    }
    else
	write_message("Field Number out of range ...") ;
}


/*******************************************************************

		procedure pour la table de couleurs

********************************************************************/


/****************************************************************/
/*								*/
/* nom	    : init_map						*/   
/*								*/								
/* fonction : initialise la table de couleur et les Pixwin	*/ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : map_panel,arrow_canvas, map_canvas,		*/								
/*	      red[],green[],blue[],ppw,arrowpw,mappw		*/								
/*								*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static void
init_map()
{
    ppw = (Pixwin *) window_get(map_panel, WIN_PIXWIN);
    pw_setcmsname(ppw, MAPPANELNAME) ;

    framepw = (Pixwin *) window_get(map_main_frame, WIN_PIXWIN);
    pw_setcmsname(framepw, MAPFRAMENAME) ;

    mappw = (Pixwin *) canvas_pixwin(map_canvas) ;
    pw_setcmsname(mappw, MAPCANVASNAME) ;

    arrowpw = (Pixwin *) canvas_pixwin(arrow_canvas) ;
    pw_setcmsname(arrowpw, ARROWCANVASNAME) ;

    red[BLACK] = 0 ;
    green[BLACK] = 0 ;
    blue[BLACK] = 0 ;
    
    red[WHITE] = 255 ;
    green[WHITE] = 255 ;
    blue[WHITE] = 255 ;

    red[254] = 255 ;
    green[254] = 255 ;
    blue[254] = 255 ;

    pw_putcolormap(ppw, 0, CMAP_SIZE, red, green, blue);
    pw_putcolormap(arrowpw, 0, CMAP_SIZE, red, green, blue);
    pw_putcolormap(framepw, 0, CMAP_SIZE, red, green, blue);
    pw_putcolormap(mappw, 0, CMAP_SIZE, red, green, blue);
   
}


/*******************************************************************

		procedures de tests sur des objets

********************************************************************/


/****************************************************************/
/*								*/
/* nom	    : lin_on						*/   
/*								*/								
/* fonction : defini le mode d'interpolation des couleurs       */ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : iter_item						*/								
/*							    	*/								
/* return   : TRUE si "lin" FALSE si "log"			*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static int
lin_on()
{
    int	index ;

    index = (int) panel_get_value(iter_item) ;

    if (index == 0)
	return(TRUE) ;
    else 
	return(FALSE);
}


/****************************************************************/
/*								*/
/* nom	    : freeze_on						*/   
/*								*/								
/* fonction : defini les couleurs sont gelees sur les fleches   */ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : freeze_item					*/								
/*							    	*/								
/* return   : TRUE si "ON" FALSE si "OFF"			*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static int
freeze_on()
{
    int	index ;

    index = (int) panel_get_value(freeze_item) ;

    if (index == 0)
	return(FALSE) ;
    else 
	return(TRUE);
}


/*************************************************************************/
/*						                         */
/* nom	    : is_in_dir                                                  */
/*							                 */
/* fonction : recherche le fichier "file" dans le directory "dir "	 */
/*							                 */
/* entrees  : file, dir	 			                         */
/*							                 */
/* globales : ---							 */
/*							                 */
/* return   : FOUND si trouve NOT_FOUND sinon (resp. TRUE et FALSE)      */
/*							                 */
/* routines : ---						         */
/*							                 */
/*************************************************************************/

static int
is_in_dir(file,dir)
char *file,*dir ;
{
    DIR		    *dirname ;
    int		    len ;
    struct direct   *dp;


    len = strlen(file);
    dirname = opendir(dir);
    for ( dp = readdir(dirname); dp!= NULL; dp = readdir(dirname))
	if (dp->d_namlen == len && !strcmp(dp->d_name, file))
	{
	    closedir(dirname);
	    return TRUE;
	}
    closedir(dirname);
    return (FALSE);
}


/****************************************************************/
/*								*/
/* nom	    : get_filename					*/   
/*								*/								
/* fonction : trouve le nom complet du fichier a utiliser	*/ 
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : dir_item, file_item				*/								
/*							    	*/								
/* return   : la chaine de char contenant le nom complet	*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static char 
*get_filename() 
{
    char *dir,*file,*filename ;

    filename = (char *) calloc (100, sizeof *dir);

    dir = (char *) panel_get_value(dir_item) ;
    file = (char *) panel_get_value(file_item) ;
    strcpy(filename,dir) ;
    strcat(filename,"/") ;
    strcat(filename,file) ;
    return(filename) ;
}


/*******************************************************************

	    procedures liees aux panneaux et evenements

*******************************************************************/


/****************************************************************/
/*								*/
/* nom	    : change_one_field					*/   
/*								*/								
/* fonction : change la table de couleur pointee par la fleche  */ 
/*	      selon la valeur des sliders			*/
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : one_number_item,	one_red_H_slider,		*/
/*	      one_green_L_slider, one_blue_S_slider, mode	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : fix_colourRGB, fix_colourHLS			*/								
/*	      freeze_on, write_message, RGB_on			*/								
/*								*/								
/****************************************************************/

static void
change_one_field()
{
    int	n,rh,gl,bs ;
    float h,l,s ;

    n = atoi((char *) panel_get(one_number_item, PANEL_VALUE)) ;

    if ((n >= 0) && (n <= 255))
    { 	
	rh = (int) panel_get(one_red_H_slider, PANEL_VALUE) ;
	gl = (int) panel_get(one_green_L_slider, PANEL_VALUE) ;
	bs = (int) panel_get(one_blue_S_slider, PANEL_VALUE) ;

 	if (RGB_on(mode))
	{

	    if (freeze_on())
	    {
		one_cred = rh ;
		one_cgreen = gl ;
		one_cblue = bs ;
	    }

	    fix_colourRGB(n, rh, gl, bs) ;	    
	}
	else
	{
	    h = (float) rh *1.0 ;
	    l = (float) gl/100 ;
	    s = (float) bs/100 ;	
	    fix_colourHLS(n,h,l,s) ;

	    if (freeze_on())
	    {
                pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;
		one_cred = red[n] ;
		one_cblue = blue[n] ;
		one_cgreen = green[n] ;
	    }	    
	}

   }
    else
	write_message("Field Number out of range ...") ;
}


/****************************************************************/
/*								*/
/* nom	    : change_multiple_field				*/   
/*								*/								
/* fonction : change la table de couleur entre les fleches	*/ 
/*	      selon la valeur des sliders			*/
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : mult_number_item,	mult_red_H_slider,		*/
/*	      mult_green_L_slider, mult_blue_S_slider, mode	*/								
/*	      red[], green[], blue[],				*/								
/*	      cred[], cgreen[], cblue[],			*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : RGB_fix_mult_map, HLS_fix_mult_map		*/								
/*	      freeze_on, write_message,RGB_on			*/								
/*								*/								
/****************************************************************/

static void
change_multiple_field()
{
    int	    n0,n1,rh0,rh1,gl0,gl1,bs0,bs1;

    float   h0,h1,l0,l1,s0,s1 ;

    n0 = atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE)) ; 
    n1 = atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE)) ; 

    if ((n0 >= 0) && (n0 <= 255) && (n1 >= 0) && (n1 <= 255))
    {
	rh0 = (int) panel_get(mult_red_H_slider[0], PANEL_VALUE) ; 
    	gl0 = (int) panel_get(mult_green_L_slider[0], PANEL_VALUE) ; 
	bs0 = (int) panel_get(mult_blue_S_slider[0], PANEL_VALUE) ; 

	rh1 = (int) panel_get(mult_red_H_slider[1], PANEL_VALUE) ; 
	gl1 = (int) panel_get(mult_green_L_slider[1], PANEL_VALUE) ; 
	bs1 = (int) panel_get(mult_blue_S_slider[1], PANEL_VALUE) ; 

	if (RGB_on(mode))
	{
	    if (n0 != n1)
	    RGB_fix_mult_map(n0,rh0,gl0,bs0,n1,rh1,gl1,bs1) ;

	    if (freeze_on())
	    {
		cred[0] = rh0 ;
		cred[1] = rh1 ;
		cgreen[0] = gl0 ;
		cgreen[1] = gl1 ;
		cblue[0] = bs0 ;
		cblue[1] = bs1 ;
	    }
	}
	else
	{
	    h0 = (float) rh0 ; 
	    l0 = (float) gl0/100 ; 
	    s0 = (float) bs0/100 ; 

	    h1 = (float) rh1 ; 
	    l1 = (float) gl1/100 ; 
	    s1 = (float) bs1/100 ; 

	    if (n0 != n1)
	    HLS_fix_mult_map(n0,h0,l0,s0,n1,h1,l1,s1) ;

	    if (freeze_on())
	    {
		pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;

		cred[0] = red[n0] ;
		cred[1] = red[n1] ;
		cgreen[0] = green[n0] ;
		cgreen[1] = green[n1] ;
		cblue[0] = blue[n0] ;
		cblue[1] = blue[n1] ;    
	    }	    
	}
    }
    else
	write_message("Field Number out of range ...") ;   
}


/****************************************************************/
/*								*/
/* nom	    : one_field_proc					*/   
/*								*/								
/* fonction : fait pointer la fleche sur le bon champ de la     */ 
/*	      table et met celle-ci a jour			*/								
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : one_number_item, one_red_H_slider,		*/
/*	      one_green_L_slider, one_blue_S_slider,		*/								
/*	      red[], green[], blue[],				*/								
/*	      mappw, mode					*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : RGB_on, one_arrow, write_message, RGB_to_HLS	*/								
/*								*/								
/****************************************************************/

static void 
one_field_proc()
{
    int	    num, h, l, s;

    one_arrow() ;

    num = atoi((char *) panel_get(one_number_item, PANEL_VALUE));
    if ((num >= 0) && (num <= 255))
    {   
	pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;

	if (RGB_on(mode))	
	{
	    panel_set(one_red_H_slider, PANEL_VALUE, red[num], 0) ;
	    panel_set(one_green_L_slider, PANEL_VALUE, green[num], 0) ;
	    panel_set(one_blue_S_slider, PANEL_VALUE, blue[num], 0) ;
	}
	else
	{    
	    RGB_to_HLS(red[num], green[num], blue[num], &h, &l, &s) ;
	    panel_set(one_red_H_slider, PANEL_VALUE, h, 0) ;
	    panel_set(one_green_L_slider, PANEL_VALUE, l, 0) ;
	    panel_set(one_blue_S_slider, PANEL_VALUE, s, 0) ;
	}
    }
    else
	write_message("Field Number out of range ...") ;  
}


/****************************************************************/
/*								*/
/* nom	    : fields_proc					*/   
/*								*/								
/* fonction : fait pointer la fleche sur le bon champ de la     */ 
/*	      table et met celle-ci a jour			*/								
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : mult_number_item, mult_red_H_slider,		*/
/*	      mult_green_L_slider, mult_blue_S_slider,		*/								
/*	      red[], green[], blue[],				*/								
/*	      mappw, mode					*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : RGB_on, arrows, write_message, RGB_to_HLS		*/								
/*								*/								
/****************************************************************/

static void 
fields_proc()
{
    int	    numl,numr,h0,h1,l0,l1,s0,s1 ;

    arrows() ;

    numl = atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE));
    numr = atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE));

    if ((numl >= 0) && (numl <= 255) && (numr >= 0) && (numr <= 255))
    { 
	pw_getcolormap(mappw, 0, CMAP_SIZE, red, green, blue) ;

	if (RGB_on(mode))
	{
	    panel_set(mult_red_H_slider[0], PANEL_VALUE, red[numl], 0) ;
	    panel_set(mult_green_L_slider[0], PANEL_VALUE, green[numl], 0) ;
	    panel_set(mult_blue_S_slider[0], PANEL_VALUE, blue[numl], 0) ;

	    panel_set(mult_red_H_slider[1], PANEL_VALUE, red[numr], 0) ;
	    panel_set(mult_green_L_slider[1], PANEL_VALUE, green[numr], 0) ;
	    panel_set(mult_blue_S_slider[1], PANEL_VALUE, blue[numr], 0) ;
	}
	else
	{
	    RGB_to_HLS(red[numl], green[numl], blue[numl],&h0,&l0,&s0) ;
	    RGB_to_HLS(red[numr], green[numr], blue[numr],&h1,&l1,&s1) ;
	    
	    panel_set(mult_red_H_slider[0], PANEL_VALUE, h0, 0) ;
	    panel_set(mult_green_L_slider[0], PANEL_VALUE, l0, 0) ;
	    panel_set(mult_blue_S_slider[0], PANEL_VALUE, s0, 0) ;

	    panel_set(mult_red_H_slider[1], PANEL_VALUE, h1, 0) ;
	    panel_set(mult_green_L_slider[1], PANEL_VALUE, l1, 0) ;
	    panel_set(mult_blue_S_slider[1], PANEL_VALUE, s1, 0) ;
	}
    }
    else
	write_message("Field Number out of range ...") ;
 
}


/****************************************************************/
/*								*/
/* nom	    : of_proc()						*/   
/*								*/								
/* fonction : appelle les procedures de mise a jour d'un champ  */ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : one_number_item, one_cred, one_cgreen, one_cblue 	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : one_field_proc, one_arrow, write_message,		*/								
/*	      change_one_field, fix_colourRGB, freeze_on	*/								
/*								*/								
/****************************************************************/

static void 
of_proc() 
{

    int	n;

    n = atoi((char *) panel_get(one_number_item, PANEL_VALUE)) ; 

    if ((n >= 0) && (n <= 255))
    {		    
	one_arrow() ;

	if (freeze_on())
	{
	    fix_colourRGB(n,one_cred, one_cgreen, one_cblue) ;
	    return ;
	}
	else
	{
	    one_field_proc() ;
	    change_one_field() ;
	}
    }   
    else
	write_message("Field Number out of range ...") ;
}


/****************************************************************/
/*								*/
/* nom	    : fs_proc						*/   
/*								*/								
/* fonction : appelle les procedures de mise a jour des champs  */ 
/*								*/								
/* entree   : ---						*/								
/*								*/								
/* globales : mult_number_item[], cred[], cgreen[], cblue[] 	*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : fields_proc, arrows, write_message, freeze_on	*/								
/*	      change_multiple_field, RGB_fix_mult_map		*/								
/*								*/								
/****************************************************************/

static void 
fs_proc() 
{
    int	    n0,n1,
	    temp ;
    char    *buf ;

    buf = (char *) calloc (3, sizeof *buf) ;

    n0 = atoi((char *) panel_get(mult_number_item[0], PANEL_VALUE));
    n1 = atoi((char *) panel_get(mult_number_item[1], PANEL_VALUE));

    if ((n0 >= 0) && (n0 <= 255) && (n1 >= 0) && (n1 <= 255))
    {
	arrows() ;
	if (n1 < n0)
	{
	    temp = n0 ;
	    n0 = n1 ;
	    n1 = temp ;
	    sprintf(buf,"%d",n0) ;
	    panel_set(mult_number_item[0], PANEL_VALUE, buf, 0) ;
	    sprintf(buf,"%d",n1) ;
	    panel_set(mult_number_item[1], PANEL_VALUE, buf, 0) ;
	}

	if (freeze_on())
	{
	    if (n0 != n1)
		RGB_fix_mult_map(n0,cred[0],cgreen[0],cblue[0],n1,cred[1],cgreen[1],cblue[1]) ;
	    return ;
	}    
	else
	{
	    fields_proc() ;
	    change_multiple_field() ;
	}
    }
    else
	write_message("Field Number out of range ...") ;
}


/*******************************************************************

	    procedures de gestion de couleurs RGB

*******************************************************************/


/****************************************************************/
/*								*/
/* nom	    : RGB_fix_mult_map					*/   
/*								*/								
/* fonction : change la table de couleur entre n1 et n2	en      */ 
/*	      interpolant lineairement ou logarithmiquement	*/
/*	      entre (r1,g1,b1) et (r2,g2,b2)			*/
/*								*/
/* entree   : int n1,n2,r1,g1,b1,r2,g2,b2			*/								
/*								*/								
/* globales : ---						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : lin_on, fix_colourRGB				*/								
/*								*/								
/****************************************************************/

static void
RGB_fix_mult_map(n1,r1,g1,b1,n2,r2,g2,b2)
int n1,n2,r1,r2,g1,g2,b1,b2 ;
{
    int	    i,u = 0, 
	    r,g,b,
            dr = r2 -r1 ,
	    dg = g2 -g1 ,
	    db = b2 -b1 ,
	    number_int = n2 - n1 ;

    float rvar, gvar, bvar, coeff ;



    if (lin_on())
    {
	rvar = (float) dr/number_int;
	gvar = (float) dg/number_int; 
	bvar = (float) db/number_int; 

    	for(i=n1 ; i<=n2 ; i++)
	{	      
	    r = r1 + (int) rint ((double) (u * rvar)) ;
	    g = g1 + (int) rint ((double) (u * gvar)) ;
	    b = b1 + (int) rint ((double) (u * bvar)) ;
	    fix_colourRGB(i,r,g,b) ;
	    u++ ;
	}
    }
    else
    {
    	for(i=n1 ; i<=n2 ; i++)
	{

	    coeff = (float) log ( 1 + ((float) u/number_int*(E - 1))) ;
	      
	    r = r1 + (int) rint ((double) ((float) dr*coeff)) ;
	    g = g1 + (int) rint ((double) ((float) dg*coeff)) ;
	    b = b1 + (int) rint ((double) ((float) db*coeff)) ;
	    fix_colourRGB(i,r,g,b) ;
	    u++ ;
	}
    }
}


/****************************************************************/
/*								*/								
/* nom	    : fix_colourRGB					*/								
/*							    	*/								
/* fonction : permet de gerer la table de couleur. "number"	*/ 
/*            designe le numero de la couleur dans la table.	*/
/*	      R,G,B sont les trois intensites de couleur qui	*/								
/*	      sont des entiers ( 0 - 255 )			*/								
/*								*/								
/* entree   : int number, R, G, B			    	*/								
/*								*/								
/* globales : ppw, spw, mappw, arrowpw  red[], green[], blue[]	*/								
/*								*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/
/*								*/								
/****************************************************************/

static void
fix_colourRGB(number,R,G,B) 
int number, R, G, B ;
{
    red[0] = (unsigned char) R ;
    green[0] = (unsigned char) G ;
    blue[0] = (unsigned char) B ;


    if (number > 1 && number < 254)
    {
	pw_putcolormap(ppw, number, 1 , red, green, blue);
	pw_putcolormap(arrowpw, number, 1 , red, green, blue);
	pw_putcolormap(framepw, number, 1 , red, green, blue);
	
    }
    pw_putcolormap(mappw, number, 1 , red, green, blue);
}


/*******************************************************************

		procedures de gestion de couleurs HLS

********************************************************************/


/****************************************************************/
/*								*/
/* nom	    : HLS_fix_mult_map					*/   
/*								*/								
/* fonction : change la table de couleur entre n1 et n2	en      */ 
/*	      interpolant lineairement ou logarithmiquement	*/
/*	      entre (h1,l1,s1) et (h2,l2,s2)			*/
/*								*/
/* entree   : int n1,n2,h1,l1,s1,h2,l2,s2			*/								
/*								*/								
/* globales : ---						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : lin_on, fix_colourHLS				*/								
/*								*/								
/****************************************************************/

static void
HLS_fix_mult_map(n1,h1,l1,s1,n2,h2,l2,s2)
int n1,n2;
float h1,h2,l1,l2,s1,s2 ;
{
    int	    i,u = 0, 
	    number_int = n2 - n1;

    float   hvar, lvar, svar,
	    h,l,s, 
            dl = l2 -l1 ;

    hvar = (float) (h2 - h1)/number_int;
    svar = (float) (s2 - s1)/number_int;

    if (lin_on())
    {
	lvar = (float) dl/number_int; 
	for(i=n1 ; i<=n2 ; i++)
	{      
	    h = (float) h1 + u*hvar ;
	    l = (float) l1 + u*lvar ;
	    s = (float) s1 + u* svar ;
	    fix_colourHLS(i,h,l,s) ;
	    u++ ;
	}
    }
    else
    {
	for(i=n1 ; i<=n2 ; i++)
	{      
	    h = (float) h1 + u*hvar ;
	    l = (float) l1 + (dl*log(1 + ((float) u/number_int*(E - 1)))) ;
	    s = (float) s1 + u* svar ;		    
	    fix_colourHLS(i,h,l,s) ;
	    u++ ;
	}	
    }
}


/****************************************************************/
/*								*/
/* nom	    : fix_colourHLS					*/
/*							    	*/								
/* fonction : permet de gerer la table de couleur. "number"	*/ 
/*            designe le numero de la couleur dans la table et	*/
/*	      H, L, S sont les composantes (tous des reals)	*/								
/*	      H : 0.0 - 360.0					*/								
/*	      L : 0.0 - 1.0					*/								
/*	      S : 0.0 - 1.0					*/								
/*								*/								
/* entree   : int number : float H, L, S			*/								
/*								*/								
/* globales : ppw, , mappw, arrowpw, red[], green[], blue[]	*/								
/*								*/								
/* return   : ---						*/								
/*								*/								
/* routines : RGB						*/								
/*								*/								
/****************************************************************/

static void
fix_colourHLS(number,H,L,S) 
int number ;
float H,L,S ;
{
    register float  M1, M2; 
 
    float	    LS = L*S ;
                 
    if (L <= 0.5)
    {
	M2 = L + LS;
    }
    else
    {
	M2 = L + S - LS ;
    }

    M1 = 2*L - M2 ;
    red[0] = (unsigned char) rint ((double) RGBconv(H + 120,M1,M2) * 255);
    green[0] = (unsigned char) rint ((double) RGBconv(H, M1,M2) * 255);
    blue[0] = (unsigned char) rint ((double) RGBconv(H -120, M1, M2) * 255);

    if (number > 1 && number < 254)
    {
	pw_putcolormap(ppw, number, 1 , red, green, blue);
	pw_putcolormap(arrowpw, number, 1 , red, green, blue);
	pw_putcolormap(framepw, number, 1 , red, green, blue);
    }
    pw_putcolormap(mappw, number, 1 , red, green, blue);

}


/****************************************************************/
/*								*/
/* nom	    : RGB_to_HLS					*/   
/*								*/								
/* fonction : converti une couleur donnee par RGB en couleur co-*/ 
/*	      dee selon HLS				        */								
/* entree   : unsigned char r,g,b				*/								
/*	      int h,l,s   : attention passage par reference !!	*/								
/*								*/								
/* globales : ---						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : ---						*/								
/*								*/								
/****************************************************************/

static void
RGB_to_HLS(r,g,b,h,l,s) 
unsigned char	r,g,b ;
int		*h,*l,*s ;
{
    int	    m,m1,m2 ;
    float   M1,M2,
	    R,G,B,
	    H,L,S,
	    Cr,Cg,Cb ;

    m = max(r,g) ;
    m1 = max(m,b) ;
    M1 = (float) m1/255 ;

    m = min(r,g) ;
    m2 = min(m,b) ;
    M2 = (float) m2/255 ;

    L = (float) (M1 + M2)/2 ;

     /* determine the saturation */

    if (m1 == m2)    /* achromatic case */
    {
	S = 0 ;
	H = 0 ;
    }
    else	    /* chromatic case */
    {
	if (L <= 0.5)
	{
	    S = (float) (M1 - M2)/(M1 + M2);
	}
        else
	{
	    S = (float) (M1 - M2)/(2 - M1 - M2) ;
	}

        R = (float) r/255 ;
        G = (float) g/255 ;
        B = (float) b/255 ;

        Cr = (float)(M1 - R)/(M1 - M2) ;
        Cg = (float)(M1 - G)/(M1 - M2) ;
        Cb = (float)(M1 - B)/(M1 - M2) ;

	if (r == m1)
	    H = Cb - Cg ;
        else if (g == m1)
	    H = 2.0 + Cr - Cb ;
	else 
	    H = 4.0 + Cg -Cr ;

	H = (float) H*60.0 ;

	if (H < 0)
	    H = H + 360.0 ;
    }
    
    *h = (int) rint ((double) H * 1.0 ) ;
    *l = (int) rint ((double) L*100) ;
    *s = (int) rint ((double) S*100) ;
	
}


/*********************************************************************/
/*								     */
/* nom	    : RGBconv			  			     */
/*								     */
/* fonction : retouner une valeur dependant de H                     */
/*								     */
/* entree   : float H, M1, M2					     */
/*								     */
/* globales : ---					             */
/*								     */
/* return   : float Value				             */
/*								     */
/* routines : ---				          	     */
/*								     */
/*********************************************************************/

static float
RGBconv(H,M1,M2)

float H, M1, M2 ;

{
    float Value ;

    /* adjust correct range */

    if (H < 0) H = H + 360 ;
    if (H > 360) H = H - 360 ;

    /* determine the Value */

    if (H < 60) Value = M1 + (M2-M1)*H/60 ;
    if (H >= 60 && H < 180) Value = M2 ;
    if (H >= 180 && H < 240) Value = M1 + (M2 - M1)*(240 - H)/60 ;
    if (H >= 240 && H <= 360) Value = M1;

    return Value; 
}


/*******************************************************************

	    procedures de test de variables globales

*******************************************************************/


/****************************************************************/
/*								*/
/* nom	    : RGB_on						*/   
/*								*/								
/* fonction : test la variable d'environnemnt			*/ 
/*								*/								
/* entree   : int mode						*/								
/*								*/								
/* globales : ---						*/
/*							    	*/								
/* return   : TRUE si mode RGB, FALSE si mode HLS		*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static int
RGB_on(mode)
int mode ;
{
    if ((mode % 2) == 0 )
        return(FALSE) ;
    else 
	return(TRUE) ;
}


/****************************************************************/
/*								*/
/* nom	    : MULT_on						*/   
/*								*/								
/* fonction : test la variable d'environnemnt			*/ 
/*								*/								
/* entree   : int mode						*/								
/*								*/								
/* globales : ---						*/
/*							    	*/								
/* return   : TRUE si mode multiple, FALSE si mode unique	*/								
/*								*/								
/* routines : ---						*/
/*								*/
/****************************************************************/

static int
MULT_on(mode)
int mode ;
{
    if (mode <= 2)
        return(TRUE) ;
    else 
	return(FALSE) ;
}


/*******************************************************************

	   procedures du panneau d'information : 16.6.89

*******************************************************************/



/****************************************************************/
/*								*/
/* nom	    : create_info_frame					*/   
/*								*/								
/* fonction : creation du panneau d'informations		*/ 
/*								*/
/* entree   : ---						*/								
/*								*/								
/* globales : info_frame, info_panel, map_main_frame		*/				
/*	      text_item, mouse_button, files_button		*/				
/*	      quit_button, info_text				*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : text_proc, quit_info_proc, insert_text_proc	*/								
/*								*/								
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

static void
create_info_frame()
{
    info_frame = window_create(map_main_frame, FRAME, 
				    WIN_X, 650 ,
				    WIN_Y, 0 ,
				    0) ;

    info_panel = window_create(info_frame, PANEL,
				    0) ;
    

    text_item = panel_create_item(info_panel, PANEL_CYCLE, 
				    PANEL_LABEL_X, ATTR_COL(0),
				    PANEL_LABEL_Y, ATTR_ROW(0) ,
                                    PANEL_LABEL_BOLD, TRUE,
                                    PANEL_LABEL_STRING, "Text  : ",
				    PANEL_CHOICE_STRINGS, "English","Francais",0,
				    PANEL_NOTIFY_PROC, text_proc,
				    0);

    
    panel_create_item(info_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, ATTR_COL(25),
				    PANEL_LABEL_Y, ATTR_ROW(0) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"Info",6,0),
				    PANEL_CLIENT_DATA, 0 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0);

     panel_create_item(info_panel, PANEL_BUTTON,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"Freeze",6,0),
				    PANEL_CLIENT_DATA, 1 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0);

     panel_create_item(info_panel, PANEL_BUTTON,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"Interp",6,0),
				    PANEL_CLIENT_DATA, 2 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0);  

     files_button = panel_create_item(info_panel, PANEL_BUTTON,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"Files",8,0),
				    PANEL_CLIENT_DATA, 3 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0); 

     mouse_button = panel_create_item(info_panel, PANEL_BUTTON,
				    PANEL_LABEL_X, ATTR_COL(25),
				    PANEL_LABEL_Y, ATTR_ROW(1) ,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"Mouse",6,0),
				    PANEL_CLIENT_DATA, 4 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0);


    panel_create_item(info_panel, PANEL_BUTTON,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"RGB - HLS",10,0),
				    PANEL_CLIENT_DATA, 5 ,
				    PANEL_NOTIFY_PROC, insert_text_proc,
				    0);

    
     quit_button = panel_create_item(info_panel, PANEL_BUTTON,
                                    PANEL_LABEL_BOLD, TRUE,
				    PANEL_LABEL_IMAGE,
                                    panel_button_image(info_panel,"QUIT",7,0),
				    PANEL_NOTIFY_PROC, quit_info_proc,
				    0);

    window_fit(info_panel) ;

    info_text = window_create(info_frame, TEXTSW,
				    WIN_BELOW, info_panel, 
				    WIN_X, 0,
				    WIN_HEIGHT, 400 ,
				    TEXTSW_READ_ONLY, TRUE, 
				    TEXTSW_SCROLLBAR, NULL,
				    TEXTSW_IGNORE_LIMIT, 
				    TEXTSW_INFINITY,
				    0) ;

    window_fit(info_frame) ;
    window_set(info_frame, WIN_SHOW, TRUE, 0) ;	    
}


/****************************************************************/
/*								*/
/* nom	    : insert_text_proc					*/   
/*								*/								
/* fonction : imprimme l'information dans la fenetre		*/ 
/*								*/
/* entree   : Panel_item item, Event *event			*/								
/*								*/								
/* globales : info_text						*/								
/*							    	*/								
/* return   : ---						*/								
/*								*/								
/* routines : clear_text() 					*/								
/*								*/								
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

static void
insert_text_proc(item,event)
Panel_item item ;
Event *event ;
{
    int index = (int) panel_get(item, PANEL_CLIENT_DATA) ;
    char *buf;

    buf = (char *) malloc (1500) ;

    strcpy(buf,"\0") ;
    clear_text() ;

    switch(index) 
    {
	case 0  : 
	switch(info_lang)
	{  
	    case FRENCH  :

	    strcat(buf,"\n                Informations Generales\n") ;
	    strcat(buf,"                ----------------------\n\n") ;
	    strcat(buf," MapEdit est un logiciel de gestion de tables de couleurs.\n");
	    strcat(buf," On l'utilise pour creer ou modifier une table de look-up.\n\n");
	    strcat(buf," Deux methodes de coloration sont a disposition, RGB et HLS\n");
	    strcat(buf," En outre on peut modifier un champ (*_one) ou plusieurs \n");
	    strcat(buf," (*_Multi), * indique la methode de coloration.\n\n");
	    strcat(buf," Cette fenetre donne des informations en anglais ou francais\n");
	    strcat(buf," modifier  \"Text  :\"  au moyen de la souris.\n");
	    strcat(buf," Cliquer les boutons pour des informations supplementaires.\n\n\n");
	    strcat(buf," Copyright : Groupe IA et Vision\n");
	    strcat(buf,"             Centre Universitaire d'Informatique (C.U.I)\n");
	    strcat(buf,"             Universite de Geneve, Suisse\n");
	    break ;

	    case ENGLISH :

	    strcat(buf,"\n                General Informations\n") ;
	    strcat(buf,"                ---------------------\n\n") ;
	    strcat(buf," MapEdit is a look-up table editor. It should be used to\n");
	    strcat(buf," create new color maps or to modify existent ones.\n\n");
	    strcat(buf," Two methods are used to define new colors : RGB and HLS.\n");
	    strcat(buf," One can edit one (*_one) or more (*_Multi) adjacent fields \n");
	    strcat(buf," * is either RGB or HLS on the control panel.\n\n");
	    strcat(buf," In this window you can read information in english/french.\n");
	    strcat(buf," To change the language click \"Text\" (left mouse button).\n");
	    strcat(buf," To know more about MapEdit click the topic buttons.\n\n\n");
	    strcat(buf," Copyright : AI and Vision Group\n");
	    strcat(buf,"             Computing Science Center (C.U.I)\n");
	    strcat(buf,"             University of Geneva, Switzerland\n");
	    break ;
	}
	break ;

	case 1  : 
	switch(info_lang)
	{  
	    case FRENCH  :

	    strcat(buf,"\n                       Freeze\n") ;
	    strcat(buf,"                       ------\n\n") ;
	    strcat(buf," \"Freeze :\" sert a geler les couleurs sur les fleches,\n");
	    strcat(buf," les bornes gardents les memes couleurs.\n\n");
	    strcat(buf," OFF : quand on deplace les indicateurs de bornes a l'aide\n");
	    strcat(buf," de la souris, les trois sliders indiquent les trois compo-\n");
	    strcat(buf," santes de la couleur actuelle pointee par la fleche.\n\n");
	    strcat(buf," ON : les sliders ne changent pas. Les deux bornes gardent\n");
	    strcat(buf," leur couleur (idem dans le cas d'un champ unique).\n\n");
	    strcat(buf," Utilisation :\n\n");
	    strcat(buf," En mode (OFF), fixer un petit intervale et colorer ses\n");
	    strcat(buf," deux bornes. Puis se mettre en mode (ON). Deplacer\n");
	    strcat(buf," les fleches pour agrandir la zone et cliquer <milieu>.\n");
	    strcat(buf," Les couleurs identiques a celles du petit intervale, seront\n");
	    strcat(buf," interpolees sur le plus grand champ. Cette methode permet\n");
	    strcat(buf," de faire du seuillage interactif multicolore.\n\n");
	    strcat(buf," Voir info Souris\n");
	    break ;

	    case ENGLISH :


	    strcat(buf,"\n                       Freeze\n") ;
	    strcat(buf,"                       ------\n\n") ;
	    strcat(buf," \"Freeze\" is used to fix a colour on the arrows, keeping\n");
	    strcat(buf," the same colours on the field limits.\n\n");
	    strcat(buf," OFF : when moving the arrows on the colour map, the sliders\n");
	    strcat(buf," will show the colour components of the pointed field.\n\n");
	    strcat(buf," ON : the sliders do no change. The colours are frozen.\n\n");
	    strcat(buf," Example :\n\n");
	    strcat(buf," In (OFF) mode select a small region and color it's\n");
	    strcat(buf," limits. Then click (ON) and move the arrows, enlarging the\n");
	    strcat(buf," region. Now click <middle> on mouse, the previous region\n");
	    strcat(buf," colours will expand to fill in the new region. This method\n");
	    strcat(buf," is very practical to perform fast multicolour thresholding.\n\n");
	    strcat(buf," See Mouse information.\n\n");
	    
	    break ;
	}
	break ;
		
	case 2  :   
    
	switch(info_lang)
	{
	    case FRENCH  :

	    strcat(buf,"\n                    INTERPOLATION\n") ;
	    strcat(buf,"                    -------------\n\n") ;
	    strcat(buf," Les couleurs d'un intervale sont interpolees entre les deux\n");
	    strcat(buf," bornes indiquees par les fleches.\n\n");
	    strcat(buf," L'objet \"Interpolation :\" indique LINeaire ou LOGarithmique.\n\n");
	    strcat(buf," RGB : les trois composantes sont interpolees selon le mode.\n");
	    strcat(buf," HLS : la Luminance est interpolee selon le mode, les deux\n");
	    strcat(buf," autres sont toujours interpolees lineairement.\n\n");
	    strcat(buf," A chaque changement de potentiometre (R,G,B,H,L, ou S),\n");
	    strcat(buf," la table est mise a jour entre les deux bornes. Ces\n");
	    strcat(buf," bornes \"Field number :\" sont les positions dans la table\n");
	    strcat(buf," de couleur, les deux bornes sont indiquees par une fleche.\n\n");
	    strcat(buf," Voir aide Freeze et souris.\n\n");   

	    break ;	

	    case ENGLISH :

	    strcat(buf,"\n                    INTERPOLATION\n") ;
	    strcat(buf,"                    -------------\n\n") ;
	    strcat(buf," The colours of a field are interpolated between the\n");
	    strcat(buf," the two field limits.\n\n");
	    strcat(buf," The \"Interpolation :\" item indicates LINear or\n");
	    strcat(buf," LOGarithmic interpolation.\n");
	    strcat(buf," RGB : all three componants are interpolated LIN or LOG.\n");
	    strcat(buf," HLS : Luminance only can be interpolated LOG.\n\n");
	    strcat(buf," When a slider (R,G,B,H,L, or S) is changed, the color table\n");
	    strcat(buf," field is instantly updated according to LIN or LOG mode.\n\n");
	    strcat(buf," The field limits are given by \"Field number :\", they can be\n");
	    strcat(buf," changed (keybord), the two arrows show both region limits\n") ;
	    strcat(buf," on the colour table.\n\n");
	    strcat(buf," More details in Freeze and Mouse information.\n\n"); 
	    break ;
	}   
	break ;
		
	case 3  :   
    
	switch(info_lang)
	{
	    case FRENCH  :

	    strcat(buf,"\n                    LES FICHIERS\n") ;
	    strcat(buf,"                    ------------\n\n") ;
	    strcat(buf," Mapedit possede les outils pour la gestion des tables sur\n");
	    strcat(buf," disque : lecture et ecriture.\n\n");
	    strcat(buf," Le fichier et directoire de travail sont indiques dans le\n");
	    strcat(buf," panneau de controle ( \"Direcory :  \"et \"File     : \").\n\n");
	    strcat(buf," Lecture : \"LOAD\" affiche la table contenue dans le fichier.\n");
	    strcat(buf," Ecriture : \"STORE\" sauve la table dans le fichier courrant.\n\n");
	    strcat(buf," Un message (+ bip !) annonce la fin d'une operation\n");
	    strcat(buf," d'entree/sortie, ainsi que d'une erreur eventuelle.\n\n");
	    strcat(buf," Fichier de 3 x 256 bytes, les valeurs RGB sont stockees\n");
	    strcat(buf," sans en-tete, selon : \n\n");
	    strcat(buf," R[0], .., R[255], G[0], ..,G[255],B[0], ..,B[255].\n\n");
	    strcat(buf," Ces formats sont utilises de maniere implicite, par\n");
	    strcat(buf," le logiciel MapEdit.\n\n");   

	    break ;	

	    case ENGLISH :

	    strcat(buf,"\n                    FILE SYSTEM\n") ;
	    strcat(buf,"                    -----------\n\n") ;
	    strcat(buf," MapEdit can create new look-up tables, edit old ones, read\n");
	    strcat(buf," or write them in files.\n\n");
	    strcat(buf," The current working file and directory are indicated in \n");
	    strcat(buf," the control panel ( \"Direcory :  \" and \"File     : \").\n\n");
	    strcat(buf," Loading : \"LOAD\" --> file colors are displayed on map.\n");
	    strcat(buf," Ecriture : \"STORE\" --> map colors are stored in file.\n\n");
	    strcat(buf," A message (+ beep !) confirms the end of a read/write\n");
	    strcat(buf," operation, or informs of an eventual error.\n\n");
	    strcat(buf," File size : 3 x 256 bytes (RGB components), no header:\n");
	    strcat(buf," R[0], .., R[255], G[0], ..,G[255],B[0], ..,B[255].\n\n");
	    strcat(buf," This format is used implicitly by Mapedit when clicking\n");
	    strcat(buf," \"LOAD\" or \"STORE\" buttons.\n\n"); 
	    break ;
	}   
	break ;

	case 4  : 
	switch(info_lang)
	{  
	    case FRENCH  :

	    strcat(buf,"\n             Les fonctions de la souris\n") ;
	    strcat(buf,"             --------------------------\n\n") ;
	    strcat(buf," La souris permet d'acceder a toutes les fonctions de\n");
	    strcat(buf," MapEdit sans utiliser le clavier.\n\n");
	    strcat(buf," Le bouton <gauche> permet de changer les objets du panneau\n");
	    strcat(buf," de commande ainsi que les potentiometres.\n\n");
	    strcat(buf," Les fonctions interessantes de la souris sont utilisees\n");
	    strcat(buf," dans la fenetre ou se trouvent les fleches. Le curseur\n");
	    strcat(buf," est alors symbolise par une fleche pointant vers le bas.\n\n");
	    strcat(buf," <gauche> deplace la fleche de la borne gauche, modifiant\n");
	    strcat(buf," les attributs de celle-ci.\n");
	    strcat(buf," <droit> idem que gauche pour la burne droite.\n");
	    strcat(buf," !! Attention, en mode \"Freeze : ON\" pas de modifications !!\n");
	    strcat(buf," <milieu> lance l'interpolation de la zone pointee.\n\n");
	    strcat(buf," Pour des raisons d'affichage et de clarte, la totalite des \n");
	    strcat(buf," 256 couleurs n'est visible que si le curseur se trouve\n");
	    strcat(buf," sur la table de couleurs.\n\n");
	    break ;

	    case ENGLISH :


	    strcat(buf,"\n                Mouse functions\n") ;
	    strcat(buf,"                --------------\n\n") ;
	    strcat(buf," All fucntions of MapEdit can be called , just by using\n");
	    strcat(buf," the three mouse buttons.\n\n");
	    strcat(buf," The <left> button is used to change the main panel items\n");
	    strcat(buf," and the field colour component sliders.\n\n");
	    strcat(buf," The most interesing mouse functions are used in the small\n");
	    strcat(buf," window containing the vertical arrows.\n\n");
	    strcat(buf," <left> will move the left limit arrow on the map, updating\n");
	    strcat(buf," it's features (number and colour attributs).\n");
	    strcat(buf," <right> same function as <left>, fbut or right limit.\n");
	    strcat(buf," <middle> calls the interpolation of the new interval.\n\n");
	    strcat(buf," For display reasons, all the 256 colours are only visible\n");
	    strcat(buf," when the cursor is moved onto the displayed colour map.\n\n");
	    
	    break ;
	}
	break ;

	case 5  : 
	switch(info_lang)
	{  
	    case FRENCH  :

	    strcat(buf,"\n                  RGB - HLS\n") ;
	    strcat(buf,"                  ---------\n\n") ;
	    strcat(buf," RGB (Red, Green, Blue)\n\n");
	    strcat(buf," Les composantes utilisees par le hardware des ecrans\n");
	    strcat(buf," sont directement utilisables. Les sliders sont gradues\n");
	    strcat(buf," de 0 a 255, leurs combinaison offre plus de 16\n");
	    strcat(buf," millions de tonalites differentes.\n\n");
	    strcat(buf," HLS (Hue, Luminance, Saturation)\n\n");
	    strcat(buf," Systeme plus proche de la representation humaine. Un \n");
	    strcat(buf," double cone vertical defini par trois parametres :\n\n");
	    strcat(buf," H teinte, les angles decoupent un disque horizontal\n");
	    strcat(buf," sur lequel on trouve rouge[0], vert[120], bleu[240], avec \n");
	    strcat(buf," toutes les couleurs intermediaires.\n");
	    strcat(buf," L luminance : 0.0 noir -- 1.0 blanc, hauteur dans le cone\n");
	    strcat(buf," S saturation : 1.0 tres pure -- 0.0 gris, achromatique;\n");
	    strcat(buf," la distance par rapport au centre du cone.\n\n");
	    strcat(buf," Total : 3'600'000 couleurs definissables aux bornes.\n");
	    break ;

	    case ENGLISH :


	    strcat(buf,"\n                RGB - HLS\n") ;
	    strcat(buf,"                ---------\n\n") ;
	    strcat(buf," MapEdit uses two color systems to define new color maps.\n\n");
	    strcat(buf," RGB (Red, Green, Blue)\n\n");
	    strcat(buf," The system used by the screen and the device hardware\n");
	    strcat(buf," is directly used by the field sliders. They are graduated\n");
	    strcat(buf," 0 to 255, their combinations offer more than 16 millon\n");
	    strcat(buf," different colours.\n\n");
	    strcat(buf," HLS (Hue, Luminance, Saturation)\n\n");
	    strcat(buf," A vertical double cone colour space, defined by :\n\n") ;
	    strcat(buf," H angle in the cone : red[0], green[120], blue[240]\n");
	    strcat(buf," with all the intermediate colours in between.\n");
	    strcat(buf," L height in the cone : 0.0 black to 1.0 white.\n");
	    strcat(buf," S distance from center of cone : 1.0 saturated colours\n");
	    strcat(buf," to 0.0 grey, achromatic colours.\n");
	    strcat(buf," 3'600'000 can be displayed at field limits.\n\n");
	    
	    break ;
	}

	break ;
    }

    textsw_insert(info_text, buf, strlen(buf)) ;
    free(buf) ;

}

/****************************************************************/
/*								*/
/* nom	    : quit_info_proc					*/   
/*								*/
/* fonction : detruit la fenetre d'info.			*/ 
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : info_frame					*/								
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/


static void
quit_info_proc()
{
    window_set(info_frame, FRAME_NO_CONFIRM, TRUE, 0) ;
    window_destroy(info_frame) ;
}


/****************************************************************/
/*								*/
/* nom	    : ret_proc						*/   
/*								*/
/* fonction : detruit la fenetre et retient la valeur du	*/ 
/*	      PANEL_CLIENT_DATA					*/
/*								*/
/* entree   :Panel_item item, Event *event			*/								
/*								*/
/* globales : ---						*/								
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

static void
ret_proc(item,event)
Panel_item item ;
Event *event ;
{
    window_return(panel_get(item, PANEL_CLIENT_DATA));
}


/****************************************************************/
/*								*/
/* nom	    : text_proc						*/   
/*								*/
/* fonction : change la langue du texte d'information		*/ 
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : info_lang, info_panel, text_item			*/
/*	      mouse_button, files_button, quit_button		*/			
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

static void
text_proc()
{
    info_lang = (int) panel_get_value(text_item) ;
		 
    switch(info_lang)
    {
	case 0 : panel_set(mouse_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"Mouse",6,0),
			   0);

		 panel_set(files_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"Files",8,0),
			   0);

		 panel_set(quit_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"QUIT",7,0),
			   0);

		 panel_set(text_item,PANEL_LABEL_STRING,
                           "Text  : ",
			   0);

		 return ;

	case 1 : panel_set(mouse_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"Souris",6,0),
			   0);

		 panel_set(files_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"Fichiers",8,0),
			   0);

		 panel_set(quit_button,PANEL_LABEL_IMAGE,
                           panel_button_image(info_panel,"QUITTER",7,0),
			   0);

		 panel_set(text_item,PANEL_LABEL_STRING,
                           "Texte : ",
			   0);

		 return ;
    }
}


/****************************************************************/
/*								*/
/* nom	    : clear_text					*/   
/*								*/
/* fonction : fait un clear_screen de la fentre d'information	*/ 
/*								*/
/* entree   : ---						*/								
/*								*/
/* globales : info_text						*/								
/*							    	*/								
/* return   : ---						*/
/*								*/								
/* routines : ---						*/
/*								*/
/* modif    : nouvelle procedure				*/								
/*								*/								
/* date	    : 16.6.89, Rene LUTZ				*/								
/*								*/								
/****************************************************************/

static void
clear_text() 
{
    textsw_scroll_lines(info_text, 25) ;
    textsw_reset(info_text, 0, 0) ;
}

