/*
  File: devXX/DEVICE.c
  Author: K.R. Sloan
  Last Modified: 10 February 1991
  Purpose: definitions for the generic graphics device used by SGP
           this version uses the "framebuffer" abstraction to
           provide a framebuffer (see devInit) and draw Pixels (see
           devRect).  
  See also: devLine.c, devTriangle.c, devCircle.c, devClip.c.
           Some of the functions promised in DEVICE.h are implemented there.
 */
#include <stdio.h>
#include "DEVICE.h"
#include "framebuffer.h"

/* X11 specific stuff */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Form.h>
#include <X11/StringDefs.h>
#include <X11/Box.h>
#include <X11/Command.h>
#include <X11/cursorfont.h>
extern XStandardColormap FB_cmap;
extern int FB_grayscale;
extern int FB_depth;
extern int FB_ncolors;
extern unsigned long FB_pixels[256];
extern XColor FB_colors[256];

extern devClipLine();
extern devClipRectangle();



/*  Device state  - made global for convenience ... */
int devLeft, devBottom, devRight, devTop;
int devClipL, devClipB, devClipR, devClipT;
int devCurrentColor;
FILE *devOutputStream;
double devAspectRatio = 1.0;

/*
   devInit initializes the display, loading color maps if necessary
   There are two modes: color and mono.  Here, we assume a color
   display, and the two modes are identical

   devInit may also need to fill in the "physical" size of the display.
 */
int devInit(color, OutputStream)
 int color;
 FILE *OutputStream;
 {
  devOutputStream = OutputStream;
  if (0 != fb_open(&devLeft, &devBottom, &devRight, &devTop, color))
   {
    fprintf(stderr, "Can't open frame buffer: %s.\n", fb_error_message);
    exit(1);
   }
  devSetClipWin(devLeft,devBottom,devRight,devTop);
  devCurrentColor = devColorCode(0.0, 0.0, 0.0); 
  return(0); /* success */ 
 }

/*
  devClose insures that all operations are complete and closes up
 */
int devClose()
 {
  (void)fb_flush();
  (void)fb_close();
  return(0); /* success */ 
 }

/*
  devSetClipWin establishes a clipping window
  restrictions such as (L<R) must be enforced here!
 */
int devSetClipWin(L, B, R, T)
 int L, B, R, T;
 {
  if (L<R) { devClipL=L; devClipR=R; } else { devClipL=R; devClipR=L;} 
  if (B<T) { devClipB=B; devClipT=T; } else { devClipB=T; devClipT=B;} 
  if (devLeft < devRight)
   {
    if (devClipL < devLeft)   devClipL = devLeft;
    if (devClipR > devRight)  devClipR = devRight;
   }
  else
   {
    if (devClipL < devRight)  devClipL = devRight;
    if (devClipR > devLeft)   devClipR = devLeft;
   }

  if (devBottom < devTop)
   {
    if (devClipB < devBottom) devClipB = devBottom;
    if (devClipT > devTop)    devClipT = devTop;
   }
  else 
   {
    if (devClipB < devTop)    devClipB = devTop;
    if (devClipT > devBottom) devClipT = devBottom;
   }

  return(0); /* success */
 }

/*
 devClearScreen fills the entire screen with a particular color
 */
int devClearScreen(color)
 int color;
 {
  int SavedColorCode;
  int SavedClipL, SavedClipB, SavedClipR, SavedClipT; 

  SavedColorCode = devCurrentColor;
  SavedClipL = devClipL;  SavedClipB = devClipB;
  SavedClipR = devClipR;  SavedClipT = devClipT;
  devSetClipWin(devLeft,devBottom,devRight,devTop);

  devCurrentColor = color;
  devRect(devLeft, devBottom, devRight, devTop);

  devCurrentColor = SavedColorCode;
  devSetClipWin(SavedClipL,SavedClipB,SavedClipR,SavedClipT);

  return(0); /* success */
 }

/*
   the following routines draw primitive shapes on the physical device
   All of them clip against the device's clipping window.
   All of them paint in the device's current color
 */

/*
  devLine draws a line - here, we use fb_drawLine.
 */

int devLine(x1, y1, x2, y2)
 int x1, y1, x2, y2;
 {
  if (devClipLine(&x1,&y1,&x2,&y2)) return(0); /* success */

  fb_drawLine(x1,y1,x2,y2,devCurrentColor);
  return(0);
 }

/*
 devRect fills a rectangle
 */
int devRect(L, B, R, T)
 int L, B, R, T;
 {
  int x,y;

  if (devClipRectangle(&L, &B, &R, &T)) return(0); /* success */

  fb_writeRectangle(L, B, R, T, devCurrentColor);
  return(0); /* success */
 }

/*
 devDisc fills a disc - see devCircle.c
 */

/*
 devCircle draws a circle - see devCircle.c
 */

/*
 devSolidTriangle fills a triangle - see devTriangle.c
 */

/*
 devShadeTriangle shades a triangle
   with color codes specified at each vertex - see devTriangle.c
 */

/*
 devSetColor establishes the current color
 */
int devSetColor(color)
 int color;
 {
  devCurrentColor = color;
  return(0); /* success */
 }

/*
   devColorCode returns a framebuffer value
 
   we depend on devInit to set up FB_cmap and FB_grayscale

   color codes are pixel values RELATIVE TO FB_cmap.base_pixel, so they
   fit in an int.

 */ 
int devColorCode(R, G, B)
 double R,G,B; 
 {
  int I;
  if (FB_grayscale)
   {
    I = ((int)((FB_ncolors*(R*77.0 + G*151.0 + B*28.0)) + 127.0)) >> 8;
    if (I >= FB_ncolors) I = FB_ncolors-1;
    return I;
   }
  else
   {
    int r_B, g_B, b_B, r,g,b;
    r_B = FB_cmap.red_max+1; 
    r = R * r_B; if (r >= r_B) r = r_B-1;
    g_B = FB_cmap.green_max+1; 
    g = G * g_B; if (g >= g_B) g = g_B-1;
    b_B = FB_cmap.blue_max+1; 
    b = B * b_B; if (b >= b_B) b = b_B-1;
    return(   r*FB_cmap.red_mult
            + g*FB_cmap.green_mult
            + b*FB_cmap.blue_mult);
   }
 }
/*
  devColorDecode generates r,g,b values in [0.0, 1.0] 
 */
int devColorDecode(code, R, G, B)
 int code;
 double *R, *G, *B;
 {
  if (FB_grayscale)
   { 
    *R = (double)code/FB_ncolors;
    *G = *R;
    *B = *R;
   }
  else
   {
    *R = (double)((int)FB_colors[code].red)/65535.0; 
    *G = (double)((int)FB_colors[code].green)/65535.0;
    *B = (double)((int)FB_colors[code].blue)/65535.0;
   }
  return (0);
 }

/*
   devPick(&x,&y,&button) produces a location and a flag
 */
int devPick(x,y,button)
 int *x,*y,*button;
 {
  return(fb_pick(x,y,button));
 }

/*       -30-         */

